/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            source.c
 *
 *  Sat Oct  2 22:40:46 2004
 *  Copyright  2004  Gergely POLONKAI
 *  polesz@techinfo.hu
 ***************************************************************************/
/***************************************************************************
 * Preferences panel
 ***************************************************************************/

#include <gtk/gtk.h>
#include <gnome.h>

#include "variables.h"
#include "functions.h"
#include "typedefs.h"

/* Define these, so it won't be a problem referring to them in callback functions as data */
static gint cb_scroll_on_output_val = 1;
static gint cb_transparent_background_val = 2;
static gint cb_change_to_message_val = 3;
static gint cb_debug_mode_val = 4;
static gint cb_autosave_on_exit_val = 5;
static global_config_data_t saved_config;
static GtkWidget *font_picker,
								 *scroll_on_output_cb,
								 *transparent_background_cb,
	               *change_to_message_cb,
	               *debug_mode_cb,
	               *autosave_on_exit_cb,
      					 *satur_bar,
								 *history_length_entry;

/*
 * cb_toggled()
 * Called whenever the user changes a checkbox value in the preferences window
 */
static void
prefs_cb_toggled(GtkWidget *widget, gpointer data)
{
	gboolean activated = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
	
	if (data == NULL)
		return;
	
	switch (*((gint *)data))
	{
		case 1: /* Scroll on output was toggled */
			config_data.scroll_on_output = activated;
			change_terminals_set_scroll_on_output(activated);
			break;
		case 2: /* Transparent background was toggled */
			config_data.transparent_background = activated;
			change_terminals_set_background_transparent(activated);
			break;
		case 3: /* Change to message mode was toggled */
			config_data.change_to_message = activated;
			break;
		case 4: /* Debug mode was toggled */
			config_data.debug_mode = activated;
			break;
		case 5: /* Save preferences on exit was toggled */
			config_data.autosave_on_exit = activated;
		  break;
	}
}

/*
 * prefs_font_changed()
 * Called when the font is changed.
 */
static void
prefs_font_changed(GnomeFontPicker *picker, gchar *font, gpointer data)
{
	change_terminals_set_font_name(font);
}

/*
 * satur_change()
 * Called when the value of the saturation bar changes
 */
static void
prefs_satur_change(GtkWidget *widget, gpointer data)
{
	double value = gtk_range_get_value(GTK_RANGE(widget)) / 100;
	
	config_data.background_saturation = value;
	change_terminals_set_background_saturation(value);
}

/*
 * prefs_response()
 * Called when a response signal arrives
 */
static void
prefs_response(GtkDialog *prefs, int id, gpointer data)
{
	tab_data_t *tab;
	
	if ((tab = get_active_tab()) == NULL)
		return;

	if (id == GTK_RESPONSE_OK)
	{
		gdouble satur = gtk_range_get_value(GTK_RANGE(satur_bar));
		
		g_free(config_data.vte_font);
		config_data.vte_font = g_strdup(gnome_font_picker_get_font_name(GNOME_FONT_PICKER(font_picker)));
		if (config_data.debug_mode)
			feed_info_to_terminal(tab, TRUE, "Configuration saved...");
		set_boolean_value("autosave_on_exit", gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(autosave_on_exit_cb)));
		set_float_value("background_saturation", satur / 100);
		set_boolean_value("change_to_message", gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(change_to_message_cb)));
		set_boolean_value("debug", gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(debug_mode_cb)));
		set_boolean_value("scroll_on_output", gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(scroll_on_output_cb)));
		set_boolean_value("transparent_background", gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(transparent_background_cb)));
		set_string_value("vte_font", gnome_font_picker_get_font_name(GNOME_FONT_PICKER(font_picker)));
		change_terminals_set_background_saturation(satur / 100);
	}
	if (id == GTK_RESPONSE_CANCEL)
	{
		change_terminals_set_background_saturation(saved_config.background_saturation);
		change_terminals_set_background_transparent(saved_config.transparent_background);
		change_terminals_set_font_name(saved_config.vte_font);
		change_terminals_set_scroll_on_output(saved_config.scroll_on_output);
		config_data = saved_config;
	}
	gtk_signal_emit_by_name(GTK_OBJECT(static_data.prefs_window), "destroy", NULL);
	static_data.prefs_window = NULL;
}

/*
 * open_prefs()
 * Opens the preferences window.
 */
void
open_prefs(void)
{
	GtkWidget *vbox,
	          *prefs_table,
	          *satur_label,
						*font_label,
	          *history_length_label;
	char *inttostr;
	
	saved_config = config_data;
	
	if (static_data.prefs_window != NULL)
	{
		gtk_window_present(GTK_WINDOW(static_data.prefs_window));
		return;
	}
	
	static_data.prefs_window = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(static_data.prefs_window), "BotCommander preferences");
	gtk_window_set_policy(GTK_WINDOW(static_data.prefs_window), FALSE, FALSE, FALSE);
	gtk_window_set_modal(GTK_WINDOW(static_data.prefs_window), TRUE);
	gtk_signal_connect(GTK_OBJECT(static_data.prefs_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &(static_data.prefs_window));
	gtk_signal_connect(GTK_OBJECT(static_data.prefs_window), "response", GTK_SIGNAL_FUNC(prefs_response), NULL);
	
	vbox = GTK_DIALOG(static_data.prefs_window)->vbox;
	
	prefs_table = gtk_table_new(9, 2, TRUE);
	gtk_box_pack_start(GTK_BOX(vbox), prefs_table, TRUE, TRUE, 0);
	
	scroll_on_output_cb = gtk_check_button_new_with_label("Scroll on output");
	gtk_table_attach_defaults(GTK_TABLE(prefs_table), scroll_on_output_cb, 0, 2, 0, 1);
	gtk_signal_connect(GTK_OBJECT(scroll_on_output_cb), "toggled", GTK_SIGNAL_FUNC(prefs_cb_toggled), &cb_scroll_on_output_val);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(scroll_on_output_cb), config_data.scroll_on_output);
	gtk_widget_show(scroll_on_output_cb);
	
	transparent_background_cb = gtk_check_button_new_with_label("Transparent background");
	gtk_table_attach_defaults(GTK_TABLE(prefs_table), transparent_background_cb, 0, 2, 1, 2);
	gtk_signal_connect(GTK_OBJECT(transparent_background_cb), "toggled", GTK_SIGNAL_FUNC(prefs_cb_toggled), &cb_transparent_background_val);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(transparent_background_cb), config_data.transparent_background);
	gtk_widget_show(transparent_background_cb);
	
	change_to_message_cb = gtk_check_button_new_with_label("Change to Message Mode after connect");
	gtk_table_attach_defaults(GTK_TABLE(prefs_table), change_to_message_cb, 0, 2, 2, 3);
	gtk_signal_connect(GTK_OBJECT(change_to_message_cb), "toggled", GTK_SIGNAL_FUNC(prefs_cb_toggled), &cb_change_to_message_val);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(change_to_message_cb), config_data.change_to_message);
	gtk_widget_show(change_to_message_cb);
	
	debug_mode_cb = gtk_check_button_new_with_label("Debug mode");
	gtk_table_attach_defaults(GTK_TABLE(prefs_table), debug_mode_cb, 0, 2, 3, 4);
	gtk_signal_connect(GTK_OBJECT(debug_mode_cb), "toggled", GTK_SIGNAL_FUNC(prefs_cb_toggled), &cb_debug_mode_val);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(debug_mode_cb), config_data.debug_mode);
	gtk_widget_show(debug_mode_cb);
	
	autosave_on_exit_cb = gtk_check_button_new_with_label("Save preferences on exit");
	gtk_table_attach_defaults(GTK_TABLE(prefs_table), autosave_on_exit_cb, 0, 2, 4, 5);
	gtk_signal_connect(GTK_OBJECT(autosave_on_exit_cb), "toggled", GTK_SIGNAL_FUNC(prefs_cb_toggled), &cb_autosave_on_exit_val);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(autosave_on_exit_cb), config_data.autosave_on_exit);
	gtk_widget_show(autosave_on_exit_cb);
	
	history_length_label = gtk_label_new("Command line history length");
	gtk_table_attach(GTK_TABLE(prefs_table), history_length_label, 0, 1, 5, 6, GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment(GTK_MISC(history_length_label), 0, 0.5);
	gtk_misc_set_padding(GTK_MISC(history_length_label), 0, 0);
	gtk_widget_show(history_length_label);
	
	inttostr = int_to_str(config_data.commandline_history_length);
	history_length_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(prefs_table), history_length_entry, 1, 2, 5, 6, GTK_FILL, 0, 0, 0);
	gtk_entry_set_text(GTK_ENTRY(history_length_entry), inttostr);
	g_free(inttostr);
	gtk_widget_show(history_length_entry);
	
	satur_label = gtk_label_new("Saturation");
	gtk_table_attach(GTK_TABLE(prefs_table), satur_label, 0, 1, 6, 7, GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment(GTK_MISC(satur_label), 0, 0.5);
	gtk_misc_set_padding(GTK_MISC(satur_label), 0, 0);
	gtk_widget_show(satur_label);
	
	satur_bar = gtk_hscale_new_with_range(0, 100, 1);
	gtk_scale_set_digits(GTK_SCALE(satur_bar), 0);
	gtk_scale_set_draw_value(GTK_SCALE(satur_bar), TRUE);
	gtk_scale_set_value_pos(GTK_SCALE(satur_bar), GTK_POS_RIGHT);
	gtk_range_set_value(GTK_RANGE(satur_bar), config_data.background_saturation * 100);
	gtk_signal_connect(GTK_OBJECT(satur_bar), "value-changed", GTK_SIGNAL_FUNC(prefs_satur_change), NULL);
	gtk_table_attach(GTK_TABLE(prefs_table), satur_bar, 1, 2, 6, 7, GTK_FILL, 0, 0, 0);
	gtk_widget_show(satur_bar);
	
	font_label = gtk_label_new("VTE Font");
	gtk_table_attach(GTK_TABLE(prefs_table), font_label, 0, 1, 7, 8, GTK_FILL, 0, 0, 0);
	gtk_misc_set_alignment(GTK_MISC(font_label), 0, 0.5);
	gtk_misc_set_padding(GTK_MISC(font_label), 0, 0);
	gtk_widget_show(font_label);
	
	font_picker = gnome_font_picker_new();
	gtk_table_attach(GTK_TABLE(prefs_table), font_picker, 1, 2, 7, 8, GTK_FILL, 0, 0, 0);
	gnome_font_picker_set_preview_text(GNOME_FONT_PICKER(font_picker), "Test text");
	gnome_font_picker_set_mode(GNOME_FONT_PICKER(font_picker), GNOME_FONT_PICKER_MODE_FONT_INFO);
	gnome_font_picker_set_font_name(GNOME_FONT_PICKER(font_picker), config_data.vte_font);
	gtk_signal_connect(GTK_OBJECT(font_picker), "font-set", GTK_SIGNAL_FUNC(prefs_font_changed), NULL);
	gtk_widget_show(font_picker);
	
	gtk_dialog_set_has_separator(GTK_DIALOG(static_data.prefs_window), FALSE);
	gtk_dialog_add_buttons(GTK_DIALOG(static_data.prefs_window), GTK_STOCK_OK, GTK_RESPONSE_OK, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, NULL);
	
	gtk_widget_show(prefs_table);
	gtk_widget_show(vbox);
	gtk_widget_show(static_data.prefs_window);
}
