/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            config.c
 *
 *  Sat Aug 21 23:59:26 2004
 *  Copyright  2004  Gergely Polonkai
 *  polesz@techinfo.hu
 ****************************************************************************/
/***************************************************************************
 * Configuration backends. Currently only GConf2 is supported.
 ****************************************************************************/

#include "botcommander.h"

#if (CONFIG_BACKEND == CB_GCONF2)
# include <gconf/gconf-client.h>
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
# include <stdio.h>
# include <stdlib.h>
# include <errno.h>
#endif
#include <gtk/gtk.h>
#include <string.h>

#include "botcommander.h"
#include "typedefs.h"
#include "functions.h"

#if (CONFIG_BACKEND == CB_GCONF2)
GConfClient *gconf_client;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
FILE *configfile;
#endif
OldTConfigData ConfigData;

#if (CONFIG_BACKEND == CB_GCONF2)
/*
 * GConfSanityCheckString()
 * Copied from gnome-mud source code
 * Checks if gconf is available and operational
 */
gboolean
GConfSanityCheckString(GConfClient *client, const gchar *key)
{
	gchar *string;
	GError *error = NULL;
	
	string = gconf_client_get_string(client, key, &error);
	
	if (error)
	{
		int err = 0;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "allow_bold", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_float(client, GCONF_CONFIG_ROOT "background_saturation", 1.0, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "change_to_message", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "debug", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "default_mode", "B", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "sanity", "Needed for sanity check, never delete!", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "scroll_on_output", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "tearoff_menus", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "transparent_background", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "vte_font", "Courier 10 Pitch 12", &error);
		if (error)
			err = 1;

		if (err == 1)
		{
			GtkWidget *dialog;
		
			dialog = gtk_message_dialog_new(NULL,
																			0,
																			GTK_MESSAGE_ERROR,
																			GTK_BUTTONS_OK,
																			"There was an error accessing GConf: %s",
																			error->message);
			gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
		
			return FALSE;
		}
	}
	
	if (!string)
	{
		int err = 0;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "allow_bold", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_float(client, GCONF_CONFIG_ROOT "background_saturation", 1.0, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "change_to_message", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "debug", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "default_mode", "B", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "sanity", "Needed for sanity check, never delete!", &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "scroll_on_output", TRUE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "tearoff_menus", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_bool(client, GCONF_CONFIG_ROOT "transparent_background", FALSE, &error);
		if (error)
			err = 1;

		error = NULL;
		gconf_client_set_string(client, GCONF_CONFIG_ROOT "vte_font", "Courier 10 Pitch 12", &error);
		if (error)
			err = 1;
		
		if (err == 1)
		{
			GtkWidget *dialog;
		
			dialog = gtk_message_dialog_new(NULL,
																			0,
																			GTK_MESSAGE_ERROR,
																			GTK_BUTTONS_OK,
																			"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
																			"The default configuration values could not be retrieved correctly.",
																			"Please check your GConf configuration, specifically the schemam have been installed correctly."
																			);
			gtk_label_set_use_markup(GTK_LABEL(GTK_MESSAGE_DIALOG(dialog)->label), TRUE);
			gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
			gtk_dialog_run(GTK_DIALOG(dialog));
			gtk_widget_destroy(dialog);
		
			return FALSE;
		}
	}
	
	g_free(string);
	
	return TRUE;
}
#endif

/*
 * GetStringValue()
 * Get a string value from the configuration backend.
 */
gboolean
GetStringValue(gchar *key, gchar **retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	char buf[1024];
	gboolean global_started = FALSE;
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;

	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if (error || !value) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_STRING) {
			if (retval)
				*retval = g_strdup(gconf_client_get_string(gconf_client, full_key, &error));
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif /* (CONFIG_BACKEND == CB_GCONF2) */
	
#if (CONFIG_BACKEND == CB_SAMBA)
	fseek(configfile, 0, SEEK_SET);
	memset(&buf, 0, 1024);

	while (fgets((char *)&buf, 1023, configfile))
	{
		char buf2[1024];
		memset(&buf2, 0, 1024);
		trim((gchar *)&buf, (gchar *)&buf2, NULL);
		strcpy((char *)&buf, (char *)&buf2);
		if (strchr(";#", buf[0]))
			continue;
		if (str_eq((gchar *)&buf, "[global]", FALSE))
		{
			global_started = TRUE;
			memset(&buf, 0, 1024);
			continue;
		}
		if ((buf[0] == '[') && (buf[strlen((char *)&buf) - 1] == ']'))
		{
			global_started = FALSE;
			memset(&buf, 0, 1024);
			continue;
		}
		if (global_started && buf[0])
		{
			char *eqsign;
			
			eqsign = strchr((char *)&buf, '=');
			if (eqsign)
			{
				int len = eqsign - (char *)&buf;
				char *foundkey, *tkey;
				
				if ((foundkey = g_malloc(len + 1)) == NULL)
				{
					memset(&buf, 0, 1024);
					continue;
				}
				if ((tkey = g_malloc(len + 1)) == NULL)
				{
					g_free(foundkey);
					memset(&buf, 0, 1024);
					continue;
				}
				
				memset(foundkey, 0, len + 1);
				strncpy(foundkey, (char *)&buf, len);
				trim(foundkey, tkey, NULL);
				g_free(foundkey);
				
				if (str_eq(tkey, key, FALSE))
				{
					char *tval;
					
					if ((tval = g_malloc(strlen(eqsign + 1))) == NULL)
					{
						g_free(tkey);
						memset(&buf, 0, 1024);
						continue;
					}
					memset(tval, 0, strlen(eqsign + 1));
					trim(eqsign + 1, tval, NULL);
					if (retval)
						*retval = tval;
					g_free(tkey);
					return TRUE;
				}
				
				g_free(tkey);
			}
		}
		memset(&buf, 0, 1024);
	}
#endif
	return FALSE;
}

/*
 * GetBooleanValue()
 * Get a boolean value from the configuration backend.
 */
gboolean
GetBooleanValue(gchar *key, gboolean *retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	char buf[1024];
	gboolean global_started = FALSE;
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_BOOL) {
			if (retval)
				*retval = gconf_client_get_bool(gconf_client, full_key, &error);
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	fseek(configfile, 0, SEEK_SET);
	memset(&buf, 0, 1024);

	while (fgets((char *)&buf, 1023, configfile))
	{
		char buf2[1024];
		memset(&buf2, 0, 1024);
		trim((gchar *)&buf, (gchar *)&buf2, NULL);
		strcpy((char *)&buf, (char *)&buf2);
		if (str_eq((gchar *)&buf, "[global]", FALSE))
		{
			global_started = TRUE;
			memset(&buf, 0, 1024);
			continue;
		}
		if ((buf[0] == '[') && (buf[strlen((char *)&buf) - 1] == ']'))
		{
			global_started = FALSE;
			memset(&buf, 0, 1024);
			continue;
		}
		if (global_started && buf[0])
		{
			char *eqsign;
			
			eqsign = strchr((char *)&buf, '=');
			if (eqsign)
			{
				int len = eqsign - (char *)&buf;
				char *foundkey, *tkey;
				
				if ((foundkey = g_malloc(len + 1)) == NULL)
				{
					memset(&buf, 0, 1024);
					continue;
				}
				if ((tkey = g_malloc(len + 1)) == NULL)
				{
					g_free(foundkey);
					memset(&buf, 0, 1024);
					continue;
				}
				
				memset(foundkey, 0, len + 1);
				strncpy(foundkey, (char *)&buf, len);
				trim(foundkey, tkey, NULL);
				g_free(foundkey);
				
				if (str_eq(tkey, key, FALSE))
				{
					char *tval;
					
					if ((tval = g_malloc(strlen(eqsign + 1))) == NULL)
					{
						g_free(tkey);
						memset(&buf, 0, 1024);
						continue;
					}
					memset(tval, 0, strlen(eqsign + 1));
					trim(eqsign + 1, tval, NULL);
					if (retval)
					{
						if (str_eq(tval, "true", FALSE) || str_eq(tval, "yes", FALSE) || str_eq(tval, "on", FALSE))
							*retval = TRUE;
						else
							*retval = FALSE;
					}
					g_free(tval);
					g_free(tkey);
					return TRUE;
				}
				
				g_free(tkey);
			}
		}
		memset(&buf, 0, 1024);
	}
#endif
	return FALSE;
}

/*
 * GetFloatValue()
 * Get a float value from the configuration backend.
 */
gboolean
GetFloatValue(gchar *key, gfloat *retval)
{
#if (CONFIG_BACKEND == CB_GCONF2)
	GError *error = NULL;
	GConfValue *value = NULL;
	gchar *full_key;
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	char buf[1024];
	gboolean global_started = FALSE;
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	if ((full_key = g_malloc(strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1)) == NULL)
		return FALSE;
	
	memset(full_key, 0, strlen(GCONF_CONFIG_ROOT) + strlen(key) + 1);
	
	strcpy(full_key, GCONF_CONFIG_ROOT);
	strcat(full_key, key);
	
	value = gconf_client_get(gconf_client, full_key, &error);
	
	if ((error != NULL) || (value == NULL)) {
		g_free(full_key);
		return FALSE;
	}

	if (value) {
		if (value->type == GCONF_VALUE_FLOAT) {
			if (retval)
				*retval = gconf_client_get_float(gconf_client, full_key, &error);
			g_free(full_key);
			return TRUE;
		}
	}
	g_free(full_key);
#endif
#if (CONFIG_BACKEND == CB_SAMBA)
	fseek(configfile, 0, SEEK_SET);
	memset(&buf, 0, 1024);

	while (fgets((char *)&buf, 1023, configfile))
	{
		char buf2[1024];
		memset(&buf2, 0, 1024);
		trim((gchar *)&buf, (gchar *)&buf2, NULL);
		strcpy((char *)&buf, (char *)&buf2);
		if (str_eq((gchar *)&buf, "[global]", FALSE))
		{
			global_started = TRUE;
			memset(&buf, 0, 1024);
			continue;
		}
		if ((buf[0] == '[') && (buf[strlen((char *)&buf) - 1] == ']'))
		{
			global_started = FALSE;
			memset(&buf, 0, 1024);
			continue;
		}
		if (global_started && buf[0])
		{
			char *eqsign;
			
			eqsign = strchr((char *)&buf, '=');
			if (eqsign)
			{
				int len = eqsign - (char *)&buf;
				char *foundkey, *tkey;
				
				if ((foundkey = g_malloc(len + 1)) == NULL)
				{
					memset(&buf, 0, 1024);
					continue;
				}
				if ((tkey = g_malloc(len + 1)) == NULL)
				{
					g_free(foundkey);
					memset(&buf, 0, 1024);
					continue;
				}
				
				memset(foundkey, 0, len + 1);
				strncpy(foundkey, (char *)&buf, len);
				trim(foundkey, tkey, NULL);
				g_free(foundkey);
				
				if (str_eq(tkey, key, FALSE))
				{
					char *tval, *ep;
					float rv;
					
					if ((tval = g_malloc(strlen(eqsign + 1))) == NULL)
					{
						g_free(tkey);
						memset(&buf, 0, 1024);
						continue;
					}
					memset(tval, 0, strlen(eqsign + 1));
					trim(eqsign + 1, tval, NULL);
					
					rv = strtod(tval, &ep);
					if ((rv == 0) && (ep == tval))
					{
						g_free(tval);
						g_free(tkey);
						return FALSE;
					}
					if (retval)
						*retval = rv;
					g_free(tval);
					g_free(tkey);
					return TRUE;
				}
				
				g_free(tkey);
			}
		}
		memset(&buf, 0, 1024);
	}
#endif
	return FALSE;
}

/*
 * ConfigInit()
 * Initiates a connection to the gconf engine.
 */
gboolean
ConfigInit(void)
{
	gboolean setting_b;
	gfloat setting_f;
	gchar *setting_s;
#if (CONFIG_BACKEND == CB_SAMBA)
	char *filename;
#endif

#if (CONFIG_BACKEND == CB_GCONF2)
	gconf_client = gconf_client_get_default();
	if (!GConfSanityCheckString(gconf_client, GCONF_CONFIG_ROOT "sanity"))
	{
		return FALSE;
	}
#endif
	
#if (CONFIG_BACKEND == CB_SAMBA)
	if ((filename=g_malloc(strlen(DEFAULT_CONFIG))) == NULL)
		return FALSE;

	strcpy(filename, DEFAULT_CONFIG);
	
	if ((filename[0] == '~') && (filename[1] == '/'))
	{
		gchar *home;
		
		home = getenv("HOME");
		g_free(filename);
		filename = g_malloc(strlen(home) + strlen(DEFAULT_CONFIG));
		memset(filename, 0, strlen(home) + strlen(DEFAULT_CONFIG));
		strcpy(filename, home);
		strcat(filename, "/");
		strcat(filename, (DEFAULT_CONFIG) + 1);
	}
	
	if ((configfile = fopen(filename, "r")) == NULL)
	{
		fprintf(stderr, "Unable to open configuration file '%s': %s. Exiting.\n", filename, strerror(errno));
		return FALSE;
	}
#endif
	
	/* Get the configuration data. */
	if (GetBooleanValue("scroll_on_output", &setting_b))
		ConfigData.scroll_on_output = setting_b;
	else
		ConfigData.scroll_on_output = DEFAULT_SCROLL_ON_OUTPUT;
	
	if (GetBooleanValue("transparent_background", &setting_b))
		ConfigData.transparent_background = setting_b;
	else
		ConfigData.transparent_background = DEFAULT_TRANSPARENT_BACKGROUND;
	
	if (GetFloatValue("background_saturation", &setting_f))
		ConfigData.background_saturation = setting_f;
	else
		ConfigData.background_saturation = DEFAULT_BACKGROUND_SATURATION;
	
	if (GetBooleanValue("debug", &setting_b))
		ConfigData.debug_mode = setting_b;
	else
		ConfigData.debug_mode = DEFAULT_DEBUG_MODE;
	
	if (GetStringValue("vte_font", &setting_s))
		ConfigData.vte_font = setting_s;
	else
		ConfigData.vte_font = g_strdup(DEFAULT_VTE_FONT);
	
	if (GetBooleanValue("change_to_message", &setting_b))
		ConfigData.change_to_message = setting_b;
	else
		ConfigData.change_to_message = DEFAULT_CHANGE_TO_MESSAGE;
	
	return TRUE;
}

/*
 * CreateVersionString()
 * Creates a version string. If need_feed is true, it will append a
 * linefeed to the end of the string, which is accepted by the vte.
 */
gchar *
CreateVersionString(gboolean need_feed)
{
	int len;
	gchar *ret;
	
	len = strlen(PACKAGE) + 1 + strlen(VERSION) + 1;
	if (need_feed)
		len += 2;
	
	if ((ret = g_malloc(len)) == NULL)
	{
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new(NULL,
		                                0,
		                                GTK_MESSAGE_ERROR,
		                                GTK_BUTTONS_OK,
		                                "Cannot allocate enough memory. This is not good. Exiting.");
		gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
		gtk_dialog_run(GTK_DIALOG(dialog));
		
		gtk_main_quit();
	}
	
	strcpy(ret, PACKAGE);
	strcat(ret, " ");
	strcat(ret, VERSION);
	if (need_feed)
		strcat(ret, "\r\n");
	
	return ret;
}
