(* Copyright (C) 2014-2015, Daniel Wyckoff, except for the portions so labeleled as (*Schepler*) which I got from Daniel Schepler.  
My functions will be labeled (*Wyckoff*)*)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)


Require Export EnsemblesImplicit.
Require Export ImageImplicit.
Require Import Classical.
Require Import Image.
Require Import LogicUtilities.
Require Import Setoid.
Require Import FunctionProperties.
Require Import Arith.
Require Import Equality.
Require Import Infinite_sets.
Require Import ArithUtilities. 
Require Import Description.
Require Import IndefiniteDescription.
Require Import DecidableDec.
Require Import Bool.
Require Import FunctionalExtensionality.


(*Wyckoff*)
Lemma incl_add : 
  forall {T:Type} (A:Ensemble T) (a:T),
    Included A (Add A a). auto with sets.
Qed.


(*Wyckoff*)
Lemma in_eq_set : 
  forall {T:Type} (A B:Ensemble T) (pf:A = B) (x:T),
    In A x <-> In B x.
intros; subst; tauto.
Qed.


(*Wyckoff*)
Lemma in_add_eq : forall {T:Type} (A:Ensemble T) (x:T),
                    In A x -> Add A x = A.
intros T A x h1.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros y h2.
destruct h2 as [y h2l | y h2r]. assumption.
destruct h2r; subst. assumption.
(* >= *)
red. intros y h2.
left. assumption.
Qed.

(*Wyckoff*)
Lemma add_not_empty : forall {T:Type} (A:Ensemble T) (x:T),
                        Add A x <> Empty_set _.
intros T A x. intro h1.
pose proof (Add_intro2 _ A x) as h2.
rewrite h1 in h2.
contradiction.
Qed.

(*Wyckoff*)
Lemma add_empty_sing : forall {T:Type} (x:T), 
                         Add (Empty_set _) x =
                         Singleton x.
intros T x.
apply Extensionality_Ensembles.
red. split.
red.
intros y h1.
destruct h1; try contradiction; auto.
red.
intros y h2. right; auto.
Qed.


(*Wyckoff*)
Lemma card_sing : forall {T:Type} (x:T), cardinal _ (Singleton x) 1.
intros T x.
pose proof (add_empty_sing x) as h1.
rewrite <- h1.
constructor.
constructor.
auto with sets.
Qed.


(*Wyckoff*)
Lemma sub_add_comm : 
  forall {T:Type} (A:Ensemble T) 
         (x y:T),  (x <> y) ->
    (Subtract (Add A x) y) = (Add (Subtract A y) x).
intros T A x y h0.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros z h1.
destruct h1 as [h1 h2].
destruct h1 as [h1 | z h3].
left. constructor; auto.
right; auto.
(* >= *)
red. intros z h1.
destruct h1 as [z h1 | z h2].
destruct h1 as [h1 h3].
constructor. constructor; auto. auto.
destruct h2; subst.
constructor. apply Add_intro2.
intro h1. destruct h1. 
contradict h0. auto.
Qed.

(*Wyckoff*)
Lemma sub_add_compat_nin : 
  forall {T:Type} (A:Ensemble T) (x:T),
    ~ In A x ->
    (Subtract (Add A x) x) = A.
intros T A x h1.
apply Extensionality_Ensembles.
red. split.
(* => *)
red.
intros y h2.
destruct h2 as [h2 ?].
destruct h2; try contradiction; auto.
(* <= *)
red.
intros y h2.
constructor. left; auto.
intro h3.
destruct h3; subst.
contradiction.
Qed.                          

(*Wyckoff*)
Lemma sub_add_compat_in :
  forall {T:Type} (A:Ensemble T) (x:T),
    In A x ->
    (Subtract (Add A x) x) = Subtract A x.
intros T A x h1.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h2.
destruct h2 as [h3 h4].
destruct h3 as [y h3 | y h5].
constructor; auto. contradiction.
(* >= *)
red.
intros y h2.
destruct h2. 
constructor; try constructor; auto.
Qed.

(*Wyckoff*)
Lemma add_sub_compat_in :
  forall {T:Type} (A:Ensemble T) (x:T),
    In A x ->
    (Add (Subtract A x) x) = A.
intros T A x h1.
apply Extensionality_Ensembles.
red. split.
red. intros x' h2.
destruct h2 as [x' h2 | x' h3].
destruct h2; auto. inversion h3. subst. assumption.
red. intros x' h2.
destruct (classic (x = x')). subst. right. constructor.
left.
constructor. assumption. intro h3.
inversion h3. subst. contradict H. reflexivity.
Qed.


(*Wyckoff*)
Lemma add_nin_sub_compat :
  forall {T:Type} (A B:Ensemble T) (x:T),
    ~Ensembles.In A x ->
    Add A x = Add B x ->
    A = Subtract B x.
intros T A B x h1 h2.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h3.
pose proof (Add_intro1 _ A x y h3) as h4.
rewrite h2 in h4.
pose proof (Add_inv _ B x y h4) as h5.
destruct h5 as [h5l | h5r].
constructor; auto.
intro h6.
destruct h6; subst; contradiction.
subst. contradiction.
(* >= *)
red. 
intros y h3.
destruct h3 as [h3 h4].
assert (h5: x <> y).
  intro. subst.
  pose proof (In_singleton _ y). contradiction.
clear h4.
pose proof (Add_intro1 _ B x y h3) as h4.
rewrite <- h2 in h4.
destruct h4 as [y h4l | y h4r]. auto.
destruct h4r; subst. 
contradict h5. auto.
Qed.

(*Wyckoff*)
Lemma add2 : forall {T:Type} (A:Ensemble T) (x:T), 
               Add (Add A x) x = Add A x.
intros T A x.
unfold Add.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h1.
destruct h1 as [y h2 | y h3].
auto.
destruct h3.
apply Union_intror. constructor.
(* >= *)
red. intros y h1.
constructor. auto.
Qed.

(*Wyckoff*)
Lemma add_comml : forall {T:Type} (A:Ensemble T) (x y:T),
                   Included (Add (Add A x) y) (Add (Add A y) x).
intros T A x y.
red. 
intros z h1.
destruct h1 as [z h2|z h3].
destruct h2 as [z h4 | z h5].
left. left. auto.
destruct h5; subst.
right. constructor.
destruct h3; subst.
left. right. constructor.
Qed.

(*Wyckoff*)
Lemma add_comm : forall {T:Type} (A:Ensemble T) (x y:T),
                   Add (Add A x) y = Add (Add A y) x.
intros;
apply Extensionality_Ensembles; red; split;
apply add_comml.
Qed.

(*Wyckoff*)
Lemma union_add_comm : forall {T:Type} (A B:Ensemble T) (x:T),
                    Union (Add A x) B = Add (Union A B) x.
intros T A B x.
apply Extensionality_Ensembles.
red. split.
(* <= *) red.
intros y h1.
destruct h1 as [y h1 | y h2].
destruct h1 as [y h1 | y h3].
left. left. assumption.
destruct h3. subst. right. constructor.
left. right. assumption.
(* >= *)
red. intros y h1.
destruct h1 as [y h1 | y h2].
destruct h1 as [y h1 | y h3].
left. left. assumption.
right. assumption.
destruct h2; subst.
left. right. constructor.
Qed.

(*Wyckoff*)
Lemma couple_add_sing : forall {T:Type} (x y:T),
                         Couple x y = Add (Singleton x) y.
intros T x y.
apply Extensionality_Ensembles.
red. split. 
red.
intros z h1.
destruct h1. left. constructor.
right. constructor.
red.
intros z h1.
destruct h1 as [z h1l | z h1r].
destruct h1l; subst. left.
destruct h1r. right.
Qed.

(*Wyckoff*)
Lemma couple_add_add : forall {T:Type} (x y:T),
                         Couple x y = Add (Add (Empty_set _) x) y.
intros T x y.
rewrite couple_add_sing.
rewrite add_empty_sing.
reflexivity.
Qed.

(*Wyckoff*)
Lemma couple_singleton : forall {T:Type} (x:T), 
                           Couple x x = Singleton x.
intros T x.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros ? h1.
destruct h1; constructor.
(* >= *)
red. intros ? h1.
destruct h1; constructor.
Qed.




(*Wyckoff*)
Lemma empty_union : forall {T:Type} (S:Ensemble T),
                      Union (Empty_set T) S = S.
intros T S.
apply Extensionality_Ensembles.
red. split.
  (* <= *)
  red.
  intros x h1.
  destruct h1; [contradiction | assumption].
  (* >= *)
  red.
  intros x h1.
  apply Union_intror; assumption.
Qed.

(*Wyckoff*)
Lemma empty_intersection : forall {T:Type} (S:Ensemble T),
                             Intersection (Empty_set T) S =
                             Empty_set T.
  intros T S.
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros x h1.
  destruct h1; contradiction.
  (* >= *)
  auto with sets.
Qed.

(*Wyckoff*)
Lemma Intersection_idempotent : 
  forall {T:Type} (A:Ensemble T),
    Intersection A A = A.
intros T A.
apply Extensionality_Ensembles.
red. split.
red. intros ? h1. destruct h1; auto.
red. intros ? h1. constructor; auto.
Qed.
                                       

(*Wyckoff*)
Lemma singleton_inj : forall {T:Type} (a b:T), 
                        Singleton a = Singleton b ->
                        a = b.
intros T a b h1.
pose proof (In_singleton _ a) as h2.
rewrite h1 in h2.
destruct h2; reflexivity. 
Qed.

(*Wyckoff*)
Lemma im_singleton : forall {T U:Type} (x:T) (f:T->U),
                       Im (Singleton x) f = Singleton (f x).
intros T U x f.
apply Extensionality_Ensembles.
red. split.
red.
intros y h1.
destruct h1 as [z h2]. subst.
destruct h2; subst.
constructor; auto.
red. 
intros y h1.
destruct h1; subst.
apply Im_intro with x; auto; constructor.
Qed.

(*Wyckoff*)
Lemma im_preserves_inclusion : 
  forall {T U:Type} (A B:Ensemble T) (f:T->U),
    Included A B -> Included (Im A f) (Im B f).
intros T U A B f h1.
red. intros u h2.
destruct h2 as [u h2]. subst.
apply h1 in h2.
apply Im_intro with u; auto.
Qed.



(*Wyckoff*)
Lemma empty_image : forall {T U:Type} (A:Ensemble T) (f:T->U),
                      Im A f = Empty_set _ -> A = Empty_set _.
intros T U A f h1.
apply NNPP.
intro h2. 
pose proof (not_empty_Inhabited _ _ h2) as h3. 
destruct h3 as [x h3].
assert (h4:Ensembles.In (Im A f) (f x)).
    apply Im_intro with x. assumption. reflexivity.
  rewrite h1 in h4.
contradiction.
Qed.

(*Wyckoff*)
Lemma finite_inh_or_empty : 
  forall {T:Type} (A:Ensemble T),
    Finite A -> 
    Inhabited A \/ A = Empty_set _.
intros T A h1.
induction h1 as [|A  h1 h2 a h3]. right; auto.
left. apply Inhabited_intro with a.
right.
constructor.
Qed.


(*Wyckoff*)
Lemma finite_cardinal_unq : 
  forall {T:Type} (S:Ensemble T), 
    Finite  S -> exists! n:nat, cardinal _ S n.
intros T S h1.
pose proof (finite_cardinal _ _ h1) as h2.
destruct h2 as [n h2].
exists n.
red.
split; try assumption.
apply cardinal_unicity; assumption.
Qed.

(*Wyckoff*)
Definition card_fun {T:Type} (S:Ensemble T)
           (pf:Finite S) : nat := 
  proj1_sig (constructive_definite_description _ (finite_cardinal_unq S pf)).

(*Wyckoff*)
Lemma card_fun_compat : 
  forall {T:Type} (S:Ensemble T) (pf:Finite S),
    cardinal _ S (card_fun _ pf).
intros T S h1.
unfold card_fun.
destruct constructive_definite_description.
simpl.
assumption.
Qed.

(*Wyckoff*)
Lemma card_fun_empty : 
  forall T:Type,
    card_fun (Empty_set T) (Empty_is_finite T) = 0.
intro T.
pose proof (card_fun_compat _ (Empty_is_finite T)) as h1.
pose proof (card_empty T) as h2.
eapply cardinal_is_functional; auto. apply h1. assumption.
Qed.

(*Wyckoff*)
Lemma card_fun1_ex : 
  forall {T:Type} (S:Ensemble T),
    exists! n:nat, (Finite S -> cardinal _ S n) /\
                   (~Finite S -> n = 0).
intros T S.
destruct (classic (Finite S)) as [h1 | h2].
exists (card_fun _ h1). red. split.
split. intro h2. apply card_fun_compat.
intro; contradiction.
intros n h2. destruct h2 as [h2l h2r]. specialize (h2l h1).
pose proof (card_fun_compat _ h1) as h3.
pose proof cardinal_is_functional.
pose proof (cardinal_is_functional _ _ _ h2l _ _ h3 (eq_refl _)). subst. reflexivity.
exists 0. red. split. split.
intros; contradiction. auto.
intros n h3. destruct h3 as [h3l h3r].
symmetry. apply h3r; auto.
Qed.

(*Wyckoff*)
Definition card_fun1 {T:Type} (S:Ensemble T) :=
  proj1_sig (constructive_definite_description _ (card_fun1_ex S)).

(*Wyckoff*)
Lemma card_fun1_compat : 
  forall {T:Type} (S:Ensemble T),
    let n:= card_fun1 S in 
    (Finite S -> cardinal _ S n) /\
    (~Finite S -> n = 0).
intros T S. unfold card_fun1. destruct constructive_definite_description. simpl. assumption.
Qed.

Lemma card_fun1_empty : 
  forall (T:Type), card_fun1 (Empty_set T) = 0.
intro T.
pose proof (card_fun1_compat (Empty_set T)) as h1.
destruct h1 as [h1l h1r].
specialize (h1l (Empty_is_finite T)).
pose proof (card_empty T) as h2.
eapply cardinal_is_functional; auto. apply h1l. apply h2.
Qed.

Lemma card_fun1_O : 
  forall {T:Type} (A:Ensemble T), card_fun1 A = 0 ->
         A = Empty_set _ \/ ~Finite A.
intros T A h1.
pose proof (card_fun1_compat A) as h2.
destruct h2 as [h2l h2r].
destruct (classic (Finite A)) as [h3 | h4].
apply h2l in h3. rewrite h1 in h3. 
apply cardinalO_empty in h3. left. assumption.
right. assumption.
Qed.

Lemma card_fun1_sing : 
  forall {T:Type} (x:T), card_fun1 (Singleton x) = 1.
intros T x.
pose proof (card_fun1_compat (Singleton x)) as h1.
destruct h1 as [h1l h1r].
specialize (h1l (Singleton_is_finite _ x)).
pose proof (card_sing x) as h2.
eapply cardinal_is_functional; auto.
apply h1l. apply h2.
Qed.

Lemma lt_card_fun1_finite : 
  forall {T:Type} (A:Ensemble T) 
         (n:nat), n < card_fun1 A -> Finite A.
intros T A n h1.
pose proof (card_fun1_compat A) as h2.
destruct h2 as [h2 h3].
apply NNPP.
intro h4. apply h3 in h4.
rewrite h4 in h1. omega.
Qed.
                                    

Lemma card_fun1_1 : 
  forall {T:Type} (A:Ensemble T), 
    card_fun1 A = 1 -> exists x:T, A = Singleton x.
intros T A h1.
assert (h0:0 < card_fun1 A). omega.
apply lt_card_fun1_finite in h0. 
revert h1.
induction h0 as [|A h2 h3 x h4].
intro h1. rewrite card_fun1_empty in h1. omega.
exists x. 
assert (h5:A = Empty_set _).
  apply NNPP. intro h5. apply not_empty_Inhabited in h5.
  destruct h5 as [y h5].
  assert (h6:x <> y). intro h6. subst. contradiction.
pose proof (card_fun1_compat (Add A x)) as h7.
destruct h7 as [h7l h7r]. clear h7r.
pose proof (Add_preserves_Finite _ _ x h2) as h8. apply h7l in h8.
rewrite h1 in h8.
pose proof card_soustr_1.
pose proof (card_soustr_1 _ _ _ h8 _ (Add_intro2 _ A x)) as h9.
simpl in h9.
rewrite sub_add_compat_nin in h9.
apply cardinalO_empty in h9. subst. contradiction. assumption.
subst.
apply add_empty_sing; auto.
Qed.


Lemma card_fun_card_fun1_compat : 
  forall {T:Type} (A:Ensemble T) (pf:Finite A),
    card_fun A pf = card_fun1 A.
intros T A h1.
pose proof (card_fun_compat A h1) as h2.
pose proof (card_fun1_compat A) as h3.
destruct h3 as [h3l h3r]. specialize (h3l h1).
eapply cardinal_is_functional; auto. apply h2. apply h3l.
Qed.

(*Wyckoff*)
Lemma card_add_nin : 
  forall {T:Type} (A:Ensemble T) (n:nat),
    cardinal _ A n -> 
    forall x:T, ~ In A x -> cardinal _ (Add A x) (S n).
intros T A n h1 x h2.
constructor; auto.
Qed.                                
           
(*Wyckoff*)
Lemma card_add_nin' : 
  forall {T:Type} (A:Ensemble T),
    Finite A -> forall x:T, 
                  ~ In A x -> 
                  card_fun1 (Add A x) = S (card_fun1 A).
intros T A h1 x h2.
pose proof (card_fun1_compat A) as h3. destruct h3 as [h3l h3r].
specialize (h3l h1).
pose proof (card_fun1_compat (Add A x)) as h4. destruct h4 as [h4l h4r].
specialize (h4l (Add_preserves_Finite _ _ _ h1)).
eapply card_add_nin in h3l.
eapply cardinal_is_functional. apply h4l. apply h3l. apply eq_refl. apply h2.
Qed.


(*Wyckoff*)
Lemma incl_card_fun1 : 
  forall {T:Type} (A B:Ensemble T),
    Finite B ->
    Included A B -> card_fun1 A <= card_fun1 B.
intros T A B h4 h1.
assert (h2:Finite A). eapply Finite_downward_closed; auto. apply h4. assumption.
pose proof (finite_cardinal _ _  h2) as h5.
pose proof (finite_cardinal _ _ h4) as h6.
destruct h5 as [m h5]. destruct h6 as [n h6].
pose proof (incl_card_le _ _ _ _ _ h5 h6 h1) as h7.
pose proof (card_fun1_compat A) as [h8].
pose proof (card_fun1_compat B) as [h9].
specialize (h8 h2). specialize (h9 h4).
pose proof (cardinal_is_functional _ _ _ h5 _ _ h8 (eq_refl _)).
subst.
pose proof (cardinal_is_functional _ _ _ h9 _ _ h6 (eq_refl _)).
subst.
assumption.
Qed.

(*Wyckoff*)                       
Lemma singleton_inc : forall {T:Type} (a:T) (A:Ensemble T),
                        Included A (Singleton a) ->
                        A = Empty_set _ \/ A = Singleton a.
intros T a A h0.
pose proof (Singleton_is_finite _ a) as h6.
assert (h7:cardinal _ (Singleton a) 1).
assert (h7:Singleton a = Add (Empty_set _) a).
unfold Add.
rewrite empty_union. reflexivity.
rewrite h7.
constructor. constructor. intro. contradiction.
pose proof (Finite_downward_closed _ _ h6 _ h0) as h8.
pose proof (finite_cardinal _ _ h8) as h9.
destruct h9 as [n h9].
pose proof (incl_card_le _ _ _ _ _ h9 h7 h0) as h10.
pose proof (le_pred n 1 h10) as h11.
simpl in h11.
pose proof (le_n_0_eq _ h11) as h12.
unfold pred in h12.

destruct (zerop n) as [? | h13].
subst.        
left.
apply cardinalO_empty. assumption.
assert (h14: n = S O). auto with arith.
subst.
right.
apply Extensionality_Ensembles.
red. split. assumption.
red.
intros x h14.
destruct h9.    
pose proof (lt_irrefl 0).
contradiction.        
assert (h15:a = x0).
unfold Included in h0.
pose proof (Add_intro2 _ A x0) as h15.
specialize (h0 _ h15).
destruct h0; auto.
subst.
destruct h14; subst.
apply Add_intro2.
Qed.








(*Wyckoff*)
Lemma singleton_inc_int : 
  forall {T:Type} (a:T) (A:Ensemble T),
    Intersection A (Singleton a) =  Empty_set _ \/ 
    Intersection A (Singleton a) = Singleton a.
intros T a A.
assert (h1:Included (Intersection A (Singleton a)) (Singleton a)). auto with sets.
apply singleton_inc.
assumption.
Qed.

(*Wyckoff*)
Lemma inclusion_intersection : forall {T:Type} (P S R : Ensemble T), 
  (Included P S /\ Included P R) <-> Included P (Intersection S R).

intros T P S R.
split.
(*left*)
intro h1.
destruct h1 as [h2 h3].
red.
intros x h4.
constructor; auto with sets.
(*right*)
intro h1.
split; red.
  (*left*)
  intros x h5.
  red in h1.
  pose proof (h1 x h5) as h6.
  inversion h6; trivial.
  (*right*)
  intros x h7.
  pose proof (h1 x h7) as h6. 
  inversion h6; trivial.
Qed.

(*Wyckoff*)
Lemma intersection_preserves_inclusion : forall {T:Type} (P Q R:Ensemble T), 
  Included P Q -> Included (Intersection R P) (Intersection R Q).
intros T P Q R h1.
red.
intros x h2.
destruct h2 as [x h2l h2r].
auto with sets.
Qed.

(*Wyckoff*)
Lemma in_intersection_iff : 
  forall {T:Type} (A B:Ensemble T) (x:T),
    In (Intersection A B) x <-> In A x /\ In B x.
intros T A B x.
split.
intros h1.
destruct h1. split; auto.
intros h1.
destruct h1. constructor; auto.
Qed.

(*Wyckoff*)
Lemma in_union_iff : 
  forall {T:Type} (A B:Ensemble T) (x:T),
    In (Union A B) x <-> In A x \/ In B x.
intros T A B x.
split.
intro h1.
destruct h1. left; auto. right; auto.
intro h1.
destruct h1. left; auto. right; auto.
Qed.


(*Wyckoff*)
Lemma intersection_preserves_finite : 
  forall {T:Type} (P Q:Ensemble T),
    Finite P -> Finite (Intersection P Q).
intros T P Q h1.
assert (h2:Included (Intersection P Q) P). auto with sets.
apply Finite_downward_closed with P; auto.
Qed.

(*Wyckoff*)
Lemma add_preserves_infinite : 
  forall {T:Type} (A:Ensemble T) (a:T),
    ~Finite A -> ~Finite (Add A a).
intros T A a h1. 
intro h2.
assert (h5:Included A (Add A a)). auto with sets.
pose proof (Finite_downward_closed _ _ h2 _  h5).
contradiction.
Qed.



(*Wyckoff*)
Lemma incl_subtract :
  forall {T:Type} (P:Ensemble T) (x:T),
    Included (Subtract P x) P.
intros T P x. red. intros x' h1.
destruct h1. assumption.
Qed.


(*Wyckoff*)
Lemma subtract_preserves_finite : 
  forall {T:Type} (P:Ensemble T) (x:T),
    Finite P -> Finite (Subtract P x).
intros T P x h1.
assert (h2:Included (Subtract P x) P).  apply incl_subtract.
apply Finite_downward_closed with P; auto.
Qed.

(*Wyckoff*)
Lemma card_sub_in : 
  forall {T:Type} (A:Ensemble T),
    Finite A -> forall x:T,
                  In A x ->
                  card_fun1 A = S (card_fun1 (Subtract A x)).
intros T A h1 x h2.
pose proof (card_fun1_compat A) as h3. destruct h3 as [h3l h3r]. clear h3r. 
specialize (h3l h1).
pose proof (card_fun1_compat (Subtract A x)) as h4. destruct h4 as [h4l h4r]. clear h4r.
pose proof (subtract_preserves_finite _ x h1) as h5.
specialize (h4l h5). 
rewrite <- (add_sub_compat_in A x) at 1; auto. 
rewrite card_add_nin'. f_equal.
assumption.
intro h6.
destruct h6 as [h6 h7]. contradict h7. constructor.
Qed.


(*Wyckoff*)
Lemma inclusion_iff_intersection_eq : forall {T:Type} (P Q:Ensemble T), 
  Included P Q <-> Intersection P Q = P.
intros T P Q.
split.
(* -> *)
intro.
apply Extensionality_Ensembles.
red. split; auto with sets.
(* <= *)
intro h1.
red.
intros x h2.
rewrite <- h1 in h2.
destruct h2; assumption.
Qed.

(*Wyckoff*)
Lemma inclusion_iff_union : forall {T:Type} (P Q:Ensemble T), 
  Included P Q <->  Union P Q = Q.
intros T P Q.
split. intro h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. destruct h2; auto with sets.
red. intros; right; auto.
intro h1. red. intros.  rewrite <- h1. auto with sets.
Qed.





(*Wyckoff*)
Lemma empty_inclusion : forall {T:Type} (S:Ensemble T), Included (Empty_set _) S.
intros T S.
red. intros; contradiction.
Qed.

(*Wyckoff*)
Lemma full_inclusion : forall {T:Type} (S:Ensemble T), 
                         Included S (Full_set _).
intros; red; intros; constructor.
Qed.                                                                


(*Wyckoff*)
Lemma inclusion_reflexive : forall {T:Type} (S:Ensemble T),
                    Included S S.
auto with sets.
Qed.

(*Wyckoff*)
Lemma intersection_full_set : forall {T:Type} (S:Ensemble T), 
  Intersection S (Full_set T) = S.
intros T S.
apply Extensionality_Ensembles.
red. split.
  (* <= *) red. intros ? h. destruct h; assumption.
  (* >= *) red. intros x ?. constructor. assumption. constructor.
Qed.

(*Wyckoff*)
Lemma included2_impl_union_included : forall {T:Type} (A B C:Ensemble T), 
  ((Included A C) /\ (Included B C)) <-> Included (Union A B) C.
intros T A B C.
split.
(* -> *)
unfold Included.
intro h1.
destruct h1 as [h1 h2].
intros x h3.
apply Union_inv in h3.
destruct h3; auto with sets.
(* <- *)
intro h1.
split.
assert (Included A (Union A B)). auto with sets.
auto with sets.
assert (Included B (Union A B)). auto with sets.
auto with sets.
Qed.



Section complement.

(*Wyckoff*)
Lemma complement_inv : 
  forall {T:Type} (A:Ensemble T) (x:T), In (Ensembles.Complement A) x -> ~In A x.
auto.
Qed.

(*Wyckoff*)
Lemma complement_inclusion: forall {Y:Type} (S T:Ensemble Y),
  Included S T -> Included (Ensembles.Complement T) (Ensembles.Complement S).
Proof.
intros.
red; intros.
red; intro.
contradiction H0.
auto with sets.
Qed.

(*Wyckoff*)
Lemma excl_middle_empty : (forall {T:Type} (S:(Ensemble T)), Intersection 
  S (Ensembles.Complement S) = Empty_set _).
intros T S.
apply Extensionality_Ensembles. red. split.
(*left*)
red.
intros x h1.
inversion h1; contradiction.
(*right*)
red.
intros; contradiction.
Qed.

(*Wyckoff*)
Lemma excl_middle_full : forall {T:Type} (S:Ensemble T), Union
  S (Ensembles.Complement S) = Full_set _.
intros T S.
apply Extensionality_Ensembles. red. split.
(*left*)
red. intros. constructor. auto with sets.
(*right*)
red. intros x h1.
case (classic (In S x)) as [h2 | h3].
  apply Union_introl; trivial.
  apply Union_intror.
  auto with sets.
Qed.

(*Wyckoff*)
Lemma complement_full : forall T:Type, Ensembles.Complement (Full_set T) = Empty_set T.
intro T.
apply Extensionality_Ensembles. red. unfold Included. split.
(* left *)
intros x h2.
unfold Ensembles.Complement in h2. red in h2.
elim h2. apply Full_intro.
(* right *)
intros x h3. contradiction.
Qed.

(*Wyckoff*)
Lemma complement_empty : forall T:Type, Ensembles.Complement (Empty_set T) = Full_set T.
intro T.
pose proof (Complement_Complement _ (Full_set T)) as h1.
rewrite complement_full in h1.
assumption.
Qed.

(*Wyckoff*)
Lemma complement_meets_non_subset : forall {T:Type} (P Q : Ensemble T), 
  (Inhabited P) ->
  (~Included P Q <-> Inhabited (Intersection (Ensembles.Complement Q) P)).
intros T P Q h1.
split.
(*left*)
intro h2. 
unfold Included in h2.
pose proof (not_all_ex_not _ _ h2) as h4.
elim h4. intros x h5.
assert (h6: (In P x) /\ (~In Q x)).
  tauto.
destruct h6 as [h7 h8].
assert (h9: In (Intersection (Ensembles.Complement Q) P) x).
  constructor; auto with sets. 
apply Inhabited_intro with x; trivial.
(*right*)
intro h2.
unfold Included.
inversion h2 as [x h3].
assert (h4: exists x:T, ~(In P x -> In Q x)).
  exists x.
  assert (h5: (In P x) /\ (~ In Q x)).
    split.
    inversion h3 as [h6 h7]. assumption.
    inversion h3 as [y h6 h7].
    compute in h7. auto.
  tauto.
apply ex_not_not_all; trivial.
Qed.


(*Wyckoff*)
Lemma included_empty_complement_int : forall {T:Type} (A B:Ensemble T), 
  Included A B <-> Intersection A (Ensembles.Complement B) = Empty_set _.
intros T A B.
split.
(* -> *)
intros h1.
apply Extensionality_Ensembles. red.
  split.
  (* <= *)
  red in h1.
  red. intros x h2.
  destruct h2 as [x h2l h2r].
  specialize (h1 _ h2l).
  assert (h3:In (Intersection B (Ensembles.Complement B)) x).
    auto with sets.
  rewrite excl_middle_empty in h3. assumption.
  (* >= *) auto with sets.
(* <- *)
intro h1.
red. 
intros x h2.
apply NNPP.
intro h3.
assert (h4:In (Intersection A (Ensembles.Complement B)) x).
  constructor; assumption.
rewrite h1 in h4.
contradiction.
Qed.



(*Wyckoff*)
Lemma setminus_inc : forall {T:Type} (A B:Ensemble T),
                       Included (Setminus A B) A.
intros T A B. red.
intros x h1.
destruct h1; auto with sets.
Qed.


(*Wyckoff*)
Lemma setminus_int_complement : forall {T:Type} (A B:Ensemble T), Setminus A B = 
  Intersection A (Ensembles.Complement B).
intros T A B.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros ? h1.
destruct h1 as [h2 h3].
constructor; auto with sets. 
(* >= *)
red. intros x h1.
destruct h1 as [h2 h3].
constructor; auto with sets.
Qed.

(*Wyckoff*)
Lemma complement_setminus_full : 
  forall {T:Type} (A:Ensemble T), 
    Ensembles.Complement A = Setminus (Full_set T) A.
intros T A.
apply Extensionality_Ensembles.
red. split.
red. intros x h1. constructor. constructor. assumption.
red. intros x h1. red. red.
destruct h1; auto.
Qed.


(*Wyckoff*)
Lemma setminus_sub_sup : forall {T:Type} (A B:Ensemble T), Included A B -> 
  Setminus A B = Empty_set _.
intros T A B h1.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros x h2.
destruct h2 as [h3 h4].
red in h1.
apply h1 in h3.
contradiction.
(* >= *) auto with sets.
Qed.

(*Wyckoff*)
Lemma int_setminus : forall {T:Type} (A X:Ensemble T),
                       Intersection A (Setminus X A) = Empty_set _.
intros T A X.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros x h1.
destruct h1 as [x h2 h3].
destruct h3. contradiction.
(* >= *)
auto with sets.
Qed.

(*Wyckoff*)
Lemma setminus_same_int : forall {T:Type} (X Y:Ensemble T),
                            Setminus X (Intersection X Y) =
                            Setminus X Y.
intros T X Y.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct h1 as [h1 h2].
constructor. assumption. auto with sets.
red.
intros x h1. constructor.
destruct h1 as [h1 h2].
assumption.
intro h2.
destruct h2; destruct h1; contradiction.
Qed.

(*Wyckoff*)
Lemma decompose_int_setminus : 
  forall {T:Type} (X Y:Ensemble T),
    X = Union (Intersection X Y) (Setminus X Y).
intros T X Y.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct (classic (In Y x)).
left. constructor; auto.
right. constructor; auto.
red.
intros x h1.
destruct h1 as [x h1l | x h1r].
destruct h1l; auto.
destruct h1r; auto.
Qed.

(*Wyckoff*)
Lemma setminus_add1 : forall {T:Type} (E:Ensemble T) (x:T),
                        ~ In E x ->
                        Setminus (Add E x) E = Singleton x.
intros T E x h1.
apply Extensionality_Ensembles.
red. split.
red.
intros y h2.
destruct h2 as [h2 h3].
destruct h2 as [y h2l | y h2r]. contradiction.
assumption.
red. intros y h2. destruct h2; subst. constructor; auto.
right; auto. constructor.
Qed.

(*Wyckoff*)
Lemma setminus_add2 : 
  forall {T:Type} (E:Ensemble T) (e:T),
         Setminus (Add E e) (Singleton e) = 
         Subtract E e.
intros T E e.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [h1 h2].
destruct h1 as [x h1a | x h1b].
constructor; auto.
contradiction.
red.
intros x h1.
red in h1. red in h1. red in h1. destruct h1 as [h1 h2].
constructor. left. assumption.
assumption.
Qed.


   
(*Wyckoff*)
Lemma add_subtract_a : 
  forall {T:Type} (A:Ensemble T) (a:T),
    Subtract (Add A a) a = Subtract A a.
intros T A a.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros x h1.
destruct h1 as [h1 h2].
destruct h1 as [x h1a | x h1b].
constructor; auto. contradiction.
(* >= *)
red.
intros x h1.
destruct h1 as [h1 h2].
pose proof (Add_intro1 _ A a x h1).
constructor; auto.
Qed.

(*Wyckoff*)
Lemma setminus_couple_eq :
  forall {T:Type} (A:Ensemble T) (a a':T), 
    Setminus A (Couple a a') = Subtract (Subtract A a) a'.
intros T A a a'.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros x h1.
destruct h1 as [h1 h2].
constructor. constructor. assumption.
intro h3. destruct h3; subst.
pose proof (Couple_l _ a a'); contradiction.
intro h3. destruct h3; subst.
pose proof (Couple_r _ a a'); contradiction.
(* >= *)
red. intros x h1.
destruct h1 as [h1 h2].
destruct h1 as [h1a h1b].
constructor. assumption.
intro h3.
destruct h3.
pose proof (In_singleton _ a). contradiction.
pose proof (In_singleton _ a'). contradiction.
Qed.

(*Wyckoff*)
Lemma subtract2_comm : 
  forall {T:Type} (S:Ensemble T) (a b:T),
    Subtract (Subtract S a) b =
    Subtract (Subtract S b) a.
  intros T S a b.
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros x h1.
  destruct h1 as [h1a h1b].
  destruct h1a as [h1aa h1ab].
  constructor. constructor. 
  assumption. assumption. assumption.
  (* >= *)
  red. intros x h1.
  destruct h1 as [h1a h1b].
  destruct h1a as [h1aa h1ab].
  constructor. constructor.
  assumption. assumption. assumption.
Qed.

(*Wyckoff*)
Lemma couple_comm : 
  forall {T:Type} (a b:T), 
    Couple a b = Couple b a.
  intros T a b.
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros ? h1.
  destruct h1; subst; constructor.
  (* >= *)
  red. intros ? h1.
  destruct h1; subst; constructor.
Qed.

(*Wyckoff*)
Lemma union_singleton_eq_setminus :
  forall {T:Type} (A B:Ensemble T) (a b:T), 
    Add A a = Add B b -> 
    Setminus A (Couple a b) = Setminus B (Couple a b).
intros T A B a b h1.
assert (h2:Setminus (Add A a) (Couple a b) = 
           Setminus (Add B b) (Couple a b)).
  f_equal; auto.
repeat rewrite setminus_couple_eq in h2.
rewrite add_subtract_a in h2.
symmetry in h2.
rewrite subtract2_comm in h2.
rewrite add_subtract_a in h2.
repeat rewrite <- setminus_couple_eq in h2.
symmetry.
rewrite couple_comm in h2.
assumption.
Qed.

End complement.

Section family1.
Variable It T:Type.
Definition Family := Ensemble (Ensemble T).

(*Schepler*)
(*Copied from Zorn's Lemma Coq user contribution *)
Inductive FamilyUnion (F:Family): Ensemble T :=
  | family_union_intro: forall (S:Ensemble T) (x:T),
    In F S -> In S x -> In (FamilyUnion F) x.

(*Schepler*)
Inductive FamilyIntersection (F:Family) : Ensemble T :=
  | family_intersection_intro: forall x:T,
    (forall S:Ensemble T, In F S -> In S x) ->
    In (FamilyIntersection F) x.

(*Schepler*)
Definition IndexedFamily := It -> Ensemble T.
Variable FI:IndexedFamily.

(*Schepler*)
Inductive IndexedUnion : Ensemble T :=
  | indexed_union_intro: forall (i:It) (x:T),
    In (FI i) x -> In IndexedUnion x.

(*Schepler*)
Inductive IndexedIntersection : Ensemble T :=
  | indexed_intersection_intro: forall (x:T),
    (forall i:It, In (FI i) x) -> In IndexedIntersection x.

(*Schepler*)
Definition ImageFamily : Family:=
  Im (Full_set _) FI.

(*Schepler*)
Lemma indexed_to_family_union: IndexedUnion = FamilyUnion (ImageFamily).
Proof.
apply Extensionality_Ensembles.
unfold Same_set.
unfold Included.
intuition.
destruct H.
apply family_union_intro with (FI i).
apply Im_intro with i.
constructor.
reflexivity.
assumption.

destruct H.
destruct H.
apply indexed_union_intro with x0.
rewrite <- H1.
assumption.
Qed.                                                        

(*Schepler*)
Lemma indexed_to_family_intersection:
  IndexedIntersection = FamilyIntersection ImageFamily.
Proof.
apply Extensionality_Ensembles.
unfold Same_set.
unfold Included.
intuition.
constructor.
intros.
destruct H.
destruct H0.
rewrite H1.
apply H.

constructor.
intro.
destruct H.
apply H.
apply Im_intro with i.
constructor.
reflexivity.
Qed.
End family1.

(*Wyckoff*)
Lemma family_union_add : forall {T:Type} (F:Family T) (S:Ensemble T),
  FamilyUnion _ (Add F S) = Union S (FamilyUnion _ F).
intros T F S.
apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red. intros x h4.
  unfold Add in h4.
  inversion h4 as [A y h5 h6].
  pose proof (Union_inv _ _ _ A h5) as h7.
  case h7 as [h8 | h9].
  (*h8*)
  apply Union_intror. 
  apply family_union_intro with A; assumption.
  (*h9*)
  left. 
  pose proof (Singleton_inv _ _ _ h9) as h12.
  rewrite <- h12 in h6; assumption.
  (* >= *)
  red. intros x h2.
  pose proof (Union_inv _ _ _ x h2) as h3.
  case h3 as [h4 | h5].
  (*h4*)
  pose proof (Add_intro2 _ F S) as h5.
  apply family_union_intro with S; assumption.
  (*h5*)
  inversion h5 as [A y h6 h7]. 
  pose proof (Add_intro1 _ F S) as h8.
  pose proof (h8 A h6) as h9.
  apply family_union_intro with A; assumption.
Qed.

(*Wyckoff*)
Lemma family_union_im : 
  forall {T U:Type} (F:Family T) (f:T->U),
    FamilyUnion _ (Im F (fun A:Ensemble T=>Im A f)) = 
    Im (FamilyUnion _ F) f.
intros T U F f.
apply Extensionality_Ensembles.
red. split.
red.
intros u h1.
destruct h1 as [A u h2 h3].
destruct h2 as [A h2]. subst.
destruct h3 as [u h3]. subst.
apply Im_intro with u. apply family_union_intro with A; auto.
reflexivity.
red.
intros u h1. destruct h1 as [u h1]. subst. 
destruct h1 as [A x h2 h3].
apply family_union_intro with (Im A f).
apply Im_intro with A; auto.
apply Im_intro with x; auto.
Qed.



Section family2.

(*Schepler*)
Lemma empty_indexed_intersection: forall {T:Type}
  (F:IndexedFamily False T),
  IndexedIntersection _ _ F = (Full_set _).
Proof.
intros.
apply Extensionality_Ensembles; red; split; red; intros;
  auto with sets.
constructor.
constructor.
destruct i.
Qed.

Variable T:Type.
(*Schepler*)
Lemma empty_family_union: FamilyUnion T (Empty_set _) =
                          (Empty_set _).
Proof.
apply Extensionality_Ensembles.
unfold Same_set.
unfold Included.
intuition.
destruct H.
contradiction H.
contradiction H.
Qed.

(*Wyckoff*)
Lemma family_union_sing : 
  forall (A:Ensemble T),
    FamilyUnion _ (Singleton  A) = A.
intros A.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [B x h1].
destruct h1; subst. assumption.
red. intros x h1.
apply family_union_intro with A; auto with sets.
Qed.


(*Wyckoff*)
Definition chain (F:Family T) : Prop := 
  forall (S1 S2:Ensemble T), In F S1 -> 
    In F S2 -> Included S1 S2 \/ Included S2 S1.

Definition disjoint {T:Type} (A B:Ensemble T) :=
  Intersection A B = Empty_set _.


(*Wyckoff*)
Definition pairwise_disjoint (F:Family T) : Prop :=
  forall (A B:Ensemble T),
    A <> B ->
    In F A -> In F B -> disjoint A B.
   
(*Wyckoff*)
Lemma pairwise_disjoint_incl : 
  forall (E F:Family T),
    pairwise_disjoint F -> Included E F -> pairwise_disjoint E.
intros E F h1 h2.
red. red in h1. red in h2.
intros A B h3 h4 h5.
pose proof (h2 _ h4) as h6. pose proof (h2 _ h5) as h7.
apply h1; auto.
Qed.

Inductive inhabited_family {T:Type} (F:Family T) : Prop :=
  inhabited_family_intro :
    Inhabited F -> (forall A:Ensemble T, In F A -> Inhabited A) ->
    inhabited_family F.


Lemma family_intersection_relativization : 
  forall (F:Family T) (S:Ensemble T),
    In F S ->
    FamilyIntersection _ F = FamilyIntersection _ (Im F (fun A => Intersection A S)).
intros F S h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. 
constructor.
intros S' h3. destruct h3 as [S' h3]. subst.
constructor; apply h2; auto.
red.
intros x h2.
destruct h2 as [x h2]. constructor.
intros S' h3.
specialize (h2 (Intersection S' S)).
assert (h4: In (Im F (fun A : Ensemble T => Intersection A S)) (Intersection S' S)).
  apply Im_intro with S'.  assumption. 
  reflexivity. 
specialize (h2 h4).
destruct h2; auto.
Qed.

Lemma family_incl_intersection_cont : 
  forall (F F':Family T),
    Included F F' -> Included (FamilyIntersection _ F') (FamilyIntersection _ F).
intros F F' h1.
red. intros x h2.
destruct h2 as [x h2]. constructor.
intros S h3. apply h2; auto.
Qed.


Definition finc {T:Type} (E:Ensemble T) : Type := 
  {F : Ensemble T | Finite F /\ Included F E}.


Lemma family_union_im_full_set_finc_proj1_sig_eq : 
  forall (E:Ensemble T),
    FamilyUnion _ (Im (Full_set (finc E)) (@proj1_sig _ _)) = E.
intro E.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [S  x h1 h2].
destruct h1 as [S h1]. subst. clear h1.
destruct S as [S [h3 h4]]. simpl in h2.
apply h4; auto.
red.
intros x h1.
assert (h2:Included (Singleton x) E). red. intros x' h2. destruct h2; auto.
pose proof (Singleton_is_finite _ x) as h3.
assert (h4:In (Full_set (finc E)) (exist _ _ (conj h3 h2))).
  constructor.
assert (h5:In (Im (Full_set (finc E))
           (proj1_sig (P:=fun F : Ensemble T => Finite F /\ Included F E))) (Singleton x)).
apply Im_intro with 
  (exist (fun F : Ensemble T => Finite F /\ Included F E)
         (Singleton x) (conj h3 h2)). constructor.
simpl. reflexivity.
apply family_union_intro with (Singleton x).
assumption.
constructor.
Qed.


End family2.

Arguments FamilyUnion [T] _ _.
Arguments FamilyIntersection [T] _ _.
Arguments IndexedUnion _ [T] _ _.
Arguments IndexedIntersection _ [T] _ _.
Arguments chain [T] _.
Arguments disjoint [T] _ _.
Arguments pairwise_disjoint [T] _.

Section Power_set_Algebra_axioms.
Variable Xt:Type.
(*Wyckoff*)
Lemma assoc_sum_psa : forall N M P:(Ensemble Xt), (Union N (Union M P)) = 
  (Union (Union N M) P).
intros N M P.
apply Extensionality_Ensembles.  unfold Same_set.
split; unfold Included.
(*left*)
intros x H.
inversion H.
assert (h1 : In (Union N M) x). 
apply Union_introl. assumption.
apply Union_introl. assumption.
inversion H0.
assert (h1 : In (Union N M) x). 
apply Union_intror. assumption.
apply Union_introl. assumption.
apply Union_intror. assumption.
(*right*)
intros x H.
inversion H. inversion H0.
apply (Union_introl _ _ _ _ H2).
assert (h2: In (Union M P) x).
apply (Union_introl _ _ _ _ H2).
apply (Union_intror _ _ _ _ h2).
assert (h2: In (Union M P) x).
apply (Union_intror _ _ _ _ H0).
apply (Union_intror _ _ _ _ h2).
Qed.

(*Wyckoff*)
Lemma assoc_prod_psa : forall N M P:(Ensemble Xt), 
  (Intersection N (Intersection M P)) = 
  (Intersection (Intersection N M) P).
intros N M P.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 h4].
inversion h3 as [x'' h5 h6 h7].
assert (h8: In (Intersection N M) x).
apply Intersection_intro; assumption.
apply Intersection_intro; assumption.
(*right*)
intros x h9.
inversion h9 as [x' h10 h11 h12].
inversion h10 as [x'' h13 h14 h15].
assert (h16: In (Intersection M P) x).
apply Intersection_intro; assumption.
apply Intersection_intro; assumption.
Qed.

(*Wyckoff*)
Lemma comm_sum_psa : forall N M:(Ensemble Xt), Union N M = Union M N.
intros N M.
apply Extensionality_Ensembles.
unfold Same_set. 
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 | x'' h4 h5].
apply Union_intror; assumption.
apply Union_introl; assumption.
(*right*)
intros x h6.
inversion h6 as [x' h7 h8 | x'' h9 h10].
apply Union_intror; assumption.
apply Union_introl; assumption.
Qed.

(*Wyckoff*)
Lemma comm_prod_psa : 
  forall N M:(Ensemble Xt), Intersection N M = Intersection M N.
intros N M.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 h4].
apply Intersection_intro; assumption.
(*right*)
intros x h5.
inversion h5 as [x' h6 h7 h8].
apply Intersection_intro; assumption.
Qed.

(*Wyckoff*)
Lemma abs_sum_psa  : 
  forall N M:(Ensemble Xt), Union N (Intersection N M) = N.
intros N M.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 | x'' h4 h5].
assumption.
inversion h4 as [x''' h6 h7].
assumption.
(*right*)
intros x h8.
apply Union_introl. assumption.
Qed.

(*Wyckoff*)
Lemma abs_prod_psa  : 
  forall N M:(Ensemble Xt), Intersection N (Union N M) = N.
intros N M.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 h4].
assumption.
(*right*)
intros x h5.
assert (h6:In (Union N M) x).
apply Union_introl; assumption.
apply Intersection_intro; assumption.
Qed.

(*Wyckoff*)
Lemma dist_sum_psa :
  forall N M P:(Ensemble Xt), 
    Intersection P (Union N M) = 
    Union (Intersection P N) (Intersection P M).
intros N M P.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 h4].
inversion h3 as [x'' h5 h6 | x'' h7 h8].
assert (h9: In (Intersection P N) x).
apply Intersection_intro; assumption.
apply Union_introl; assumption.
assert (h10: In (Intersection P M) x).
apply Intersection_intro; assumption.
apply Union_intror; assumption.
(*right*)
intros x h11.
inversion h11 as [x' h12 h13 | x'' h14 h15].
inversion h12 as [x''' h16 h17 h18].
assert (h19 : In (Union N M) x).
apply Union_introl; assumption.
apply Intersection_intro; assumption.
inversion h14 as [x''' h20 h21 h22].
assert (h23: In (Union N M) x).
apply Union_intror; assumption.
apply Intersection_intro; assumption.
Qed.

(*Wyckoff*)
Lemma dist_prod_psa : 
  forall N M P:(Ensemble Xt), 
    Union P (Intersection N M) = 
    Intersection (Union P N) (Union P M).
intros N M P.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
inversion h1 as [x' h2 h3 | x'' h4 h5].
assert (h6: In (Union P N) x). auto with sets.
assert (h7: In (Union P M) x). auto with sets.
auto with sets.
inversion h4.
assert (h8: In (Union P N) x). auto with sets.
assert (h9: In (Union P N) x). auto with sets.
auto with sets.
(*right*)
intros x h10.
inversion h10 as [x' h11 h12 h13].
inversion h11. 
auto with sets.
inversion h12.
auto with sets.
assert (h14: In (Intersection N M) x). 
auto with sets. auto with sets.
Qed.

(*Wyckoff*)
Lemma comp_sum_psa: 
  forall N:(Ensemble Xt), Union N (Ensembles.Complement N) = Full_set Xt.
intros N.
apply Extensionality_Ensembles.
unfold Same_set. 
split; unfold Included.
(*left*)
intros x h1.
apply Full_intro.
(*right*)
intros x h2.
assert (h3: ~(In N x) \/ (In N x)). 
    tauto.
elim h3; auto with sets.
Qed.

(*Wyckoff*)
Lemma comp_prod_psa:
  forall N:(Ensemble Xt), Intersection N (Ensembles.Complement N) = Empty_set Xt.
intros N.
apply Extensionality_Ensembles.
unfold Same_set.
split; unfold Included.
(*left*)
intros x h1.
contradict h1.
intro h2.
inversion h2 as [x' h3 h4].
unfold In in h3.  unfold In in h4. unfold Ensembles.Complement in h4.
tauto.
(*right*)
intros x h5.
inversion h5.
Qed.
End Power_set_Algebra_axioms.

(*Wyckoff*)
Lemma foil_union : 
  forall {T:Type} (A B C D:Ensemble T),
    Intersection (Union A B) (Union C D) = 
    Union (Intersection A C) (Union (Intersection A D) (Union (Intersection B C) (Intersection B D))).
intros T A B C D.
rewrite dist_sum_psa. 
rewrite comm_prod_psa at 1. rewrite (comm_prod_psa _ (Union A B) D).
do 2 rewrite dist_sum_psa.
rewrite (comm_prod_psa _ C A). rewrite (comm_prod_psa _ C B).
rewrite (comm_prod_psa _ D A). rewrite (comm_prod_psa _ D B).
rewrite <- assoc_sum_psa.
rewrite (assoc_sum_psa _ (Intersection B C) (Intersection A D) (Intersection B D)).
rewrite (comm_sum_psa _ (Intersection B C) (Intersection A D)).
rewrite <- assoc_sum_psa.
reflexivity.
Qed.

(*Wyckoff*)
Lemma foil_intersection : 
  forall {T:Type} (A B C D:Ensemble T),
    Union (Intersection A B) (Intersection C D) = 
    Intersection (Union A C) (Intersection (Union A D) (Intersection (Union B C) (Union B D))).
intros T A B C D.
rewrite dist_prod_psa. 
rewrite comm_sum_psa at 1. rewrite (comm_sum_psa _ (Intersection A B) D).
do 2 rewrite dist_prod_psa.
rewrite (comm_sum_psa _ C A). rewrite (comm_sum_psa _ C B).
rewrite (comm_sum_psa _ D A). rewrite (comm_sum_psa _ D B).
rewrite <- assoc_prod_psa.
rewrite (assoc_prod_psa _ (Union B C) (Union A D) (Union B D)).
rewrite (comm_prod_psa _ (Union B C) (Union A D)).
rewrite <- assoc_prod_psa.
reflexivity.
Qed.

(*Wyckoff*)
Lemma elimination_union : 
  forall {T:Type} (A B C:Ensemble T),
    A = Union B C -> C = Intersection A (Union (Ensembles.Complement B) C).
intros T A B C h1.
pose proof (f_equal (fun D => Intersection D (Union (Ensembles.Complement B) C)) h1) as h2. simpl in h2.
rewrite foil_union in h2.
rewrite excl_middle_empty in h2.
rewrite Intersection_idempotent in h2.
rewrite <- (comm_sum_psa _ C (Intersection C (Ensembles.Complement B))) in h2.
rewrite abs_sum_psa in h2.
rewrite (comm_sum_psa _ (Intersection B C) C) in h2.
rewrite (comm_prod_psa _ B C) in h2.
rewrite abs_sum_psa in h2.
rewrite empty_union in h2.
symmetry.
assumption.
Qed.



(*Wyckoff*)
Lemma decompose_setminus_inc : 
  forall {T:Type} (X Y:Ensemble T),
    Included Y X ->
    X = Union Y (Setminus X Y).
intros T X Y h1.
rewrite (decompose_int_setminus X Y) at 1.
rewrite inclusion_iff_intersection_eq in h1.
rewrite comm_prod_psa in h1.
rewrite h1.
reflexivity.
Qed.

(*Wyckoff*)
Lemma union_eq_empty : forall {T:Type} (A B:Ensemble T),
  Union A B = Empty_set _ -> A = Empty_set _ /\ B = Empty_set _.
intros T A B h1.
split; apply Extensionality_Ensembles; split; auto with sets.
red. intros x h2. pose proof (Union_introl _ A B _ h2) as h3.
rewrite h1 in h3. assumption.
red. intros x h2. pose proof (Union_introl _ B A _ h2) as h3.
rewrite comm_sum_psa in h3.
rewrite h1 in h3. assumption.
Qed.

(*Wyckoff*)
Lemma intersection_eq_full : forall {T:Type} (A B:Ensemble T),
  Intersection A B = Full_set _ -> A = Full_set _ /\ B = Full_set _. 
intros T A B h1. split; apply Extensionality_Ensembles; split;
try (red; intros; constructor).
red. intros x h2.
rewrite <- h1 in h2. destruct h2; assumption.
red. intros x h2.
rewrite <- h1 in h2. destruct h2; assumption.
Qed.

(*Wyckoff*)
Lemma add_a_nin_inj : forall {T:Type} (A A':Ensemble T) (a:T),
                    ~In A a -> ~In A' a ->
                    Add A a = Add A' a -> A = A'.
intros T A A' a h0 h0' h1. 
pose proof (f_equal (fun S => Subtract S a) h1) as h2.
simpl in h2.
do 2 rewrite sub_add_compat_nin in h2; auto.
Qed. 


(*Wyckoff*)
Lemma card_disj_union : 
  forall {T:Type} (A B:Ensemble T) (m n:nat),
    Intersection A B = Empty_set _ ->
         cardinal _ A m ->
         cardinal _ B n ->
         cardinal _ (Union A B) (m+n).
intros T A B m n h1 h2 h3.
revert h1.
induction h2 as [|A m h2 h4 a h5].
intro h4.
assert (h5:0 + n = n). auto with arith.
rewrite h5.
rewrite empty_union.
assumption.
intro h6.
rewrite union_add_comm.
assert (h7:S m + n = S(m+n)). auto with arith.
rewrite h7.
constructor.
assert (h8:Included A (Add A a)). auto with sets.
pose proof (intersection_preserves_inclusion _ _ B h8) as h9.
rewrite (comm_prod_psa _ B (Add A a)) in h9.
rewrite (comm_prod_psa _ B A) in h9.
rewrite h6 in h9.
assert (h10:Intersection A B = Empty_set _). auto with sets.
apply h4; auto.
intro h8.
destruct h8 as [a h8l | a h8r].
contradiction.
pose proof (Add_intro2 _ A a) as h9.
assert (h10:In (Intersection (Add A a) B) a). auto with sets.
rewrite h6 in h10.
contradiction.
Qed.

(*Wyckoff*)
Lemma card_disj_union' : 
  forall {T:Type} (A B:Ensemble T) 
         (pfa:Finite A) (pfb:Finite B),
    Intersection A B = Empty_set _ ->
    (card_fun A pfa) + (card_fun B pfb) = 
    card_fun (Union A B) (Union_preserves_Finite _ _ _ pfa pfb).
intros T A B h1 h2 h3.
pose proof (card_disj_union A B (card_fun A h1) (card_fun B h2) h3) as h4.
pose proof (card_fun_compat A h1) as h5. pose proof (card_fun_compat B h2) as h6.
specialize (h4 h5 h6).
pose proof (card_fun_compat (Union A B) (Union_preserves_Finite T A B h1 h2)) as h7.
eapply cardinal_is_functional; auto. apply h4. apply h7.
Qed.

(*Wyckoff*)
Lemma card_decompose_int_setminus :
  forall {T:Type} (X Y:Ensemble T) (m n p:nat),
    cardinal _ X m -> cardinal _ (Intersection X Y) n ->
    cardinal _ (Setminus X Y) p ->
    m = n + p.
intros T X Y m n p h1 h2 h3.
pose proof (decompose_int_setminus X Y) as h4.
rewrite h4 in h1.
pose proof (setminus_same_int X Y) as h5.
rewrite <- h5 in h1.
assert (h6:(Intersection (Intersection X Y) (Setminus X (Intersection X Y))) = Empty_set _).
apply int_setminus.
rewrite <- h5 in h3.
pose proof (card_disj_union _ _ _ _ h6 h2 h3) as h7.
apply (cardinal_unicity _ _ _ h1 _ h7).
Qed.

Lemma card_decompose_int_setminus' :
  forall {T:Type} (X Y:Ensemble T) (m n:nat),
    cardinal _ X m -> cardinal _ (Intersection X Y) n ->
    cardinal _ (Setminus X Y) (m-n).
intros T X Y m n h1 h2.
pose proof (card_decompose_int_setminus X Y m n) as h3.
pose proof (cardinal_finite _ _ _ h1) as h4.
pose proof (setminus_inc X Y) as h5.
pose proof (Finite_downward_closed).
pose proof (Finite_downward_closed _ _ h4 _  h5) as h6.
pose proof (finite_cardinal _ _ h6) as h7.
destruct h7 as [r h7].
specialize (h3 _ h1 h2 h7).
subst.
assert (h8:n+r-n = r). auto with arith.
rewrite h8.
assumption.
Qed.


Section Symdiff.

(*Wyckoff*)
Definition Symdiff {T:Type} (A B:Ensemble T) :=
  Union (Setminus A B) (Setminus B A).

(*Wyckoff*)
Lemma symdiff_ref : forall {T:Type} (A:Ensemble T),
                        Symdiff A A = Empty_set _.
intros T A.
unfold Symdiff.
rewrite setminus_sub_sup; auto with sets.
Qed.

(*Wyckoff*)
Lemma symdiff_comm : forall {T:Type} (A B:Ensemble T),
                       Symdiff A B = Symdiff B A.
intros T A B. unfold Symdiff. auto with sets.
Qed.

(*Wyckoff*)
Lemma symdiff_empty_iff_eq : 
  forall {T:Type} (A B:Ensemble T),
    Symdiff A B = Empty_set _ <-> A = B.
intros T A B.
split.
(* -> *)
intro h1.
unfold Symdiff in h1.
apply Extensionality_Ensembles.
red. split.
(* <= *) red.
intros x h2.
apply NNPP.
intro h3.
pose proof (Setminus_intro _ A B x h2 h3) as h4.
pose proof (Union_introl _ (Setminus A B) (Setminus B A) x h4) as h5.
rewrite h1 in h5. contradiction.
(* >= *)
intros x h2.
apply NNPP.
intro h3.
pose proof (Setminus_intro _ B A x h2 h3) as h4.
pose proof (Union_introl _ (Setminus B A) (Setminus A B) x h4) as h5.
rewrite comm_sum_psa in h5.
rewrite h1 in h5. contradiction.
(* <- *)
intro h1.
rewrite h1.
apply symdiff_ref.
Qed.

(*Wyckoff*)
Definition Setminus_full {T:Type} (F X Y:Ensemble T) := Intersection X (Setminus F Y).

(*Wyckoff*)
Lemma setminus_full_ref : forall {T:Type} (F X:Ensemble T),
                            Setminus_full F X X = Empty_set _.
intros T F X.
apply Extensionality_Ensembles; split; auto with sets.
red.
intros x h1.
destruct h1 as [x h1a h1b].
destruct h1b; contradiction.
Qed.

(*Wyckoff*)
Lemma setminus_full_empty_inc : 
  forall {T:Type} (F X Y:Ensemble T),
    Setminus_full F X Y = Empty_set _ -> Included (Intersection F X) Y.
intros T F X Y h1.
red.
intros x h2.
apply NNPP. intro h3.
unfold Setminus_full in h1.
destruct h2 as [x h2l h2r].
assert (h4:In (Intersection X (Setminus F Y)) x).
  constructor; auto; constructor; auto.
rewrite h1 in h4.
contradiction.
Qed.

(*Wyckoff*)
Lemma setminus_full_empty_inc' : 
  forall {T:Type} (F X Y:Ensemble T),
    Setminus_full F X Y = Empty_set _ -> Included X F -> 
    Included X Y.
intros T F X Y h1 h2.
pose proof (setminus_full_empty_inc F X Y h1) as h3.
rewrite inclusion_iff_intersection_eq in h2.
rewrite comm_prod_psa in h2.
rewrite h2 in h3.
assumption.
Qed.

(*Wyckoff*)
Definition Symdiff_full {T:Type} (F X Y:Ensemble T) := 
  Union (Setminus_full F X Y) (Setminus_full F Y X).

(*Wyckoff*)
Lemma symdiff_full_ref : forall {T:Type} (F A:Ensemble T),
                        Symdiff_full F A A = Empty_set _.
intros T F A. apply Extensionality_Ensembles; split; auto with sets.
red. intros x h1.
destruct h1 as [x h1 |x h2].
rewrite setminus_full_ref in h1. assumption.
rewrite setminus_full_ref in h2. assumption.
Qed.

(*Wyckoff*)
Lemma symdiff_full_comm : forall {T:Type} (F A B:Ensemble T),
                       Symdiff_full F A B = Symdiff_full F B A.
intros T F A B.
unfold Symdiff_full. auto with sets.
Qed.

(*Wyckoff*)
Lemma symdiff_full_empty_iff_eq : 
  forall {T:Type} (F A B:Ensemble T),
    Included A F -> Included B F ->
    (Symdiff_full F A B = Empty_set _ <-> A = B).
intros T F A B h1 h2. split.
(* -> *)
intro h3.
unfold Symdiff_full in h3.
pose proof (union_eq_empty _ _ h3) as h4.
destruct h4 as [h4l h4r].
pose proof (setminus_full_empty_inc' _ _ _ h4l h1) as h5.
pose proof (setminus_full_empty_inc' _ _ _ h4r h2) as h6.
apply Extensionality_Ensembles; red; split; assumption.
(* <- *)
intro h3.
rewrite h3.
apply symdiff_full_ref.
Qed.


End Symdiff.



Section DeMorgan.
Variable T:Type.

(*Wyckoff*)
Lemma comp_union : forall (M N : Ensemble T), Ensembles.Complement (Union M N) =
  Intersection (Ensembles.Complement M) (Ensembles.Complement N).
intros M N.
apply Extensionality_Ensembles. red. split.
(*left*)
red. intros x h1.
assert (h2: Included M (Union M N)).
  auto with sets.
assert (h3: Included N (Union M N)).
  auto with sets.
split.
pose proof (complement_inclusion _ _ h2) as h4.
auto with sets.
pose proof (complement_inclusion _ _ h3) as h5.
auto with sets.
(*right*)
red. intros x h6.
inversion h6 as [y h7 h8].
unfold Ensembles.Complement in h7. red in h7.
unfold Ensembles.Complement in h8. red in h8.
unfold Ensembles.Complement. red.
intro h9.
inversion h9.
contradiction. contradiction.
Qed.

(*Wyckoff*)
Lemma comp_int : forall (M N : Ensemble T), Ensembles.Complement (Intersection M N) =
  Union (Ensembles.Complement M) (Ensembles.Complement N).
intros M N.
pose proof (Complement_Complement _ M) as h1.
pose proof (Complement_Complement _ N) as h2.
rewrite <- h1 at 1.
rewrite <- h2 at 1.
pose proof (comp_union (Ensembles.Complement M) (Ensembles.Complement N)) as h3.
rewrite <- h3.
apply Complement_Complement.
Qed.
End DeMorgan.


(*Wyckoff*)
Lemma incl_union_inv : 
  forall {T:Type} (A B C:Ensemble T),
    Included A (Union B C) -> 
    Included A B \/ Included A C \/ (Intersection A (Symdiff B C)) <> Empty_set _.
intros T A B C h1. 
destruct (classic (Included A (Union B (Ensembles.Complement C)))) as [h2 | h3].
pose proof (conj h1 h2) as h3.
rewrite inclusion_intersection in h3.
rewrite <- dist_prod_psa in h3.
rewrite excl_middle_empty in h3.
rewrite comm_sum_psa in h3.
rewrite empty_union in h3.
left; auto.
destruct (classic (Included A (Union (Ensembles.Complement B) C))) as [h4 | h5].
rewrite comm_sum_psa in h1. rewrite comm_sum_psa in h4.
pose proof (conj h1 h4) as h5.
rewrite inclusion_intersection in h5.
rewrite <- dist_prod_psa in h5.
rewrite excl_middle_empty in h5.
rewrite comm_sum_psa in h5.
rewrite empty_union in h5.
right. left. auto.
rewrite included_empty_complement_int in h3.
rewrite included_empty_complement_int in h5. 
rewrite comp_union in h5. rewrite comp_union in h3.
rewrite Complement_Complement in h3. rewrite Complement_Complement in h5.
pose proof (union_eq_empty (Intersection A (Intersection (Ensembles.Complement B) C)) (Intersection A (Intersection B (Ensembles.Complement C)))) as h6.
pose proof (contrapos _ _ h6) as h7. clear h6.
assert (h8:  ~
       (Intersection A (Intersection (Ensembles.Complement B) C) =
        Empty_set T /\
        Intersection A (Intersection B (Ensembles.Complement C)) =
        Empty_set T) ).
  apply or_not_and.
  left. auto.
specialize (h7 h8).
rewrite <- dist_sum_psa in h7.
right. right.
unfold Symdiff.
do 2 rewrite setminus_int_complement.
rewrite comm_sum_psa in h7.
rewrite (comm_prod_psa _ (Ensembles.Complement B) C) in h7.
assumption.
Qed.

(*Wyckoff*)
Lemma symdiff_sings : 
  forall {T:Type} (b c:T), b <> c ->
    Symdiff (Singleton b) (Singleton c) = Couple b c.
intros T b c h1.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2. 
destruct h2 as [x h2l | x h2r].
destruct h2l as [h2l]. destruct h2l. left.
destruct h2r as [h2r]. destruct h2r. right.
red. intros x h2.  
destruct h2. left. constructor. 
constructor. intro h2. destruct h2. contradict h1. 
reflexivity.
right. constructor. constructor.
intro h2. destruct h2. contradict h1.
reflexivity.
Qed.



Section Infinite_analogues.
Variable It T:Type.
Variable FI : (IndexedFamily It T).

(*Wyckoff*)
Definition Int_Fam_Ens (P:Ensemble T) (Q:Family T) : Family T := 
  (Im Q (fun (S:Ensemble T) => (Intersection P S))).

(*Wyckoff*)
Definition Union_Fam_Ens (P:Ensemble T) (Q:Family T) : Family T :=
  (Im Q (fun (S:Ensemble T) => (Union P S))).

(*Wyckoff*)
Definition Subtract_Fam_Ens (F:Family T) (x:T) : Family T :=
  (Im F (fun S => Subtract S x)).


(*Wyckoff*)
Definition Int_Ind_Ens (P:Ensemble T) (i:It) : (Ensemble T) :=
  Intersection P (FI i).

(*Wyckoff*)
Definition Union_Ind_Ens (P:Ensemble T) (i:It) : (Ensemble T) :=
  Intersection P (FI i).
  
(*Wyckoff*)
Lemma dist_union_infnt_1 : 
  forall (P:Ensemble T) (Q:Family T) , 
    Intersection P (FamilyUnion Q) = FamilyUnion (Int_Fam_Ens P Q).
intros P Q.
apply Extensionality_Ensembles.
unfold Same_set.
split.
(*left*)
unfold Included.
intros x h1.
inversion h1 as [y h2 h3 h4].
inversion h3 as [S z h5 h6 h7].
assert (h8: In (Intersection P S) x).
  auto with sets.
apply family_union_intro with (S := (Intersection P S)).
unfold Int_Fam_Ens.
apply Im_intro with (x := S).
assumption. reflexivity. assumption.
(*right*)
unfold Included.
intros x h8.
inversion h8 as [S y h9 h10 h11].
unfold Int_Fam_Ens in h9.
apply Intersection_intro.
inversion h9 as [A h12 B h13 h14].
rewrite h13 in h10.
inversion h10.
assumption.
inversion h9 as [A h12 B h13 h14].
rewrite h13 in h10.
assert (h15: In A x).
  inversion h10. assumption.
apply family_union_intro with (S := A).
assumption. assumption.
Qed.

(*Wyckoff*)
Lemma dist_int_infnt_1 : 
  forall (P:Ensemble T) (Q:Family T), 
    Union P (FamilyIntersection Q) = FamilyIntersection (Union_Fam_Ens P Q).
intros P Q.
apply Extensionality_Ensembles.
unfold Same_set.
split.
(*left*)
unfold Included.
intros x h1.
inversion h1 as [y h2 | y h3].
apply family_intersection_intro.
intros S h4.
unfold Union_Fam_Ens in h4.
inversion h4 as [S' z h5 h6 h7].
assert (h8: In (Union P S') x). auto with sets.
rewrite h6. assumption.
inversion h3 as [z h9 h10].
apply family_intersection_intro.
intros S h11.
unfold Union_Fam_Ens in h11.
inversion h11 as [A h12 B h13 h14].
assert (h15: In A x).
  apply h9.  assumption.
assert (h16: In (Union P A) x).  auto with sets.
rewrite h13. assumption.
(*right*)
unfold Included.
intros x h17.
inversion h17 as [y h18 h19].
unfold Union_Fam_Ens in h18.
assert (h20: forall (S':Ensemble T), (In Q S') -> In (Union P S') x).
  intros S' h50.
  apply h18.
  apply Im_intro with (x := S'). 
  assumption. reflexivity.
assert (h21 : 
    forall S : Ensemble T, In Q S -> In P x \/ In S x).
  intros S h22.
  apply Union_inv.
  apply h20. assumption.
assert (h22: In P x \/ (forall S : Ensemble T, (In Q S -> In S x))).
  apply prop_dis_forall.
  intro S.
  assert (h23: (In Q S -> In P x \/ In S x)).
    apply h21.
  tauto.
case h22.
  auto with sets.
intro h24.
apply Union_intror.
apply family_intersection_intro.
assumption.
Qed.

(*Wyckoff*)
Lemma dist_union_infnt_1_ind : 
  forall (P:Ensemble T), 
    Intersection P (IndexedUnion _ FI) = IndexedUnion _ (Int_Ind_Ens P).
intro P.
rewrite (indexed_to_family_union _ _ FI).
rewrite indexed_to_family_union.
rewrite (dist_union_infnt_1 P (ImageFamily It T FI)).
apply Extensionality_Ensembles. unfold Same_set.
unfold Int_Fam_Ens.
unfold Int_Ind_Ens.
unfold Included.
split.
(*left*)
intros x h1.
inversion h1.
inversion H.
unfold ImageFamily.
apply family_union_intro with (S := S).
unfold ImageFamily in H2.
rewrite H3.
inversion H2.
apply Im_intro with (X := Full_set It) (x := x2).
assumption.
rewrite H6. reflexivity.
assumption.
(*right*)
intros x h20.
unfold ImageFamily in h20.
inversion h20.
apply family_union_intro with (S := S).
inversion H.
rewrite H3.
apply Im_intro with (x := (FI x1)).
unfold ImageFamily.
apply Im_intro with (x := x1).
assumption.
reflexivity.
reflexivity.
assumption.
Qed.

(*Wyckoff*)
Lemma subtract_family_union :
  forall (x:T) (F:Family T),
    Subtract (FamilyUnion F) x = 
    FamilyUnion (Subtract_Fam_Ens F x).
intros x F.
apply Extensionality_Ensembles.
red. split.
red. intros a h1.
destruct h1 as [h1 h2].
destruct h1 as [A a h1 h3].
apply family_union_intro with (Subtract A x).
apply Im_intro with A; auto. constructor; auto.
red. intros a h1. 
destruct h1 as [A a h1 h2].
destruct h1 as [A h1 ? h4]. subst.
destruct h2 as [h2l h2r].
constructor. apply family_union_intro with A; auto.
assumption.
Qed.


End Infinite_analogues.

(*Wyckoff*)
Lemma pairwise_disjoint_add_family_union : 
  forall {T:Type} (F:Family T) (A:Ensemble T),
    ~Ensembles.In F A ->
    pairwise_disjoint (Add F A) ->
    disjoint (FamilyUnion F) A.
intros T F A h0 h1. red.
destruct (eq_dec F (Empty_set _)) as [hem | hnem].
rewrite hem.
rewrite empty_family_union. auto with sets.
pose proof (not_empty_Inhabited _ _ hnem) as hinh.
destruct hinh as [B hin]. 
rewrite comm_prod_psa.
rewrite dist_union_infnt_1.
pose proof (Add_intro2 _ F A) as hadd.
red in h1.
assert (h2:Int_Fam_Ens T A F = Singleton (Empty_set _)).
  apply Extensionality_Ensembles.
  red. split.
  red. intros X h2.
  destruct h2 as [X h2 ? ?]. subst.
  pose proof (Add_intro1 _ _ A _ h2) as h3.
  assert (h4:A <> X).
    intro h4.  subst. contradiction.
  specialize (h1 _ _ h4 hadd h3).
  red in h1. rewrite h1. constructor.
  red. intros X h2.
  destruct h2; subst.
  unfold Int_Fam_Ens.
  apply Im_intro with B. assumption.
  assert (h2:A <> B). intro h3. subst. contradiction.
  pose proof (Add_intro1 _ _ A _ hin) as h3.
  specialize (h1 _ _ h2 hadd h3).
  red in h1. rewrite h1. reflexivity.
rewrite h2.
apply family_union_sing.
Qed.



(*Section Characteristic. Commented to keep the notation top-level*)

(*Schepler*)
Inductive characteristic_function_abstraction {X:Type} (P:X->Prop) (x:X) : Prop :=
  | intro_characteristic_sat: P x ->
    In (characteristic_function_abstraction P) x.

(*Schepler*)
Definition characteristic_function_to_ensemble {X:Type} (P:X->Prop) : Ensemble X :=
  characteristic_function_abstraction P.

(*Schepler*)
Notation "[ x : X | P ]" :=
  (characteristic_function_to_ensemble (fun x:X => P))
  (x ident).

(*Schepler*)
Lemma characteristic_function_to_ensemble_is_identity:
  forall {X:Type} (P:X->Prop),
    [ x:X | P x ] = P.
Proof.
intros.
apply Extensionality_Ensembles; split; red; intros.
destruct H.
exact H.
constructor.
exact H.
Qed.

(*Wyckoff*)
Lemma sat_nin :
  forall {T:Type} (P:T->Prop) (x:T),
    ~ In [t:T | P t] x -> ~P x.
intros T P x h1.
intro h2.
contradict h1. constructor.  assumption.
Qed.


(*Wyckoff*)
Lemma eq_sat : forall {T:Type} (A:Ensemble T),
                 A = [x:T | In A x].
intros T A.
rewrite characteristic_function_to_ensemble_is_identity.
reflexivity.
Qed.

(*Wyckoff*)
Lemma sat_iff : 
  forall {T:Type} (P Q:T->Prop),
    (forall x:T, P x <-> Q x) ->
    [x : T | P x] = [x : T | Q x].
intros T P Q h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. destruct h2 as [h2]. rewrite h1 in h2.
constructor; auto.
red. intros x h2. destruct h2 as [h2]. rewrite <- h1 in h2.
constructor; auto.
Qed.


(*Wyckoff*)
Definition power_set {T:Type} (A:Ensemble T) :=
  [S:Ensemble T | Included S A].

(*Wyckoff*)
Lemma power_set_empty : 
  forall (T:Type),
    power_set (Empty_set T) = Singleton (Empty_set T).
intro T. unfold power_set.
apply Extensionality_Ensembles.
red. split.
red. intros S h1. destruct h1 as [h1].
assert (h2:Included (Empty_set _) S). auto with sets.
assert (h3:S = Empty_set T). apply Extensionality_Ensembles; red; split; auto. subst. constructor.
red. intros S h1. destruct h1. constructor. auto with sets.
Qed.

(*Wyckoff*)
Lemma in_power_full : forall {T:Type} (A:Ensemble T),
                        In (power_set (Full_set T)) A.
intros T A.
constructor.
red; intros; constructor.
Qed.

(*Wyckoff*)
Lemma in_power_incl : 
  forall {T:Type} (A B X:Ensemble T),
    In (power_set X) B ->
    Included A B ->
    In (power_set X) A.
intros T A B X h1 h2.
constructor. destruct h1.
auto with sets.
Qed.


(*Wyckoff*)
Lemma in_power_intersection : 
  forall {T:Type} (A B X:Ensemble T),
    In (power_set X) A ->
    In (power_set X) (Intersection A B).
intros T A B X h1.
constructor. 
eapply (in_power_incl _ _ _ h1).
auto with sets.
Qed.


(*Wyckoff*)
Lemma in_power_union : 
  forall {T:Type} (A B X:Ensemble T),
    In (power_set X) A ->
    In (power_set X) B ->
    In (power_set X) (Union A B).
intros T A B X h1 h2.
destruct h1 as [h1]. destruct h2 as [h2].
constructor.
auto with sets.
Qed.

(*Wyckoff*)
Lemma in_power_comp : 
  forall {T:Type} (A X:Ensemble T),
    In (power_set X) A ->
    In (power_set X) (Setminus X A).
intros T A X h1.
constructor.
destruct h1 as [h1].
apply setminus_inc.
Qed.


(*Wyckoff*)
Definition sig_set {T:Type} (A:Ensemble T) :=
  {x:T | In A x}.

(*Wyckoff*)
Lemma sig_set_eq : forall {T:Type} (A B:Ensemble T),
                     A = B ->
                     sig_set A = sig_set B.
intros; subst; auto.
Defined.


(*Wyckoff*)
Definition full_sig {T:Type} (A:Ensemble T) :=
  Full_set (sig_set A).

(*Wyckoff*)
Definition full_set_sig_conv {T:Type} (x:T) : sig_set (Full_set T) := exist _ _ (Full_intro _ x).

Section Finite_Families.
(*Wyckoff*)
Lemma Finite_Finite_Union : forall {T:Type} (F:Family T), 
  (forall (S:(Ensemble T)), In F S -> Finite  S) ->
  Finite F -> Finite (FamilyUnion F).
intros T F h1 h2.
induction h2.
(*Empty*)
pose proof (empty_family_union T) as h3.
rewrite h3.
constructor.
(*Add*)
rename x into S.
rename A into F.
rename H into h3.
rename IHh2 into h4.
pose proof (family_union_add F S) as h5.
rewrite h5.
apply Union_preserves_Finite.
apply h1.
auto with sets.
assert (h6:forall S:Ensemble T, In F S -> Finite S).
  intros S0 h7.
  apply h1.
  auto with sets.
apply (h4 h6).
Qed.



  


(*Wyckoff*)
(* Singleton Family decomposition of a set *)
Definition SingF {T:Type} (S:Ensemble T) := Im S (@Singleton T).  

(*Wyckoff*)
Lemma union_singF_eq : forall {T:Type} (S:Ensemble T), 
  S = FamilyUnion (SingF S).
intros T S.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros x h1.
assert (h2:In (SingF S) (Singleton x)).
  unfold SingF.
  apply Im_intro with x; trivial.
pose proof (In_singleton _ x) as h3.
apply family_union_intro with (Singleton x); assumption.
(* >= *)
red. intros x h1.

inversion h1 as [S0 ? h2 h3].
unfold SingF in h2.
inversion h2 as [? h4 ? h5].
rewrite h5 in h3.
inversion h3 as [h6].
rewrite <- h6; trivial.
Qed.

(*Wyckoff*)
Lemma fin_fin_singf : forall {T:Type} (S:Ensemble T), Finite S 
  -> Finite (SingF S).
intros.
apply finite_image; assumption.
Qed.

End Finite_Families.



Section Relations.

(*Wyckoff*)
Definition dom_rel {T U:Type} (S:Ensemble (T*U)) : Ensemble T :=
  [x:T | exists y:U, In S (x, y)].

(*Wyckoff*)
Definition ran_rel {T U:Type} (S:Ensemble (T*U)) : Ensemble U :=
  [y:U | exists x:T, In S (x, y)].

(*Wyckoff*)
Lemma dom_rel_eq : forall {T U:Type} (S:Ensemble (T*U)),
                     dom_rel S = Im S (@fst _ _).  
intros T U S.
apply Extensionality_Ensembles.
red. split. 
red. intros x h1.
destruct h1 as [h1].
destruct h1 as [y h1].
apply Im_intro with (x, y); auto. 
red. intros x h1.
destruct h1 as [pr h1 x]. subst.
constructor.
exists (snd pr). rewrite (surjective_pairing pr) in h1.
assumption.
Qed.

(*Wyckoff*)
Lemma ran_rel_eq : forall {T U:Type} (S:Ensemble (T*U)),
                     ran_rel S = Im S (@snd _ _).  
intros T U S.
apply Extensionality_Ensembles.
red. split. 
red. intros y h1.
destruct h1 as [h1].
destruct h1 as [x h1].
apply Im_intro with (x, y); auto. 
red. intros x h1.
destruct h1 as [pr h1 y]. subst.
constructor.
exists (fst pr). rewrite (surjective_pairing pr) in h1.
assumption.
Qed.

(*Wyckoff*)
Lemma dom_rel_empty : forall (T U:Type),
                    dom_rel (Empty_set (T*U)) = Empty_set _.
intros T U.
rewrite dom_rel_eq.
apply image_empty.
Qed.

(*Wyckoff*)
Lemma ran_rel_empty : forall (T U:Type),
                    ran_rel (Empty_set (T*U)) = Empty_set _.
intros T U.
rewrite ran_rel_eq.
apply image_empty.
Qed.

(*Wyckoff*)
Lemma dom_rel_finite : 
  forall {T U:Type} (S:Ensemble (T*U)),
    Finite S -> Finite (dom_rel S).
intros T U S h1.
rewrite dom_rel_eq.
apply finite_image; auto.
Qed.

(*Wyckoff*)
Lemma ran_rel_finite : 
  forall {T U:Type} (S:Ensemble (T*U)),
    Finite S -> Finite (ran_rel S).
intros T U S h1.
rewrite ran_rel_eq.
apply finite_image; auto.
Qed.

(*Wyckoff*)
Lemma dom_rel_add : 
  forall {T U:Type} (S:Ensemble (T*U)) (pr:T*U),
    dom_rel (Add S pr) = Add (dom_rel S) (fst pr).
intros T U S pr. unfold dom_rel.
apply Extensionality_Ensembles.
red. split.
red. intros x h1. 
destruct h1 as [h1].
destruct h1 as [y h1]. 
inversion h1 as [pr' h1l | pr' h1r]. subst.
left. constructor. exists y. assumption.
inversion h1r. subst. simpl.
right. constructor.
red. intros x h1.
destruct h1 as [x h1l | x h1r].
constructor.
destruct h1l as [h1l].
destruct h1l as [y h1l].
exists y. left. assumption.
inversion h1r. subst. constructor.
exists (snd pr). right. rewrite surjective_pairing at 1.
constructor.
Qed.

(*Wyckoff*)
Lemma ran_rel_add : 
  forall {T U:Type} (S:Ensemble (T*U)) (pr:T*U),
    ran_rel (Add S pr) = Add (ran_rel S) (snd pr).
intros T U S pr. unfold dom_rel.
apply Extensionality_Ensembles.
red. split.
red. intros y h1. 
destruct h1 as [h1].
destruct h1 as [x h1]. 
inversion h1 as [pr' h1l | pr' h1r]. subst.
left. constructor. exists x. assumption.
inversion h1r. subst. simpl.
right. constructor.
red. intros y h1.
destruct h1 as [y h1l | y h1r].
constructor.
destruct h1l as [h1l].
destruct h1l as [x h1l].
exists x. left. assumption.
inversion h1r. subst. constructor.
exists (fst pr). right. rewrite surjective_pairing at 1.
constructor.
Qed.


(*Wyckoff*)
Lemma dom_rel_incl : 
  forall {T U:Type} (S S':Ensemble (T*U)),
    Included S S' -> Included (dom_rel S) (dom_rel S').
intros T U S S' h1.
red.
intros x h2.
destruct h2 as [h2]. destruct h2 as [y h2].
constructor. exists y. auto with sets.
Qed.

(*Wyckoff*)
Lemma ran_rel_incl : 
  forall {T U:Type} (S S':Ensemble (T*U)),
    Included S S' -> Included (ran_rel S) (ran_rel S').
intros T U S S' h1.
red.
intros y h2.
destruct h2 as [h2]. destruct h2 as [x h2].
constructor. exists x. auto with sets.
Qed.

End Relations.

Section CartProd.
(*Wyckoff*)
Definition cart_prod {T U:Type} (A:Ensemble T) (B:Ensemble U) :
  Ensemble (T*U) := [pr:T*U | 
                      In A (fst pr) /\ In B (snd pr)].

(*Wyckoff*)
Lemma in_cart_prod : forall {T U:Type} 
                            (A:Ensemble T) (B:Ensemble U)
                            (x:T) (y:U),
                       In A x -> In B y ->
                       In (cart_prod A B) (x, y).
intros A B x y h1 h2.
constructor.
simpl.
split; assumption.
Qed.

(*Wyckoff*)
Lemma in_cart_prod_comm : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) (pr:T*U),
    In (cart_prod A B) pr -> In (cart_prod B A) (snd pr, fst pr).
intros T U A B pr h1.
constructor.
simpl.
destruct h1 as [h1]. 
destruct h1; split; assumption.
Qed.
                     
(*Wyckoff*)
Lemma cart_prod_sing : 
  forall {T U:Type} (A:Ensemble T) (y:U), 
    cart_prod A (Singleton y) = 
    Im A (fun x:T => (x, y)).
  intros T U A y.
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros pr h1.
  unfold cart_prod in h1.
  destruct h1 as [h1].
  destruct h1 as [h1 h2].
  apply  Im_intro with (fst pr). assumption.
  pose proof (Singleton_inv _ _ _ h2) as h3.
  subst.
  apply surjective_pairing.
  (* >= *)
  red.
  intros pr h1.
  destruct h1 as [x h2 pr].
  subst.
  unfold cart_prod.
  constructor. split. simpl. assumption.
  simpl. constructor.
Qed.

  
(*Wyckoff*)
Lemma cart_prod_sing' : 
  forall {T U:Type} (B:Ensemble U) (x:T), 
    cart_prod (Singleton x) B = 
    Im B (fun y:U => (x, y)).
  intros T U B x.
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros pr h1.
  unfold cart_prod in h1.
  destruct h1 as [h1].
  destruct h1 as [h1 h2].
  apply  Im_intro with (snd pr). assumption.
  pose proof (Singleton_inv _ _ _ h1) as h3.
  subst.
  apply surjective_pairing.
  (* >= *)
  red.
  intros pr h1.
  destruct h1 as [z h2 pr].
  subst.
  unfold cart_prod.
  constructor. split. simpl. constructor.
  simpl. assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_sing_rev : 
  forall {T U:Type} (A:Ensemble T) (y:U), 
    A = Im (cart_prod A (Singleton y)) (fun pr:T*U => fst pr).
intros T U A y.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros x h1.
apply Im_intro with (x, y).
unfold cart_prod. constructor. simpl.
split; auto with sets. reflexivity.
(* >= *)
red. 
intros x h1.
destruct h1 as [pr h1]; subst.
destruct h1 as [h1].
destruct h1; assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_sing_rev' : 
  forall {T U:Type} (B:Ensemble U) (x:T), 
    B = Im (cart_prod (Singleton x) B) (fun pr:T*U => snd pr).
intros T U B x.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h1.
apply Im_intro with (x, y).
unfold cart_prod. constructor. simpl.
split; auto with sets. reflexivity.
(* >= *)
red. 
intros y h1.
destruct h1 as [pr h1]; subst.
destruct h1 as [h1].
destruct h1; assumption.
Qed.

(*Wyckoff*)
Lemma card_cart_prod_sing : 
  forall {T U:Type} (A:Ensemble T) (n:nat) (b:U),
    cardinal _ A n -> cardinal _ (cart_prod A (Singleton b)) n.
intros T U A n b h1.
rewrite cart_prod_sing.
pose proof (injective_preserves_cardinal).
pose proof (cardinal_finite _ _ _ h1) as h2.
pose proof (finite_image _ _ _ (fun x:T=>(x, b)) h2) as h3.
pose proof (finite_cardinal _ _ h3) as h4.
destruct h4 as [n' h4].
assert (h5:Image.injective _ _ (fun x:T=>(x, b))).
  red. intros x' y' h5. injection h5. auto.
pose proof (injective_preserves_cardinal _ _ _ _ _ h5 h1 _ h4) as h6. 
subst.
assumption.
Qed.





(*Wyckoff*)
Lemma cart_prod_sing_fin : forall {T U:Type} (A:Ensemble T) (y:U),
                             Finite A ->
                             Finite (cart_prod A (Singleton y)).
intros T U A y h1.
pose proof (cart_prod_sing A y) as h2.
rewrite h2.
apply finite_image.
assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_sing_fin' : forall {T U:Type} (B:Ensemble U) (x:T),
                             Finite B ->
                             Finite (cart_prod (Singleton x) B).
intros T U B x h1.
pose proof (cart_prod_sing' B x) as h2.
rewrite h2.
apply finite_image.
assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_sing_fin_rev : forall {T U:Type} (A:Ensemble T) (y:U),
                                 Finite (cart_prod A (Singleton y)) ->                            
                                 Finite A.
                            
intros T U A y h1.
pose proof (cart_prod_sing_rev A y) as h2.
rewrite h2.
apply finite_image.
assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_sing_fin_rev' : forall {T U:Type} (B:Ensemble U) (x:T),
                                 Finite (cart_prod (Singleton x) B) ->                            
                                 Finite B.
                            
intros T U B x h1.
pose proof (cart_prod_sing_rev' B x) as h2.
rewrite h2.
apply finite_image.
assumption.
Qed.


(*Wyckoff*)
Lemma card_cart_prod_sing' : 
  forall {T U:Type} (A:Ensemble T),
    Finite A -> forall (b:U),
    card_fun1 (cart_prod A (Singleton b)) = card_fun1 A.
intros T U A h1 b.
pose proof (finite_cardinal _ _ h1) as h2.
destruct h2 as [n h2].
pose proof (card_cart_prod_sing A n b h2) as h3.
pose proof (card_fun1_compat (cart_prod A (Singleton b))) as h4.
pose proof (card_fun1_compat A) as h5.
destruct h4 as [h4l h4r].
destruct h5 as [h5l h5r].
specialize (h5l h1).
pose proof (cart_prod_sing_fin _ b h1) as h6.
specialize (h4l h6).
pose proof (cardinal_is_functional _ _ _ h2 _ _ h5l (eq_refl)). subst.
pose proof (cardinal_is_functional _ _ _ h3 _ _ h4l (eq_refl)). symmetry.
assumption.
Qed.


(*Wyckoff*)
Lemma cart_prod_inc_add1 : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) (x:T),
    Included (cart_prod A B) (cart_prod (Add A x) B).
intros T U A B x.
red. intros pr h1.
destruct h1 as [h1].
destruct h1 as [h1l h1r].
constructor.
split; try assumption.
left. assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_inc_add2 : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U) (y:U),
    Included (cart_prod A B) (cart_prod A (Add B y)).
intros T U A B y.
red. intros pr h1.
destruct h1 as [h1].
destruct h1 as [h1l h1r].
constructor.
split; try assumption.
left. assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_eq : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U), 
    cart_prod A B =
    FamilyUnion 
      [S:Ensemble (T*U) | 
       exists y:U,
                In B y /\ S = cart_prod A (Singleton y)].
intros T U A B.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros pr h1.
destruct h1 as [h1l h1r].
destruct h1l as [h1ll h1lr].
assert (h2:In (cart_prod A (Singleton (snd pr))) pr).
  rewrite cart_prod_sing.
  apply Im_intro with (fst pr). assumption.
  apply surjective_pairing.
  apply family_union_intro with 
  (cart_prod A (Singleton (snd pr))).
  constructor.
  exists (snd pr). 
  split; [assumption | reflexivity]. assumption.
(* >= *)
red. intros pr h1.
destruct h1 as [S pr h1 h2].
destruct h1 as [h1].
destruct h1 as [y h1].
destruct h1 as [h1l h1r].
subst.
rewrite cart_prod_sing in h2.
destruct h2 as [x].
subst.
constructor; split; assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_eq' : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U), 
    cart_prod A B =
    FamilyUnion 
      [S:Ensemble (T*U) | 
       exists x:T,
                In A x /\ S = cart_prod (Singleton x) B].

intros T U A B.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros pr h1.
destruct h1 as [h1l h1r].
destruct h1l as [h1ll h1lr].
assert (h2:In (cart_prod (Singleton (fst pr)) B) pr).
  rewrite cart_prod_sing'.
  apply Im_intro with (snd pr). assumption.
  apply surjective_pairing.
  apply family_union_intro with 
  (cart_prod (Singleton (fst pr)) B).
  constructor.
  exists (fst pr). 
  split; [assumption | reflexivity]. assumption.
(* >= *)
red. intros pr h1.
destruct h1 as [S pr h1 h2].
destruct h1 as [h1].
destruct h1 as [y h1].
destruct h1 as [h1l h1r].
subst.
rewrite cart_prod_sing' in h2.
destruct h2 as [x].
subst.
constructor; split; assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_empty : forall {T U:Type} (B:Ensemble U),
                          cart_prod (Empty_set T) B =
                          Empty_set _.
intros T U B.
apply Extensionality_Ensembles.
red. split.
red.
intros pr h1.
unfold cart_prod in h1.
destruct h1 as [h1].
destruct h1; contradiction.
auto with sets.
Qed.

(*Wyckoff*)
Lemma cart_prod_empty_empty : forall {T U:Type} (B:Ensemble U),
                          cart_prod (Empty_set T) B =
                          cart_prod (Empty_set _) (Empty_set _).
intros T U B.
rewrite cart_prod_empty.
rewrite <- (cart_prod_empty (Empty_set _)).
reflexivity.
Qed.

(*Wyckoff*)
Lemma cart_prod_empty' : forall {T U:Type} (A:Ensemble T),
                          cart_prod A (Empty_set U)=
                          Empty_set _.
intros T U A.
apply Extensionality_Ensembles.
red. split.
red.
intros pr h1.
unfold cart_prod in h1.
destruct h1 as [h1].
destruct h1; contradiction.
auto with sets.
Qed.

(*Wyckoff*)
Lemma cart_prod_empty_rev : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
    cart_prod A B = Empty_set _ -> 
    (A = Empty_set _ \/ B = Empty_set _).
intros T U A B h1.
apply NNPP.
intro h2.
pose proof (not_or_and _ _ h2) as h3.
destruct h3 as [h3 h4].
pose proof (not_empty_Inhabited _ _ h3) as h5.
pose proof (not_empty_Inhabited _ _ h4) as h6.
destruct h5 as [a h5].
destruct h6 as [b h6].
assert (h7:In (cart_prod A B) (a, b)).
  constructor. simpl. split; assumption.
rewrite h1 in h7.
contradiction.
Qed.

(*Wyckoff*)
Lemma cart_prod_empty_empty' : forall {T U:Type} (A:Ensemble U),
                          cart_prod A (Empty_set U) =
                          cart_prod (Empty_set _) (Empty_set _).
intros T U B.
rewrite cart_prod_empty'.
rewrite <- (cart_prod_empty' (Empty_set _)).
reflexivity.
Qed.

(*Wyckoff*)
Lemma add_ex_family : 
  forall {T U:Type} (A:Ensemble U) (a:U)
         (f:U -> Ensemble T),
    [S:Ensemble T | exists u:U, Ensembles.In (Add A a) u /\ S = f u] =
    Add [S:Ensemble T | exists u:U, Ensembles.In A u /\ S = f u]
        (f a).
  intros T U A a f.
    apply Extensionality_Ensembles.
    red. split.
    (* <= *)
    red.
    intros S h6.
    destruct h6 as [h6].
    destruct h6 as [y h6].
    destruct h6 as [h6l h6r].
    destruct h6l as [y h6la | y h6lb].
    left.
    constructor. exists y. split; assumption.
    destruct h6lb.
    right. rewrite h6r.
    constructor.
    (* >= *)
    red.
    intros S h6.
    destruct h6 as [S h6 | S h7'].
    destruct h6 as [h6].
    destruct h6 as [y h6].
    destruct h6 as [h6a h6b].
    constructor.
    exists y.
    split. left. assumption. assumption.
    destruct h7'.
    constructor.
    exists a. split.
    right. constructor. reflexivity.
Qed.

(*Wyckoff*)
Lemma cart_prod_fin : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U), 
    Finite A -> Finite B -> Finite (cart_prod A B).
  intros T U A B h1.
  induction h1.
    intros h2. rewrite cart_prod_empty.
    constructor.
  intro h2.
  rewrite cart_prod_eq.
  apply Finite_Finite_Union.
  intros S h3.
  destruct h3 as [h3].
  destruct h3 as [y h3].
  destruct h3 as [h3l h3r].
  rewrite h3r.
  apply cart_prod_sing_fin.
  apply Add_preserves_Finite. assumption.
  induction h2 as [|B h3 h4 z h5].
  assert (h3: [S : Ensemble (T * U)
     | exists y : U, In (Empty_set U) y /\ S = cart_prod (Add A x) (Singleton y)] = Empty_set _ ).
  apply Extensionality_Ensembles.
  red. split.
  red.
  intros S h2. destruct h2 as [h2].
  destruct h2 as [y h2].
  destruct h2; contradiction.
  auto with sets.
  rewrite h3.
  constructor.
  pose proof (Add_preserves_Finite _ B z h3) as h0.
  specialize (IHh1 h0).
  rewrite cart_prod_eq in IHh1.
  assert (hs : Included (FamilyUnion [S:Ensemble (T*U) |
                            exists y:U, In B y /\ 
                                        S = cart_prod A (Singleton y)]) (FamilyUnion
              [S : Ensemble (T * U)
              | exists y : U, In (Add B z) y /\ S = cart_prod A (Singleton y)])).
    red. intros pr h6.
    destruct h6 as [S pr h7].
    apply family_union_intro with S.
    destruct h7 as [h7].
    destruct h7 as [y h7].
    constructor. exists y.
    split.
    left. tauto. tauto. assumption.
  pose proof (Finite_downward_closed _ _ IHh1 _ hs) as h7.
  rewrite <- cart_prod_eq in h7.
  assert (h8:Finite B -> Finite (cart_prod A B)).
    tauto.
  specialize (h4 h8).
  pose proof (@add_ex_family).
  pose proof (add_ex_family B z (fun u:U => cart_prod (Add A x)
                                                      (Singleton u))) as h6.
  simpl in h6.
  rewrite h6.
  constructor. assumption.
  intro h9.
  destruct h9 as [h9].
  destruct h9 as [a h9].
  destruct h9 as [h9a h9b].
  assert (h10:In (cart_prod (Add A x) (Singleton z)) (x, z)).
    constructor.
    simpl. split.
    right. constructor.
    constructor.
  rewrite h9b in h10.
  destruct h10 as [h10].
  destruct h10 as [h10a h10b].
  simpl in h10b.
  destruct h10b.
  contradiction.
Qed.

(*Wyckoff*)
Lemma cart_prod_fin_rev1 : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U), 
    Finite (cart_prod A B) -> Inhabited B ->
    Finite A.
intros T U A B h1 h2.

dependent induction h1.
destruct h2 as [y h2].

assert (h3:A = Empty_set _).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros z h1.
  assert (h3:In (cart_prod A B) (z, y)). 
    constructor. simpl. split; assumption.
  rewrite <- x in h3.
  contradiction.
  (* >= *)
  auto with sets.
rewrite h3. constructor.
pose proof (Add_preserves_Finite _ _ x0 h1) as h3.
rewrite x in h3.
rewrite cart_prod_eq in h3.
assert (h4:Included (cart_prod A (Singleton (snd x0)))
(FamilyUnion
            [S : Ensemble (T * U)
            | exists y : U, In B y /\ S = cart_prod A (Singleton y)])).
  red.
  intros pr h4.
  apply family_union_intro with (cart_prod A (Singleton (snd x0))).
  constructor.
  exists (snd x0).
  split.
  pose proof (Add_intro2 _ A0 x0) as h5.
  rewrite x in h5.
  destruct h5 as [h5]. destruct h5; assumption.
  reflexivity.
  assumption.
pose proof (Finite_downward_closed _ _ h3 _ h4) as h5.
apply cart_prod_sing_fin_rev with (snd x0).
assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_fin_rev2 : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U), 
    Finite (cart_prod A B) -> Inhabited A ->
    Finite B.
intros T U A B h1 h2.

dependent induction h1.
destruct h2 as [a h2].

assert (h3:B = Empty_set _).
  apply Extensionality_Ensembles.
  red. split.
  (* <= *)
  red.
  intros b h1.
  assert (h3:In (cart_prod A B) (a, b)). 
    constructor. simpl. split; assumption.
  rewrite <- x in h3.
  contradiction.
  (* >= *)
  auto with sets.
rewrite h3. constructor.
pose proof (Add_preserves_Finite _ _ x0 h1) as h3.
rewrite x in h3.
rewrite cart_prod_eq' in h3.
assert (h4:Included (cart_prod (Singleton (fst x0)) B)
(FamilyUnion
            [S : Ensemble (T * U)
            | exists x1 : T, In A x1 /\ S = cart_prod (Singleton x1) B])).
  red.
  intros pr h4.
  apply family_union_intro with (cart_prod (Singleton (fst x0)) B).
  constructor.
  exists (fst x0).
  split.
  pose proof (Add_intro2 _ A0 x0) as h5.
  rewrite x in h5.
  destruct h5 as [h5]. destruct h5; assumption.
  reflexivity.
  assumption.
pose proof (Finite_downward_closed _ _ h3 _ h4) as h5.
apply cart_prod_sing_fin_rev' with (fst x0).
assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_fin_rev : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U), 
    Finite (cart_prod A B) -> Inhabited (cart_prod A B) ->
    Finite A /\ Finite B.
intros T U A B h1 h2.
destruct h2 as [pr h2].
destruct h2 as [h2].
destruct h2 as [h2l h2r].
pose proof (Inhabited_intro _ _ _ h2l) as h3.
pose proof (Inhabited_intro _ _ _ h2r) as h4.
split.
apply cart_prod_fin_rev1 with B; assumption.
apply cart_prod_fin_rev2 with A; assumption.
Qed.

(*Wyckoff*)
Lemma cart_prod_fin_rev_or : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
    Finite (cart_prod A B) ->
    Finite A \/ Finite B.
intros T U A B h1.
destruct (classic (Inhabited A)) as [h2 | h3].
right.
apply cart_prod_fin_rev2 with A; assumption.
left.
assert (h4:A = Empty_set _).
  apply Extensionality_Ensembles;split. 
  red. intros x h4. 
  pose proof (Inhabited_intro _ A _ h4).
  contradiction. 
  auto with sets. 
rewrite h4.
constructor.
Qed.

(* See TypeUtilties for 
Lemma cart_prod_fin_comm : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
          Finite (cart_prod A B) ->
          Finite (cart_prod B A).*)

(*Wyckoff*) 
Lemma cart_prod_proj1_surj : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
    let f := (fun pr:T*U => fst pr) in
    Inhabited B -> Im (cart_prod A B) f = A.
intros T U A B f h1.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h2.
destruct h2 as [x h2].
subst.
destruct h2 as [h2].
destruct h2 as [h2l h2r].
unfold f. simpl.
assumption.
(* >= *)
red.
intros x h2.
unfold f.
destruct h1 as [y h1].
apply Im_intro with (x, y).
split; simpl; split; assumption.
simpl. reflexivity.
Qed.

(*Wyckoff*)
Lemma cart_prod_proj2_surj : 
  forall {T U:Type} (A:Ensemble T) (B:Ensemble U),
    let f := (fun pr:T*U => snd pr) in
    Inhabited A -> Im (cart_prod A B) f = B.
intros T U A B f h1.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros y h2.
destruct h2 as [x h2].
subst.
destruct h2 as [h2].
destruct h2 as [h2l h2r].
unfold f. simpl.
assumption.
(* >= *)
red.
intros y h2.
unfold f.
destruct h1 as [x h1].
apply Im_intro with (x, y).
split; simpl; split; assumption.
simpl. reflexivity.
Qed.


(*Wyckoff*)
Lemma cart_prod_im : 
  forall {T U V W:Type} 
         (A:Ensemble T) (B:Ensemble U) (f:T->V) (g:U->W),
         cart_prod (Im A f) (Im B g) = 
         Im (cart_prod A B) (fun pr => (f (fst pr), g (snd pr))).
intros T U V W A B f g.
apply Extensionality_Ensembles.

red. split.
(* <= *)
red.
intros pr h1.
inversion h1 as [h2]. clear h1.
destruct h2 as [h2l h2r].
inversion h2l as [x h3 v h4 h5]. clear h2l.
inversion h2r as [y h6 w h7 h8]. clear h2r.
rewrite (surjective_pairing pr).
subst. rewrite h4. rewrite h7.
apply Im_intro with (x, y). constructor; auto.
simpl. reflexivity.
(* >= *)
red. intros pr h1.
inversion h1 as [prx  h3 pry  h4 h5]. clear h1. 
subst.
rewrite (surjective_pairing prx) in h3.
destruct h3 as [h3].
simpl in h3.
destruct h3 as [h3l h3r].
constructor. simpl. split.
apply Im_intro with (fst prx).  assumption. reflexivity.
apply Im_intro with (snd prx).  assumption. reflexivity.
Qed.

(*Wyckoff*)
Lemma full_set_prod : 
  forall {T U:Type}, Full_set (T*U) = cart_prod (Full_set T) (Full_set U).
intros T U.
apply Extensionality_Ensembles.
red. split.
red. intros pr h1.
constructor. split; constructor.
red. intros pr h1. 
constructor.
Qed.


End CartProd.
 



(*Wyckoff*)
Lemma surj_iff : forall {T U:Type} (f:T->U), surjective f <-> Im (Full_set T) f = 
  Full_set U.
intros T U f. 
split.
(* -> *)
intro h1.
apply Extensionality_Ensembles.
red. split.
  (* <= *)
  red.
  intros x ?. constructor.
  (* >= *)
  red.
  intros u ?.
  red in h1.
  pose proof (h1 u) as h2.
  destruct h2 as [t].
  apply Im_intro with t.
  constructor.
  symmetry; assumption.
(* <- *)
intro h1.
red.
intro u.
pose proof (Full_intro _ u) as h2.
rewrite <- h1 in h2.
inversion h2 as [t].
exists t. symmetry. assumption.
Qed.



(*Wyckoff*)
Lemma not_inhabited_empty : forall {T:Type} (S:Ensemble T),
                              ~ Inhabited S -> S = Empty_set _.
intros T S h1. 
apply Extensionality_Ensembles.
red. split.
red.
intros x h2.
pose proof (Inhabited_intro _ _ _ h2). contradiction.
auto with sets.
Qed.


(*Wyckoff*) 
Lemma empty_alt :
  forall {T:Type} (A:Ensemble T),
    (forall x, ~In A x) -> A = Empty_set _.
intros T A h1. red in h1.
destruct (classic (Inhabited A)) as [h2 | h3]. 
destruct h2 as [x h2].
contradict h1; auto.
intro h1.  apply (h1 _ h2).
apply not_inhabited_empty; auto.
Qed.


Lemma inhabited_family_cond :
  forall {T:Type} (F:Family T),
         inhabited_family F ->
  forall A:sig_set F, exists y:T, In (proj1_sig A) y.
intros T F h1 A.
destruct h1 as [h1 h2].
destruct A as [A h3].
simpl.
apply h2 in h3.
destruct h3 as [y h3].
exists y; auto.
Qed.




Section RandomFacts.

(*Wyckoff*)
Lemma  bool_couple : Full_set bool = Couple true false.
apply Extensionality_Ensembles.
red. split. red. intros x ?. destruct x. left. right.
red. intros; constructor.
Qed.

(*Wyckoff*)
Lemma cardinal_couple : forall {T:Type} (x y:T),
                             x <> y ->
                             cardinal _ (Couple x y) 2.
intros T x y h1.
rewrite couple_add_sing.
constructor. apply card_sing. intro h2. destruct h2; auto.
Qed.

Lemma cardinal_2 : 
  forall {T:Type} (A:Ensemble T),
    cardinal _ A 2 ->
    exists (x y:T), x <> y /\ A = Couple x y.
intros T A h1. 
assert (h2:Inhabited A).
  apply not_empty_Inhabited.
  intro h2.
  subst. inversion h1. subst.
  pose proof (Add_intro2 _ A x) as h3. rewrite H in h3. contradiction.
destruct h2 as [x h2].
pose proof (cardinal_finite _ _ _ h1) as h4.
pose proof (card_sub_in A h4 x h2) as h5.
assert (ha:Finite (Subtract A x)).
  pose proof (incl_subtract A x) as h6.
  apply (Finite_downward_closed _ _ h4 _ h6).
pose proof (card_fun1_compat A) as h6. simpl in h6.
destruct h6 as [h6 h7]. clear h7.
specialize (h6 h4).
assert (h3:Inhabited (Subtract A x)).
  apply not_empty_Inhabited.
  intro h3.
  rewrite h3 in h5.
  rewrite card_fun1_empty in h5.
  rewrite h5 in h6.
  pose proof (cardinal_unicity _  _ _ h1  _ h6). omega.
destruct h3 as [y h3]. 
pose proof (card_sub_in (Subtract A x) ha y h3) as hb.
rewrite hb in h5.
pose proof (card_fun1_compat (Subtract A x)) as h6'. simpl in h6'.
destruct h6' as [h6' h7']. clear h7'.
specialize (h6' ha).

exists x, y. split.
destruct h3.
intro h8. subst.
contradict H0. constructor.
pose proof (cardinal_unicity _ _ _ h1 _ h6) as h7.
rewrite h5 in h7.
assert (h8:card_fun1 (Subtract (Subtract A x) y) = 
           0). omega.
apply card_fun1_O in h8.
destruct h8 as [h8l | h8r].
apply Extensionality_Ensembles.
red. split.
red. intros a h4'.
destruct (eq_dec a x), (eq_dec a y); subst. 
left. left. right.
assert (h5':In (Setminus A (Couple x y)) a).
  constructor; auto.
  intro h5'. destruct h5'; auto. 
rewrite setminus_couple_eq in h5'.
rewrite h8l in h5'. contradiction.
red.
intros a h9.
destruct h9. assumption.
destruct h3; auto.
pose proof (add_preserves_infinite _ y h8r) as h10.
rewrite add_sub_compat_in in h10; auto.
contradiction.
Qed.


(*Wyckoff*)
Lemma finite_couple : forall {T:Type} (x y:T),
                        Finite (Couple x y).
intros T x y.
destruct (eq_dec x y) as [he | hne].
subst.
rewrite couple_singleton.
apply Singleton_is_finite.
eapply cardinal_finite.
apply cardinal_couple; auto.
Qed.

(*Wyckoff*)
Lemma finite_bool : Finite (Full_set bool).
rewrite bool_couple.
apply finite_couple.
Qed.

(*Wyckoff*)
Lemma card_fun1_couple : 
  forall {T:Type} (x y:T),
    x <> y ->
    card_fun1 (Couple x y) = 2.
intros T x y h1.
pose proof (card_fun1_compat (Couple x y)) as h2.
destruct h2 as [h2l h2r].
specialize (h2l (finite_couple x y)).
pose proof (cardinal_couple _ _ h1) as h3.
eapply cardinal_is_functional; auto.
apply h2l. apply h3.
Qed.





(*Wyckoff*)
Lemma im_full_sig_proj1_sig : forall {T:Type} (S:Ensemble T), 
  S = Im (full_sig S) (@proj1_sig _ _).
intros T S.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros x h1.
apply Im_intro with (exist _ x h1).
constructor. simpl. reflexivity.
(* >= *)
red. intros x h1.
inversion h1 as [x' h2 ? h3 h4].
rewrite h3.
apply proj2_sig.
Qed.

(*Wyckoff*)
Lemma im_id : forall {T:Type} (S:Ensemble T),
                S = Im S id.
intros T S.
apply Extensionality_Ensembles.
red. split.
red. intros x h1. apply Im_intro with x; auto.
red. intros x h1. destruct h1 as [x h1]. subst. unfold id.
assumption.
Qed.

(*Wyckoff*)
Lemma im_ext_in : 
  forall {T U:Type} (A:Ensemble T) (f g:T->U),
    (forall x:T, In A x -> f x = g x) ->
    Im A f = Im A g.
intros T U A f g h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. subst.
rewrite h1; auto. apply Im_intro with x; auto.
red. intros x h2.
destruct h2 as [x h2]. subst.
rewrite <- h1; auto. apply Im_intro with x; auto.
Qed.


(*Wyckoff*)
Definition im_proj2_sig 
           {T:Type} {P:T->Prop} (A:Ensemble T)  
           (pfx:(forall x:T, Ensembles.In A x -> P x)) :
  Ensemble {x | P x} :=
 Im (full_sig A) (fun p => (exist _ _ (pfx _ (proj2_sig p)))).


(*Wyckoff*)
Definition im_in {T U:Type} (A:Ensemble T) 
           (P:T->Prop) (f:{x:T|P x}->U)
           (pfx:(forall x:T, Ensembles.In A x -> P x)) : 
  Ensemble U :=
  Im (full_sig A) 
     (fun x:{t:T|In A t}=>f (exist _ (proj1_sig x) (pfx _ (proj2_sig x)))).  


(*Wyckoff*)
(*assigns image to range*)
Definition sig_im_fun {T U:Type} (f:T->U) : T->sig_set (Im (Full_set T) f) :=
  fun x => exist _ _ (Im_intro _ _ _  f _ (Full_intro _ x) _ (@eq_refl _ _)).


(*Wyckoff*)
Lemma surj_sig_im_fun : 
  forall {T U:Type} (f:T->U),
    surjective (sig_im_fun f).
intros T U f. red. intro y. destruct y as [y h1].
destruct h1 as [x]. subst.
exists x. unfold sig_im_fun.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma inj_bij_sig_im_fun_iff :
  forall {T U:Type} (f:T->U),
    injective f <-> bijective (sig_im_fun f).
intros T U f.
split.
intro h1. red. split. red in h1. red.
intros a b h2. unfold sig_im_fun in h2. apply exist_injective in h2.
apply h1; auto.
apply surj_sig_im_fun.
intro h1. red in h1. destruct h1 as [h1l h1r].
red in h1l. red.
intros a b h2. specialize (h1l a b).
apply h1l. unfold sig_im_fun. apply proj1_sig_injective.
simpl.
assumption.
Qed.


(*Wyckoff*)
Definition extends_sig {T U:Type} {A B:Ensemble T}
           (f:sig_set B->U) (g:sig_set A->U) :=
  exists (pf:Included A B),
    forall x:sig_set A,
      g x = f (exist _ _ (pf _ (proj2_sig x))).

(*Wyckoff*)
Lemma refl_extends_sig : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U),
    extends_sig f f.
intros T U A f.
red. exists (inclusion_reflexive _ ).
intro x.
destruct x. simpl. f_equal. apply proj1_sig_injective.
simpl.
reflexivity.
Qed.                               



(*Wyckoff*)
Definition extends_sig1 {T U:Type} {A:Ensemble T} 
           (f:T->U) (g:sig_set A->U)  : Prop :=
  forall x:sig_set A, g x = f (proj1_sig x).

(*Wyckoff*)
(*[prim] fefers to the primitive type of the domain, as opposed
  to assuming a priori that the extended function's domain is 
  a sigma type.*)
Definition extends_prim {T T' U:Type} (f:T->U) (g:T'->U) : Prop :=
  exists A:Ensemble T, 
    T' = sig_set A /\
    forall (x:T) (x':T') (pf:In A x),
      existT id _ x' = existT id _ (exist _ _ pf) ->
      f x = g x'.


(*Wyckoff*)
Lemma extends_sig1_impl_extends_prim : 
  forall {T U:Type} {A:Ensemble T}
         (f:T->U) (g:sig_set A->U),
    extends_sig1 f g -> extends_prim f g.
intros T U A f g h1.
red in h1. red.
exists A. split; auto.
intros x x' h2 h3.
specialize (h1 (exist _ _ h2)).
simpl in h1.
rewrite <- h1.
f_equal.
apply inj_pair2 in h3. symmetry.
assumption.
Qed.

(*See TypeUtilities for the converse of the above*)

(*Wyckoff*)
Definition common_extension2 {T1 T2 T U:Type}
           (f:T->U) (g1:T1->U) (g2:T2->U) : Prop :=
  extends_prim f g1 /\ extends_prim f g2.

(*Wyckoff*)
Definition fam_fun_one_ran {U:Type} :=
  Ensemble {T:Type & T->U}.



(*Wyckoff*)
Definition common_extension_fam {T U:Type} (f:T->U) 
           (F:Ensemble {V:Type & V->U}) :=
  forall g, In F g -> extends_prim f (projT2 g).         

(*Wyckoff*)
Lemma common_extension2_fam_compat_iff : 
  forall {T1 T2 T U:Type}
         (f:T->U) (g1:T1->U) (g2:T2->U),
    common_extension2 f g1 g2 <->
    common_extension_fam f (Couple (existT _ _ g1) (existT _ _ g2)).
intros T1 T2 T U f g1 g2.
split.
intro h1.
red in h1. red. intros g h2. 
destruct h1 as [h1a h1b]. destruct h2; auto.
intro h2.
red in h2. red.  
pose proof (h2 (existT _ _ g1) (Couple_l _ _ _)) as h3. 
pose proof (h2 (existT _ _ g2) (Couple_r _ _ _)) as h4.
simpl in h3. simpl in h4.
split; auto.
Qed.

(*Wyckoff*)
Definition directed_fun_fam {U:Type} 
           (F:Ensemble {V:Type & V->U}) : Prop :=
  forall f g, In F f -> In F g -> 
              exists k, In F k /\
                        common_extension2 (projT2 k) (projT2 f) 
                                          (projT2 g).


(*Wyckoff*)
Definition agree_on {T U:Type} (f g:T->U)
           (A:Ensemble T) : Prop :=
  forall x:T, In A x -> f x = g x.


(*Wyckoff*)
Definition char_fun {T:Type} (A:Ensemble T)
           : T->bool := 
  fun x:T => if (classic_dec (In A x)) then true else false.


(*Wyckoff*)
Lemma char_fun_int : 
  forall {T:Type} (A B:Ensemble T) (x:T),
    char_fun (Intersection A B) x = char_fun A x && char_fun B x.
intros T A B x.  
pose proof (in_intersection_iff A B x) as h1.
unfold char_fun.
destruct (classic_dec (In (Intersection A B) x)) as [h2 | h3].
rewrite h1 in h2.
destruct h2.
destruct (classic_dec (In A x)); destruct (classic_dec (In B x));
try reflexivity; try contradiction.
rewrite h1 in h3.
apply not_and_or in h3.
destruct h3 as [h3 | h4].
destruct (classic_dec (In A x)); destruct (classic_dec (In B x)); 
  try reflexivity; try contradiction.
destruct (classic_dec (In A x)); destruct (classic_dec (In B x)); 
  try reflexivity; try contradiction.
Qed.

(*Wyckoff*)
Lemma char_fun_union : 
  forall {T:Type} (A B:Ensemble T) (x:T),
    char_fun (Union A B) x = char_fun A x || char_fun B x.
intros T A B x.  
unfold char_fun.
destruct (classic_dec (In (Union A B) x)) as [h2 | h3].
destruct h2; destruct (classic_dec (In A x)); destruct (classic_dec (In B x));
try reflexivity; try contradiction. 
rewrite in_union_iff in h3.
apply not_or_and in h3.
destruct h3.
destruct (classic_dec (In A x)); destruct (classic_dec (In B x));
try reflexivity; try contradiction.
Qed.

(*Wyckoff*)
Lemma char_fun_comp : 
  forall {T:Type} (A:Ensemble T) (x:T),
    char_fun (Ensembles.Complement A) x = negb (char_fun A x).
intros T A x.
unfold char_fun.
destruct (classic_dec (In A x)) as [h1 | h2].
assert (h2:~In (Ensembles.Complement A) x). intro h2. destruct h2.
assumption.
destruct (classic_dec (In (Ensembles.Complement A) x)); try reflexivity; try contradiction.
assert (h3:In (Ensembles.Complement A) x). assumption.
destruct (classic_dec (In (Ensembles.Complement A) x)); try reflexivity; try contradiction.
Qed.


(*Wyckoff*)
Lemma feq_im_aux : forall {T U:Type} (A B:Ensemble T) (f:T->U),
                 A = B -> Included (Im A f) (Im B f).
intros T U A B f h1.
red.
intros y h2.
destruct h2; subst.
apply Im_intro with x; auto.
Qed.

(*Wyckoff*)
Lemma feq_im : forall {T U:Type} (A B:Ensemble T) (f:T->U),
                 A = B -> Im A f = Im B f.
intros T U A B f h1.
apply Extensionality_Ensembles; red; split; apply feq_im_aux; auto.
Qed.

(*Wyckoff*)
Lemma im_inj_inj : forall {T U:Type} (A B:Ensemble T) (f:T->U),
                     FunctionProperties.injective f ->
                     Im A f = Im B f -> A = B.
intros T U A B f h1 h2.
apply Extensionality_Ensembles.
red. split.
red. intros x h3.
assert (h4:In (Im A f) (f x)). apply Im_intro with x; auto.
rewrite h2 in h4.
inversion h4 as [y h5 ha h6]. subst.
specialize (h1 _ _ h6). subst.
assumption.
red. intros x h3.
assert (h4:In (Im B f) (f x)). apply Im_intro with x; auto.
rewrite <- h2 in h4.
inversion h4 as [y h5 ha h6]. subst.
specialize (h1 _ _ h6). subst.
assumption.
Qed.

                          
(*Wyckoff*)
Lemma im_union : forall {T U:Type} (A B:Ensemble T) (f:T->U),
                      Im (Union A B) f = Union (Im A f) (Im B f).
intros T U A B f.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red.
intros x h1.
destruct h1 as [x h1 y h3].
subst.
destruct h1 as [x h1a | x h1b].
left. apply Im_intro with x; auto.

right. apply Im_intro with x; auto.
(* >= *)
red. 
intros x h1. 
destruct h1 as [x h1a | x h1b].
destruct h1a as [x h1a y h2]. subst.
apply Im_intro with x. left; auto. reflexivity.
destruct h1b as [x h1b y h2]. subst.
apply Im_intro with x; try auto; right; auto.
Qed.                       

(*Wyckoff*)
Lemma im_intersection_incl : 
  forall {T U:Type} (A B:Ensemble T) (f:T->U),
    Included (Im (Intersection A B) f)
             (Intersection (Im A f) (Im B f)).
intros T U A B f.
red. 
intros b h1. 
destruct h1 as [x h1]. subst.
destruct h1 as [x h1l h1r].
constructor.
apply Im_intro with x; auto. apply Im_intro with x; auto.
Qed.


(*Wyckoff*)
Lemma im_intersection_inj : 
  forall {T U:Type} (A B:Ensemble T) (f:T->U),
    injective f ->
    Im (Intersection A B) f = Intersection (Im A f) (Im B f).
intros T U A B f h0.
apply Extensionality_Ensembles.
red. split.
apply im_intersection_incl.
red. intros b h1.
destruct h1 as [b h1 h2].
destruct h1 as [b h1]. subst.
apply Im_intro with b.
inversion h2 as [a h3 c h4]. subst. clear h2.
red in h0. apply h0 in h4. subst.
constructor; auto.
reflexivity.
Qed.


(*Wyckoff*)
Lemma im_setminus : 
  forall {T U:Type} (A B:Ensemble T) (f:T->U),
    Included (Im (Setminus A B) f) (Intersection (Im A f) (Im (Ensembles.Complement B) f)).
intros T U A B f.
red. intros x h1.
destruct h1 as [x h1]. subst. destruct h1 as [h1 h2].
constructor. apply Im_intro with x; auto.
apply Im_intro with x. auto. reflexivity.
Qed.


(*Wyckoff*)
Lemma incl_im_setminus : 
  forall {T U:Type} (A B:Ensemble T),
    Included B A ->
    forall (f:T->U),
    Included (Setminus (Im A f) (Im B f)) (Im (Setminus A B) f).
intros T U A B h1 f.
assert (h2:Im A f = Union (Im B f) (Im (Setminus A B) f)).
  rewrite (decompose_setminus_inc _ _ h1) at 1.
  rewrite im_union. reflexivity.  
apply elimination_union in h2. 
rewrite dist_sum_psa in h2. 
pose proof (setminus_inc A B) as h3.
pose proof h3 as h3'. 
rewrite inclusion_iff_union in h3'. rewrite comm_sum_psa in h3'.
apply (im_preserves_inclusion _ _ f) in h3.
rewrite inclusion_iff_intersection_eq in h3.
rewrite comm_prod_psa in h3.
rewrite h3 in h2.
symmetry in h2.
rewrite <- inclusion_iff_union in h2.
rewrite <- setminus_int_complement in h2. 
assumption.
Qed.


(*Wyckoff*)
Lemma im_complement_incl_inj : 
  forall {T U:Type} (A:Ensemble T) (f:T->U),
    injective f ->
    Included (Im (Ensembles.Complement A) f) (Ensembles.Complement (Im A f)).
intros T U A f h0. 
red. intros u h1.
destruct h1 as [u h1]. subst.
red in h1. red in h1. red. red.
intro h2.  inversion h2 as [x h3 h4 h5]. subst. 
apply h0 in h5. subst.
contradiction.
Qed.


(*Wyckoff*)
Lemma im_complement_incl_bij : 
  forall {T U:Type} (A:Ensemble T) (f:T->U),
    bijective f ->
    Included (Im (Ensembles.Complement A) f) 
             (Setminus (Im (Full_set _) f) (Im A f)).
intros T U A f h0.
red in h0. destruct h0 as [h0 h0'].
pose proof (im_complement_incl_inj A f h0) as h1.
rewrite setminus_int_complement.
rewrite comm_prod_psa. 
rewrite surj_iff in h0'. rewrite h0'. 
rewrite intersection_full_set.
assumption.
Qed.


(*Wyckoff*)
Lemma im_complement_bij : 
  forall {T U:Type} (A:Ensemble T) (f:T->U),
    bijective f ->
    Im (Ensembles.Complement A) f = 
    Setminus (Im (Full_set _) f) (Im A f).
intros T U A f h0.
apply Extensionality_Ensembles.
red. split.
apply im_complement_incl_bij. assumption.
red. intros u h1.
red in h0. destruct h0 as [h0 h0']. 
rewrite surj_iff in h0'.
pose proof (Full_intro _ u) as h2.
rewrite <- h0' in h2.
destruct h2 as [u h2]. subst.
apply Im_intro with u.
intro h3. 
inversion h1 as [h4 h5].
contradict h5.
apply Im_intro with u; auto.
reflexivity.
Qed.


(*Wyckoff*)
Lemma im_set_minus_inj : 
  forall {T U:Type} (A B:Ensemble T) (f:T->U),
    injective f ->
    Im (Setminus A B) f = Intersection (Im A f) (Im (Ensembles.Complement B) f).
intros T U A B f h0.
apply Extensionality_Ensembles.
red. split. 
apply im_setminus.
red. intros x h1.
destruct h1 as [u h1 h2].
destruct h1 as [u h1]. subst.
inversion h2 as [y h3 x h4]. subst. 
rewrite h4. 
apply h0 in h4. subst. 
apply Im_intro with y. constructor; auto.
reflexivity.
Qed.


(*Wyckoff*)
Lemma excl_middle_sat : forall {T:Type} (P:T->Prop),
                          Full_set T = Union [x:T | P x] [x:T | ~ (P x)].
intros T P. apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct (classic (P x)) as [h2 | h3].
left. constructor. assumption.
right. constructor. assumption.
red. intros; constructor.
Qed.

(*Wyckoff*)
Lemma excl_middle_sat' : forall {T:Type} (S:Ensemble T)
                         (P:T->Prop),
                           S = Union [x:T | In S x /\ P x]
                                     [x:T | In S x /\ ~ (P x)].
intros T S P.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct (classic (P x)) as [h2 | h3].
left. constructor. split; auto.
right. constructor. split; auto.
red. intros x h1.
destruct h1 as [x h1l | x h1r].
destruct h1l as [h1l]. destruct h1l; assumption.
destruct h1r as [h1r]. destruct h1r; assumption.
Qed.

(*Wyckoff*)
Lemma im_im : 
  forall {T U V:Type} (A:Ensemble T) (f:T->U) (g:U->V),
    Im (Im A f) g = Im A (fun x => g (f x)).
intros T U V A f g.
apply Extensionality_Ensembles.
red. split.
(* <= *)
red. intros v h1.
inversion h1 as [x h2 v' h3 h4].  subst.
destruct h2 as [x h2 u h3]. subst.
apply Im_intro with x. assumption. reflexivity.
(* >= *)
red. intros v h1.
destruct h1 as [x h2 v' h3]. subst.
apply Im_intro with (f x). apply Im_intro with x; auto.
reflexivity.
Qed.



(*Wyckoff*)
Definition im_proj1_sig {T:Type} {A:Ensemble T} 
           (S:Ensemble (sig_set A)) : Ensemble T :=
  Im S (@proj1_sig _ _).

(*Wyckoff*)
Lemma im_proj1_sig_injective : 
  forall {T:Type} {A:Ensemble T} (R S:Ensemble (sig_set A)),
    im_proj1_sig R = im_proj1_sig S ->
    R = S.
intros T A R S h1.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. destruct x as [x h3]. 
assert (h4:In (im_proj1_sig R) x).
  apply Im_intro with (exist _ _ h3).
  assumption. simpl. reflexivity.
rewrite h1 in h4.
destruct h4 as [x h4]. subst.
rewrite unfold_sig in h4.
assert (h3 = proj2_sig x). apply proof_irrelevance. subst.
assumption.
red. intros x h2. destruct x as [x h3]. 
assert (h4:In (im_proj1_sig S) x).
  apply Im_intro with (exist _ _ h3).
  assumption. simpl. reflexivity.
rewrite <- h1 in h4.
destruct h4 as [x h4]. subst.
rewrite unfold_sig in h4.
assert (h3 = proj2_sig x). apply proof_irrelevance. subst.
assumption.
Qed.



(*Wyckoff*)
Lemma im_proj1_sig_undoes_im_proj2_sig : 
  forall {T:Type} (A B:Ensemble T)
         (pf:Included A B),
    A = im_proj1_sig (im_proj2_sig A pf).
intros T A B h1.
unfold im_proj1_sig, im_proj2_sig.
rewrite im_im.
rewrite im_full_sig_proj1_sig at 1.
simpl.
apply im_ext_in.
intros; auto.
Qed.


(*Wyckoff*)
Lemma incl_im_proj1_sig_ens_sig_set : 
  forall {T:Type} {A:Ensemble T} (B:Ensemble (sig_set A)),
    Included (im_proj1_sig B) A.
intros T A B.
red. intros x h1.
destruct h1 as [x h1]. subst. apply proj2_sig.
Qed.



(*Wyckoff*)
Lemma im_proj2_sig_undoes_im_proj1_sig : 
  forall {T:Type} {B:Ensemble T} (A:Ensemble (sig_set B)),
    A = im_proj2_sig (im_proj1_sig A) (incl_im_proj1_sig_ens_sig_set A).
intros T B A.
unfold im_proj1_sig, im_proj2_sig.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct x as [x h2].
assert (h3:In (Im A (@proj1_sig _ _)) x).
  apply Im_intro with (exist _ _ h2). assumption.
  simpl. reflexivity.
apply Im_intro with (exist _ _ h3). constructor.
apply proj1_sig_injective. simpl. reflexivity.
red. intros x h1.
destruct h1 as [x h1]. subst.
destruct x as [x h2]. simpl.
destruct h2 as [x h2]. subst.
pose proof h2 as h2'. rewrite unfold_sig in h2'.
assert (h3:incl_im_proj1_sig_ens_sig_set A (proj1_sig x)
           (Im_intro (sig_set B) T A (proj1_sig (P:=fun x0 : T => In B x0)) x
              h2 (proj1_sig x) eq_refl) =
           proj2_sig x). apply proof_irrelevance.
rewrite h3.
assumption.
Qed.

(*Wyckoff*)
Lemma im_proj2_sig_undoes_im_proj1_sig' : 
  forall {T:Type} (B:Ensemble T) (A:Ensemble (sig_set B))
         (pf:Included (im_proj1_sig A) B),
    A = im_proj2_sig (im_proj1_sig A) pf.
intros T B A h1.
pose proof (im_proj2_sig_undoes_im_proj1_sig A) as h2.
assert (h3:h1 = incl_im_proj1_sig_ens_sig_set A). apply proof_irrelevance.
rewrite h3.
assumption.
Qed.



(*Wyckoff*)
Lemma union_setminus : forall {T:Type} (A X:Ensemble T),
                         Union A (Setminus X A) = Union A X.
intros T A X.
rewrite setminus_int_complement.
rewrite dist_prod_psa.
rewrite excl_middle_full.
rewrite intersection_full_set.
reflexivity.
Qed.

(*Wyckoff*)
Lemma union_setminus_incl : forall {T:Type} (A X:Ensemble T),
                          Included A X ->
                          Union A (Setminus X A) = X.
intros T A X h1.
rewrite union_setminus.
rewrite <- inclusion_iff_union.
assumption.
Qed.

(*Wyckoff*)
Lemma setminus_union : 
  forall {T:Type} (A B:Ensemble T),
    Setminus (Union A B) B = Setminus A B.
intros T A B.
apply Extensionality_Ensembles.
red; split; red.
intros x h1.
destruct h1 as [h1 h2].
destruct h1 as [x h1l | x h1r].
constructor; auto. contradiction.
intros x h1.
destruct h1 as [h1 h2].
constructor; auto. left; auto.
Qed.

(*Wyckoff*)
Lemma incl_union_setminus :
  forall {T:Type} (A B C:Ensemble T),
    Included A (Union B C) ->
    Included (Setminus A C) B.
intros T A B C h1.
red.
intros x h2.
destruct h2 as [h2 h3].
apply h1 in h2.
destruct h2; auto with sets. contradiction.
Qed.


(*Wyckoff*)
Lemma card_union :
  forall {T:Type} (A B:Ensemble T) (m n p:nat),
         cardinal _ A m ->
         cardinal _ B n ->
         cardinal _ (Intersection A B) p ->
         cardinal _ (Union A B) (m+n-p).
intros T A B m n p h1 h2 h3.
pose proof (decompose_int_setminus (Union A B) B) as h4. 
assert (h5:Included B (Union A B)). auto with sets.
rewrite comm_prod_psa in h4.
rewrite inclusion_iff_intersection_eq in h5.
rewrite h5 in h4.
rewrite setminus_union in h4.
pose proof (int_setminus B (Union A B)) as h6.
rewrite setminus_union in h6.
pose proof (card_decompose_int_setminus' _ _ _ _ h1 h3) as h7.
pose proof (card_disj_union _ _ _ _ h6 h2 h7) as h8.
rewrite h4.
assert (h9:n + (m-p) = m + n - p).
  rewrite (plus_comm m n).    
  apply plus_minus_assoc.
  assert (h9:Included (Intersection A B) A). auto with sets.
  apply (incl_card_le _ _ _ _ _ h3 h1 h9).
rewrite <- h9.
assumption.
Qed.

(*Wyckoff*)
Lemma card_gt_O_inh : 
  forall {T:Type} (A:Ensemble T) (n:nat),
    n > O -> cardinal _ A n -> Inhabited A.
intros T A n; destruct n as [|n]; simpl.
intros h1 h2. omega.
intros ? h1. inversion h1. apply Inhabited_intro with x. right.
constructor.
Qed.



(*Wyckoff*)
Lemma subtract_nin : forall {T:Type} (A:Ensemble T) (x:T),
                       ~In A x -> Subtract A x = A.
intros T A x h1.
apply Extensionality_Ensembles.
red. split. red. intros y h2. destruct h2; auto.
red. intros y h2. constructor; auto.
intro h3.
destruct h3; subst.
contradiction.
Qed.


(*Wyckoff*)
Lemma sub_empty_sing_or_empty :
  forall {T:Type} (A:Ensemble T) (x:T),
    Subtract A x = Empty_set _ ->
    A = Singleton x \/ A = Empty_set _.
intros T A x h1.
destruct (classic (In A x)) as [h2 | h3].
left.
apply Extensionality_Ensembles; red; split.
red. intros a h3. 
apply NNPP.
intro h4.
assert (h5:In (Subtract A x) a). constructor; auto.
rewrite h1 in h5. contradiction.
red. intros a h3. destruct h3; subst. assumption.
right.
rewrite subtract_nin in h1; auto.
Qed.


(*Wyckoff*)
Lemma subtract_preserves_inclusion : 
  forall {T:Type} (A B:Ensemble T) (x:T), 
    Included A B -> Included (Subtract A x) (Subtract B x).
intros T A B x h1.
red. intros y h2.
destruct h2 as [h2 h3].
constructor; auto with sets.
Qed.


(*Wyckoff*)
Lemma add_preserves_inclusion : 
  forall {T:Type} (A B:Ensemble T) (x:T), 
    Included A B -> Included (Add A x) (Add B x).
intros T A  B x h1.
red. intros x' h2.
destruct h2 as [x' h2 | x' h3].
left. auto with sets. destruct h3. right. constructor.
Qed.



(*Wyckoff*)
Lemma incl_card_fun1_eq :
  forall {T:Type} (A B:Ensemble T),
    Finite B -> Included A B ->
    card_fun1 A = card_fun1 B ->
    A = B.
intros T A B h1. revert A.
induction h1 as [|B h2 h3 x h4].
intros; auto with sets.
intros A h1 h5. 
pose proof (Finite_downward_closed _ _ (Add_preserves_Finite _  _ x h2) _ h1) as hf.
eapply subtract_preserves_inclusion in h1.  
rewrite sub_add_compat_nin in h1; auto.
destruct (classic (In A x)) as [h6 | h7].
rewrite card_add_nin' in h5; auto.
rewrite (card_sub_in _ hf x) in h5.
apply S_inj in h5.
specialize (h3 _ h1 h5).
rewrite <- (add_sub_compat_in _ _ h6).
f_equal. assumption. 
assumption.
rewrite subtract_nin in h1; auto.
apply incl_card_fun1 in h1; auto.
rewrite h5 in h1; auto.
rewrite card_add_nin' in h1; auto.
omega.
Qed.


(*Wyckoff*)
Lemma incl_couple_inv : 
  forall {T:Type} (A:Ensemble T) (b c:T),
    Included A (Couple b c) ->
    A = Empty_set _ \/ A = Singleton b \/ A = Singleton c \/
    A = Couple b c.
intros T A b c h1. 
destruct (eq_dec b c) as [he | hne].
subst.
rewrite couple_singleton. rewrite couple_singleton in h1.
apply singleton_inc in h1. 
destruct h1 as [h1l | h1r]. left. assumption.  
right. left. assumption. 
pose proof (finite_couple b c) as h2.
pose proof (card_fun1_couple _ _ hne) as h3.
pose proof (Finite_downward_closed _ _ h2 _ h1) as h4.
pose proof (incl_card_fun1 _ _ h2 h1) as h5.
rewrite h3 in h5. 
destruct (le_lt_eq_dec (card_fun1 A) 2 h5) as [h6 | h7]. 
assert (h7:card_fun1 A <= 1). auto with arith. clear h6.
destruct (le_lt_eq_dec (card_fun1 A) 1 h7) as [h8 | h9].
apply lt_n_1 in h8. clear h7. 
left.
apply card_fun1_O in h8. destruct h8; auto. contradiction.
apply card_fun1_1 in h9.
destruct h9 as [x h9]. subst.
red in h1. specialize (h1 _ (In_singleton _ x)).
destruct h1. right. left. reflexivity.
right. right. left. reflexivity. 
apply incl_card_fun1_eq in h1; auto.
rewrite h7. symmetry. assumption.
Qed.


(*Wyckoff*)
Lemma atom_sing : 
  forall {T:Type} (E A:Ensemble T), 
    Finite E -> Included A E -> A <> Empty_set _ ->
    ((forall F:Ensemble T, Included F E ->
                          Intersection F A = A \/
                          Intersection F A = Empty_set _) <->
    (exists (a:T), In E a /\ A = Singleton a)).
intros T E A h1. revert A.
induction h1 as [|E h2 h3 x h4].
intros A h1 h2.
assert (h3:A = Empty_set _). auto with sets.
contradiction.
intros A h1 h5. 
split. 
(* -> *)
intro h6.
pose proof (subtract_preserves_inclusion A (Add E x) x h1) as h7.
rewrite sub_add_compat_nin in h7; auto.
destruct (classic (In A x)) as [h8 | h9].
exists x. split. right. constructor. 
pose proof (Inhabited_intro _ _ _ h8) as h9.

assert (h10:Included (Singleton x) (Add E x)).
  red. intros x' h11. destruct h11. right. constructor.
specialize (h6 (Singleton x) h10).
destruct h6 as [h6l | h6r].
assert (h11:Intersection (Singleton x) A = Singleton x).
  apply Extensionality_Ensembles.
  red. split. auto with sets.
  red. intros x' h11. destruct h11. constructor; auto.
  constructor.
rewrite h11 in  h6l.
rewrite h6l. reflexivity.
assert (h11:In (Intersection (Singleton x) A) x).
  constructor; auto. constructor.
rewrite h6r in h11. contradiction.
pose proof (subtract_preserves_inclusion A (Add E x) x h1) as h10.
rewrite subtract_nin in h10; auto.
rewrite sub_add_compat_nin in h10; auto.
specialize (h3 _ h10 h5).
assert (h11:(forall F : Ensemble T,
        Included F E ->
        Intersection F A = A \/ Intersection F A = Empty_set T)).
  intros F h12. 
  assert (h13:Included F (Add E x)). red. intros; auto with sets.
  apply h6; auto.
rewrite h3 in h11.
destruct h11 as [a h11].
destruct h11 as [h11l h11r].
exists a. split; auto. left. assumption.
intros h6.
destruct h6 as [a h6].
destruct h6 as [h6l h6r].
intros F h7.
subst.
destruct (classic (In F a)) as [h8 | h9].
left. apply Extensionality_Ensembles. red. split.  auto with sets.
red. intros a' h9. destruct h9; subst. constructor; auto.
constructor.
right.
apply Extensionality_Ensembles.
red. split; auto with sets.
red. intros a' h10. destruct h10 as [a' h10l h10r].
destruct h10r; subst.
contradiction.
Qed.

(*Wyckoff*)
Inductive incl_sig {T:Type} (A B:Ensemble T)
          (pfi:Included A B) : Ensemble {x:T|In B x} :=
  incl_sig_intro : forall (x:T) (pfa:In A x),
                    In (incl_sig A B pfi) 
                       (exist _ _ (pfi _ pfa)).

(*Wyckoff*)
Lemma incl_sig_iff :   
  forall {T:Type} (A B:Ensemble T) (pf:Included A B)
         x, In (incl_sig A B pf) x <->
            In A (proj1_sig x).
intros T A B h0 x.
split. intros h1.
destruct h1. simpl. assumption.
intro h1. 
pose proof (incl_sig_intro _ _ h0 _ h1) as h2.
assert (h3:x = (exist (In B) (proj1_sig x) (h0 (proj1_sig x) h1))). 
  rewrite unfold_sig at 1. apply proj1_sig_injective. simpl.
  reflexivity.
rewrite <- h3 in h2.
assumption.
Qed.


(*Wyckoff*)
Definition subset_sig {T:Type} (A B:Ensemble T)
  (pf:Included A B) : Ensemble (sig_set B).
red in pf.
refine (Im (full_sig A) (fun x => exist _ _ (pf (proj1_sig x) (proj2_sig x)))). 
Defined.


(*Wyckoff*)
Lemma subset_sig_compat : 
  forall {T:Type} (A B:Ensemble T) (pf:Included A B)
         x, In (subset_sig A B pf) x <->
                In A (proj1_sig x).
intros T A B h0 x.
split.
intro h1.
destruct h1 as [x h2 y h4]. subst. simpl.
inversion h2. subst. apply proj2_sig.
intro h1.
unfold subset_sig.
apply Im_intro with  (exist _ _ h1).
constructor. simpl.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma incl_sig_eq_subset_sig : forall {T:Type} (A B:Ensemble T)
  (pf:Included A B), incl_sig A B pf = subset_sig A B pf.
intros T A B pf.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct h1 as [x h1].
unfold subset_sig.
apply Im_intro with (exist _ _ h1).
constructor. apply proj1_sig_injective. simpl. reflexivity.
red. 
intros x h1.
destruct h1 as [x h1]. subst.
constructor.
Qed.




(*Wyckoff*)
Lemma full_sig_family_union_eq :
  forall {T:Type} (F:Family T),
    full_sig (FamilyUnion F) =
    FamilyUnion
      [S:Ensemble (sig_set (FamilyUnion F)) | In F (Im S (@proj1_sig _ _))].
intros T F. 
apply Extensionality_Ensembles. red. split.
Focus 2.
red. intros x h3. constructor.
red. intros x h3.
destruct x as [x h4].
destruct h4 as [S x h4 h5].
assert (h7:forall s:T, In S s -> In S s). auto.
apply family_union_intro with (im_in _ _  (fun x':{s:T|In S s}=> (exist _ _ (family_union_intro _ _ _ (proj1_sig x') h4 (proj2_sig x')))) h7).
constructor.
assert (h8:S =  (Im
                   (im_in S (In S)
                          (fun x' : {s : T | In S s} =>
                             exist (In (FamilyUnion F)) (proj1_sig x')
                                   (family_union_intro T F S (proj1_sig x') h4 (proj2_sig x')))
                          h7) (proj1_sig (P:=fun x0 : T => In (FamilyUnion F) x0)))).
apply Extensionality_Ensembles.
red. split.
red. intros s h9.
apply Im_intro with (exist _ _ (family_union_intro _ _ _ _ h4 h9)).
unfold im_in. simpl.
apply Im_intro with (exist _ _ h9).
constructor.
apply proj1_sig_injective.
simpl. reflexivity. simpl. reflexivity.
red. intros s h9.
destruct h9 as [s h9]. subst.
destruct h9 as [s h9]. subst.
simpl.
destruct s. simpl. assumption.
pose proof h4 as h4'. rewrite h8 in h4'.
assumption.
unfold im_in. simpl.
apply Im_intro with (exist _ _ h5).
constructor.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.

(*Wyckoff*)
Lemma full_sig_decompose_setminus_incl : 
  forall {T:Type} (A B:Ensemble T) (pf:Included B A),
    full_sig A = Union (Im (full_sig (Setminus A B)) 
                           (fun x => exist _ _ ((setminus_inc A B) _ (proj2_sig x)))) (Im (full_sig B) (fun x => exist _ _ (pf _ (proj2_sig x)))).
intros T A B h1.
apply Extensionality_Ensembles.
red. split.
red.
intros x h2. clear h2.
destruct x as [x h2].
pose proof (decompose_setminus_inc _ _ h1) as h3.
pose proof h2 as h2'. rewrite h3 in h2'.
destruct h2' as [x h4 | x h5].
right.
apply Im_intro with (exist _ _ h4). constructor. apply proj1_sig_injective.
simpl. reflexivity.
left. apply Im_intro with (exist _ _ h5).
constructor.
apply proj1_sig_injective. simpl. reflexivity.
red. intros x h2. constructor.
Qed.


(*Wyckoff*)
Lemma full_false_empty : 
  forall (T:Type), 
    Full_set {_ : T | False} = Empty_set _.
intro T.
apply Extensionality_Ensembles.
red. split; auto with sets.
red.
intros x. destruct x. contradiction.
Qed.


(*Wyckoff*)
Lemma full_empty_sig_empty : forall (T:Type),
                                     full_sig (Empty_set T) = Empty_set _.
intros T.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
pose proof (proj2_sig x) as h2. simpl in h2.
contradiction.
auto with sets.
Qed.




(*Wyckoff*)
Lemma finite_image_rev_inj : 
  forall {T U:Type} (A:Ensemble T) (f:T->U), 
    injective f -> Finite (Im A f) ->
    Finite A.
intros T U A f h1 h2. 
apply NNPP.
intro h3. 
contradict h2.
intro h4.
pose proof (Pigeonhole_bis _ _ _ _ h3 h4).
contradiction.
Qed.

(*Wyckoff*)
Definition inv_im {T U:Type} (B:Ensemble U) (f:T->U) :=
  [x : T | In B (f x)].

(*Wyckoff*)
Lemma inv_im_empty : forall {T U:Type} (f:T->U),
                       inv_im (Empty_set U) f = Empty_set T.
intros T U f.
apply Extensionality_Ensembles; red; split; auto with sets.
red.
intros x h1.
destruct h1. contradiction.
Qed.


(*Wyckoff*)
Lemma im_inv_im : 
  forall {T U:Type} (f:T->U) (B:Ensemble U),
    Im (inv_im B f) f = Intersection B (Im (Full_set T) f).
intros T U  f B.
apply Extensionality_Ensembles.
unfold inv_im.
red. split.
(* <= *)
red.
intros b h1. 
inversion h1 as [a h2 b' h3]. subst.
constructor. destruct h2. assumption.
apply Im_intro with a. constructor. reflexivity.
red.
intros u h1.
inversion h1 as [u' h2 h3]. subst.
inversion h3 as [x h4 y h5 h6]. subst.
apply Im_intro with x.
constructor. assumption.
reflexivity.
Qed.

(*Wyckoff*)
Lemma inv_im_int : 
  forall {T U:Type} (A B:Ensemble U) (f:T->U),
    inv_im (Intersection A B) f = Intersection (inv_im A f) (inv_im B f).
intros T U A B f.
apply Extensionality_Ensembles.
red. split.
red.
intros x h1.
destruct h1 as [h1].
inversion h1 as [? h2 h3]. subst.
constructor; constructor; auto.
red.
intros x h1.
destruct h1 as [x h1 h2].
constructor.
destruct h1 as [h1]. destruct h2 as [h2].
constructor; auto.
Qed.

(*Wyckoff*)
Lemma inv_im_union : 
  forall {T U:Type} (A B:Ensemble U) (f:T->U),
    inv_im (Union A B) f = Union (inv_im A f) (inv_im B f).
intros T U A B f.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [h1].
inversion h1 as [? h2 |? h3]. subst.
left. constructor; auto.
subst. right. constructor; auto.
red. intros x h1.
destruct h1 as [x h1 | x h2].
destruct h1 as [h1].
constructor. left. auto.
destruct h2 as [h2].
constructor. right.
assumption.
Qed.

(*Wyckoff*)
Lemma inv_im_comp : 
  forall {T U:Type} (A:Ensemble U) (f:T->U),
    inv_im (Ensembles.Complement A) f = Ensembles.Complement (inv_im A f).
intros T U A f.
apply Extensionality_Ensembles.
red. split.
red. intros x h1.
destruct h1 as [h1]. 
intro h2.
destruct h2 as [h2].
auto with sets.
red. intros x h1. 
constructor.
intro h2. 
unfold inv_im in h1.
apply complement_inv in h1. 
apply sat_nin in h1.
contradiction.
Qed.

(*Wyckoff*)
Lemma inv_im_surj_inj : 
  forall {T U:Type} (f:T->U),
    surjective f ->
    forall (A B:Ensemble U),
      inv_im A f= inv_im B f->
      A = B.
intros T U f h1 B C h2.
red in h1.
apply Extensionality_Ensembles.
red. split.
red. intros y h3.
specialize (h1 y).
destruct h1 as [x h1r].
subst.
assert (h4:In (inv_im B f) x).
  constructor; auto.
rewrite h2 in h4.
destruct h4.
assumption.
red. intros y h3.
specialize (h1 y).
destruct h1 as [x h1r]. subst.
assert (h4:In (inv_im C f) x).
  constructor; auto.
rewrite <- h2 in h4.
destruct h4.
assumption.
Qed.

(*Wyckoff*)
Lemma inv_im_inj_sing : 
  forall {T U:Type} (f:T->U),
    injective f -> 
    forall y:U,
      {exists! x:T, f x = y} +
      {inv_im (Singleton y) f = Empty_set _}.
intros T U f h1 y.
destruct (classic_dec (exists! x:T, f x = y)) as [h2 | h3].
left; auto. right.
apply Extensionality_Ensembles; red; split; auto with sets.
red. intros x h4. destruct h4 as [h4]. inversion h4; clear h4.
subst.
contradict h3.
exists x.
red; split; auto.
Qed.

(*Wyckoff*)
Lemma inv_im_inj_sing' : 
  forall {T U:Type} (f:T->U),
    injective f -> 
    forall y:U,
      card_fun1 (inv_im (Singleton y) f) <= 1.
intros T U f h1 y.
pose proof (inv_im_inj_sing f h1 y) as h2.
destruct h2 as [h2a | h2b].
destruct h2a as [x h3]. red in h3. destruct h3 as [? h3]. subst.
assert (h4:inv_im (Singleton (f x)) f = Singleton x).
  apply Extensionality_Ensembles.
  red. split. red. intros x' h4.
  destruct h4 as [h4].
  inversion h4 as [h5]. clear h4.
  symmetry in h5. apply h3 in h5. subst. constructor.
  red. intros x' h4. destruct h4. constructor. 
  constructor. 
rewrite h4.
rewrite card_fun1_sing. auto with arith.
rewrite h2b.
rewrite card_fun1_empty. auto with sets.
Qed.



(*Wyckoff*)
(*Maybe move this to another file*)
Lemma left_inverse_ex :
  forall {T U:Type} (f:T->U),
    injective f ->
    forall (A:Ensemble T),
      exists! f':sig_set (Im A f)->T,
        forall (x:T) (pf:In A x),
          f' (exist _ _ (Im_intro _ _ A f _ pf _ (eq_refl _))) = x.
intros T U f h1 A.
assert (h2:forall y:sig_set (Im A f), exists! x:T, f x = proj1_sig y).
  intro y.
  pose proof (inv_im_inj_sing _  h1 (proj1_sig y)) as h2.
  destruct h2 as [h2l | h2r]; auto.
  destruct y as [y h3]. destruct h3 as [x h3]. subst.
  simpl in h2r. simpl.
  contradict h2r.
  apply Inhabited_not_empty.
  apply Inhabited_intro with x. constructor. constructor.
exists (fun x => (proj1_sig (constructive_definite_description _ (h2 x)))). red. split.
intros x h3. destruct constructive_definite_description as [x' h5]. simpl in h5.
simpl.
apply h1.
assumption.
intros f' h3.
apply functional_extensionality.
intro x.
destruct constructive_definite_description as [x' h4].
simpl.
destruct x as [x h5]. simpl in h4.
destruct h5 as [x h5]. subst.
apply h1 in h4.
subst.
symmetry.
apply h3.
Qed.

(*Wyckoff*)
Definition left_inverse {T U:Type} (f:T->U)
           (pf:injective f) (A:Ensemble T) : sig_set (Im A f)-> T :=
  proj1_sig (constructive_definite_description _ (left_inverse_ex f pf A)).


(*Wyckoff*)
Lemma left_inverse_compat : 
  forall {T U:Type} (f:T->U) (pf:injective f)
         (A:Ensemble T),
      let f' := left_inverse f pf A in
      forall (x:T) (pf:In A x),
        f' (exist _ _ (Im_intro _ _ A f _ pf _ (eq_refl _))) = x.
intros T U f h1 S f'.
unfold f'. unfold left_inverse.
destruct constructive_definite_description as [f'' h2].
simpl.
apply h2.
Qed.


(*Wyckoff*)
Definition left_inverse_full 
           {T U:Type} (f:T->U) (pf:injective f) :
  sig_set (Im (Full_set T) f) -> T :=
  left_inverse f pf (Full_set T).

(*Wyckoff*)
Lemma left_inverse_inj : 
  forall {T U:Type} (f:T->U) (pf:injective f)
         (A:Ensemble T),
    injective (left_inverse f pf A).
intros T U f h1 A.
red.
intros y y' h0.
destruct y as [y h2], y' as [y' h3].
pose proof (left_inverse_compat f h1 A) as h4.
simpl in h4.
destruct h2 as [x h2], h3 as [x' h3]. subst.
do 2 rewrite h4 in h0.
subst.
apply proj1_sig_injective. simpl.
reflexivity.
Qed.


(*Wyckoff*)
Lemma incl_im_full_sig_im_left_inverse :
  forall {T U:Type} (f:T->U) (pf:injective f) (A:Ensemble T),
    Included (Im (full_sig (Im A f)) (left_inverse f pf A)) A.
intros T U f h1 A.
red. intros x h2.
destruct h2 as [x h2]. subst. clear h2.
destruct x as [x h2].
destruct h2 as [x h2]. subst.
rewrite left_inverse_compat.
assumption.
Qed.




(*Wyckoff*)
Lemma inv_im_inj_im_compat : 
  forall {T U:Type} (f:T->U)
         (pf:injective f) (B:Ensemble U),
    inv_im B f = Im [u:sig_set (Im (Full_set T) f) | In B (proj1_sig u)] (left_inverse_full f pf).
intros T U f h1 B.
apply Extensionality_Ensembles.
red. split.
red. intros x h2. 
destruct h2 as [h2].
apply Im_intro with (exist _ _ (Im_intro _ _ (Full_set T) f _ (Full_intro _ x) _ (eq_refl _))).
constructor; simpl; auto. 
unfold left_inverse_full.
pose proof (left_inverse_compat f h1 (Full_set T)) as h3.
simpl in h3.
rewrite  h3.
reflexivity.
red.
intros x h2.
destruct h2 as [y h2]. subst.
destruct y as [y h3]. destruct h3 as [x h3]. subst. 
inversion h2 as [h4]. clear h2. simpl in h4.
constructor.
pose proof (left_inverse_compat f h1 (Full_set T)) as h3'.
simpl in h3'.
unfold left_inverse_full. rewrite  h3'.
assumption.
Qed.

Lemma finite_im_full_set_f_in_b : 
  forall {T U:Type} (B:Ensemble U),
    Finite B ->
    forall (f:T->U)
           (pf:injective f),
      Finite  (Im [u : sig_set (Im (Full_set T) f) | In B (proj1_sig u)]
        (left_inverse f pf (Full_set T))).
intros T U B h1.
induction h1 as [|B h2 h3 b h4].
intros.
assert (h1: (Im [u : sig_set (Im (Full_set T) f) | In (Empty_set U) (proj1_sig u)]
        (left_inverse f pf (Full_set T))) = Empty_set _).
  apply Extensionality_Ensembles; red; split; auto with sets.
  red. intros x h2. destruct h2 as [x h2]. destruct h2 as [h2].
  contradiction. 
rewrite h1. constructor.
intros f h1. 
assert (h7: [u : sig_set (Im (Full_set T) f) | In (Add B b) (proj1_sig u)] = Union  [u : sig_set (Im (Full_set T) f) | In B (proj1_sig u)] [u : sig_set (Im (Full_set T) f) | proj1_sig u = b]).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h7. destruct h7 as [h7]. inversion h7 as [b' h8 | b' h9]; clear h7.
  subst. left. constructor. assumption.
  inversion h9. subst.
  right. constructor. reflexivity.
  red.
  intros x h7.
  destruct h7 as [x h7l | x h7r]. destruct h7l as [h7l].
  constructor. left. assumption.
  destruct h7r as [h7r]. subst.
  constructor. right. constructor.
rewrite h7.
rewrite im_union.
apply Union_preserves_Finite.
apply h3.
pose proof (inv_im_inj_im_compat _ h1 (Singleton b)) as h8.
assert (h9:Im [u : sig_set (Im (Full_set T) f) | In (Singleton b) (proj1_sig u)]
         (left_inverse_full f h1) =  (Im [u : sig_set (Im (Full_set T) f) | proj1_sig u = b]
        (left_inverse f h1 (Full_set T)))).
  f_equal. apply Extensionality_Ensembles. red. split.
  red. intros x h9. destruct h9 as [h9]. inversion h9. subst.
  constructor. reflexivity.
  red. intros x h9. destruct h9 as [h9]. subst. constructor. constructor.
rewrite <- h9.
rewrite <- h8.
pose proof (inv_im_inj_sing' _ h1 b) as h10.
apply le_lt_eq_dec in h10.
destruct h10 as [h10l | h10r].
assert (h11:card_fun1 (inv_im (Singleton b) f) = 0). omega.
apply card_fun1_O in h11.
destruct h11 as [h11l | h11r].
rewrite h11l. constructor. 
pose proof (approximant_can_be_any_size _ (inv_im (Singleton b) f)(inv_im (Singleton b) f)   h11r 2) as h12.
destruct h12 as [X h13].
destruct h13 as [h13l h13r].
destruct h13r as [h14 h15].
apply cardinal_2 in h13l.
destruct h13l as [x [y [h13a h13b]]]. subst.
pose proof (h15 _ (Couple_l _ x y)) as h16.
pose proof (h15 _ (Couple_r _ x y)) as h17.
destruct h16 as [h16], h17 as [h17]. 
inversion h16; subst; clear h16.
inversion h17; subst; clear h17.
apply h1 in H0.
contradiction.
apply card_fun1_1 in h10r.
destruct h10r as [x' h10r].
rewrite h10r.
apply Singleton_is_finite.
Qed.


Definition sig_fun_app {T U:Type} {A:Ensemble T} (f:sig_set A->U) (def:U) : T->U.
intro x.
destruct (classic_dec (In A x)) as [h1 | h2].
refine (f (exist _ _ h1)).
refine def.
Defined.


Definition restriction_fun {T U:Type} (f:T->U) (A:Ensemble T) :
  sig_set A -> U :=
  fun x => f (proj1_sig x).

Definition restriction_fun2 {T U:Type} (f:T->T->U) (A:Ensemble T) :
  sig_set A -> sig_set A -> U :=
  fun x y => f (proj1_sig x) (proj1_sig y).


Definition restriction_sig 
           {T U:Type} {A:Ensemble T}
           (f:sig_set A->U) (B:Ensemble T)
           (pf:Included B A) : sig_set B->U :=
  fun x=>f (exist _ _ (pf _ (proj2_sig x))).

Definition restriction_sig2 
           {T U:Type} {A:Ensemble T}
           (f:sig_set A->sig_set A->U) (B:Ensemble T)
           (pf:Included B A) : sig_set B->sig_set B->U :=
  fun x y=>f (exist _ _ (pf _ (proj2_sig x))) (exist _ _ (pf _ (proj2_sig y))).


Definition closed_fun {T:Type} {A B:Ensemble T} 
           (f:sig_set A->sig_set A) (pf:Included B A) : Prop :=
           forall (x:T) (pfi:In B x), 
                     In B (proj1_sig (f (exist _ _ (pf _ pfi)))).


Definition closed_fun2 {T:Type} {A B:Ensemble T} 
           (f:sig_set A->sig_set A->sig_set A) (pf:Included B A) : Prop :=
           forall (x y:T) (pfx:In B x) (pfy:In B y), 
                     In B (proj1_sig (f (exist _ _ (pf _ pfx))
                                        (exist _ _ (pf _ pfy)))).


Definition restriction_sig' 
           {T:Type} {A:Ensemble T}
           (f:sig_set A->sig_set A) (B:Ensemble T)
           (pf:Included B A) 
           (pfcl:closed_fun f pf)
: sig_set B->sig_set B :=
  fun x=>exist _ _ (pfcl _ (proj2_sig x)).


Definition restriction_sig2' 
           {T:Type} {A:Ensemble T}
           (f:sig_set A->sig_set A->sig_set A) (B:Ensemble T)
           (pf:Included B A) 
           (pfcl:closed_fun2 f pf)
: sig_set B->sig_set B->sig_set B :=
  fun x y=>exist _ _ (pfcl _ _  (proj2_sig x) (proj2_sig y)).



Definition fun_injectors {T U:Type} (f:T->U) :=
  [pr:{A:Ensemble T & sig_set A->U} | injective (projT2 pr) /\
                                      extends_sig1 f (projT2 pr)
                                      /\ Im (Full_set T) f =
                                         Im (full_sig (projT1 pr)) (projT2 pr)].



Definition sig_fun_injectors {T U:Type} {A:Ensemble T} 
           (f:sig_set A->U) :=
  [pr:{B:Ensemble T & sig_set B->U} | injective (projT2 pr) /\
                                      extends_sig f (projT2 pr)
                                      /\ Im (full_sig A) f =
                                         Im (full_sig (projT1 pr)) (projT2 pr)].




         
Definition same_im_subsets_sig 
           {T U:Type} {A:Ensemble T}
           (f:sig_set A->U) :=
  [B:Ensemble T | exists pf:Included B A, 
                            Im (full_sig A) f =
                            Im (full_sig B) (restriction_sig f B pf)].

Lemma im_full_sig_add : 
  forall {T U:Type} {A:Ensemble T} {a:T}
         (f:sig_set (Add A a)->U),
         Im (full_sig (Add A a)) f =
         Add (Im (full_sig A) (restriction_sig f _ (incl_add A a)))
             (f (exist _ _ (Add_intro2 _ A a))).
intros T U A a f.
apply Extensionality_Ensembles.
red. split.
red. intros y h1.
destruct h1 as [y h1]. subst. clear h1.
destruct y as [y h1]. destruct h1 as [y h1 | y h2]. left.
apply Im_intro with (exist _ _ h1). constructor.
unfold restriction_sig. f_equal. apply proj1_sig_injective.
simpl. reflexivity.
destruct h2. right.
assert (h1: f (exist (In (Add A a)) a (Add_intro2 T A a)) = 
            f
               (exist (fun x : T => In (Add A a) x) a
                      (Union_intror T A (Singleton a) a (In_singleton T a)))).
  f_equal. apply proj1_sig_injective. simpl. reflexivity.
rewrite h1. constructor.
red. intros y h1. 
destruct h1 as [y h1 | y h2].
destruct h1 as [x h1]. subst. clear h1. destruct x as [x h1].
apply Im_intro with (exist _ _ (Add_intro1 _ A a x h1)). constructor.
unfold restriction_sig. f_equal. apply proj1_sig_injective.
simpl. reflexivity.
destruct h2.
apply Im_intro with (exist (In (Add A a)) a (Add_intro2 T A a)); auto.
constructor.
Qed.



Lemma setminus_im_full_sig_incl_setminus : 
  forall {T U:Type} (A B:Ensemble T) (pf:Included B A)
         (f:sig_set A->U),
    Included (Setminus (Im (full_sig A) f) 
                       (Im (full_sig B) (restriction_sig f _ pf)))
             (Im (full_sig (Setminus A B)) (restriction_sig f _ (setminus_inc A B))).
intros T U A B h1 f. red. intros y h2.
destruct h2 as [h2 h3].
destruct h2 as [y h2 u]. subst. clear h2.
destruct y as [y h4].
assert (h5:~In B y).
  intro h5.
  contradict h3.
  apply Im_intro with (exist _ _ h5). constructor.
  unfold restriction_sig. simpl.
  f_equal. apply proj1_sig_injective. simpl. reflexivity.
assert (h6:In (Setminus A B) y).
  constructor; auto.
apply Im_intro with (exist _ _ h6). constructor.
unfold restriction_sig. simpl.
f_equal.
apply proj1_sig_injective. simpl. reflexivity.
Qed.


Section EquivalenceClass.


Definition rel_classes 
           {T:Type} (A:Ensemble T) (R:T->T->Prop) :=
  Im A (fun x:T => [y:T | In A y /\ R x y]).

Lemma refl_rel_classes_inhabited : 
  forall {T:Type} (A:Ensemble T),
    Inhabited A ->
  forall (R:T->T->Prop),
    Reflexive R ->
    inhabited_family (rel_classes A R).
intros T A h1 R h2.
constructor.
destruct h1 as [a h1].
specialize (h2 a).
unfold rel_classes.
apply Inhabited_intro with [y : T | In A y /\ R a y].
apply Im_intro with a; auto.
intros A' h3.
destruct h3 as [a' h3]. subst.
apply Inhabited_intro with a'.
constructor.
specialize (h2 a').
split; auto.
Qed.

Lemma equiv_classes_inhabited : 
  forall {T:Type} (A:Ensemble T),
    Inhabited A ->
  forall (R:T->T->Prop),
    Equivalence R ->
    inhabited_family (rel_classes A R).
intros T A h1 R h2.
destruct h2 as [h2].
apply refl_rel_classes_inhabited; auto.
Qed.


Definition partition {T:Type} (F:Family T) (A:Ensemble T) : Prop 
  := 
    (forall S S':Ensemble T, 
       In F S -> In F S' -> S <> S' -> disjoint S S') /\
    FamilyUnion F = A.

Lemma equiv_classes_partition : 
  forall {T:Type} (A:Ensemble T) (R:T->T->Prop),
    Equivalence R -> partition (rel_classes A R) A.
intros T A R h1.
destruct h1 as [h1a h1b h1c].
red. split. 
intros S S' h2 h3 h4.
destruct h2 as [a h2 S]. subst. destruct h3 as [a' h3 S']. subst.
red.
assert (h0:~R a a').
  intro h5.
  contradict h4.
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h6. destruct h6 as [h6]. destruct h6 as [h6a h6b].
  apply h1b in h6b. specialize (h1c _ _ _ h6b h5).
  constructor; auto.
  red. intros x h6. destruct h6 as [h6]. destruct h6 as [h6a h6b].
  apply h1b in h6b. apply h1b in h5. specialize (h1c _ _ _ h6b h5).
  constructor; auto.
apply Extensionality_Ensembles; red; split; auto with sets. 
red. intros x h5.
destruct h5 as [x h5 h6]. destruct h5 as [h5]. destruct h6 as [h6].
destruct h5 as [h5a h5b]. destruct h6 as [h6a h6b].
apply h1b in h6b.
pose proof (h1c _ _ _ h5b h6b) as h7.
contradiction.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [S x h2 h3]. destruct h2 as [y h2 S]. subst.
destruct h3 as [h3]. destruct h3; auto.
red.
intros x h2.  
apply family_union_intro with [y:T |In A y /\ R x y].
unfold rel_classes.
apply Im_intro with x; auto.
specialize (h1a x).
constructor; auto.
Qed.


Definition im_rel {T U:Type} (f:T->U) := (fun x y:T=>f x = f y).

Lemma equiv_im_rel :
  forall {T U:Type} (f:T->U),
    Equivalence (im_rel f).
intros T U f. constructor.
red. intro x. red. reflexivity.
red. intros x y h1.
red in h1. red. rewrite h1. reflexivity.
red. intros x y z h1 h2. red in h1, h2. red. rewrite h1, h2.
reflexivity.
Qed.

Definition rel_classes_im_rel 
           {T U:Type} (f:T->U) : Ensemble (Ensemble T) := 
  rel_classes (Full_set T) (im_rel f).

Definition rel_classes_im_rel_set
           {T U:Type} (A:Ensemble T) (f:T->U) : 
  Ensemble (Ensemble T) := 
  rel_classes A (im_rel f).

Lemma partition_rel_classes_im_rel_set : 
  forall {T U:Type} (A:Ensemble T) (f:T->U),
    partition (rel_classes_im_rel_set A f) A.
intros T U A f. 
apply equiv_classes_partition.
apply equiv_im_rel.
Qed.


Definition rel_classes_im_rel_sig_fun 
           {T U:Type} {A:Ensemble T} (f:sig_set A->U) : Ensemble (Ensemble T) := 
  Im (rel_classes (full_sig A) (im_rel f)) (fun A=>Im A (@proj1_sig _ _)).


Lemma rel_classes_im_rel_eq_inv_im_singletons : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U),
    rel_classes_im_rel_sig_fun f = 
    [S:Ensemble T | exists u:U, In (Im (full_sig A) f) u /\
                                S = (Im (inv_im (Singleton u) f) (@proj1_sig _ _))].
intros T U A f.
apply Extensionality_Ensembles.
red. split.
red. intros B h1.
destruct h1 as [B h1]. subst.
destruct h1 as [x h1]. subst. clear h1.
constructor.
exists (f x). split.
apply Im_intro with x. constructor. reflexivity.
f_equal. unfold inv_im. unfold im_rel.
apply Extensionality_Ensembles.
red. split.
red.
intros x' h1. destruct h1 as [h1]. destruct h1 as [h1l h1r].
clear h1l. constructor. rewrite h1r. constructor.
red.
intros x' h1. destruct h1 as [h1']. inversion h1'.
constructor. rewrite H0.
split; auto. constructor.
red.
intros B h1.
destruct h1 as [h1].
destruct h1 as [u h1].
destruct h1 as [h1l h1r]. subst. 
destruct h1l as [x h1]. subst. clear h1.
unfold rel_classes_im_rel_sig_fun.
unfold rel_classes. rewrite im_im.
apply Im_intro with x. constructor.
f_equal.
unfold inv_im.
apply Extensionality_Ensembles.
red. split.
red. intros x' h1.
destruct h1 as [h1]. inversion h1 as [h2].
constructor. split; try constructor.
red. assumption.
red. intros x' h1.
destruct h1 as [h1].
destruct h1 as [h0 h1]. clear h0.
constructor. red in h1. rewrite h1.
constructor.
Qed.

Lemma rel_classes_im_rel_eq_inv_im_singletons' : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U),
    rel_classes_im_rel_sig_fun f = 
    Im (Im (full_sig A) f) (fun u:U => (Im (inv_im (Singleton u) f) (@proj1_sig _ _))).
intros T U A f.
rewrite rel_classes_im_rel_eq_inv_im_singletons.
rewrite im_im.
apply Extensionality_Ensembles.
red. split.
red.
intros B h1.
destruct h1 as [h1]. destruct h1 as [u h1]. destruct h1 as [h1l h1r].
subst.
destruct h1l as [u h1]. subst.
apply Im_intro with u. assumption.
reflexivity.
red.
intros B h1.
destruct h1 as [x h1]. subst. clear h1.
constructor.
exists (f x). split. apply Im_intro with x. constructor.
reflexivity.
reflexivity.
Qed.


Lemma empty_rel_classes_im_rel_eq_inv_im_singletons : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U),
    rel_classes_im_rel_sig_fun f =
    Empty_set _ -> A = Empty_set _.
intros T U A f h1.
apply Extensionality_Ensembles; red; split; auto with sets.
red.
intros x h2.
unfold rel_classes_im_rel_sig_fun in h1.
unfold rel_classes in h1.
rewrite im_im in h1.
assert (h3:In (Im (full_sig A)
                  (fun x : sig_set A =>
                     Im [y : sig_set A | In (full_sig A) y /\ im_rel f x y]
                        (proj1_sig (P:=fun x0 : T => In A x0))))
               ((fun x : sig_set A =>
          Im [y : sig_set A | In (full_sig A) y /\ im_rel f x y]
            (proj1_sig (P:=fun x0 : T => In A x0))) (exist _ _ h2))).
  apply Im_intro with (exist _ _ h2).
  constructor.
  reflexivity.
rewrite h1 in h3.
contradiction.
Qed.

Lemma inh_rel_classes_im_rel_sig_fun : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U),
    forall (S:Ensemble T),
      Ensembles.In (rel_classes_im_rel_sig_fun f) S ->
      Inhabited S.
intros T U A f S h2.
destruct h2 as [S h2]. subst.
destruct h2 as [x h2]. subst. clear h2.
apply Inhabited_intro with (proj1_sig x).
apply Im_intro with x.
constructor. split. constructor. red. reflexivity.
reflexivity.
Qed.

Lemma inh_rel_classes_im_rel_set : 
  forall {T U:Type} {A:Ensemble T} (f:T->U),
    forall (S:Ensemble T),
      Ensembles.In (rel_classes_im_rel_set A f) S ->
      Inhabited S.
intros T U A f S h1.
destruct h1 as [a h1]. subst.
apply Inhabited_intro with a. constructor.
split; auto. red. auto.
Qed.


Lemma rel_classes_im_rel_sig_fun_consistent_membership : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U)
         (B:Ensemble T),
    In (rel_classes_im_rel_sig_fun f) B ->
    forall (x y:sig_set A),
      (f x = f y) -> In B (proj1_sig x) ->
      In B (proj1_sig y).
intros T U A f B h1 x y h2 h3.
destruct h1 as [B h1]. subst.
destruct h1 as [x' h1]. subst.
apply Im_intro with y.
constructor. split. constructor.
inversion h3 as [y' h5]. subst.
destruct h5 as [h5]. destruct h5 as [h5l h5r]. clear h5l.
red in h5r. red.
rewrite <- h2. rewrite h5r.
f_equal.
destruct y', x. simpl in H. subst.
apply proj1_sig_injective. simpl. reflexivity.
reflexivity.
Qed.

Lemma partition_rel_classes_im_rel_sig_fun : 
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U),
    partition (rel_classes_im_rel_sig_fun f) A.
intros T U A f.
pose proof (equiv_classes_partition (full_sig A) _ (equiv_im_rel f)) as h1.
red in h1. destruct h1 as [h1l h1r].
red.
split.
intros S S' h2 h3 h4.
destruct h2 as [S h2]. destruct h3 as [S' h3]. subst.
specialize (h1l _ _ h2 h3). 
assert (h5:S <> S'). intro h5. subst. contradict h4. reflexivity.
specialize (h1l h5).
red in h1l. red.
apply NNPP.
intro h6.
apply not_empty_Inhabited in h6.
destruct h6 as [x h6].
destruct h6 as [x h6 h7]. 
destruct h6 as [x h6]. destruct x as [x h8].
subst. simpl in h7. destruct h7 as [x h7]. destruct x as [x h9]. subst.
simpl in h6, h8. assert (h10:h8 = h9). apply proof_irrelevance.
subst.
assert (h10:In (Intersection S S') (exist _ _ h9)). auto with sets.
rewrite h1l in h10. contradiction.
unfold rel_classes_im_rel_sig_fun.
rewrite family_union_im.
rewrite h1r.
apply Extensionality_Ensembles.
red. split.
red. intros x h2.
destruct h2 as [x h2]. subst.
apply proj2_sig.
red.
intros x h2. apply Im_intro with (exist _ _ h2); auto.
constructor.
Qed.

Lemma im_elt_of_rel_classes_im_rel_sig_fun :
  forall {T U:Type} {A:Ensemble T} (f:sig_set A->U) (B:Ensemble T)
         (pfinc:Included B A),
    In (rel_classes_im_rel_sig_fun f) B ->
    forall (b:T) (pfb:In B b),
        Im (full_sig B) (fun x => f (exist _ _ (pfinc _ (proj2_sig x)))) = 
      Singleton (f (exist _ _ (pfinc _ pfb))).
intros T U A f B h1 h2 b h3.
apply Extensionality_Ensembles.
red. split.
red.
intros u h4.
destruct h4 as [u h4]. subst. clear h4. destruct u as [u h4].
destruct h2 as [B h2]. subst. simpl. 
destruct h2 as [x h2]. subst. clear h2.
destruct h3 as [b h3]. subst. destruct h3 as [h3]. destruct h3 as [h3a h3b]. 
destruct h4 as [u h4]. subst. destruct h4 as [h4]. destruct h4 as [h4a h4b].
red in h3b. red in h4b.
destruct b as [b h5]. destruct u as [u h6].
simpl.
pose proof h4b as h4b'. rewrite h3b in h4b'.
assert (h7:exist (In A) b
              (h1 b
                 (Im_intro (sig_set A) T
                    [y : sig_set A | In (full_sig A) y /\ im_rel f x y]
                    (proj1_sig (P:=fun x0 : T => In A x0))
                    (exist (fun x0 : T => In A x0) b h5)
                    (intro_characteristic_sat
                       (fun y : sig_set A =>
                        In (full_sig A) y /\ im_rel f x y)
                       (exist (fun x0 : T => In A x0) b h5) 
                       (conj h3a h3b)) b eq_refl)) =
           exist _ _ h5). apply proj1_sig_injective.
  simpl. reflexivity.
assert (h8:exist (In A) u
           (h1 u
              (Im_intro (sig_set A) T
                 [y : sig_set A | In (full_sig A) y /\ im_rel f x y]
                 (proj1_sig (P:=fun x0 : T => In A x0))
                 (exist (fun x0 : T => In A x0) u h6)
                 (intro_characteristic_sat
                    (fun y : sig_set A => In (full_sig A) y /\ im_rel f x y)
                    (exist (fun x0 : T => In A x0) u h6) 
                    (conj h4a h4b)) u eq_refl)) =
           exist _ _ h6).  apply proj1_sig_injective.
  simpl. reflexivity.
rewrite h7, h8 at 1.
rewrite h4b' at 1.
constructor.
red.
intros x h4.
destruct h4.
apply Im_intro with (exist _ _ h3). constructor.
simpl.
reflexivity.
Qed.



Lemma rel_classes_im_rel_sig_fun_setminus : 
  forall {T U:Type} (A B:Ensemble T) (f:sig_set A -> U),
    In (rel_classes_im_rel_sig_fun f) B ->
    rel_classes_im_rel_sig_fun (restriction_sig f (Setminus A B) (setminus_inc A B)) = 
         Subtract (rel_classes_im_rel_sig_fun f) B.
intros T U A B f h1.
unfold restriction_sig.
do 2 rewrite  rel_classes_im_rel_eq_inv_im_singletons.
apply Extensionality_Ensembles.
red. split. 
red. intros S h2.
destruct h2 as [h2]. destruct h2 as [u h2]. destruct h2 as [h2 h3].
subst.
destruct h2 as [u h2]. subst. clear h2.
destruct u as [u h2]. destruct h2 as [h2 h3].
constructor.
constructor. 
exists (f (exist _ _ h2)).
split. apply Im_intro with (exist _ _ h2). constructor. reflexivity.
unfold restriction_sig. simpl. 
assert (h4:h2 = setminus_inc A B u (conj h2 h3)). apply proof_irrelevance. rewrite <- h4. 
apply Extensionality_Ensembles.
red. split.
red.
intros x h5. destruct  h5 as [x h5].  clear h4. subst.
destruct x as [x h6]. destruct h6 as [h6 h7]. simpl.
apply Im_intro with (exist _ _ h6).
inversion h5 as [h8]. clear h5. simpl in h8.
constructor.
inversion h8 as [h9]. clear h8. rewrite h9.
assert (h10:h6 = setminus_inc A B x (conj h6 h7)). apply proof_irrelevance. rewrite <- h10.
constructor. simpl. reflexivity.
red.
intros x h5.
destruct h5 as [x h5]. clear h4. subst.
destruct h5 as [h5]. inversion h5 as [h6]. clear h5.
assert (h7:~In B (proj1_sig x)).
  intro h8.
  symmetry in h6.
  pose proof (rel_classes_im_rel_sig_fun_consistent_membership f B h1 _ _ h6 h8) as h9.
  simpl in h9. contradiction.
destruct x as [x h8]. simpl in h7. simpl.
assert (h9:In (Setminus A B) x). auto with sets.
apply Im_intro with (exist _ _ h9).
constructor.
simpl.
rewrite h6.
assert (h10:h8 = setminus_inc A B x h9). apply proof_irrelevance.
subst.
constructor. simpl. reflexivity.
intro h4.
inversion h4 as [h5]. clear h4. simpl in h5.
assert (h6:In  (Im
         (inv_im
            (Singleton (f (exist (In A) u (setminus_inc A B u (conj h2 h3)))))
            (fun x : sig_set (Setminus A B) =>
             f
               (exist (In A) (proj1_sig x)
                  (setminus_inc A B (proj1_sig x) (proj2_sig x)))))
         (proj1_sig (P:=fun x : T => In (Setminus A B) x))) u).
  assert (h6:In (Setminus A B) u). auto with sets.
  apply Im_intro with (exist _ _ h6).
  constructor. simpl.
  assert (h7:h6 = conj h2 h3). apply proof_irrelevance. rewrite h7.
  constructor. simpl. reflexivity.
rewrite <- h5 in h6.
contradiction.
red.
intros S h2.
destruct h2 as [h2 h3]. assert (h4:S <> B). intro h4. subst. contradict h3. constructor. clear h3.
destruct h2 as [h2].
destruct h2 as [u h2].
destruct h2 as [h2 h3]. subst.
destruct h2 as [x h2]. subst. clear h2. 
destruct x as [x h2].
constructor. 
exists (f (exist _ _ h2)). split. 
assert (h3:~In B x).
  assert (h5:In (Im (inv_im (Singleton (f (exist (fun x0 : T => In A x0) x h2))) f)
         (proj1_sig (P:=fun x0 : T => In A x0))) x).
   apply Im_intro with (exist _ _ h2).
   constructor. constructor. simpl. reflexivity.
   intro h6.
   assert (h7:In (rel_classes_im_rel_sig_fun f) 
                 (Im (inv_im (Singleton (f (exist (fun x0 : T => In A x0) x h2))) f)
            (proj1_sig (P:=fun x0 : T => In A x0)))). 
     rewrite rel_classes_im_rel_eq_inv_im_singletons.
     constructor.
     exists (f (exist _ _ h2)).
     split. apply Im_intro with (exist _ _ h2).
     constructor. reflexivity.
     f_equal.
  pose proof (partition_rel_classes_im_rel_sig_fun f) as h8.
  red in h8.
  destruct h8 as [h8 h9].
  specialize (h8 _ _ h7 h1 h4).
  red in h8.
  assert (h10:In  (Intersection
         (Im (inv_im (Singleton (f (exist (fun x0 : T => In A x0) x h2))) f)
            (proj1_sig (P:=fun x0 : T => In A x0))) B) x).
    auto with sets.
  rewrite h8 in h10.
  contradiction.
assert (h5:In (Setminus A B) x). auto with sets.
apply Im_intro with (exist _ _ h5).
constructor. simpl.
f_equal. apply proj1_sig_injective. reflexivity.
apply Extensionality_Ensembles.
red. split.
red. intros x' h5.
destruct h5 as [x' h5]. subst. destruct x' as [x' h6].
inversion h5 as [h7]. clear h5. inversion h7 as [h8]. clear h7.
simpl.
   assert (h7:In (rel_classes_im_rel_sig_fun f) 
                 (Im (inv_im (Singleton (f (exist (fun x0 : T => In A x0) x h2))) f)
            (proj1_sig (P:=fun x0 : T => In A x0)))). 
     rewrite rel_classes_im_rel_eq_inv_im_singletons.
     constructor.
     exists (f (exist _ _ h2)).
     split. apply Im_intro with (exist _ _ h2).
     constructor. reflexivity.
     f_equal.
pose proof (partition_rel_classes_im_rel_sig_fun f) as h9.
red in h9. destruct h9 as [h9 h10].
specialize (h9 _ _ h7 h1 h4). red in h9.
assert (h11:In (Im (inv_im (Singleton (f (exist (fun x0 : T => In A x0) x h2))) f)
            (proj1_sig (P:=fun x0 : T => In A x0))) x).
  apply Im_intro with (exist _ _ h2).
  constructor. constructor. reflexivity.
  pose proof (rel_classes_im_rel_sig_fun_consistent_membership _ _ h7 _ _ h8 h11) as h12.
  simpl in h12.
  assert (h13:~In B x').
    intro h14.
    assert (h15:In  (Intersection
         (Im (inv_im (Singleton (f (exist (fun x0 : T => In A x0) x h2))) f)
            (proj1_sig (P:=fun x0 : T => In A x0))) B) x').
      auto with sets.
    rewrite h9 in h15. contradiction.
  assert (h14:In (Setminus A B) x'). auto with sets.
  apply Im_intro with (exist _ _ h14).
  constructor.
  simpl.
  rewrite h8 at 1.
  assert (h15:h6 = setminus_inc A B x' h14). apply proof_irrelevance.
  rewrite h15.
  constructor.
  reflexivity.
red.
intros x' h5.
destruct h5 as [x' h5]. subst.
destruct h5 as [h5]. destruct x' as [x' h6]. simpl in h5.
inversion h5 as [h7]. clear h5. simpl.
destruct h6 as [h6 h8].
apply Im_intro with (exist _ _ h6).
constructor. rewrite h7 at 1.
assert (h9:h6 = setminus_inc A B x' (conj h6 h8)). apply proof_irrelevance. 
rewrite h9 at 2.
constructor.
reflexivity.
Qed.


End EquivalenceClass.
End RandomFacts.


Notation "pr ||-> x" := (sig_fun_app (fst pr) (snd pr) x) (at level 20).

Section RandomFacts'.


(*Wyckoff*)
Lemma incl_im_im_left_inverse_compose :
  forall {T U V:Type} (f:T->U) (pff:injective f) (A:Ensemble T)
         (deft:T) (g:T->V),
    Included (Im (Im A f) (fun u : U => g ((left_inverse f pff A, deft) ||-> u))) (Im A g).
intros T U V f h1 A deft g.
red. intros y h2. rewrite im_im in h2.
destruct h2 as [x h2]. subst.
apply Im_intro with x; auto.
simpl. unfold sig_fun_app.
destruct classic_dec as [h3 | h4]. 
assert (h4:Im_intro _ _ A f x h2 (f x) eq_refl = h3).
  apply proof_irrelevance. subst.
rewrite left_inverse_compat.
reflexivity.
contradict h4. apply Im_intro with x; auto.
Qed.



Definition seg_set_weak (n:nat) := [m:nat | m <= n].
Definition seg_set (n:nat) := [m:nat | m < n].

Lemma seg_set_incl_seg_set_weak : 
  forall (n:nat),
    Included (seg_set n) (seg_set_weak n).
intros n. red. intros m h1.
constructor. destruct h1. omega.
Qed.

Lemma seg_set_weak_O : seg_set_weak 0 = Singleton 0.
apply Extensionality_Ensembles.
red. split.
red. intros m h1.
destruct h1 as [h1]. assert (m = 0). omega. subst.
constructor.
red. intros m h1.
destruct h1; subst.
constructor. auto with arith.
Qed.

Lemma seg_set_weak_Sn : 
  forall n:nat, 
    seg_set_weak (S n) = Add (seg_set_weak n) (S n).
intro n.
apply Extensionality_Ensembles.
red. split.
red. intros m h1.
destruct h1 as [h1].
destruct (le_lt_eq_dec _ _ h1) as [h2 | h3].
left. constructor. omega.
subst. right. constructor.
red. intros m h1.
destruct h1 as [m h1l | m h1r].
destruct h1l as [h1l]. constructor. omega.
destruct h1r; subst.
constructor. auto with arith.
Qed.


Lemma seg_set_O : seg_set O = Empty_set _.
apply Extensionality_Ensembles.
red; split; auto with sets.
red. intros m h1. destruct h1; omega.
Qed.

Lemma seg_set_Sn : 
  forall n:nat, 
    seg_set (S n) = Add (seg_set n) n.
intro n.
apply Extensionality_Ensembles.
red. split.
red. intros m h1.
destruct h1 as [h1].
assert (h2: m <= n). omega.
destruct (le_lt_eq_dec _ _ h2) as [h3 | h4].
left. constructor. omega. 
subst. right. constructor.
red. intros m h1.
destruct h1 as [m h1l | m h1r].
destruct h1l as [h1l]. constructor. omega.
destruct h1r; subst.
constructor. auto with arith.
Qed.


Lemma finite_seg_set_weak : 
  forall n:nat, Finite (seg_set_weak n).
intro n.
induction n as [|n h1].
rewrite seg_set_weak_O.
apply Singleton_is_finite.
rewrite seg_set_weak_Sn.
constructor; auto.
intro h2.
destruct h2 as [h2]. omega.
Qed.

Lemma finite_seg_set : 
  forall n:nat, Finite (seg_set n).
intros n.
pose proof (seg_set_incl_seg_set_weak n) as h1.
pose proof (finite_seg_set_weak n) as h2.
apply (Finite_downward_closed _ _ h2 _ h1).
Qed.
 
Lemma in_eq_rect : 
  forall {T U:Type} (pf:T=U) (A:Ensemble T) (x:U),
    In (eq_rect _ (fun V=>Ensemble V) A _ pf) x <->
    In A (eq_rect_r id x pf).
intros; subst; simpl; tauto.
Qed.

Lemma in_eq_rect_r : 
  forall {T U:Type} (pf:T=U) (A:Ensemble U) (x:T),
    In (eq_rect_r (fun V=>Ensemble V) A pf) x <->
    In A (eq_rect _ id x _ pf).
intros; subst; simpl; tauto.
Qed.

End RandomFacts'.



Section ChoiceFamily.



(*Axiom of choice applied to a non_empty family of non_empty sets*)
Definition choice_family
           {T:Type} (F:Family T)
           (pf:inhabited_family F) :=
  (Im (full_sig F) (fun A => (proj1_sig (constructive_indefinite_description _ (inhabited_family_cond _ pf A))))).

Definition refl_rel_classes_reps 
           {T:Type} (A:Ensemble T) (R:T->T->Prop)
           (pfr:Reflexive R) (pfi:Inhabited A)
           := choice_family _ (refl_rel_classes_inhabited _ pfi _ pfr).

Definition equiv_classes_reps 
           {T:Type} (A:Ensemble T) (R:T->T->Prop)
           (pfr:Equivalence R) (pfi:Inhabited A)
           := choice_family _ (equiv_classes_inhabited _ pfi _ pfr).

Lemma equiv_classes_reps_not_r : 
  forall {T:Type} (A:Ensemble T) (R:T->T->Prop)
         (pfr:Equivalence R) (pfi:Inhabited A)
         (x y:T), 
      In (equiv_classes_reps _ _ pfr pfi) x ->
      In (equiv_classes_reps _ _ pfr pfi) y ->
      x <> y ->
      ~ R x y.
intros T A R h1 h2 x y h3 h4 h5.
destruct h3 as [C h3 x h3'].
destruct h4 as [D h4 y h4'].
destruct C as [C h10]. destruct D as [D h11].  simpl. simpl in h5.
  simpl in h3'. simpl in h4'.
assert (h12:In C x).
  subst. destruct constructive_indefinite_description. simpl. assumption.
assert (h13:In D y).
  subst. destruct constructive_indefinite_description. 
  destruct constructive_indefinite_description. 
  simpl. assumption.
intro h14.
assert (h15:C = D).
  destruct h10 as [a h10 C']. destruct h11 as [b h11 D']. subst.
  apply Extensionality_Ensembles.
  red. split.
  red. intros r h15.
  destruct h15 as [h15]. destruct h15 as [h15a h15b].
  destruct constructive_indefinite_description as [a' h16]. destruct constructive_indefinite_description as [b' h17].
  simpl in h14. simpl in h5. simpl in h12.
  destruct h16 as [h16]. destruct h16 as [h16a h16b].
  simpl in h13.
  destruct h13 as [h13]. destruct h13 as [h13a h13b].
  constructor. split; auto.
  destruct h12 as [h12]. destruct h12 as [h12a h12b].
  destruct h1 as [h1a h1b h1c]. 
  apply h1b in h15b. 
  pose proof (h1c _ _ _ h15b h12b) as h16. 
  pose proof (h1c _ _ _ h16 h14) as h18.
  apply h1b in h16b. apply h1b in h13b.
  pose proof (h1c _ _ _ h18 h13b) as h19.
  apply (h1b _ _ h19).
  red. intros r h15.
  destruct h15 as [h15]. destruct h15 as [h15a h15b].
  destruct constructive_indefinite_description as [a' h16]. destruct constructive_indefinite_description as [b' h17].
  simpl in h14. simpl in h5. simpl in h12.
  destruct h16 as [h16]. destruct h16 as [h16a h16b].
  simpl in h13.
  destruct h13 as [h13]. destruct h13 as [h13a h13b].
  constructor. split; auto.
  destruct h12 as [h12]. destruct h12 as [h12a h12b].
  destruct h1 as [h1a h1b h1c]. 
  apply h1b in h15b.  
  pose proof (h1c _ _ _ h15b h13b) as h18. apply h1b in h14.
  pose proof (h1c _ _ _ h18 h14) as h19.
  apply h1b in h12b. pose proof (h1c _ _ _ h19 h12b) as h20.
  apply (h1b _ _ h20).
subst.
contradict h5.
assert (h16:h10 = h11). apply proof_irrelevance. subst.
reflexivity.
Qed.



End ChoiceFamily.


Arguments FamilyUnion [T] F _.
Arguments FamilyIntersection [T] F _.
Arguments IndexedUnion [It] [T] FI _.
Arguments IndexedIntersection [It] [T] FI _.
Arguments assoc_sum_psa [Xt] N M P.
Arguments assoc_prod_psa [Xt] N M P.
Arguments comm_sum_psa [Xt] N M. 
Arguments comm_prod_psa [Xt] N M.
Arguments abs_sum_psa [Xt] N M. 
Arguments abs_prod_psa [Xt] N M.
Arguments dist_sum_psa [Xt] N M P.
Arguments dist_prod_psa [Xt] N M P.
Arguments comp_sum_psa [Xt] N. 
Arguments comp_prod_psa [Xt] N.

