(* Copyright (C) 2014, Daniel Wyckoff *)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)



Require Export BoolAlgBasics.
Require Import SetUtilities.
Require Export Image.
Require Import FunctionalExtensionality.
Require Export Basics.
Require Import DecidableDec.
Require Import Description.
Require Import LogicUtilities.
Require Import TypeUtilities.


(*This section uses the indexed function notation from Givant/Halmos;
  for ensemble notation, see the subsequent top-level section:
  "Section infinite_set"*)


Definition Bcompi {It:Type} {B:Bool_Alg} (p: It->Btype (Bc B)) : 
  It->(Btype (Bc B)) :=
  fun (i:It) => -(p i).

Notation "-. x" := (Bcompi x) (at level 30).

Section infinite.
Variable B:Bool_Alg.
Let Bt := Btype (Bc B).
Variable It : Type.
Let IS := Full_set It.


Lemma doub_neg_ind : forall (p : It->Bt), -.(-.p) = p.
intros p.
unfold Bcompi.
cut ((fun i : It => - - p i) = (fun i : It => p i)).
intro h.
apply h.
apply functional_extensionality.
intro x.
rewrite doub_neg. reflexivity.
Qed.


Definition SupGen {Gt:Type} (S:Ensemble Gt) (p:Gt->Bt) (b:Bt) : Prop := 
sup (Im S p) b.

Definition InfGen {Gt:Type} (S:Ensemble Gt) (p:Gt->Bt) (b:Bt) : Prop := 
inf (Im S p) b.

Definition Sup (p: It->Bt) (b:Bt) : Prop :=
SupGen IS p b.

Definition Inf (p: It->Bt) (b:Bt) : Prop :=
InfGen IS p b.

Lemma comp_ind : 
  forall (p : It->Bt) (x:Bt), Im IS p (-x) <-> Im IS (-.p) x.
intros p x.
split.
(* left *)
intro h.
inversion h.
assert (h1: --x = -(p x0)).
rewrite H0. reflexivity.
assert (h2: --x = x).
rewrite doub_neg. reflexivity.
assert (h3: x = (-.p) x0).
unfold Bcompi. congruence.
apply Im_intro with (x := x0).
apply H. apply h3.
(* right *)
intro h4.
inversion h4.
unfold Bcompi in H0.
assert (h5: -x = --p x0).
rewrite H0. reflexivity.
assert (h6: --p x0 = p x0).
rewrite doub_neg. reflexivity.
assert (h7: p x0 = -x). congruence.
apply Im_intro with (x := x0).
assumption. rewrite h7. reflexivity.
Qed.

Lemma infnt_de_mor1 : forall (p : It->Bt) (b:Bt), Sup p b -> Inf (-.p) (-b).
intros p b h1.
unfold Sup in h1. unfold SupGen in h1. unfold sup in h1. elim h1.
intros h1l h1r.
unfold ub in h1l.
unfold Inf. unfold inf. split. unfold lb.
(* -b is a lower bound for -.p *)
intros t h2.
cut (le (-t) b).
rewrite doub_neg with (x := t) at 2.
apply mono_comp.
unfold In in h2.
pose proof (comp_ind p t) as h0. unfold Bt in h0. unfold Bt in h2.
rewrite <- h0 in h2.
apply h1l with (s := -t).
unfold In. assumption.
(* next subgoal: -b is the greatest lower bound *)
intros b'.
unfold lb.
unfold In.
intro h3.
rewrite doub_neg with (x := b').
apply mono_comp.
apply h1r with (b' := -b').
unfold ub.
unfold In.
intros s.
intro h4.
rewrite doub_neg in h4.
rewrite comp_ind in h4.
assert (h5: le b' (-s)).
apply h3 with (s := -s).
assumption.
rewrite doub_neg with (x := s).
apply mono_comp.
assumption.
Qed.

(*Dual of the previous of course.*)
Lemma infnt_de_mor2 : forall (p : It->Bt) (b : Bt), Inf p b -> Sup (-.p) (-b).
intros p b h1.
unfold Inf in h1. unfold InfGen in h1. unfold inf in h1. elim h1.
intros h1l h1r.
unfold lb in h1l.
unfold Sup. unfold sup. split.  unfold ub.
(* -b is an upper bound for -.p*)
intros t h2.
cut (le b (-t)).
rewrite doub_neg with (x :=t) at 2.
apply mono_comp.
unfold In in h2.
pose proof (comp_ind p t) as h0. unfold Bt in h0. unfold Bt in h2.
rewrite <- h0 in h2.
apply h1l with (s := -t).
unfold In.  assumption.
(* next suboal: -b is the greatest lower bound *)
intros b'.
unfold ub.
unfold In.
intro h3.
rewrite doub_neg with (x := b').
apply mono_comp.
apply h1r with (b' := -b').
unfold lb.
unfold In.
intros s.
intro h4.
rewrite doub_neg in h4.
rewrite comp_ind in h4.
assert (h5: le (-s) b').
apply h3 with (s := -s).
assumption.
rewrite doub_neg with (x := s).
apply mono_comp.
assumption.
Qed.

Lemma infnt_de_mor3 : forall (p : It->Bt) (b : Bt), Sup (-.p) b -> Inf p (-b).
intros p b h1.
rewrite <- doub_neg_ind with (p := p).
apply infnt_de_mor1 with (p := (-.p)).
assumption.
Qed.

Lemma infnt_de_mor4 : forall (p : It->Bt) (b : Bt), Inf (-.p) b -> Sup p (-b).
intros p b h1.
rewrite <- doub_neg_ind with (p := p).
apply infnt_de_mor2 with (p := (-.p)).
assumption.
Qed.

Definition ind_complete : Prop := forall (p : It->Bt), (exists b_, Inf p b_) /\ 
                                                   (exists b', Sup p b').

Lemma ind_complete_sup : (forall (p : It->Bt), (exists b', Sup p b')) -> ind_complete.
intros h1.
unfold ind_complete.
intro p. split.
assert (h2: exists b':Bt, Sup (-.p) b').
apply h1 with (p := -.p).
elim h2.
intros q.
intro h3.
clear h2.
assert (h4: Inf p (-q)).
apply infnt_de_mor3. assumption.
exists (-q).  assumption.
apply h1.
Qed.

(*Dual*)
Lemma ind_complete_inf : (forall (p : It->Bt), (exists b_, Inf p b_)) -> ind_complete.
intros h1. 
unfold ind_complete.
intro p. split.
apply h1.  
assert (h2: exists b_:Bt, Inf (-.p) b_).
apply h1 with (p := -.p).
elim h2.
intros q.  intro h3. clear h2.
assert (h4: Sup p (-q)).
apply infnt_de_mor4. assumption.
exists (-q). assumption.
Qed.


Section infnt_assoc.
Variable Jt:Type.
Variable IIS:Ensemble (Ensemble It).
Hypothesis covers : (FamilyUnion IIS) = IS.


Lemma infnt_assoc_sup : 
    (forall  (p: It -> Bt) (q:Jt->Bt), 
    (forall (S:Ensemble It), In IIS S ->       
     (exists (j:Jt), SupGen S p (q j))) -> 
    (forall (j:Jt), exists (S:Ensemble It), In IIS S 
                          /\ SupGen S p (q j)) ->
     (forall (q':Bt), SupGen (Full_set Jt) q q' ->
      Sup p q')).
intros p q h1 h2' q'. intro h4'.
destruct (eq_dec IIS (Empty_set _)) as [h3 | h4]. 
rewrite h3 in covers. rewrite h3 in h2'. clear h3.
assert (h3:Full_set Jt = Empty_set _).
  apply Extensionality_Ensembles; red; split; auto with sets.
  red. intros x ?. specialize (h2' x).
  destruct h2' as [S h2]. destruct h2; contradiction.
rewrite h3 in h4'.
unfold SupGen in h4'.
rewrite image_empty in h4'.
pose proof (sup_empty B) as h5.
pose proof (sup_unq _ _ _ h4' h5) as heq. 
rewrite heq. 
red. red.
rewrite empty_family_union in covers.
subst. rewrite <- covers.
rewrite image_empty. assumption. 
apply not_empty_Inhabited in h4.
destruct h4 as [S' h4].
red. red. 

assert (h7: ub (Im IS p) q').
  unfold ub.
  intros b h8.
  unfold In in h8. inversion h8.
  assert (h5:forall (i:It), (exists (S1:(Ensemble It)), (In IIS S1) /\
    In S1 i)).  
    intro i.
    assert (h6: In (FamilyUnion IIS) i).
    rewrite covers. constructor.
    inversion h6.
    exists S.
    tauto.
  assert (h10:exists S1 : Ensemble It, In IIS S1 /\ In S1 x).
  apply h5.
  elim h10.
  intros S1 h.
  assert (h11: exists j : Jt, SupGen S1 p (q j)).
  apply h1. apply h.
  elim h11. intros j h12.
  assert (h13: le b (q j)).
    apply h12.
    unfold In. 
    apply Im_intro with (x := x) (y := b).
    apply h.
    unfold In.
    assumption.
  assert (h14: le (q j) q').
    apply h4'.
    unfold In. apply Im_intro with (x := j).
    apply Full_intro.  reflexivity.
  apply trans_le with (x := b) (y := (q j)) (z := q').
  assumption.  assumption.
  assert (h15: forall r:Bt, ub (Im IS p) r -> le q' r).
    intros r h16.
    assert (h17: forall (S: Ensemble It), In IIS S ->
                 exists j : Jt, le (q j) r).
      intros S h30.
      assert (h31: exists j : Jt, SupGen S p (q j)).
        apply h1. apply h30.
        elim h31.
        intros j h32.
    assert (h21: le (q j) r).
      apply h32. unfold ub.
        intros b' h22. unfold In in h22. inversion h22.        
        apply h16.           
      unfold In.
      assert (h23: In IS x).
        unfold IS.
      apply Full_intro.
      apply Im_intro with (x := x).
      assumption. assumption.
    exists j. assumption.
    assert (h24: ub (Im (Full_set Jt) q) r).
      unfold ub.  
      intros b h25. unfold In in h25. inversion h25. 
      assert (h26: exists S : Ensemble It,
        In IIS S /\ SupGen S p (q x)).
      apply h2'.
      elim h26. intros S h27.  
      elim h27. intros h27l h27r.
      rewrite <- H0 in h27r.
      apply h27r.  unfold ub.  
      intros b' h28. unfold In in h28. apply h16.
      inversion h28. rewrite H3. unfold In. apply Im_intro with (x:=x0).
      assert (h29: (FamilyUnion IIS = IS)).
        apply covers.
      unfold IS. apply Full_intro. reflexivity. 
    apply h4'. apply h24.
 
unfold Sup. unfold SupGen. unfold sup. split.
assumption.
apply h15.
Qed.

(*dual*)
Lemma infnt_assoc_inf : 
    (forall  (p: It -> Bt) (q:Jt->Bt), 
    (forall (S:Ensemble It), In IIS S ->       
     (exists (j:Jt), InfGen S p (q j))) -> 
    (forall (j:Jt), exists (S:Ensemble It), In IIS S 
                          /\ InfGen S p (q j)) ->
     (forall (q':Bt), InfGen  (Full_set Jt) q q' ->
      Inf p q')).
intros p q h1 h2' q' h4'.  

destruct (eq_dec IIS (Empty_set _)) as [h3 | h4]. 
rewrite h3 in covers. rewrite h3 in h1. rewrite h3 in h2'. clear h3.
assert (h3:Full_set Jt = Empty_set _).
  apply Extensionality_Ensembles; red; split; auto with sets.
  red. intros x ?. specialize (h2' x).
  destruct h2' as [S h2]. destruct h2; contradiction.
rewrite h3 in h4'.
unfold InfGen in h4'.
rewrite image_empty in h4'.
pose proof (inf_empty B) as h5.
pose proof (inf_unq _ _ _ h4' h5). 
red. red.
rewrite empty_family_union in covers.
subst. rewrite <- covers.
rewrite image_empty. assumption. 
apply not_empty_Inhabited in h4.
destruct h4 as [S' h4].
red. red. 

assert (h7: lb (Im IS p) q').
  unfold lb.
  intros b h8.
  unfold In in h8. inversion h8.
  assert (h5:forall (i:It), (exists (S1:(Ensemble It)), (In IIS S1) /\
    In S1 i)).  
    intro i.
    assert (h6: In (FamilyUnion IIS) i).
    rewrite covers.
    constructor.
    inversion h6.
    exists S.
    tauto.
  assert (h10:exists S1 : Ensemble It, In IIS S1 /\ In S1 x).
  apply h5.
  elim h10.
  intros S1 h.
  assert (h11: exists j : Jt, InfGen S1 p (q j)).
  apply h1. apply h.
  elim h11. intros j h12.
  assert (h13: le (q j) b).
    apply h12.
    unfold In. 
    apply Im_intro with (x := x) (y := b).
    apply h.
    unfold In.
    assumption.
  assert (h14: le q' (q j)).
    apply h4'.
    unfold In. apply Im_intro with (x := j).
    apply Full_intro.  reflexivity.
  apply trans_le with (x := q') (y := (q j)) (z := b).
  assumption.  assumption.
  assert (h15: forall r:Bt, lb (Im IS p) r -> le r q').
    intros r h16.
    assert (h17: forall (S: Ensemble It), In IIS S ->
                 exists j : Jt, le r (q j)).
      intros S h30.
      assert (h31: exists j : Jt, InfGen S p (q j)).
        apply h1. apply h30.
        elim h31.
        intros j h32.
    assert (h21: le r (q j)).
      apply h32. unfold lb.
        intros b' h22. unfold In in h22. inversion h22.        
        apply h16.           
      unfold In.
      assert (h23: In IS x).
        unfold IS.
      apply Full_intro.
      apply Im_intro with (x := x).
      assumption. assumption.
    exists j. assumption.
    assert (h24: lb (Im (Full_set Jt) q) r).
      unfold lb.  
      intros b h25. unfold In in h25. inversion h25. 
      assert (h26: exists S : Ensemble It,
        In IIS S /\ InfGen S p (q x)).
      apply h2'.
      elim h26. intros S h27.  
      elim h27. intros h27l h27r.
      rewrite <- H0 in h27r.
      apply h27r.  unfold lb.  
      intros b' h28. unfold In in h28. apply h16.
      inversion h28. rewrite H3. unfold In. apply Im_intro with (x:=x0).
      assert (h29: (FamilyUnion IIS = IS)).
        apply covers.
      unfold IS. apply Full_intro. reflexivity. 
    apply h4'. apply h24.
unfold Inf. unfold InfGen. unfold inf. split.
assumption.
apply h15.
Qed.


End infnt_assoc.


Lemma infnt_distr_1_sup : forall {T:Type} (p:Bt) (q:T->Bt) (q':Bt), 
  let pq := (compose (fun x:Bt => p*x) q) in  
  SupGen (Full_set T) q q' -> SupGen (Full_set T) pq (p*q').
intros T p q q' pq h1.
assert (h2:ub (Im (Full_set T) pq) (p*q')).
  unfold ub.
  intro s. unfold In. intro h3. inversion h3. 
  unfold pq in H0. unfold compose in H0.
  rewrite H0.
  apply mono_prod. apply refl_le.
  apply h1. unfold In. apply Im_intro with (x:=x). apply H.
    reflexivity.
assert (h4: forall r:Bt, ub (Im (Full_set T) pq) r -> le (p*q') r).
 intros r h5.
 unfold ub in h5. 
 assert (h6: ub (Im (Full_set T) q) (r + (-p))).
   unfold ub. intros b h7.
   unfold In in h7. inversion h7.
   assert (h8: b = (p + (-p)) * b). 
     rewrite comp_sum.  rewrite comm_prod.  
     rewrite one_prod. reflexivity. 
   assert (h9: (p + (-p))*b = (p*b) + (-p)*b).
     apply dist_sum_r.
   assert (h10: b = (p*b) + (-p)*b).
     congruence.
   assert (h11:le (p*b) r).
     apply h5.
     unfold In.
     rewrite H0.
     apply Im_intro with (x := x).
     apply H.
     unfold pq.
     unfold compose. reflexivity.
   assert (h12:le (-p*b) (-p)).
     unfold le. apply eq_ord. 
     assert (h13: (-p)*b*(-p) = (-p)*(-p)*b).  
       rewrite <- assoc_prod.
       rewrite (comm_prod _ b (-p)). 
       rewrite assoc_prod.  reflexivity. 
     assert (h14: (-p)*(-p) = (-p)).
       apply idem_prod.
     rewrite h14 in h13.
     apply h13.
   assert (h15 : le (p*b + (-p*b)) (r + -p)).
     apply mono_sum.
   assumption.  assumption.
   rewrite  h10. assumption.
assert (h11: le q' (r+ (-p))).
  apply h1. assumption.
assert (h12: le p p).
  apply refl_le.
assert (h13: le (p*q') (p*(r+-p))).
  apply mono_prod. assumption. assumption.
assert (h14: p*(r+ -p) = p*r + p*-p).
  apply dist_sum.
assert (h15: p*r + p*-p = p*r).  
  rewrite comp_prod.  rewrite zero_sum.  reflexivity. 
assert (h16: le (p*r) r). unfold le.  apply eq_ord.
    assert (h18: r*r = r).  apply idem_prod.
    assert (h19: p*r*r = p*(r*r)). 
      rewrite assoc_prod. reflexivity.
    rewrite h18 in h19. assumption.
rewrite <- h15 in h16.
rewrite <- h14 in h16.
apply trans_le with (x := (p*q')) (y := (p*(r+-p))) (z:=r).
assumption. assumption.
unfold Sup. unfold SupGen. unfold sup. split.
assumption. assumption.
Qed.

Lemma infnt_distr_1_inf : forall {T:Type} (p:Bt) (q:T->Bt) (q':Bt), 
  let pq := (compose (fun x:Bt => p+x) q) in  
  InfGen (Full_set T) q q' -> InfGen (Full_set T) pq (p+q').
intros T p q q' pq h1.
assert (h2:lb (Im (Full_set T) pq) (p+q')).
  unfold lb.
  intro s. unfold In. intro h3. inversion h3. 
  unfold pq in H0. unfold compose in H0.
  rewrite H0.
  apply mono_sum. apply refl_le.
  apply h1. unfold In. apply Im_intro with (x:=x). apply H.
    reflexivity.
assert (h4: forall r:Bt, lb (Im (Full_set T) pq) r -> le r (p+q')).
 intros r h5.
 unfold lb in h5. 
 assert (h6: lb (Im (Full_set T) q) (r * (-p))).
   unfold ub. intros b h7.
   unfold In in h7. inversion h7.
   assert (h8: b = (p * (-p)) + b). 
     rewrite comp_prod. rewrite comm_sum. rewrite zero_sum.
     reflexivity.
   assert (h9: (p * (-p))+b = (p+b) * ((-p)+b)).
     apply dist_prod_r.
   assert (h10: b = (p+b) * ((-p)+b)).
     congruence.
   assert (h11:le r (p+b)).
     apply h5.
     unfold In.
     rewrite H0.
     apply Im_intro with (x := x).
     apply H.
     unfold pq.
     unfold compose. reflexivity.
   assert (h12:le (-p) (-p+b)).
     unfold le.  
     assert (h13: (-p)+((-p)+b) = ((-p)+(-p))+b).  
       rewrite assoc_sum.  reflexivity. 
     assert (h14: (-p)+(-p) = (-p)).      
       apply idem_sum.
     rewrite h14 in h13.
     apply h13.
   assert (h15 : le (r * -p) ((p+b) * (-p+b))).
     apply mono_prod.
   assumption.  assumption.
   rewrite  h10. assumption.
assert (h11: le (r * (-p)) q').
  apply h1. assumption.
assert (h12: le p p).
  apply refl_le.
assert (h13: le (p+(r*-p)) (p+q')).
  apply mono_sum. assumption. assumption.
assert (h14: p+(r * -p) = (p+r) *( p+-p)).
  apply dist_prod.
assert (h15: (p+r) * (p+-p) = p+r).  
  rewrite comp_sum.  rewrite one_prod.  reflexivity. 
assert (h16: le r (p+r)). unfold le.  
    assert (h18: r+r = r).  apply idem_sum.
    assert (h19: r + (p + r) = (r + r) + p). 
      rewrite (comm_sum _ p r).
      apply assoc_sum. 
    rewrite h19. rewrite h18. 
      rewrite (comm_sum _ p r). reflexivity. 
rewrite <- h15 in h16.
rewrite <- h14 in h16.
apply trans_le with (x := r) (y := (p+(r*-p))) (z:=p+q').
assumption. assumption.
unfold Inf. unfold InfGen. unfold inf. split.
assumption. assumption.
Qed.



(*Probably erase these two things at some point*)
Variable Jt:Type.
Definition Sup' {T:Type} (f:T->Bt) (b:Bt) := 
  SupGen (Full_set T) f b.

Lemma sup_le : forall {T:Type} (f:T->Bt) (b:Bt), 
  Sup' f b -> forall (j:T), le (f j) b.
intros T f b h1 j.
    unfold Sup' in h1. red in h1. red in h1. destruct h1 as [h1 ?].
    red in h1. 
    apply (h1 (f j)). 
    pose proof (Full_intro _ j) as h2.
    apply Im_intro with j. assumption. reflexivity.
Qed.


End infinite.


Arguments Bcompi [It] [B] _ _.
Arguments SupGen [B] [Gt]  _ _ _.
Arguments InfGen [B] [Gt]  _ _ _.
Arguments Sup [B] [It] _ _.
Arguments Inf [B] [It] _ _.
Arguments comp_ind [B] _ _ _.
Arguments infnt_de_mor1 [B] _ _ _ _.
Arguments infnt_de_mor2 [B] _ _ _ _.
Arguments infnt_de_mor3 [B] _ _ _ _.
Arguments infnt_de_mor4 [B] _ _ _ _.
Arguments ind_complete [B] _.
Arguments ind_complete_sup [B] _ _ _.
Arguments ind_complete_inf [B] _ _ _.
Arguments infnt_assoc_sup [B] _ _ _ _ _ _ _ _ _ _.
Arguments infnt_assoc_inf [B] _ _ _ _ _ _ _ _ _ _.
Arguments infnt_distr_1_sup [B] _ _ _ _ _.
Arguments infnt_distr_1_inf [B] _ _ _ _ _.
Arguments Sup' [B] _ _ _.
Arguments sup_le [B] _ _ _ _ _.


(*This section contains much of "Section infinite",
only it uses ensembles instead of indexed functions.*)
Section infinite_set.
Variable B:Bool_Alg.
Let Bt := (Btype (Bc B)).

Lemma sup_Sup_iff :
  forall (A:Ensemble Bt) (b:Bt),
    sup A b <-> @Sup _ (sig_set A) (@proj1_sig _ _) b. 
intros A b. 
assert (h2:(Im (full_sig A) (proj1_sig (P:=fun x : Bt => In A x))) = A).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h2. destruct h2 as [x h2]. subst. destruct x as [x].
  simpl. assumption.
  red. intros x h2.
  apply Im_intro with (exist _ _ h2).
  constructor. simpl. reflexivity.
split.
intro h1.
red. red.
unfold Bt in h2. unfold Bt. unfold full_sig in h2. rewrite h2.
assumption.
intro h1.
red in h1. red in h1.
unfold Bt in h2. unfold Bt in h1. unfold full_sig in h2. rewrite h2 in h1.
assumption.
Qed.

Lemma inf_Inf_iff :
  forall (A:Ensemble Bt) (b:Bt),
    inf A b <-> @Inf  _ (sig_set A) (@proj1_sig _ _) b. 
intros A b. 
assert (h2:(Im (full_sig A) (proj1_sig (P:=fun x : Bt => In A x))) = A).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h2. destruct h2 as [x h2]. subst. destruct x as [x].
  simpl. assumption.
  red. intros x h2.
  apply Im_intro with (exist _ _ h2).
  constructor. simpl. reflexivity.
split.
intro h1.
red. red.
unfold Bt in h2. unfold Bt. unfold full_sig in h2. rewrite h2.
assumption.
intro h1.
red in h1. red in h1.
unfold Bt in h2. unfold Bt in h1. unfold full_sig in h2. rewrite h2 in h1.
assumption.
Qed.

Lemma set_infnt_de_mor1 : forall (A:Ensemble Bt) (b:Bt), sup A b -> inf (comp_set A) (-b).
intros A b h1.
pose proof (infnt_de_mor1 (sig_set A) (@proj1_sig _ _) b) as h2.
rewrite <- sup_Sup_iff in h2.
specialize (h2 h1).  
rewrite inf_Inf_iff. 
red. red. red. red in h2. red in h2. red in h2.
destruct h2 as [h2l h2r]. 
split.
red. red in h2l. intros s h3.
destruct h3 as [s h3]. subst.
destruct s as [s h4]. simpl. 
destruct h4 as [s h4]. subst. 
assert (h5: In
          (Im (full_sig A)
             (Bcompi (proj1_sig (P:=fun x : Bt => In A x)))) (-s)).
  apply Im_intro with (exist _ _ h4).
  constructor. unfold Bcompi. simpl. reflexivity.
apply h2l; auto.
intros b' h3.
assert (h4:lb
          (Im (full_sig A)
             (Bcompi  (proj1_sig (P:=fun x : Bt => In A x)))) b').
  red. red in h3. intros s h4.
  destruct h4 as [s h4]. subst. 
  unfold Bcompi. destruct s as [s h5]. simpl.  
  apply h3.  
  assert (h6:In (comp_set A) (-s)).
    apply Im_intro with s; auto.
  apply Im_intro with (exist _ _ h6).
  constructor. simpl. reflexivity.
apply h2r; auto.
Qed.

Lemma set_infnt_de_mor2 : forall (A:Ensemble Bt) (b : Bt), inf A b -> sup (comp_set A) (-b).
intros A b h1.
pose proof (infnt_de_mor2 (sig_set A) (@proj1_sig _ _) b) as h2.
rewrite <- inf_Inf_iff in h2.
specialize (h2 h1).  
rewrite sup_Sup_iff. 
red. red. red. red in h2. red in h2. red in h2.
destruct h2 as [h2l h2r]. 
split.
red. red in h2l. intros s h3.
destruct h3 as [s h3]. subst.
destruct s as [s h4]. simpl. 
destruct h4 as [s h4]. subst. 
assert (h5: In
          (Im (full_sig A)
             (Bcompi (proj1_sig (P:=fun x : Bt => In A x)))) (-s)).
  apply Im_intro with (exist _ _ h4).
  constructor. unfold Bcompi. simpl. reflexivity.
apply h2l; auto.
intros b' h3. 
assert (h4:ub
          (Im (full_sig A)
             (Bcompi (proj1_sig (P:=fun x : Bt => In A x)))) b').
  red. red in h3. intros s h4.
  destruct h4 as [s h4]. subst. 
  unfold Bcompi. destruct s as [s h5]. simpl.  
  apply h3.  
  assert (h6:In (comp_set A) (-s)).
    apply Im_intro with s; auto.
  apply Im_intro with (exist _ _ h6).
  constructor. simpl. reflexivity.
apply h2r; auto.
Qed.


Lemma set_infnt_de_mor3 : forall (A:Ensemble Bt) (b : Bt), sup (comp_set A) b -> inf A (-b).
intros A b h1.
rewrite <- comp_set_comp_set with (A:=A).
apply set_infnt_de_mor1.
assumption.
Qed.

Lemma set_infnt_de_mor4 : forall (A:Ensemble Bt) (b : Bt), inf (comp_set A) b -> sup A (-b).
intros A b h1.
rewrite <- comp_set_comp_set with (A:=A).
apply set_infnt_de_mor2.
assumption.
Qed.


Definition set_complete : Prop := 
  forall A:Ensemble Bt, 
    exists b' b_:Bt, sup A b' /\ inf A b_.

Lemma sup_set_complete_ex : 
  forall (A:Ensemble Bt),
    set_complete ->
    exists! p:Bt, sup A p.
intros A h1.
specialize (h1 A).
destruct h1 as [b h1]. destruct h1 as [b' h1]. destruct h1 as [h1 h2].
exists b.
red.
split; auto.
intros b'' h3.
apply sup_unq with A; auto.
Qed.

Definition sup_set_complete (A:Ensemble Bt) (pf:set_complete) :=
  proj1_sig (constructive_definite_description _ (sup_set_complete_ex A pf)).

Lemma sup_set_complete_compat : 
  forall (A:Ensemble Bt) (pf:set_complete),
    sup A (sup_set_complete A pf).
intros A h1. 
unfold sup_set_complete.
destruct constructive_definite_description as [a h2].
simpl.
assumption.
Qed.


Lemma inf_set_complete_ex : 
  forall (A:Ensemble Bt),
    set_complete ->
    exists! p:Bt, inf A p.
intros A h1.
specialize (h1 A).
destruct h1 as [b h1]. destruct h1 as [b' h1]. destruct h1 as [h1 h2].
exists b'.
red.
split; auto.
intros b'' h3.
apply inf_unq with A; auto.
Qed.

Definition inf_set_complete (A:Ensemble Bt) (pf:set_complete) :=
  proj1_sig (constructive_definite_description _ (inf_set_complete_ex A pf)).

Lemma inf_set_complete_compat : 
  forall (A:Ensemble Bt) (pf:set_complete),
    inf A (inf_set_complete A pf).
intros A h1. 
unfold inf_set_complete.
destruct constructive_definite_description as [a h2].
simpl.
assumption.
Qed.
    

Lemma set_complete_sup : (forall A:Ensemble Bt, exists b', sup A b') -> set_complete.
intros h1.  
red. intro A.  
assert (h2:exists b_:Bt, sup (comp_set A) b_).
  apply (h1 (comp_set A)).
specialize (h1 A). 
destruct h1 as [b' h1]. exists b'; auto.
destruct h2 as [b_ h2].
exists (-b_). split; auto.
apply set_infnt_de_mor3 in h2; auto.
Qed.

Lemma set_complete_inf : (forall A:Ensemble Bt, (exists b_, inf A b_)) -> set_complete.
intros h1.  
red. intro A.   
assert (h2:exists b':Bt, inf (comp_set A) b').
  apply (h1 (comp_set A)).
specialize (h1 A). 
destruct h2 as [b' h2].
exists (-b'). 
destruct h1 as [b_ h1].
exists b_.
split; auto.
apply set_infnt_de_mor4; auto.
Qed.

Lemma set_infnt_assoc_sup : 
  forall (F:Family Bt)
    (pf:forall A:Ensemble Bt, In F A -> exists p:Bt, sup A p),
    forall q:Bt, sup (Im (full_sig F)
                            (fun A => 
                               (proj1_sig (constructive_definite_description _
                                                                             (ex_sup_unq _ _ (pf _ (proj2_sig A))))))) q ->
    sup (FamilyUnion F) q.
intros F h1 q h2.
destruct (eq_dec F (Empty_set _)) as [he | hne].
subst.
rewrite empty_family_union. 
rewrite full_empty_sig_empty in h2.
rewrite image_empty in h2.
unfold Bt. assumption.
unfold full_sig in h2. unfold sig_set in h2.
pose proof (full_sig_family_union_eq F) as h3.
symmetry in h3. unfold full_sig in h3.
pose proof (infnt_assoc_sup (sig_set (FamilyUnion F)) (sig_set F) _ h3 (@proj1_sig _ _) (fun A : {x : Ensemble Bt | In F x} =>
                                                                                           proj1_sig
                                                                                             (constructive_definite_description (sup (proj1_sig A))
                                                                                                                                (ex_sup_unq B (proj1_sig A)
                                                                                                                                            (h1 (proj1_sig A) (proj2_sig A)))))) as h4.
assert (h5:(forall S : Ensemble (sig_set (FamilyUnion F)),
        In
          [S0 : Ensemble (sig_set (FamilyUnion F))
          | In F (Im S0 (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x)))]
          S ->
        exists j : sig_set F,
          SupGen  S
            (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x))
            ((fun A : {x : Ensemble Bt | In F x} =>
              proj1_sig
                (constructive_definite_description 
                   (sup (proj1_sig A))
                   (ex_sup_unq B (proj1_sig A)
                      (h1 (proj1_sig A) (proj2_sig A))))) j))).
  intros S h6.
  destruct h6 as [h6].
  exists (exist _ _ h6).
  simpl.
  destruct constructive_definite_description as [x h8].
  simpl. unfold SupGen. assumption.
assert (h6: (forall j : sig_set F,
        exists S : Ensemble (sig_set (FamilyUnion F)),
          In
            [S0 : Ensemble (sig_set (FamilyUnion F))
            | In F
                (Im S0 (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x)))]
            S /\
          SupGen  S
            (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x))
            ((fun A : {x : Ensemble Bt | In F x} =>
              proj1_sig
                (constructive_definite_description 
                   (sup (proj1_sig A))
                   (ex_sup_unq B (proj1_sig A)
                      (h1 (proj1_sig A) (proj2_sig A))))) j))).
  intro S.
  destruct S as [S h7].
  simpl.
  destruct constructive_definite_description as [x h8].
  simpl.
  pose proof family_union_intro.
  exists (Im (full_sig S) (fun x => (exist _ (proj1_sig x) (family_union_intro _ _ _ _ h7 (proj2_sig x))))).
  split.
  constructor.
  rewrite im_im. simpl.
  rewrite <- im_full_sig_proj1_sig. assumption.
  red. rewrite im_im. simpl.
  rewrite <- im_full_sig_proj1_sig.
  assumption.
specialize (h4 h5 h6). clear h5 h6.
specialize (h4 q).
unfold SupGen in h4.
specialize (h4 h2).
red in h4. red in h4.
fold (full_sig (FamilyUnion F)) in h4.
rewrite <- im_full_sig_proj1_sig in h4.
assumption.
Qed.


Lemma set_infnt_assoc_inf : 
  forall (F:Family Bt)
    (pf:forall A:Ensemble Bt, In F A -> exists p:Bt, inf A p),
    forall q:Bt, inf (Im (full_sig F)
                            (fun A => 
                               (proj1_sig (constructive_definite_description _
                                                                             (ex_inf_unq _ _ (pf _ (proj2_sig A))))))) q ->
    inf (FamilyUnion F) q.
intros F h1 q h2.
destruct (eq_dec F (Empty_set _)) as [he | hne].
subst.
rewrite empty_family_union. 
rewrite full_empty_sig_empty in h2.
rewrite image_empty in h2.
unfold Bt. assumption.
unfold full_sig in h2. unfold sig_set in h2.
pose proof (full_sig_family_union_eq F) as h3.
symmetry in h3. unfold full_sig in h3.
pose proof (infnt_assoc_inf (sig_set (FamilyUnion F)) (sig_set F) _ h3 (@proj1_sig _ _) (fun A : {x : Ensemble Bt | In F x} =>
                                                                                           proj1_sig
                                                                                             (constructive_definite_description (inf (proj1_sig A))
                                                                                                                                (ex_inf_unq B (proj1_sig A)
                                                                                                                                            (h1 (proj1_sig A) (proj2_sig A)))))) as h4.
assert (h5:(forall S : Ensemble (sig_set (FamilyUnion F)),
        In
          [S0 : Ensemble (sig_set (FamilyUnion F))
          | In F (Im S0 (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x)))]
          S ->
        exists j : sig_set F,
          InfGen  S
            (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x))
            ((fun A : {x : Ensemble Bt | In F x} =>
              proj1_sig
                (constructive_definite_description 
                   (inf (proj1_sig A))
                   (ex_inf_unq B (proj1_sig A)
                      (h1 (proj1_sig A) (proj2_sig A))))) j))).
  intros S h6.
  destruct h6 as [h6].
  exists (exist _ _ h6).
  simpl.
  destruct constructive_definite_description as [x h8].
  simpl. unfold InfGen. assumption.
assert (h6: (forall j : sig_set F,
        exists S : Ensemble (sig_set (FamilyUnion F)),
          In
            [S0 : Ensemble (sig_set (FamilyUnion F))
            | In F
                (Im S0 (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x)))]
            S /\
          InfGen  S
            (proj1_sig (P:=fun x : Bt => In (FamilyUnion F) x))
            ((fun A : {x : Ensemble Bt | In F x} =>
              proj1_sig
                (constructive_definite_description 
                   (inf (proj1_sig A))
                   (ex_inf_unq B (proj1_sig A)
                      (h1 (proj1_sig A) (proj2_sig A))))) j))).
  intro S.
  destruct S as [S h7].
  simpl.
  destruct constructive_definite_description as [x h8].
  simpl.
  pose proof family_union_intro.
  exists (Im (full_sig S) (fun x => (exist _ (proj1_sig x) (family_union_intro _ _ _ _ h7 (proj2_sig x))))).
  split.
  constructor.
  rewrite im_im. simpl.
  rewrite <- im_full_sig_proj1_sig. assumption.
  red. rewrite im_im. simpl.
  rewrite <- im_full_sig_proj1_sig.
  assumption.
specialize (h4 h5 h6). clear h5 h6.
specialize (h4 q).
unfold InfGen in h4.
specialize (h4 h2).
red in h4. red in h4.
fold (full_sig (FamilyUnion F)) in h4.
rewrite <- im_full_sig_proj1_sig in h4.
assumption.
Qed.

Lemma sup_add_assoc : 
  forall (A:Ensemble Bt) (a p:Bt),
    sup A p ->
    sup (Add A a) (p+a).
intros A a p h1.
pose proof (set_infnt_assoc_sup (Couple A (Singleton a))) as h2.
assert (h3:forall A0 : Ensemble Bt,
               In (Couple A (Singleton a)) A0 -> exists p0 : Bt, sup A0 p0).
  intros C h3.
  destruct h3.
  exists p. assumption.
  exists a. apply sup_singleton.
specialize (h2 h3).
specialize (h2 (p + a)).
assert (h4:sup
         (Im (full_sig (Couple A (Singleton a)))
            (fun A0 : sig_set (Couple A (Singleton a)) =>
             proj1_sig
               (constructive_definite_description (sup (proj1_sig A0))
                  (ex_sup_unq B (proj1_sig A0)
                     (h3 (proj1_sig A0) (proj2_sig A0)))))) 
         (p + a)).
red. split.
red.
intros s h4.
destruct h4 as [A' h4 s]. subst.
destruct constructive_definite_description as [p' h6].
simpl.
destruct A' as [A' h7].
destruct h7. simpl in h6.
pose proof (sup_unq _ _ _ h1 h6). subst. apply le_plus.
simpl in h6.
pose proof (sup_singleton _ a) as h7.
pose proof (sup_unq _ _ _ h6 h7). subst.
rewrite comm_sum.
apply le_plus.
intros b' h8. red in h8.
pose proof (h8 a) as h9.
assert (h10:In
         (Im (full_sig (Couple A (Singleton a)))
            (fun A0 : sig_set (Couple A (Singleton a)) =>
             proj1_sig
               (constructive_definite_description (sup (proj1_sig A0))
                  (ex_sup_unq B (proj1_sig A0)
                     (h3 (proj1_sig A0) (proj2_sig A0)))))) a).
  assert (h10:In (Couple A (Singleton a)) (Singleton a)). right.
  apply Im_intro with (exist _ _ h10).
  constructor. simpl.
  destruct constructive_definite_description as [a' h12].
  simpl.
  apply sup_unq with (Singleton a).
  apply sup_singleton. assumption.
specialize (h9 h10).
pose proof (h8 p) as h11.
assert (h12: In
          (Im (full_sig (Couple A (Singleton a)))
             (fun A0 : sig_set (Couple A (Singleton a)) =>
              proj1_sig
                (constructive_definite_description 
                   (sup (proj1_sig A0))
                   (ex_sup_unq B (proj1_sig A0)
                      (h3 (proj1_sig A0) (proj2_sig A0)))))) p).
  assert (h13:In (Couple A (Singleton a)) A). left.
  apply Im_intro with (exist _ _ h13).
  constructor.
  simpl.
  destruct constructive_definite_description as [p' h14].
  simpl.
  apply sup_unq with A; auto.
specialize (h11 h12).
rewrite <- (idem_sum b').
apply mono_sum; auto.
specialize (h2 h4).
rewrite couple_add_sing in h2.
rewrite family_union_add in h2.
rewrite family_union_sing in h2.
unfold Add.
rewrite comm_sum_psa.
assumption.
Qed.

Lemma inf_add_assoc : 
  forall (A:Ensemble Bt) (a p:Bt),
    inf A p ->
    inf (Add A a) (p*a).
intros A a p h1.
pose proof (set_infnt_assoc_inf (Couple A (Singleton a))) as h2.
assert (h3:forall A0 : Ensemble Bt,
               In (Couple A (Singleton a)) A0 -> exists p0 : Bt, inf A0 p0).
  intros C h3.
  destruct h3.
  exists p. assumption.
  exists a. apply inf_singleton.
specialize (h2 h3).
specialize (h2 (p * a)).
assert (h4:inf
         (Im (full_sig (Couple A (Singleton a)))
            (fun A0 : sig_set (Couple A (Singleton a)) =>
             proj1_sig
               (constructive_definite_description (inf (proj1_sig A0))
                  (ex_inf_unq B (proj1_sig A0)
                     (h3 (proj1_sig A0) (proj2_sig A0)))))) 
         (p * a)).
red. split.
red.
intros s h4.
destruct h4 as [A' h4 s]. subst.
destruct constructive_definite_description as [p' h6].
simpl.
destruct A' as [A' h7].
destruct h7. simpl in h6.
pose proof (inf_unq _ _ _ h1 h6). subst. apply times_le.
simpl in h6.
pose proof (inf_singleton _ a) as h7.
pose proof (inf_unq _ _ _ h6 h7). subst.
rewrite comm_prod.
apply times_le.
intros b' h8. red in h8.
pose proof (h8 a) as h9.
assert (h10:In
         (Im (full_sig (Couple A (Singleton a)))
            (fun A0 : sig_set (Couple A (Singleton a)) =>
             proj1_sig
               (constructive_definite_description (inf (proj1_sig A0))
                  (ex_inf_unq B (proj1_sig A0)
                     (h3 (proj1_sig A0) (proj2_sig A0)))))) a).
  assert (h10:In (Couple A (Singleton a)) (Singleton a)). right.
  apply Im_intro with (exist _ _ h10).
  constructor. simpl.
  destruct constructive_definite_description as [a' h12].
  simpl.
  apply inf_unq with (Singleton a).
  apply inf_singleton. assumption.
specialize (h9 h10).
pose proof (h8 p) as h11.
assert (h12: In
          (Im (full_sig (Couple A (Singleton a)))
             (fun A0 : sig_set (Couple A (Singleton a)) =>
              proj1_sig
                (constructive_definite_description 
                   (inf (proj1_sig A0))
                   (ex_inf_unq B (proj1_sig A0)
                      (h3 (proj1_sig A0) (proj2_sig A0)))))) p).
  assert (h13:In (Couple A (Singleton a)) A). left.
  apply Im_intro with (exist _ _ h13).
  constructor.
  simpl.
  destruct constructive_definite_description as [p' h14].
  simpl.
  apply inf_unq with A; auto.
specialize (h11 h12).
rewrite <- (idem_prod b').
apply mono_prod; auto.
specialize (h2 h4).
rewrite couple_add_sing in h2.
rewrite family_union_add in h2.
rewrite family_union_sing in h2.
unfold Add.
rewrite comm_sum_psa.
assumption.
Qed.

 
Lemma set_infnt_distr_1_sup : 
  forall (A:Ensemble Bt) (p q:Bt),
    sup A q -> sup (Im A (fun x => p*x)) (p*q).
intros A p q h1.
pose proof infnt_distr_1_sup.
pose proof (infnt_distr_1_sup {x:Bt|In A x} p (@proj1_sig _ _) q) as h2.
assert (h3: SupGen (Full_set {x : Bt | In A x})
         (proj1_sig (P:=fun x : Bt => In A x)) q).
  red.
  fold (sig_set A). fold (full_sig A).
  rewrite <- im_full_sig_proj1_sig.
  assumption.
specialize (h2 h3).
red in h2. simpl in h2. 
unfold compose in h2.
assert (h4: (Im (Full_set {x : Bt | In A x})
            (fun x : {x : Bt | In A x} => p * proj1_sig x)) =
             (Im A (fun x : Bt => p * x))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h4.
  destruct h4 as [x h4].
  subst. apply Im_intro with (proj1_sig x).
  apply proj2_sig. 
  reflexivity.
  red. intros x h4.
  destruct h4 as [x h4]. subst.
  apply Im_intro with (exist _ _ h4).
  constructor. simpl. reflexivity.
rewrite h4 in h2.
assumption.
Qed.

Lemma set_infnt_distr_1_inf : 
  forall (A:Ensemble Bt) (p q:Bt),
    inf A q -> inf (Im A (fun x => p+x)) (p+q).
intros A p q h1.
pose proof infnt_distr_1_inf.
pose proof (infnt_distr_1_inf {x:Bt|In A x} p (@proj1_sig _ _) q) as h2.
assert (h3: InfGen (Full_set {x : Bt | In A x})
         (proj1_sig (P:=fun x : Bt => In A x)) q).
  red.
  fold (sig_set A). fold (full_sig A).
  rewrite <- im_full_sig_proj1_sig.
  assumption.
specialize (h2 h3).
red in h2. simpl in h2. 
unfold compose in h2.
assert (h4: (Im (Full_set {x : Bt | In A x})
            (fun x : {x : Bt | In A x} => p + proj1_sig x)) =
             (Im A (fun x : Bt => p + x))).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h4.
  destruct h4 as [x h4].
  subst. apply Im_intro with (proj1_sig x).
  apply proj2_sig. 
  reflexivity.
  red. intros x h4.
  destruct h4 as [x h4]. subst.
  apply Im_intro with (exist _ _ h4).
  constructor. simpl. reflexivity.
rewrite h4 in h2.
assumption.
Qed.


End infinite_set.

(*Theorems*)
Arguments sup_Sup_iff [B] _ _.
Arguments inf_Inf_iff [B] _ _.
Arguments set_infnt_de_mor1 [B] _ _ _.
Arguments set_infnt_de_mor2 [B] _ _ _.
Arguments set_infnt_de_mor3 [B] _ _ _.
Arguments set_infnt_de_mor4 [B] _ _ _.
Arguments sup_set_complete_ex [B] _ _.
Arguments sup_set_complete_compat [B] _ _.
Arguments inf_set_complete_ex [B] _ _.
Arguments inf_set_complete_compat [B] _ _.
Arguments set_complete_sup [B] _ _.
Arguments set_complete_inf [B] _ _.
Arguments set_infnt_assoc_sup [B] _ _ _ _.
Arguments set_infnt_assoc_inf [B] _ _ _ _.
Arguments sup_add_assoc [B] _ _ _ _.
Arguments inf_add_assoc [B] _ _ _ _.
Arguments set_infnt_distr_1_sup [B] _ _ _ _.
Arguments set_infnt_distr_1_inf [B] _ _ _ _.

(*Definitions*)
Arguments sup_set_complete [B] _ _.
Arguments inf_set_complete [B] _ _.



Definition Bcompi_p {It T:Type} {Bp:Bool_Alg_p T} 
           (p: It->Btype_p T (Bc_p T Bp)) : 
              It->Btype_p T (Bc_p T Bp) :=
                fun (i:It) => %- (p i).

Notation "%~ x" := (Bcompi_p x) (at level 30).

(*This is a section of dual versions of each of the theorems,
  this time with [Bool_Alg_p]s instead of [Bool_Alg]s.
  The below theorems are just wrappers.*)
Section ParametricAnalogues.
Variable T:Type.
Variable Bp:Bool_Alg_p T.
Let Btp := Btype_p T (Bc_p T Bp).
Variable It : Type.
Let IS := Full_set It.


Definition ba_conv_ind {Tt:Type} (p:Tt->Btp) : Tt->(Btype (Bc (ba_conv Bp))) :=
  transfer_dep (U:=fun T=>(Tt->T)) (ba_conv_type Bp) p.

(*I'm not sure why the notion "-." is not working*)
Lemma Bcompi_p_eq : 
  forall (p:It->Btp), 
  (ba_conv_ind (%~ p)) =  Bcompi (ba_conv_ind p).
intro p.
unfold ba_conv_ind. unfold ba_conv_type.
do 2 rewrite transfer_dep_eq_refl.
unfold Bcompi. unfold Bcompi_p.
reflexivity.
Qed.

Lemma Bcompi_p_eq' : 
  forall p:It->Btp,
    Bcompi_p p = Bcompi (ba_conv_ind p).
auto.
Qed.

Lemma doub_neg_ind_p : forall (p : It->Btp), %~(%~p) = p.
intro p.
do 2 rewrite Bcompi_p_eq'.
apply (@doub_neg_ind (ba_conv Bp)).
Qed.

Definition SupGen_p {Gt:Type} (S:Ensemble Gt) (p:Gt->Btp) (b:Btp) : Prop :=
sup_p (Im S p) b.

Definition InfGen_p {Gt:Type} (S:Ensemble Gt) (p:Gt->Btp) (b:Btp) : Prop :=
inf_p (Im S p) b.

Definition Sup_p (p: It->Btp) (b:Btp) : Prop :=
SupGen_p IS p b.

Definition Inf_p (p: It->Btp) (b:Btp) : Prop :=
InfGen_p IS p b.

Lemma SupGen_p_iff : 
  forall (Gt:Type) (S:Ensemble Gt) (p:Gt->Btp) (b:Btp),
    SupGen_p S p b <-> SupGen S (ba_conv_ind p) (ba_conv_elt b).
intros Gt S p b.
unfold SupGen_p. unfold SupGen.
rewrite sup_p_iff.
tauto.
Qed.

Definition InfGen_p_iff : 
  forall (Gt:Type) (S:Ensemble Gt) (p:Gt->Btp) (b:Btp),
  InfGen_p S p b <-> InfGen S (ba_conv_ind p) (ba_conv_elt b).
intros Gt S p b.
unfold InfGen_p. unfold InfGen.
rewrite inf_p_iff.
tauto.
Qed.


Definition Sup_p_iff : 
  forall (p: It->Btp) (b:Btp),
    Sup_p p b <-> Sup (ba_conv_ind p) (ba_conv_elt b).
intros p b.
unfold Sup_p. unfold Sup.
rewrite SupGen_p_iff.
tauto.
Qed.

Definition Inf_p_iff : 
  forall (p: It->Btp) (b:Btp),
    Inf_p p b <-> Inf (ba_conv_ind p) (ba_conv_elt b).
intros p b.
unfold Inf_p. unfold Inf.
rewrite InfGen_p_iff.
tauto.
Qed.


Lemma comp_ind_p :
  forall (p : It->Btp) (x:Btp), Im IS p (%-x) <-> Im IS (%~p) x.
intros p x.
rewrite ba_conv_comp. rewrite Bcompi_p_eq'.
apply (@comp_ind (ba_conv Bp)).
Qed.

Lemma infnt_de_mor1_p : forall (p : It->Btp) (b:Btp), Sup_p p b -> Inf_p (%~p) (%-b).
intros p b.
rewrite Sup_p_iff. rewrite Inf_p_iff.
apply (@infnt_de_mor1 (ba_conv Bp)).
Qed.

Lemma infnt_de_mor2_p : forall (p : It->Btp) (b : Btp), Inf_p p b -> Sup_p (%~p) (%-b).
intros p b.
rewrite Sup_p_iff. rewrite Inf_p_iff.
apply (@infnt_de_mor2 (ba_conv Bp)).
Qed.

Lemma infnt_de_mor3_p : forall (p : It->Btp) (b : Btp), Sup_p (%~p) b -> Inf_p p (%-b).
intros p b.
rewrite Sup_p_iff. rewrite Inf_p_iff.
apply (@infnt_de_mor3 (ba_conv Bp)).
Qed.

Lemma infnt_de_mor4_p : forall (p : It->Btp) (b : Btp), Inf_p (%~p) b -> Sup_p p (%-b).
intros p b.
rewrite Sup_p_iff. rewrite Inf_p_iff.
apply (@infnt_de_mor4 (ba_conv Bp)).
Qed.

Definition ind_complete_p : Prop := 
  forall (p : It->Btp), 
    (exists b_, Inf_p p b_) /\
    (exists b', Sup_p p b').

Lemma ind_complete_p_iff : 
  ind_complete_p <-> (@ind_complete (ba_conv Bp) It).
unfold ind_complete_p. unfold ind_complete.
split.
intros h1 p.
specialize (h1 p).
assumption.
intros h1 p.
specialize (h1 p).
assumption.
Qed.


Lemma ind_complete_sup_p : 
  (forall (p : It->Btp), 
     (exists b', Sup_p p b')) -> ind_complete_p.
rewrite ind_complete_p_iff.
intro h1.
assert (h2:forall p:It -> (Btype (Bc (ba_conv Bp))), exists b', Sup p b').
intro p.
specialize (h1 p).
assumption.
apply ind_complete_sup; auto.
Qed.

Lemma ind_complete_inf_p : 
  (forall (p : It->Btp), 
     (exists b_, Inf_p p b_)) -> ind_complete_p.
rewrite ind_complete_p_iff.
intro h1.
assert (h2:forall p:It -> (Btype (Bc (ba_conv Bp))), exists b', Inf p b').
intro p.
specialize (h1 p).
assumption.
apply ind_complete_inf; auto.
Qed.

Variable Jt:Type.
Variable IIS:Ensemble (Ensemble It).
Hypothesis covers : (FamilyUnion IIS) = IS.

Lemma infnt_assoc_sup_p :
    (forall  (p: It -> Btp) (q:Jt->Btp),
    (forall (S:Ensemble It), In IIS S ->
     (exists (j:Jt), SupGen_p S p (q j))) ->
    (forall (j:Jt), exists (S:Ensemble It), In IIS S
                          /\ SupGen_p S p (q j)) ->
     (forall (q':Btp), SupGen_p (Full_set Jt) q q' ->
      Sup_p p q')).
intros p  q h1 h2 q' h3.
rewrite Sup_p_iff.
rewrite SupGen_p_iff in h3.
assert (h1':forall S : Ensemble It, In IIS S -> exists j : Jt, SupGen_p S p (q j)).
  intros S h4. specialize (h1 _ h4).
  destruct h1 as [j h1]. rewrite SupGen_p_iff in h1.
  exists j. assumption.
assert (h2': forall j : Jt, exists S : Ensemble It, In IIS S /\ SupGen_p S p (q j)).
  intro j. specialize (h2 j). destruct h2 as [S h2].
  rewrite SupGen_p_iff in h2.
  exists S. assumption.
eapply infnt_assoc_sup; auto. apply covers.
apply h1'.
Qed.

Lemma infnt_assoc_inf_p :
    (forall  (p: It -> Btp) (q:Jt->Btp),
    (forall (S:Ensemble It), In IIS S ->
     (exists (j:Jt), InfGen_p S p (q j))) ->
    (forall (j:Jt), exists (S:Ensemble It), In IIS S
                          /\ InfGen_p S p (q j)) ->
     (forall (q':Btp), InfGen_p (Full_set Jt) q q' ->
      Inf_p p q')).
intros p  q h1 h2 q' h3.
rewrite Inf_p_iff.
rewrite InfGen_p_iff in h3.
assert (h1':forall S : Ensemble It, In IIS S -> exists j : Jt, InfGen_p S p (q j)).
  intros S h4. specialize (h1 _ h4).
  destruct h1 as [j h1]. rewrite InfGen_p_iff in h1.
  exists j. assumption.
assert (h2': forall j : Jt, exists S : Ensemble It, In IIS S /\ InfGen_p S p (q j)).
  intro j. specialize (h2 j). destruct h2 as [S h2].
  rewrite InfGen_p_iff in h2.
  exists S. assumption.
eapply infnt_assoc_inf; auto. apply covers.
apply h1'.
Qed.

Lemma infnt_distr_1_sup_p : forall {T':Type} (p:Btp) (q:T'->Btp) (q':Btp),
  let pq := (compose (fun x:Btp => p%*x) q) in
  SupGen_p (Full_set T') q q' -> SupGen_p (Full_set T') pq (p%*q').
intros T' p q q' pq.
do 2 rewrite SupGen_p_iff.
rewrite ba_conv_times.
apply (@infnt_distr_1_sup (ba_conv Bp)).
Qed.

Lemma infnt_distr_1_inf_p : forall {T':Type} (p:Btp) (q:T'->Btp) (q':Btp),
  let pq := (compose (fun x:Btp => p%+x) q) in
  InfGen_p (Full_set T') q q' -> InfGen_p (Full_set T') pq (p%+q').
intros T' p q q' pq.
do 2 rewrite InfGen_p_iff.
rewrite ba_conv_plus.
apply (@infnt_distr_1_inf (ba_conv Bp)).
Qed.

(*ommitted sup_le and Sup', because I believe they're obsolete.*)
End ParametricAnalogues.

(*Theorems*)
Arguments Bcompi_p_eq [T] [Bp] [It] _.
Arguments Bcompi_p_eq' [T] [Bp] [It] _.
Arguments doub_neg_ind_p [T] [Bp] [It] _.
Arguments SupGen_p_iff [T] [Bp] [Gt] _ _ _.
Arguments InfGen_p_iff [T] [Bp] [Gt] _ _ _.
Arguments Sup_p_iff [T] [Bp] [It] _ _.
Arguments Inf_p_iff [T] [Bp] [It] _ _.
Arguments comp_ind_p [T] [Bp] [It] _ _.
Arguments infnt_de_mor1_p [T] [Bp] [It] _ _ _.
Arguments infnt_de_mor2_p [T] [Bp] [It] _ _ _.
Arguments infnt_de_mor3_p [T] [Bp] [It] _ _ _.
Arguments infnt_de_mor4_p [T] [Bp] [It] _ _ _.
Arguments ind_complete_p_iff [T] _ _.
Arguments ind_complete_sup_p [T] [Bp] [It] _ _.
Arguments ind_complete_inf_p [T] [Bp] [It] _ _.
Arguments infnt_assoc_sup_p [T] [Bp] [It] _ _ _ _ _ _ _ _ _.
Arguments infnt_assoc_inf_p [T] [Bp] [It] _ _ _ _ _ _ _ _ _.
Arguments infnt_distr_1_sup_p [T] [Bp] [T'] _ _ _ _.
Arguments infnt_distr_1_inf_p [T] [Bp] [T'] _ _ _ _.


(*Definitions*)
Arguments ba_conv_ind [T] [Bp] [Tt] _ _.
Arguments SupGen_p [T] [Bp] [Gt] _ _ _.
Arguments InfGen_p [T] [Bp] [Gt] _ _ _.
Arguments Sup_p [T] [Bp] [It] _ _.
Arguments Inf_p [T] [Bp] [It] _ _.
Arguments ind_complete_p [T] _ _.

Section ParametricAnalogues'.
Variable T:Type.
Variable Bp:Bool_Alg_p T.
Let Btp := Btype_p T (Bc_p T Bp).

Lemma sup_Sup_iff_p :
  forall (A:Ensemble Btp) (b:Btp),
    sup_p A b <-> @Sup_p _ _ (sig_set A) (@proj1_sig _ _) b.
intros A b.
rewrite sup_p_iff. rewrite Sup_p_iff.
apply (@sup_Sup_iff (ba_conv Bp)).
Qed.

Lemma inf_Inf_iff_p :
  forall (A:Ensemble Btp) (b:Btp),
    inf_p A b <-> @Inf_p _ _ (sig_set A) (@proj1_sig _ _) b.
intros A b.
rewrite inf_p_iff. rewrite Inf_p_iff.
apply (@inf_Inf_iff (ba_conv Bp)).
Qed.

Lemma set_infnt_de_mor1_p : 
  forall (A:Ensemble Btp) (b:Btp), 
    sup_p A b -> inf_p (comp_set_p A) (%-b).
intros A b.
rewrite sup_p_iff. rewrite inf_p_iff. rewrite ba_conv_comp.
rewrite comp_set_p_eq.
apply (@set_infnt_de_mor1 (ba_conv Bp)).
Qed.

Lemma set_infnt_de_mor2_p : 
  forall (A:Ensemble Btp) (b : Btp), inf_p A b -> sup_p (comp_set_p A) (%-b).
intros A b.
rewrite sup_p_iff. rewrite inf_p_iff. rewrite ba_conv_comp.
rewrite comp_set_p_eq.
apply (@set_infnt_de_mor2 (ba_conv Bp)).
Qed.

Lemma set_infnt_de_mor3_p : 
  forall (A:Ensemble Btp) (b : Btp), sup_p (comp_set_p A) b -> inf_p A (%-b).
intros A b.
rewrite sup_p_iff. rewrite inf_p_iff. rewrite ba_conv_comp.
rewrite comp_set_p_eq.
apply (@set_infnt_de_mor3 (ba_conv Bp)).
Qed.

Lemma set_infnt_de_mor4_p : 
  forall (A:Ensemble Btp) (b : Btp), inf_p (comp_set_p A) b -> sup_p A (%-b).
intros A b.
rewrite sup_p_iff. rewrite inf_p_iff. rewrite ba_conv_comp.
rewrite comp_set_p_eq.
apply (@set_infnt_de_mor4 (ba_conv Bp)).
Qed.

Definition set_complete_p : Prop :=
  forall A:Ensemble Btp,
    exists b' b_:Btp, sup_p A b' /\ inf_p A b_.

Lemma set_complete_p_iff : 
  set_complete_p <-> set_complete (ba_conv Bp).
unfold set_complete_p, set_complete.
split.
intros h1 A.
specialize (h1 A).
assumption.
intros h2 A.
specialize (h2 A).
assumption.
Qed.

Lemma sup_set_complete_ex_p :
  forall (A:Ensemble Btp),
    set_complete_p ->
    exists! p:Btp, sup_p A p.
intros A h1.
rewrite set_complete_p_iff in h1.
pose proof (@sup_set_complete_ex (ba_conv Bp) A h1) as h2.
assumption.
Qed.

Definition sup_set_complete_p (A:Ensemble Btp) (pf:set_complete_p) :=
  proj1_sig (constructive_definite_description _ (sup_set_complete_ex_p A pf)).

Definition sup_set_complete_p_eq :
  forall (A:Ensemble Btp) (pf:set_complete_p),
    sup_set_complete_p A pf =
    sup_set_complete (ba_conv_set A) (iff1 set_complete_p_iff pf).
intros A h1.
unfold sup_set_complete_p. unfold sup_set_complete.
destruct constructive_definite_description as [x h2].
destruct constructive_definite_description as [y h3].
simpl.
rewrite sup_p_iff in h2.
pose proof (sup_unq _ _ _ h2 h3) as h4.
assumption.
Qed.


Lemma sup_set_complete_compat_p :
  forall (A:Ensemble Btp) (pf:set_complete_p),
    sup_p A (sup_set_complete_p A pf).
intros A h1.
rewrite sup_p_iff.
rewrite sup_set_complete_p_eq. 
apply sup_set_complete_compat.
Qed.


Lemma inf_set_complete_ex_p :
  forall (A:Ensemble Btp),
    set_complete_p ->
    exists! p:Btp, inf_p A p.
intros A h1.
rewrite set_complete_p_iff in h1.
pose proof (@inf_set_complete_ex (ba_conv Bp) A h1) as h2.
assumption.
Qed.

Definition inf_set_complete_p (A:Ensemble Btp) (pf:set_complete_p) :=
  proj1_sig (constructive_definite_description _ (inf_set_complete_ex_p A pf)).

Definition inf_set_complete_p_eq :
  forall (A:Ensemble Btp) (pf:set_complete_p),
    inf_set_complete_p A pf =
    inf_set_complete (ba_conv_set A) (iff1 set_complete_p_iff pf).
intros A h1.
unfold inf_set_complete_p. unfold inf_set_complete.
destruct constructive_definite_description as [x h2].
destruct constructive_definite_description as [y h3].
simpl.
rewrite inf_p_iff in h2.
pose proof (inf_unq _ _ _ h2 h3) as h4.
assumption.
Qed.

Lemma inf_set_complete_compat_p :
  forall (A:Ensemble Btp) (pf:set_complete_p),
    inf_p A (inf_set_complete_p A pf).
intros A h1.
rewrite inf_p_iff.
rewrite inf_set_complete_p_eq. 
apply inf_set_complete_compat.
Qed.

Lemma set_complete_sup_p : 
  (forall A:Ensemble Btp, exists b', sup_p A b') -> set_complete_p.
intros h1.
rewrite set_complete_p_iff.
apply set_complete_sup; auto.
Qed.

Lemma set_complete_inf_p : 
  (forall A:Ensemble Btp, (exists b_, inf_p A b_)) -> set_complete_p.
intros h1.
rewrite set_complete_p_iff.
apply set_complete_inf; auto.
Qed.

Lemma set_infnt_assoc_sup_p :
  forall (F:Family Btp)
    (pf:forall A:Ensemble Btp, In F A -> exists p:Btp, sup_p A p),
    forall q:Btp, sup_p (Im (full_sig F)
                            (fun A =>
                               (proj1_sig (constructive_definite_description _
                                                                             (ex_sup_unq_p _ _ _ (pf _ (proj2_sig A))))))) q ->
    sup_p (FamilyUnion F) q.
intros F h1 q h2.
rewrite sup_p_iff in h2.
rewrite sup_p_iff.
unfold ba_conv_set in h2. unfold ba_conv_type in h2.
rewrite transfer_dep_eq_refl in h2.
assert (h3:(fun A : sig_set F =>
             proj1_sig
               (constructive_definite_description (sup_p (proj1_sig A))
                  (ex_sup_unq_p T Bp (proj1_sig A)
                     (h1 (proj1_sig A) (proj2_sig A)))))  =
             (fun A : sig_set F =>
             proj1_sig
               (constructive_definite_description (sup_p (proj1_sig A))
                  (ex_sup_unq (ba_conv Bp) (proj1_sig A)
                     (h1 (proj1_sig A) (proj2_sig A)))))).
  apply functional_extensionality.
  intro A.
  destruct constructive_definite_description as [x h3].
  destruct constructive_definite_description as [x' h4].
  simpl.
  eapply sup_unq_p. apply h3. apply h4.
rewrite h3 in h2.
eapply set_infnt_assoc_sup; auto. 
apply h2.
Qed.

Lemma set_infnt_assoc_inf_p :
  forall (F:Family Btp)
    (pf:forall A:Ensemble Btp, In F A -> exists p:Btp, inf_p A p),
    forall q:Btp, inf_p (Im (full_sig F)
                            (fun A =>
                               (proj1_sig (constructive_definite_description _
                                                                             (ex_inf_unq_p _ _ _ (pf _ (proj2_sig A))))))) q ->
    inf_p (FamilyUnion F) q.
intros F h1 q h2.
rewrite inf_p_iff in h2.
rewrite inf_p_iff.
unfold ba_conv_set in h2. unfold ba_conv_type in h2.
rewrite transfer_dep_eq_refl in h2.
assert (h3:(fun A : sig_set F =>
             proj1_sig
               (constructive_definite_description (inf_p (proj1_sig A))
                  (ex_inf_unq_p T Bp (proj1_sig A)
                     (h1 (proj1_sig A) (proj2_sig A)))))  =
             (fun A : sig_set F =>
             proj1_sig
               (constructive_definite_description (inf_p (proj1_sig A))
                  (ex_inf_unq (ba_conv Bp) (proj1_sig A)
                     (h1 (proj1_sig A) (proj2_sig A)))))).
  apply functional_extensionality.
  intro A.
  destruct constructive_definite_description as [x h3].
  destruct constructive_definite_description as [x' h4].
  simpl.
  eapply inf_unq_p. apply h3. apply h4.
rewrite h3 in h2.
eapply set_infnt_assoc_inf; auto. 
apply h2.
Qed.

Lemma sup_add_assoc_p :
  forall (A:Ensemble Btp) (a p:Btp),
    sup_p A p ->
    sup_p (Add A a) (p%+a).
intros A a p.
do 2 rewrite sup_p_iff.
apply sup_add_assoc.
Qed.

Lemma inf_add_assoc_p :
  forall (A:Ensemble Btp) (a p:Btp),
    inf_p A p ->
    inf_p (Add A a) (p%*a).
intros A a p.
do 2 rewrite inf_p_iff.
apply inf_add_assoc.
Qed.

Lemma set_infnt_distr_1_sup_p :
  forall (A:Ensemble Btp) (p q:Btp),
    sup_p A q -> sup_p (Im A (fun x => p%*x)) (p%*q).
intros A p q.
do 2 rewrite sup_p_iff.
apply set_infnt_distr_1_sup.
Qed.

Lemma set_infnt_distr_1_inf_p :
  forall (A:Ensemble Btp) (p q:Btp),
    inf_p A q -> inf_p (Im A (fun x => p%+x)) (p%+q).
intros A p q.
do 2 rewrite inf_p_iff.
apply set_infnt_distr_1_inf.
Qed.


End ParametricAnalogues'.

(*Theorems*)
Arguments sup_Sup_iff_p [T] [Bp] _ _.
Arguments inf_Inf_iff_p [T] [Bp] _ _.
Arguments set_infnt_de_mor1_p [T] [Bp] _ _ _.
Arguments set_infnt_de_mor2_p [T] [Bp] _ _ _.
Arguments set_infnt_de_mor3_p [T] [Bp] _ _ _.
Arguments set_infnt_de_mor4_p [T] [Bp] _ _ _.
Arguments sup_set_complete_ex_p [T] [Bp] _ _.
Arguments sup_set_complete_compat_p [T] [Bp] _ _.
Arguments inf_set_complete_ex_p [T] [Bp] _ _.
Arguments set_complete_sup_p [T] [Bp] _ _.
Arguments set_complete_inf_p [T] [Bp] _ _.
Arguments set_infnt_assoc_sup_p [T] [Bp] _ _ _ _.
Arguments set_infnt_assoc_inf_p [T] [Bp] _ _ _ _.
Arguments sup_add_assoc_p [T] [Bp] _ _ _ _.
Arguments inf_add_assoc_p [T] [Bp] _ _ _ _.
Arguments set_infnt_distr_1_sup_p [T] [Bp] _ _ _ _.
Arguments set_infnt_distr_1_inf_p [T] [Bp] _ _ _ _.


(*Definitions*)
Arguments sup_set_complete_p [T] [Bp] _ _.
Arguments inf_set_complete_p [T] [Bp] _ _.