(* Copyright (C) 2014-2015 Daniel Wyckoff *)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)

Require Import SetUtilities.
Require Import FunctionalExtensionality.
Require Import Description.
Require Import DecidableDec.
Require Import ProofIrrelevance.
Require Import FunctionProperties.
Require Import TypeUtilities.
Require Import LogicUtilities.
Require Import FiniteMaps.
Require Import ArithUtilities.
Require Import ListUtilities.


Definition nonzero {T:Type} (S:Ensemble (T*nat)) :=
   (forall x:T, ~ Ensembles.In S (x, 0)).

Lemma nonzero_incl : 
  forall {T:Type} (S S':Ensemble (T*nat)),
    Included S S' -> nonzero S' -> nonzero S.
intros T S S' h1 h2.
red. red in h2.
intro x. specialize (h2 x).
intro h3.
red in h1. specialize (h1 _ h3).
contradiction.
Qed.

Inductive Bag (T:Type) :=
  bag_intro :
    forall S:Ensemble (T*nat), Finite S -> self_fp S -> nonzero S ->
    Bag T.

Lemma bag_intro_functional : 
  forall {T:Type} (S S':Ensemble (T*nat)) 
         (hf:Finite S) (hf':Finite S')
         (hsf:self_fp S) (hsf':self_fp S')
         (hnz:nonzero S) (hnz':nonzero S'),
    S = S' ->
    bag_intro _ S hf hsf hnz = bag_intro _ S' hf' hsf' hnz'.
intros T S S' h1 h2 h3 h4 h5 h6 h7.
subst.
assert (h2 = h1). apply proof_irrelevance.
assert (h4 = h3). apply proof_irrelevance.
assert (h6 = h5). apply proof_irrelevance.
subst.
reflexivity.
Qed.


Lemma bag_to_pair_set {T:Type} (B:Bag T) : Ensemble (T*nat).
destruct B. refine S.
Defined.

Lemma bag_to_pair_set_inj : 
  forall (T:Type), 
    FunctionProperties.injective (@bag_to_pair_set T).
intro T. red. intros B B' h1.
destruct B as [S h2 h3 h4]; destruct B' as [S' h5 h6 h7].
simpl in h1. subst.
assert (h2 = h5). apply proof_irrelevance.
assert (h3 = h6). apply proof_irrelevance.
assert (h4 = h7). apply proof_irrelevance.
subst. reflexivity.
Qed.                              

Lemma finite_bag_to_pair_set : forall {T:Type} (B:Bag T), 
                          Finite (bag_to_pair_set B).
intros T B.
unfold bag_to_pair_set. destruct B. assumption.
Qed.

Lemma self_fp_bag_to_pair_set : forall {T:Type} (B:Bag T),
                      self_fp (bag_to_pair_set B).
intros T B.
unfold bag_to_pair_set. destruct B. assumption.
Qed.

Lemma nonzero_bag_to_pair_set : forall {T:Type} (B:Bag T),
                      nonzero (bag_to_pair_set B).
intros T B.
unfold bag_to_pair_set. destruct B. assumption.
Qed.


Definition bag_to_set {T:Type} (B:Bag T) :=
  Im (bag_to_pair_set B) (@fst _ _).

Lemma bag_to_set_dom_rel_compat : 
  forall {T:Type} (B:Bag T),
    bag_to_set B = dom_rel (bag_to_pair_set B).
unfold bag_to_set.
intros.
rewrite <- dom_rel_eq.
reflexivity.
Qed.


Lemma finite_bag_to_set : 
  forall {T:Type} (B:Bag T),
    Finite (bag_to_set B).
intros T B.
apply finite_image.
apply finite_bag_to_pair_set.
Qed.                 


Lemma bag_to_fin_map_ex :
  forall {T:Type} (B:Bag T),
    let S:= bag_to_pair_set B in 
    exists! F:Fin_map (dom_rel S) (ran_rel S) 0,
      S = fin_map_to_fps F.
intros T B S.
destruct B as [S' hf hfp hnz]. 
pose proof (dom_rel_finite _ hf) as h3.
pose proof (ran_rel_finite _ hf) as h4.
exists (fin_map_intro _ _ _ h3 h4 _  hfp).
red.
split.
apply (fin_map_to_fps_compat_s _ _ 0 h3 h4 _ hfp).
intros B h5.  
destruct B as [h6 h7 S'' h8].
assert (h9:h3 = h6). apply proof_irrelevance.
assert (h10:h4 = h7). apply proof_irrelevance.
pose proof (fin_map_to_fps_compat_s _ _ 0 h6 h7 _ h8) as h11.
rewrite <- h11 in h5.
 clear h11. subst. subst.
assert (h9:hfp = h8). apply proof_irrelevance. 
unfold bag_to_pair_set. rewrite h9.
reflexivity.
Qed.


Definition bag_to_fin_map {T:Type}
           (B:Bag T) :=
  proj1_sig 
    (constructive_definite_description _
                                       (bag_to_fin_map_ex B)).


Lemma bag_to_fin_map_compat :
  forall {T:Type} (B:Bag T),
    (bag_to_pair_set B) = fin_map_to_fps (bag_to_fin_map B).
intros T B.
unfold bag_to_fin_map.
destruct constructive_definite_description.
simpl. assumption.
Qed.



Lemma bag_to_fin_map_inj : 
  forall (T:Type) (B B':Bag T),
    (fin_map_app (bag_to_fin_map B)) = (fin_map_app (bag_to_fin_map B')) ->
    B = B'.
intros T B B' h1.
pose proof (bag_to_fin_map_compat B) as h2.
pose proof (bag_to_fin_map_compat B') as h3.
destruct B as [S h4 h5 h6]; destruct B' as [S' h7 h8 h9].
unfold bag_to_pair_set in h2. unfold bag_to_pair_set in h3. 
assert (h10:S = S').
  apply Extensionality_Ensembles.
  red. split.
  red. intros pr h11.  
  destruct (zerop (snd pr)) as [h12 | h13].
  red in h6. pose proof (h6 (fst pr)) as h13. rewrite <- h12 in h13. rewrite <-surjective_pairing in h13. contradiction.
  rewrite h2 in h11.
  assert (h12: snd pr = fin_map_app (bag_to_fin_map (bag_intro T S h4 h5 h6)) (fst pr)).
    symmetry.
    apply fin_map_to_fps_fin_map_app_compat. assumption.
  rewrite surjective_pairing. rewrite h12.
  rewrite h1.
  rewrite h3 at 1. 
  apply in_fin_map_to_fps.
  unfold bag_to_pair_set. 
  apply NNPP.
  intro h14.
  pose proof (fin_map_app_def ((bag_to_fin_map (bag_intro T S' h7 h8 h9))) _ h14) as h15.
  rewrite <- h1 in h15. rewrite <- h12 in h15. omega.
  red. intros pr h11.  
  destruct (zerop (snd pr)) as [h12 | h13].
  red in h9. pose proof (h9 (fst pr)) as h13. rewrite <- h12 in h13. rewrite <-surjective_pairing in h13. contradiction.
  rewrite h3 in h11.
  assert (h12: snd pr = fin_map_app (bag_to_fin_map (bag_intro T S' h7 h8 h9)) (fst pr)).
    symmetry.
    apply fin_map_to_fps_fin_map_app_compat. assumption.
  rewrite surjective_pairing. rewrite h12.
  rewrite <- h1.
  rewrite h2 at 1. 
  apply in_fin_map_to_fps.
  unfold bag_to_pair_set. 
  apply NNPP.
  intro h14.
  pose proof (fin_map_app_def ((bag_to_fin_map (bag_intro T S h4 h5 h6))) _ h14) as h15.
  rewrite h1 in h15. rewrite <- h12 in h15. omega.
clear h2 h3. subst.
assert (h4 = h7). apply proof_irrelevance.
assert (h5 = h8). apply proof_irrelevance.
assert (h6 = h9). apply proof_irrelevance.
subst.
reflexivity.
Qed.


Notation "B {->} x" := ((bag_to_fin_map B) |-> x) (at level 20).

Lemma bag_to_fin_map_compat' :
  forall {T:Type} (B:Bag T) (x:T),
    B{->}x = fps_to_f (bag_to_pair_set B) (self_fp_bag_to_pair_set B) 0 x.
intros T B x. 
pose proof (fin_map_app_compat (bag_to_fin_map B) x) as h1.
rewrite h1.
pose proof (bag_to_fin_map_compat B) as h2.
symmetry in h2.
pose proof (subsetT_eq_compat _ _ _ _ (fp_fin_map_to_fps (bag_to_fin_map B)) (self_fp_bag_to_pair_set B) h2) as h3.
dependent rewrite -> h3.
reflexivity.
Qed.

Lemma bag_app_functional : 
  forall {T:Type} (B B':Bag T),
    B = B' -> forall x:T, B{->}x = B'{->}x.
intros T B B' h1.
subst.
auto.
Qed.


Definition Inb1 {T:Type} (B:Bag T) (x:T) : Prop := 
  B {->} x > 0. 

Lemma inb1_dom_rel_compat :
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 B x -> Ensembles.In (dom_rel (bag_to_pair_set B)) x.
intros T B x h1.
red in h1.
destruct B as [S h2' h3' h4'].
unfold bag_to_fin_map in h1.
destruct constructive_definite_description as [F h5]. simpl in h1.
apply NNPP.
intro h6.
pose proof (fin_map_app_def F x h6) as h7. 
destruct F.
simpl in h1. simpl in h7.
omega.
Qed.




Lemma not_inb1 : forall {T:Type} (B:Bag T) (x:T),
                   ~Inb1 B x -> B {->} x = 0.
intros T B x h1. 

destruct (zerop (B {->} x)) as [h2 | h3]. assumption.
contradict h1.
red. omega.
Qed.

Definition Inb2 {T:Type} (B:Bag T) (pr:(T*nat)) : Prop :=
  B {->} (fst pr) = snd pr /\ snd pr > 0.

Lemma inb2_impl_inb1 : 
  forall {T:Type} (B:Bag T) (pr:(T*nat)),
    Inb2 B pr -> Inb1 B (fst pr).
intros T B pr h1.
red.
red in h1.
destruct h1 as [h1l h1r].
rewrite h1l.
assumption.
Qed.



Lemma inb1_inb2_compat : 
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 B x ->
    Inb2 B (x, B{->}x).
intros T B x h1.
red. simpl. split; auto.
Qed.

Lemma inb1_dec : 
  forall {T:Type} (B:Bag T) (x:T),
    {Inb1 B x} + {~Inb1 B x}.
intros T B x.
destruct (zerop (B{->}x)) as [h1 | h2].
right. red. intro h2. red in h2. omega.
left. red. auto with arith.
Qed.

Lemma inb2_inj : 
  forall {T:Type} (B:Bag T) (x:T) (m n:nat),
    Inb2 B (x, m) -> Inb2 B (x, n) ->
    m = n.
intros T B x m n h1 h2.
red in h1. red in h2. simpl in h1. simpl in h2.
destruct h1; destruct h2.
subst.
reflexivity.
Qed.


Definition Inb2_le {T:Type} (B:Bag T) (pr:(T*nat)) : Prop :=
  snd pr <= B {->} (fst pr) /\ 0 < snd pr.

Lemma inb2_impl_inb2_le : 
  forall {T:Type} (B:Bag T) (pr:(T*nat)),
    Inb2 B pr -> Inb2_le B pr.
intros T B pr h1.
red in h1. destruct h1 as [h1l h1r]. red.
split; auto with arith.
rewrite <- h1l. 
auto with arith.
Qed.



Lemma inb1_impl_inb2_le_1 : 
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 B x -> Inb2_le B (x, 1).
intros T B x h1.
red in h1. red. split; auto with sets.
Qed.

Lemma inb1_impl_inb2_le : 
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 B x -> Inb2_le B (x, B {->} x).
intros; red; simpl; split; auto.
Qed.

Lemma inb2_le_impl_inb1 : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    Inb2_le B (x, n) -> Inb1 B x. 
intros T B x n h1. red in h1. simpl in h1. red; destruct h1; omega.
Qed.


Definition Inclb {T:Type} (B B':Bag T) : Prop :=
  forall x:T, Inb1 B x -> Inb2_le B' (x, B{->}x).


Lemma inclb_refl :
  forall {T:Type} (B:Bag T),
    Inclb B B.
intros; red; apply inb1_impl_inb2_le.
Qed.

Lemma inclb_trans : 
  forall {T:Type} (B C D:Bag T),
    Inclb B C -> Inclb C D -> Inclb B D.
intros T B C D h1 h2.
red. red in h1. red in h2.
intros x h3.
specialize (h1 _ h3).
pose proof (inb2_le_impl_inb1 _ _ _ h1) as h4.
specialize (h2 _ h4).
red in h1. red in h2. simpl in h1. simpl in h2.
destruct h1 as [h1l h1r]. destruct h2 as [h2l h2r].
red. simpl. split.
omega.
assumption.
Qed.


(* maybe add a reverse implication.*)
Lemma bag_in_compat : 
  forall {T:Type} (B:Bag T) (pr:T*nat),
    let S:= (bag_to_pair_set B) in
    Ensembles.In S pr -> snd pr = B {->} (fst pr).
intros T B pr S h1.
unfold bag_to_fin_map.
destruct constructive_definite_description as [F h2]. 
simpl. 
symmetry.
apply fin_map_to_fps_fin_map_app_compat.
rewrite <- h2.
assumption.
Qed.


Lemma in_dom_rel_inb1_compat :
  forall {T:Type} (B:Bag T) (x:T),
    Ensembles.In (dom_rel (bag_to_pair_set B)) x ->
    Inb1 B x.
intros T B x h1. 
red.
destruct h1 as [h1].
destruct h1 as [y h1].
pose proof (bag_in_compat _ _ h1) as h2.
simpl in h2.
rewrite <- h2.
pose proof (nonzero_bag_to_pair_set B) as h3. red in h3. specialize (h3 x).
destruct (zerop y) as [h4|]; auto with sets.
rewrite h4 in h1.
contradiction.
Qed.



Lemma inb1_dom_rel_compat_iff : 
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 B x <->
    Ensembles.In (dom_rel (bag_to_pair_set B)) x.
intros T B x. split.
intro h1. apply inb1_dom_rel_compat; auto.
apply in_dom_rel_inb1_compat.
Qed.    

Lemma inb1_in_compat : 
  forall {T:Type} (B:Bag T) (x:T),
    let S := bag_to_pair_set B in
    Inb1 B x ->
    Ensembles.In (bag_to_pair_set B) (x, B {->} x).
intros T B x S h1.
rewrite (bag_to_fin_map_compat B) at 1.
apply in_fin_map_to_fps.
apply inb1_dom_rel_compat.
assumption.
Qed.


Lemma in_bag_to_set_iff :
  forall {T:Type} (B:Bag T) (x:T),
    Ensembles.In (bag_to_set B) x <->
    Inb1 B x.
intros T B x. split.
intro h1.
rewrite bag_to_set_dom_rel_compat in h1.
apply in_dom_rel_inb1_compat.
assumption.
intro h1.
apply Im_intro with (x, B{->}x); auto.
apply inb1_in_compat; auto.
Qed.


Lemma inb2_in_compat : 
  forall {T:Type} (B:Bag T) (pr:(T*nat)),
    Inb2 B pr -> Ensembles.In (bag_to_pair_set B) pr.
intros T B pr h1.
pose proof (inb2_impl_inb1 B pr h1) as h2.
pose proof (inb1_in_compat B _ h2) as h3.
rewrite bag_to_fin_map_compat.
red in h1.
destruct h1 as [h1l h1r].
rewrite surjective_pairing.
rewrite <- h1l.
apply in_fin_map_to_fps.
constructor.
exists (B{->}fst pr).
assumption.
Qed.

Lemma in_inb2_compat :
  forall {T:Type} (B:Bag T) (pr:(T*nat)),
    Ensembles.In (bag_to_pair_set B) pr ->
    Inb2 B pr.
intros T B pr h1.
pose proof (self_fp_bag_to_pair_set B) as h2.
assert (h3:Ensembles.In (dom_rel (bag_to_pair_set B)) (fst pr)).
  constructor. exists (snd pr). rewrite <- surjective_pairing.
  assumption.
pose proof (in_dom_rel_inb1_compat _ _ h3) as h4.
pose proof (inb1_in_compat _ _ h4) as h5.
rewrite surjective_pairing in h1.
pose proof (fp_functional h2 _ _ _ h1 h5) as h6.
red. split.
rewrite h6. reflexivity.
pose proof (nonzero_bag_to_pair_set B) as h7.
red in h7.
destruct (zerop (snd pr)) as [h8|]; auto with arith.
specialize (h7 (fst pr)).
rewrite <- h8 in h7.
contradiction.
Qed.


Definition bag_set_unq_set {T:Type} (S:Ensemble (T*nat)) :
  Ensemble (T*nat) :=
  [pr:(T*nat) | Ensembles.In S pr /\ (snd pr) > 0].

Lemma nonzero_bag_set_unq_set : 
  forall {T:Type} (S:Ensemble (T*nat)),
    nonzero (bag_set_unq_set S).
intros T S. red.
intros x h1.
inversion h1 as [h2]. clear h1. simpl in h2.
destruct h2; omega.
Qed.

Lemma bag_set_unq_set_bag_to_pair_set : 
  forall {T:Type} (B:Bag T),
    bag_set_unq_set (bag_to_pair_set B) = 
    bag_to_pair_set B.
intros T B.
destruct B as [S h1 h2 h3].
simpl.
unfold bag_set_unq_set.
red in h3.
apply Extensionality_Ensembles.
red. split.
red. intros pr h4.
destruct h4 as [h4]. destruct h4; assumption.
red. intros pr h4.
constructor. split; auto.
destruct (zerop (snd pr)) as [h5|]; auto with arith.
rewrite surjective_pairing in h4. rewrite h5 in h4.
specialize (h3 (fst pr)).
contradiction.
Qed.
        
Lemma bag_set_unq_set_incl : 
  forall {T:Type} (S:Ensemble (T*nat)),
    Included (bag_set_unq_set S) S.
intros T S pr h1.
destruct h1 as [h1]. destruct h1; assumption.
Qed.



Lemma bag_set_unq_set_preserves_inclusion : 
  forall {T:Type} (R S:Ensemble (T*nat)),
    Included R S ->
    Included (bag_set_unq_set R) (bag_set_unq_set S).
intros T R S h1.
red. intros pr h2.
destruct h2 as [h2]. destruct h2 as [h2 h3].
constructor; auto with sets.
Qed.



Lemma bag_set_unq_set_self_fp : 
  forall {T:Type} {S:Ensemble (T*nat)},
    self_fp S ->
    self_fp (bag_set_unq_set S).
intros T S h1.
pose proof (bag_set_unq_set_incl S) as h2.
apply (self_fp_incl _ _ h2).
assumption.
Qed.


Lemma finite_bag_set_unq_set : 
  forall {T:Type} (S:Ensemble (T*nat)),
    Finite S ->
    Finite (bag_set_unq_set S).
intros T S h0.
pose proof (bag_set_unq_set_incl S) as h1.
apply (Finite_downward_closed _ _ h0 _ h1). 
Qed.

Lemma im_fun1_nonzero : 
  forall {T:Type} (A:Ensemble T), 
    nonzero (Im A (fun x => (x, 1))).
intros T A x. intro h1.
inversion h1 as [x' h4 pr h5]. subst.
inversion h5.
Qed.

Lemma im_fun1_fp :
  forall {T:Type} (A:Ensemble T), self_fp (Im A (fun x => (x, 1))).
intros T A. red.
constructor.
intros x h1.
destruct h1 as [h2]. destruct h2 as [n h2].
inversion h2 as [x' h3 pr h4]. 
subst. inversion h4. subst. clear h4.
exists 1. red. split. split. constructor.
exists x'. assumption. assumption.
intros n h4. destruct h4 as [h4l h4r].
inversion h4l as [h5]. destruct h5 as [x h5].
inversion h5 as [x'' h6 pr h7].
inversion h7. reflexivity.
intros pr h1.  
inversion h1 as [x h2 pr' h3]. subst. simpl.
split.
constructor. exists 1. assumption.
constructor. exists x. assumption.
Qed.


Definition set_to_bag {T:Type} (A:Ensemble T) (pf:Finite A) :=
  bag_intro _ (Im A (fun x=>(x, 1))) (finite_image _ _ _ _ pf) (im_fun1_fp A)
            (im_fun1_nonzero A).


Lemma empty_set_nonzero : 
  forall (T:Type), nonzero (Empty_set (T*nat)).
intros; red; auto with sets.
Qed.


Definition empty_bag (T:Type) : Bag T := bag_intro _ _ (Empty_is_finite (T*nat)) (self_fp_empty T nat) (empty_set_nonzero T).

Lemma empty_pair_set_empty_bag_compat : 
  forall {T:Type} (B:Bag T),
    (bag_to_pair_set B) = Empty_set _ ->
    B = empty_bag T.
intros T B h1.
destruct B as [S h2 h3 h4].
simpl in h1. subst.
unfold empty_bag.
assert (h2 = Empty_is_finite (T*nat)). apply proof_irrelevance.
assert (h3 = (self_fp_empty T nat)). apply proof_irrelevance.
assert (h4 = (empty_set_nonzero T)). apply proof_irrelevance.
subst.
reflexivity.
Qed.

Lemma empty_bag_empty_pair_set_compat : 
  forall (T:Type),
    (bag_to_pair_set (empty_bag T)) = Empty_set _.
intros T.
unfold empty_bag. unfold bag_to_pair_set.
reflexivity.
Qed.

Lemma empty_set_empty_bag_compat : 
  forall {T:Type} (B:Bag T),
    bag_to_set B = Empty_set _ ->
    B = empty_bag T.
intros T B h1.
unfold bag_to_set in h1.
pose proof (empty_image _ _ h1) as h2.
apply empty_pair_set_empty_bag_compat.
assumption.
Qed.

Lemma empty_bag_empty_set_compat :
  forall T:Type,
    bag_to_set (empty_bag T) = Empty_set T.
intro T.
unfold bag_to_set.
rewrite empty_bag_empty_pair_set_compat.
apply image_empty.
Qed.


Lemma empty_set_empty_bag_compat_iff : 
  forall {T:Type} (B:Bag T),
    (bag_to_pair_set B) = Empty_set _ <->
    B = empty_bag T.
split.
apply empty_pair_set_empty_bag_compat.
intros; subst.
apply empty_bag_empty_pair_set_compat.
Qed.


Lemma non_empty_bag_inhabited : 
  forall {T:Type} (B:Bag T),
    B <> empty_bag T -> exists x:T, Inb1 B x.
intros T B h1.
rewrite <- empty_set_empty_bag_compat_iff in h1.
apply not_empty_Inhabited in h1.
destruct h1 as [pr h1].
exists (fst pr).
red.
pose proof (nonzero_bag_to_pair_set B) as h2.
red in h2. specialize (h2 (fst pr)).
pose proof (bag_in_compat _ _ h1) as h3. rewrite <- h3.
destruct (zerop (snd pr)) as [h4 | h5]. rewrite <- h4 in h2.
rewrite <- surjective_pairing in h2. contradiction.
red. assumption.
Qed.



Lemma all_not_inb1_empty_bag :
  forall {T:Type} (B:Bag T),
    (forall x:T, ~Inb1 B x) ->
    B = empty_bag T.
intros T B h1.
apply empty_pair_set_empty_bag_compat.
apply Extensionality_Ensembles. red. split.
red. intros pr h2.
destruct B as [S h3 h4 h5]. simpl in h2.
red in h5. 
specialize (h1 (fst pr)). 
apply not_inb1 in h1.  
assert (h6:bag_to_pair_set (bag_intro T S h3 h4 h5) = S).
simpl. reflexivity.
rewrite <- h6 in h2.
pose proof (bag_in_compat _ _ h2) as h7.
rewrite surjective_pairing in h2.
rewrite h7 in h2. rewrite h1 in h2.
pose proof (h5 (fst pr)). contradiction.
auto with sets.
Qed.

Lemma empty_bag_O :
  forall {T:Type} (x:T),
    (empty_bag T) {->} x = 0.
intros T x.
unfold bag_to_fin_map.
destruct (constructive_definite_description) as [F h1]. simpl.
simpl in F.
destruct F as [h2 h3 S h4].
pose proof h4 as h4'.
rewrite dom_rel_empty in h4'.
pose proof (fp_empty1_s _ _ h4'). subst.
simpl.
apply fps_to_f_def.
intro h5.
rewrite dom_rel_empty in h5.
contradiction.
Qed.


Lemma inhabited_non_empty_bag : 
  forall {T:Type} (B:Bag T),
    (exists x:T, Inb1 B x) ->
    B <> empty_bag T.
intros T B h1 h2.
destruct h1 as [x h1].
subst. red in h1.
pose proof (empty_bag_O x). omega.
Qed.


Lemma not_inb1_empty_bag :
  forall (T:Type),
    forall x:T, ~Inb1 (empty_bag T) x.
intros T x h1. red in h1.
pose proof (empty_bag_O x). omega.
Qed.


Lemma empty_bag_inclb :
  forall {T:Type} (B:Bag T), 
    Inclb (empty_bag T) B.
intros T B.
red.
intros x ?.
pose proof (not_inb1_empty_bag T x).
contradiction.
Qed.

Section BagAdd.


(*The peculiar definition accounts for "zero adds" and duplicate adds. 
  For "cumulative" pair adds, see [bag_add_n] functionality *)
Definition bag_to_pair_set_add {T:Type} (B:Bag T) (pr:T*nat) :=
  bag_set_unq_set (Add ((Subtract (bag_to_pair_set B) (fst pr, B {->} fst pr))) pr).

Lemma finite_bag_to_pair_set_add :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    Finite (bag_to_pair_set_add B pr).
intros T B pr.
unfold bag_to_pair_set_add.
pose proof (bag_set_unq_set_incl  (Add (Subtract (bag_to_pair_set B) (fst pr, B{->}fst pr)) pr)) as h1.
eapply Finite_downward_closed.
pose proof (finite_bag_to_pair_set B) as h2.
pose proof (incl_subtract (bag_to_pair_set B) (fst pr, B{->}fst pr)) as h3.
pose proof (Finite_downward_closed _ _ h2 _ h3) as h4.
pose proof (Add_preserves_Finite _ _ pr h4) as h5.
apply h5.
assumption.
Qed.


Lemma self_fp_bag_to_pair_set_add :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    self_fp (bag_to_pair_set_add B pr).
intros T B pr. unfold bag_to_pair_set_add.
pose proof (bag_set_unq_set_incl (Add (Subtract (bag_to_pair_set B) (fst pr, B{->}fst pr)) pr)) as h1.
eapply self_fp_incl; simpl. apply h1.
rewrite (surjective_pairing pr).
apply self_fp_add. simpl.
pose proof (incl_subtract (bag_to_pair_set B) (fst pr, B{->}fst pr)) as h2. 
eapply self_fp_incl; simpl. apply h2.
apply self_fp_bag_to_pair_set.
intro h2. simpl in h2.
destruct h2 as [h2].
destruct h2 as [n h2].
inversion h2 as [h3 h4].
pose proof (bag_in_compat _ _ h3) as h5. simpl in h5.
contradict h4. subst.
constructor.
Qed.


Lemma nonzero_bag_to_pair_set_add : 
  forall {T:Type} (B:Bag T) (pr:T*nat),
    nonzero (bag_to_pair_set_add B pr).
intros T B pr.
red. intros x h1.
inversion h1 as [h2]. clear h1.
simpl in h2.
destruct h2; omega.
Qed.


Definition bag_pair_add
           {T:Type} (B:Bag T) (pr:T*nat) :=
  bag_intro _ _ (finite_bag_to_pair_set_add B pr)
            (self_fp_bag_to_pair_set_add B pr)
            (nonzero_bag_to_pair_set_add B pr).


Lemma bag_pair_add_app_eq :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    bag_pair_add B pr{->}fst pr = snd pr.
intros T B pr. 
destruct (zerop (snd pr)) as [h1 | h2]. rewrite h1.
pose proof (bag_to_fin_map_compat' (bag_pair_add B pr) (fst pr)) as h2. simpl in h2. simpl. rewrite h2. 
apply fps_to_f_def.
intro h3.  
destruct h3 as [h3].
destruct h3 as [n h3]. inversion h3 as [h4]. clear h3. simpl in h4; destruct h4 as [h4].
inversion h4 as [? h5 | ? h6]; subst. clear h4.
contradict h5. intro h3.
inversion h3 as [h4 h5]. contradict h5.  
pose proof (bag_in_compat _ _ h4) as h5. simpl in h5. subst.
constructor.
inversion h6 as [h7]. clear h6. rewrite (surjective_pairing pr) in h7. simpl in h7. inversion h7. subst.
omega. pose proof h2 as h1. clear h2.
pose proof (bag_to_fin_map_compat' (bag_pair_add B pr) (fst pr)) as h2. simpl in h2. simpl. rewrite h2. 
assert (h3:Ensembles.In (bag_to_pair_set_add B pr) pr).
constructor. split; auto with arith. right.
constructor.
pose proof (fin_map_s_compat _ _ _ 0 pr (self_fp_bag_to_pair_set (bag_pair_add B pr)) h3).
assumption.
Qed.


Lemma bag_pair_add_intro1 :
  forall {T:Type} (B:Bag T) (pr pr':T*nat),
    (fst pr <> fst pr') ->
    Inb2 B pr -> Inb2 (bag_pair_add B pr') pr. 
intros T B pr pr' h0 h1. red. red in h1. destruct h1 as [h1l h1r].
split; auto.  
assert (h5:Ensembles.In (bag_to_pair_set_add B pr') pr).
  constructor. split; auto with arith.
  left.
  constructor. 
  rewrite <- h1l in h1r.  pose proof (inb1_in_compat _ _ h1r) as h5.
  rewrite h1l in h5.
  rewrite surjective_pairing. assumption. 
  intro h5.  
  rewrite surjective_pairing  in h5.
  inversion h5. apply neq_sym in h0. contradiction. 
pose proof (fin_map_s_compat _ _ _ 0 pr (self_fp_bag_to_pair_set (bag_pair_add B pr')) h5).
pose proof (bag_to_fin_map_compat' (bag_pair_add B pr') (fst pr)) as h6. rewrite h6.
assumption.
Qed.


Lemma bag_pair_add_intro2 :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    (snd pr > 0) ->
    Inb2 (bag_pair_add B pr) pr.
intros T B pr h1.
apply in_inb2_compat.
constructor. split; auto.
right. constructor.
Qed.


Lemma bag_pair_add_app_neq :
  forall {T:Type} (B:Bag T) (pr:T*nat) (x:T),
    x <> fst pr -> (bag_pair_add B pr) {->} x = 
                   B {->} x.
intros T B pr x h1.
destruct (inb1_dec B x) as [h2 | h3].
pose proof (inb1_inb2_compat _ _ h2) as h3.
assert (h4:x = fst (x, B{->}x)). reflexivity. rewrite h4 in h1.
pose proof (bag_pair_add_intro1 _ _ _ h1 h3) as h5.
pose proof (inb2_in_compat _ _ h5) as h6.
pose proof (bag_in_compat _ _ h6) as h7. simpl in h7.
rewrite h7.
reflexivity.
pose proof (not_inb1 _ _ h3) as h4. rewrite h4.
assert (h5:~Inb1 (bag_pair_add B pr) x).
intro h5.
pose proof (inb1_dom_rel_compat _ _ h5) as h6.
destruct h6 as [h6l].
destruct h6l as [n h6l].
inversion h6l as [h7]. clear h6l.
simpl in h7. destruct h7 as [h7l h7r].
inversion h7l as [? h8 | ? h9]; subst.  clear h7l.
inversion h8 as [h9]. clear h8.
pose proof (bag_in_compat _ _ h9) as h10. simpl in h10.
omega.
inversion h9. subst. simpl in h1. contradiction h1. reflexivity.
apply not_inb1. assumption.
Qed.



Definition bag_self_fp_add_set
           {T:Type} {C:Ensemble (T*nat)}
           (pfsf:self_fp C) (x:T) :=
  let n := fps_to_f _ pfsf 0 x in
  (Subtract (Add C (x, S n)) (x, n)).

Lemma in_bag_self_fp_add_set :
  forall {T:Type} {C:Ensemble (T*nat)}
         (pfsf:self_fp C) (x:T) (n:nat),
    Ensembles.In (bag_self_fp_add_set pfsf x) (x, n) ->
    n = S (fps_to_f _ pfsf 0 x).
intros T C h1 x n h2. 
unfold bag_self_fp_add_set in h2.  
rewrite sub_add_comm in h2.
inversion h2 as [pr h2l | pr h2r]. subst.
destruct h2l as [h3 h4].
rewrite (fps_to_f_compat h1 0) in h3.
destruct h3 as [h3]. simpl in h3.
destruct h3; subst.
contradict h4. constructor. subst.
inversion h2r. reflexivity.
intro h3. inversion h3; omega.
Qed.


Lemma in_bag_self_fp_add_set_neq :
  forall {T:Type} {C:Ensemble (T*nat)}
         (pfsf:self_fp C) (x x':T) (n:nat),
    x <> x' ->
    Ensembles.In (bag_self_fp_add_set pfsf x) (x', n) ->
    n = (fps_to_f _ pfsf 0 x').
intros T C h1 x x' n h2 h3. 
unfold bag_self_fp_add_set in h3.
rewrite sub_add_comm in h3.
inversion h3 as [? h4 | ? h5]. subst.
inversion h4 as [h5 h6].
rewrite (fps_to_f_compat h1 0) in h5.
destruct h5 as [h5].  simpl in h5. destruct h5 as [h5l h5r].
assumption. subst.
inversion h5. subst.  contradict h2. reflexivity.
intro h4. inversion h4. omega.
Qed.


Lemma in_bag_self_fp_add_set_neq' : 
  forall {T:Type} {C:Ensemble (T*nat)}
         (pfsf:self_fp C) (x x':T),
    x <> x' ->
    Ensembles.In (dom_rel C) x' ->
    Ensembles.In (bag_self_fp_add_set pfsf x) (x', fps_to_f _ pfsf 0 x').
intros T C h1 x x' h2 h3.
constructor.
left.
rewrite (fps_to_f_compat h1 0) at 1.
constructor. simpl. split; auto.
intro h4.
inversion h4. contradiction.
Qed.



Lemma dom_rel_bag_self_fp_add :
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C) (x:T),
    dom_rel (bag_self_fp_add_set pf x) = Add (dom_rel C) x.
intros T C h1 x.
destruct (classic (Ensembles.In (dom_rel C) x)) as [h2 | h3].
unfold bag_self_fp_add_set.
rewrite sub_add_comm. 
apply Extensionality_Ensembles. red. split.
red. intros x' h3.
rewrite dom_rel_add in h3.
inversion h3 as [? h4 | ? h5].  subst.
pose proof (incl_subtract C (x, fps_to_f _ h1 0 x)) as h5.
pose proof (dom_rel_incl _ _ h5) as h6.
left. auto with sets. subst.
simpl in h5. inversion h5. subst. right. constructor.   
rewrite dom_rel_add.  simpl. 
rewrite in_add_eq.   
rewrite <- (dom_rel_add _ (x, fps_to_f _ h1 0 x)).
apply dom_rel_incl. 
rewrite add_sub_compat_in. auto with sets.
rewrite (fps_to_f_compat h1 0) at 1.
constructor. simpl. split; auto. assumption.
intro h3. inversion h3.
omega.
unfold bag_self_fp_add_set.
rewrite subtract_nin.
rewrite dom_rel_add. simpl.
reflexivity.
intro h4.
inversion h4 as [? h5 | ? h6]; subst.
pose proof (fp_in_dom _ _ _ h1 (x, fps_to_f _ h1 0 x) h5) as h6.
simpl in h6. contradiction.
inversion h6. omega.
Qed.

Lemma finite_bag_self_fp_add_set : 
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C) (x:T),
    Finite C -> Finite (bag_self_fp_add_set pf x).
intros T C h1 x h2.
unfold bag_self_fp_add_set.
assert (h3:Included (Subtract (Add C (x, S (fps_to_f _ h1 0 x))) (x, fps_to_f _ h1 0 x))  (Add C (x, S (fps_to_f _ h1 0 x)))).
  apply incl_subtract.
assert (h4:Finite  (Add C (x, S (fps_to_f _ h1 0 x)))).
  apply Add_preserves_Finite. assumption.
apply (Finite_downward_closed _ _ h4 _ h3).
Qed.


Lemma self_fp_bag_self_fp_add_set :
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C) (x:T),
    self_fp (bag_self_fp_add_set pf x).
intros T C h1 x. 
destruct (classic (Ensembles.In (dom_rel C) x)) as [h2 | h3]. 
constructor.
intros x' h3.
rewrite dom_rel_bag_self_fp_add in h3. 
rewrite in_add_eq in h3.
destruct (classic (x = x')) as [h4 | h5].
subst.
exists (S (fps_to_f _ h1 0 x')). red.
split. split.
constructor.
exists x'. unfold bag_self_fp_add_set.
rewrite sub_add_comm.
right. constructor.
intro h4. inversion h4. omega.
unfold bag_self_fp_add_set. rewrite sub_add_comm. right. constructor.
intro h4. inversion h4. omega. 
intros n h4. destruct h4 as [h4l h4r].  
symmetry.
apply in_bag_self_fp_add_set. assumption.
exists (fps_to_f _ h1 0 x').
red. split. split.
constructor. exists x'.
apply in_bag_self_fp_add_set_neq'; auto.
apply in_bag_self_fp_add_set_neq'; auto.
intros n h6.
destruct h6 as [h6l h6r]. symmetry. 
apply (in_bag_self_fp_add_set_neq h1 x x' n).
assumption. assumption. assumption. 
intros pr h3. split.
constructor. exists (snd pr). rewrite <- surjective_pairing. assumption.
constructor. exists (fst pr). rewrite <- surjective_pairing. assumption.
unfold bag_self_fp_add_set.
rewrite subtract_nin.
apply self_fp_add. assumption. assumption.
intro h4.
inversion h4 as [? h5 | ? h6].
subst.
rewrite (fps_to_f_compat h1 0) in h5 at 1.
destruct h5 as [h5]. simpl in h5. destruct h5; contradiction.
subst.
inversion h6 as [h7].
omega.
Qed.

Lemma nonzero_bag_self_fp_add_set :
  forall {T:Type} (C:Ensemble (T*nat)),
    nonzero C -> 
    forall (pffp:self_fp C) (x:T),
         nonzero (bag_self_fp_add_set pffp x).
intros T C h1 h2 x. red. red in h1.
intros x' h3.
inversion h3 as [h4 h5]. clear h3. 
inversion h4 as [? h6 | ? h7]. clear h4. subst.
specialize (h1 x'). contradiction. subst.
inversion h7; subst.
Qed.


Definition bag_add {T:Type}  
           (B:Bag T) (x:T) :=
  bag_intro _ _ (finite_bag_self_fp_add_set (self_fp_bag_to_pair_set B) x (finite_bag_to_pair_set B)) (self_fp_bag_self_fp_add_set (self_fp_bag_to_pair_set B) x)
            (nonzero_bag_self_fp_add_set _ (nonzero_bag_to_pair_set B) (self_fp_bag_to_pair_set B) x).


Lemma bag_add_intro1 :
  forall {T:Type} (B:Bag T) (a:T),
    Inb1 B a -> forall x:T,
                  Inb1 (bag_add B x) a.
intros T B a h1 x.
apply in_dom_rel_inb1_compat.
pose proof (inb1_dom_rel_compat _ _ h1) as h2.
simpl.
rewrite dom_rel_bag_self_fp_add.
left.
assumption.
Qed.

Lemma bag_add_intro2 : 
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 (bag_add B x) x.
intros T B x.
apply in_dom_rel_inb1_compat. simpl.
rewrite dom_rel_bag_self_fp_add.
right.
constructor.
Qed.

Lemma bag_add_app_eq :
  forall {T:Type} (B:Bag T) (x:T),
    (bag_add B x) {->} x = S (B {->} x).
intros T B x.
pose proof (bag_add_intro2 B x) as h1.
pose proof (self_fp_bag_to_pair_set B) as h2.
pose proof (in_bag_self_fp_add_set h2 x ((bag_add B x) {->} x)) as h3.
pose proof (inb1_in_compat _ _ h1) as h4.
simpl in h4.
assert (h2 = (self_fp_bag_to_pair_set B)). apply proof_irrelevance. subst.
specialize (h3 h4).
rewrite h3.
f_equal.
symmetry.
apply bag_to_fin_map_compat'.
Qed.

Lemma bag_add_app_neq :
  forall {T:Type} (B:Bag T) (a x:T),
    a <> x -> (bag_add B a) {->} x = B {->} x.
intros T B a x h1. 
destruct (zerop (B{->} x)) as [h2 | h3].
rewrite h2.
  destruct (zerop (bag_add B a{->}x)) as [h4 | h5].
  rewrite h4. reflexivity. 
assert (h6:Inb1 (bag_add B a) x). red. auto with arith.
pose proof (inb1_dom_rel_compat _ _ h6) as h7.
simpl in h7.
rewrite dom_rel_bag_self_fp_add in h7.
destruct h7 as [x h7l | x h7r].
apply in_dom_rel_inb1_compat in h7l.
red in h7l. omega.
destruct h7r. contradict h1. reflexivity.
assert (h4:Inb1 B x). red. auto with arith.
pose proof (bag_add_intro1 B x h4 a) as h7.
pose proof (inb1_inb2_compat _ _ h7) as h8.
pose proof (inb2_in_compat _ _ h8) as h9.
simpl in h9.
pose proof (in_bag_self_fp_add_set_neq (self_fp_bag_to_pair_set B) _ _ _ h1 h9) as h10.
rewrite h10 at 1.
rewrite bag_to_fin_map_compat'.
reflexivity.
Qed.


Lemma bag_add_inb2_compat : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    Inb2 B (x, n) -> Inb2 (bag_add B x) (x, S n).
intros T B x n h1.
pose proof (inb2_in_compat _ _ h1) as h2.
pose proof (bag_in_compat _ _ h2) as h3.
apply in_inb2_compat.
simpl in h3.
rewrite h3.
simpl.
constructor. right.
pose proof (bag_to_fin_map_compat' B x) as h4.
rewrite h4.
constructor.
intro h4.
rewrite (bag_to_fin_map_compat' B x) in h4.
inversion h4 as [h5].
omega.
Qed.

Lemma bag_add_inv : 
  forall {T:Type} (B:Bag T) (x:T) (pr:T*nat),
    Inb2 (bag_add B x) pr ->
    (Inb2 B pr \/ fst pr = x) /\
    (fst pr = x <-> ~Inb2 B pr).
intros T B x pr h1. split.
red in h1. destruct h1 as [h1l h1r].
destruct (eq_dec (fst pr) x) as [h2 | h3].
right. assumption.
rewrite bag_add_app_neq in h1l.
left.
assert (h4:Inb1 B (fst pr)). red. rewrite h1l. assumption.
pose proof (inb1_inb2_compat _ _ h4) as h5. rewrite h1l in h5.
rewrite surjective_pairing. assumption.
apply neq_sym.
assumption.
split. intros h2 h3.
subst. 
pose proof (inb2_in_compat _ _ h3) as h4. clear h3.
pose proof (inb2_in_compat _ _ h1) as h5. clear h1. 
simpl in h5.
destruct h5 as [h5 h6].
contradict h6.
pose proof (bag_in_compat _ _ h4) as h6.
rewrite <- bag_to_fin_map_compat'.
rewrite <- h6.
rewrite surjective_pairing. constructor.
intro h2.
red in h1. destruct h1 as [h1l h1r].
apply NNPP.
intro h3.
apply neq_sym in h3.
rewrite (bag_add_app_neq B _ _  h3) in h1l.
assert (h4:Inb2 B pr). red. split; auto.
contradiction.
Qed.


Lemma bag_to_set_add :
  forall {T:Type} (B:Bag T) (x:T),
    bag_to_set (bag_add B x) = Add (bag_to_set B) x.
intros T B x.
unfold bag_to_set.
assert (h4:x = fst (x, B{->}x)). reflexivity.
rewrite h4 at 2.
rewrite <- Im_add.  
rewrite <- dom_rel_eq.
rewrite <- dom_rel_eq.
destruct (inb1_dec B x) as [h1 | h2]. 
rewrite in_add_eq. 
simpl.
rewrite dom_rel_bag_self_fp_add.
pose proof (inb1_dom_rel_compat _ _ h1) as h5.
rewrite in_add_eq; auto. 
apply inb2_in_compat. apply inb1_inb2_compat. assumption.
pose proof (not_inb1 _ _ h2) as h3.
rewrite h3. 
simpl. rewrite dom_rel_add. simpl.
rewrite dom_rel_bag_self_fp_add.
f_equal.
Qed.



Fixpoint bag_add_n {T:Type} (B:Bag T) (x:T) (n:nat) :=
  match n with
    | O => B
    | S n => bag_add (bag_add_n B x n) x
  end.


Lemma le_bag_add_n_app :
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    n > 0 ->
    bag_add B x {->} x <= bag_add_n B x n {->} x.
intros T B x n. revert T B x.
induction n as [|n h1]; simpl.
intros; omega.
intros T B x h2. 
pose proof (bag_add_app_eq (bag_add_n B x n) x) as he. 
simpl. simpl in he. 
rewrite he. clear he.
destruct (zerop n) as [h3 | h4].
subst. simpl. pose proof (bag_add_app_eq B x) as he. simpl in he. omega.
specialize (h1 _ B x h4). 
simpl in h1. omega.
Qed.

Lemma bag_add_n_app_neq :
  forall {T:Type} (B:Bag T) (a x:T) (n:nat),
    a <> x ->
    (bag_add_n B a n) {->} x = B {->} x.
intros  T B a x n. revert T B a x.
induction n as [|n h1].
simpl. auto.
intros T B a x h2.
specialize (h1 T B a x h2).
simpl.
pose proof (bag_add_app_neq (bag_add_n B a n) _ _ h2) as h3.
simpl in h3. rewrite h3.
assumption.
Qed.

(* Maybe add a version of this to bag_subtract*)
Lemma bag_add_n_app_eq : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    (bag_add_n B x n) {->} x = B {->} x + n.
intros T B x n. revert B x.
induction n as [|n h1]; simpl; auto with arith.
intros B x.
assert (h2:B{->}x + S n = S (B{->}x + n)). omega.
rewrite h2.
specialize (h1 B x). 
pose proof (bag_add_app_eq (bag_add_n B x n) x) as h3.
simpl in h3. simpl. rewrite h3.
rewrite h1.
reflexivity.
Qed.


Lemma bag_add_n_bag_pair_add_compat :
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    ~Inb1 B x ->
    bag_add_n B x n = bag_pair_add B (x, n).
intros T B a n h0.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro x.
destruct (eq_dec a x) as [h1 | h2]. subst.
rewrite bag_pair_add_app_eq. simpl.
rewrite bag_add_n_app_eq.  
pose proof (not_inb1 _ _ h0) as h1.
rewrite h1. simpl. reflexivity.
rewrite bag_pair_add_app_neq; auto.
rewrite bag_add_n_app_neq; auto.
Qed.


Lemma bag_pair_add_bag_add_n_compat :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    B{->}fst pr <= snd pr ->
         bag_pair_add B pr =
         bag_add_n B (fst pr) (snd pr - (B{->}(fst pr))).
intros T B pr h0.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro x.
destruct (eq_dec (fst pr) x) as [h1 | h2]. subst. 
rewrite bag_pair_add_app_eq.
rewrite bag_add_n_app_eq.
simpl. omega.
rewrite bag_pair_add_app_neq.
rewrite bag_add_n_app_neq.
reflexivity. assumption.
auto.
Qed.


End BagAdd.

Section BagSubtract.
Lemma finite_bag_to_pair_set_subtract :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    Finite (Subtract (bag_to_pair_set B) pr).
intros T B pr.
eapply Finite_downward_closed.
apply finite_bag_to_pair_set.
apply incl_subtract.
Qed.

Lemma self_fp_bag_to_pair_set_subtract :
  forall {T:Type} (B:Bag T) (pr:T*nat),
    self_fp (Subtract (bag_to_pair_set B) pr).
intros T B pr.
eapply self_fp_incl.
apply incl_subtract.
apply self_fp_bag_to_pair_set.
Qed.

Lemma nonzero_bag_to_pair_set_subtract : 
  forall {T:Type} (B:Bag T) (pr:T*nat),
    nonzero (Subtract (bag_to_pair_set B) pr).
intros T B R.
eapply nonzero_incl.
apply incl_subtract.
apply nonzero_bag_to_pair_set.
Qed.




Definition bag_pair_subtract 
           {T:Type} (B:Bag T) (pr:T*nat) :=
  bag_intro _ _ (finite_bag_to_pair_set_subtract B pr)
            (self_fp_bag_to_pair_set_subtract B pr)
            (nonzero_bag_to_pair_set_subtract B pr).

Lemma bag_pair_subtract_O :
  forall {T:Type} (B:Bag T) (x:T),
    bag_pair_subtract B (x, O) = B.
intros T B x.
destruct B as [S h1 h2 h3].
unfold bag_pair_subtract.
apply bag_intro_functional.
simpl.
assert (h4:~Ensembles.In S (x, 0)).
  intro h5. red in h3. specialize (h3 x). contradiction.
apply subtract_nin.
assumption.
Qed.




Lemma bag_pair_subtract_ninb2 : 
  forall {T:Type} (B:Bag T) (pr:T*nat),
    ~Inb2 B pr -> bag_pair_subtract B pr = B.
intros T B pr h1.
destruct B as [S h2 h3 h4]. unfold bag_pair_subtract.
apply bag_intro_functional.
unfold bag_to_pair_set.
apply Extensionality_Ensembles.
red. split. 
apply incl_subtract.
red. intros pr' h5.
constructor; auto.
intro h6. destruct h6; subst.
contradict h1.
red. split. symmetry.
apply bag_in_compat.
simpl.
assumption.
red in h4. specialize (h4 (fst pr)).
red in h3.
pose proof h5 as h5'.
rewrite (fps_to_f_compat h3 0) in h5'.
destruct h5' as [h5'].
destruct h5' as [h5l h5r].
destruct (zerop (snd pr)) as [h6|]; auto with arith.
rewrite <- h6 in h4.
rewrite <- surjective_pairing in h4.
contradiction.
Qed.


Lemma bag_to_pair_set_subtract_cons : 
  forall {T:Type} (B:Bag T) (l:list (T*nat)) (pr:T*nat),
    NoDup (pr::l) -> bag_to_pair_set B = list_to_set (pr::l) ->
    bag_to_pair_set (bag_pair_subtract B pr) = list_to_set l.
intros T B l pr h0 h1.
pose proof (no_dup_cons_nin _ _ h0) as h0'.
pose proof (self_fp_bag_to_pair_set B) as h2'.
rewrite h1 in h2'.
simpl in h1.
apply Extensionality_Ensembles.
red. split.
red. intros pr' h2.
simpl in h2.
destruct h2 as [h2 h3].
rewrite h1 in h2.
destruct h2 as [pr' h2l | pr' h2r].
assumption. contradiction.
red. intros pr' h2. simpl.
constructor.
pose proof (Add_intro1 _ _ pr _ h2) as h3.
rewrite <- h1 in h3.
assumption. 
rewrite <- list_to_set_in_iff in h2.
intro h3.
destruct h3.
contradiction.
Qed.

Definition bag_self_fp_subtract_set
           {T:Type} {C:Ensemble (T*nat)}
           (pfsf:self_fp C) (x:T) :=
  let n := fps_to_f _ pfsf 0 x in
  bag_set_unq_set (Subtract (Add C (x, pred n)) (x, n)).



Lemma finite_bag_self_fp_subtract_set : 
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C) (x:T),
    Finite C -> Finite (bag_self_fp_subtract_set pf x).
intros T C h1 x h2.
unfold bag_self_fp_subtract_set.
assert (h3:Included  
             (bag_set_unq_set
                (Subtract (Add C (x, pred (fps_to_f C h1 0 x)))
                          (x, fps_to_f C h1 0 x)))
             (bag_set_unq_set (Add C (x, pred (fps_to_f C h1 0 x))))).
  assert (h3:Included 
               (Subtract (Add C (x, pred (fps_to_f C h1 0 x)))
                          (x, fps_to_f C h1 0 x))
               (Add C (x, pred (fps_to_f C h1 0 x)))).
    apply incl_subtract.
    apply bag_set_unq_set_preserves_inclusion.
    assumption.
assert (h4:Finite (Add C (x, pred (fps_to_f C h1 0 x)))).
  apply Add_preserves_Finite. assumption.
pose proof (finite_bag_set_unq_set _ h4) as h5.
apply (Finite_downward_closed _ _ h5 _ h3).
Qed.


Lemma dom_rel_subtract_self_fp : 
  forall {T U:Type} (S:Ensemble (T*U)) (pr:T*U),
    self_fp S ->
    (~Ensembles.In S pr -> ~Ensembles.In (dom_rel S) (fst pr)) ->
    dom_rel (Subtract S pr) = Subtract (dom_rel S) (fst pr).
intros T U S pr h0 him. 
destruct (classic_dec (Ensembles.In S pr)) as [hin | hnin].
apply Extensionality_Ensembles. 
red. split. 
red. intros x h1.  
destruct h1 as [h1].
destruct h1 as [y h1].  
inversion h1 as [h2 h3]. 
constructor. constructor. exists y. assumption.
intro h5.
destruct h5.
rewrite surjective_pairing in hin.
pose proof (fp_functional h0 _ _ _ hin h2).  subst.
contradict h3.
rewrite <- surjective_pairing. constructor.
red.
intros x h1.
constructor. destruct h1 as [h1 h2].
destruct h1 as [h1]. destruct h1 as [y h1].
exists y. constructor; auto.
intro h3.
inversion h3. subst. simpl in h2. contradict h2. constructor. 
rewrite subtract_nin. 
specialize (him hnin).
rewrite subtract_nin; auto. assumption.
Qed.


Lemma bag_set_unq_set_sub_x_O : 
  forall {T:Type} (C:Ensemble (T*nat)) (x:T),
    bag_set_unq_set (Subtract C (x, O)) =
    bag_set_unq_set C.
intros T C x.
unfold bag_set_unq_set.
apply Extensionality_Ensembles.
red. split.
red. intros pr h1.
constructor.
destruct h1 as [h1]. destruct h1 as [h1l h1r].
destruct h1l; split; auto.
red. intros pr h1.
destruct h1 as [h1]. destruct h1 as [h1l h1r].
constructor. split.
constructor. assumption.
intros h2. destruct h2. simpl in h1r. omega.
assumption.
Qed.

Lemma bag_set_unq_set_sub_x_n : 
  forall {T:Type} (C:Ensemble (T*nat)) (x:T)
         (n:nat),
    bag_set_unq_set (Subtract C (x, n)) =
    Subtract (bag_set_unq_set C) (x, n).
intros T C x n.
apply Extensionality_Ensembles.
red. split.
red. intros pr h1.
destruct h1 as [h1]. destruct h1 as [h1 h2].
constructor. constructor.
destruct h1 as [h1l h1r].
split; auto.
intro h3. destruct h3. destruct h1 as [h1l h1r].
contradict h1r. constructor. 
red. intros pr h1.
destruct h1 as [h1]. destruct h1 as [h1].
destruct h1 as [h1l h1r].
constructor. split. constructor; auto.
assumption.
Qed.



Lemma bag_set_unq_set_add_x_O : 
  forall {T:Type} (C:Ensemble (T*nat)) (x:T),
    bag_set_unq_set (Add C (x, O)) =
    bag_set_unq_set C.
intros T C x.
unfold bag_set_unq_set.
apply Extensionality_Ensembles.
red. split. 
red. intros pr h1.
destruct h1 as [h1]. destruct h1 as [h1l h1r].
destruct h1l as [pr h1a | pr h1b].
constructor. split; auto.
destruct h1b. simpl in h1r. omega.
red. intros pr h1.
destruct h1 as [h1]. destruct h1 as [h1l h1r].
constructor. split. left. assumption.
assumption.
Qed.

Lemma bag_set_unq_set_add_x_Sn : 
  forall {T:Type} (C:Ensemble (T*nat)) (x:T) (n:nat),
    0 < n ->
    bag_set_unq_set (Add C (x, n)) =
    Add (bag_set_unq_set C) (x, n).
intros T C x n h1.
apply Extensionality_Ensembles.
red. split.
red. intros pr h2.
destruct h2 as [h2]. destruct h2 as [h2l h2r].
destruct h2l as [pr h2a | pr h2b].
left. constructor. split; auto.
destruct h2b; subst. right. constructor.

red. intros pr h2.   
constructor. split. 
destruct h2 as [pr h2l | pr h2r].
destruct h2l as [h2l]. 
left. destruct h2l; auto.
destruct h2r; subst. right. constructor.
destruct h2 as [pr h2l | pr h2r].
destruct h2l as [h2l]. destruct h2l; auto.
destruct h2r. simpl. omega.
Qed.


Lemma dom_rel_bag_self_fp_subtract :
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C) (x:T),
     let n := fps_to_f _ pf O x in
     dom_rel (bag_self_fp_subtract_set pf x) = 
     if (zerop (pred n)) then Subtract (dom_rel (bag_set_unq_set C)) x
     else dom_rel (bag_set_unq_set C).
intros T C h1 x n.
destruct (classic (Ensembles.In (dom_rel C) x)) as [h2 | h3];
destruct (zerop (pred n)) as [h4 | h5]. 
pose proof (pred_n_O_dec _ h4) as h5. clear h4. 
destruct h5 as [h5l | h5r]; subst.
unfold bag_self_fp_subtract_set.
rewrite h5l. simpl.
destruct (classic (Ensembles.In C (x, O))) as [h6 | h7].
rewrite sub_add_compat_in.
rewrite bag_set_unq_set_sub_x_O.
assert (h7:~Ensembles.In (dom_rel (bag_set_unq_set C)) x).
  intro h8.
  destruct h8 as [h8]. destruct h8 as [y h8].
  destruct h8 as [h8]. destruct h8 as [h8l h8r].
  simpl in h8r. 
  pose proof (fp_functional h1 _ _ _ h6 h8l). subst.
  omega.
rewrite subtract_nin.
reflexivity. assumption. assumption.
contradict h7.
rewrite <- h5l.
apply fps_to_f_s_compat. assumption.
unfold bag_self_fp_subtract_set. rewrite h5r. simpl.
rewrite sub_add_comm.
rewrite bag_set_unq_set_add_x_O.
rewrite bag_set_unq_set_sub_x_n.
rewrite dom_rel_subtract_self_fp. simpl. reflexivity.
apply self_fp_incl with C.  red.  intros pr h6. destruct h6 as [h6].
destruct h6; auto.
assumption.
simpl. rewrite <- h5r.
intro h6. intro h7.
contradict h6.
pose proof (bag_set_unq_set_incl C) as h8.
pose proof (dom_rel_incl _ _ h8) as h9.
assert (h10:Ensembles.In (dom_rel C) x). auto with sets.
pose proof (fps_to_f_s_compat h1 0 _ h10) as h11.
constructor. split; auto. simpl. rewrite h5r. omega.
intro h6. inversion h6.  
apply Extensionality_Ensembles.
red. split.
red. intros x' h6. 
destruct h6 as [h6]. destruct h6 as [m h6]. 
inversion h6 as [h6l].
destruct h6l as [h6l h6r].
rewrite sub_add_comm in h6l.
inversion h6l as [pr h7 | pr h8]. subst.
destruct h7 as [h7l h7r].
constructor. exists m. constructor. split; auto.
subst.
inversion h8 as [h9]. 
constructor. exists n. constructor. split.
rewrite <- h9.
apply fps_to_f_s_compat. assumption. simpl.
omega.
intro h7. inversion h7. 
pose proof (O_lt_pred_n_lt_n _ h5).
unfold n in H. omega. 
unfold bag_self_fp_subtract_set. 
rewrite bag_set_unq_set_sub_x_n.
rewrite bag_set_unq_set_add_x_Sn.
red. intros x' h6.
constructor.
destruct (classic_dec (x = x')) as [h7 | h8]. rewrite <- h7.
exists (pred n). constructor. right. constructor.
intro h8. inversion h8 as [h9].
pose proof (O_lt_pred_n_lt_n _ h5) as h10. unfold n in h9.
unfold n in h10. omega. 
destruct h6 as [h6].
destruct h6 as [r h6].
exists r.
constructor. left.
constructor.
destruct h6 as [h6]. assumption.
intro h9. inversion h9. contradiction.
unfold n in h5. assumption.
destruct (pred_n_O_dec _ h4) as [h5 | h6]. subst.
unfold bag_self_fp_subtract_set. rewrite h5. simpl.
rewrite bag_set_unq_set_sub_x_O.
rewrite bag_set_unq_set_add_x_O.
rewrite subtract_nin. reflexivity.
intro h6.
pose proof (bag_set_unq_set_incl C) as h7.
pose proof (dom_rel_incl _ _ h7) as h8.
auto with sets. subst.
unfold bag_self_fp_subtract_set. rewrite h6. simpl.
rewrite sub_add_comm.
rewrite bag_set_unq_set_add_x_O.
rewrite bag_set_unq_set_sub_x_n.
rewrite dom_rel_subtract_self_fp. simpl. reflexivity.
apply bag_set_unq_set_self_fp; auto.
simpl.
intro.
intro h7.
pose proof (bag_set_unq_set_incl C) as h8.
pose proof (dom_rel_incl _ _ h8) as h9.
auto with sets. 
intro h7. inversion h7.
unfold n in h5.
rewrite fps_to_f_def in h5. simpl in h5. omega.
assumption.
Qed.



Lemma self_fp_bag_self_fp_subtract_set :
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C) (x:T),
    self_fp (bag_self_fp_subtract_set pf x).
intros T C h1 x.  
constructor.
intros x' h3.
rewrite dom_rel_bag_self_fp_subtract in h3.
destruct (zerop (pred (fps_to_f C h1 0 x))) as [h4 | h5].
pose proof (pred_n_O_dec _ h4) as h5. clear h4.
destruct h5 as [h5l | h5r]. 
assert (h6:~Ensembles.In (dom_rel (bag_set_unq_set C)) x).
  intro h6.
  destruct h6 as [h6]. destruct h6 as [n h6]. destruct h6 as [h6].
  simpl in h6. destruct h6 as [h6l  h6r].
  pose proof (fp_in_dom _ _ _ h1 _ h6l) as h7. simpl in h7.
  pose proof (fps_to_f_s_compat h1 0 _ h7) as h8.
  pose proof (fp_functional h1 _  _ _ h6l h8). subst.
  omega.
rewrite subtract_nin in h3.
assert (h7:x <> x').
  intro h7. subst. contradiction.
destruct h3 as [h3]. destruct h3 as [n h3].
exists n. red. split. split.  
constructor. exists x'. constructor. simpl. split. rewrite h5l.
simpl.
rewrite add_subtract_a.
constructor.
pose proof (bag_set_unq_set_incl  C). auto with sets.
intro h9. inversion h9; subst. contradict h7. reflexivity.
destruct h3 as [h3]. destruct h3; auto. 
constructor. rewrite h5l. simpl.
rewrite add_subtract_a. split.
constructor. destruct h3 as [h3]. simpl in h3. destruct h3; auto.
intro h8. inversion h8. contradiction.
destruct h3 as [h3]. simpl in h3. destruct h3; auto.
intros m h8.
destruct h8 as [h8l h8r].
unfold bag_self_fp_subtract_set in h8r.
inversion h8r as [h9]. destruct h9 as [h9l h9r].
rewrite h5l in h9l. simpl in h9l.
rewrite add_subtract_a in h9l.
destruct h9l as [h9l].
destruct h3 as [h3]. destruct h3 as [h3l h3r].
apply (fp_functional h1 _ _ _ h3l h9l).
assumption. 
unfold bag_self_fp_subtract_set.
rewrite h5r. simpl.
destruct h3 as [h3 h4]. 
destruct h3 as [h3].
destruct h3 as [n h3].
exists n. red. split. split.
constructor.
exists x'.
constructor. split.
rewrite sub_add_comm. left. constructor.
pose proof (bag_set_unq_set_incl C). auto with sets.
intro h6. inversion h6. subst. contradict h4. constructor.
intro h6. inversion h6.
destruct h3 as [h3]. destruct h3; auto.
rewrite bag_set_unq_set_sub_x_n.
rewrite bag_set_unq_set_add_x_O.
constructor. assumption.
intro h6. inversion h6. subst. contradict h4. constructor. 
intros m h6.
destruct h6 as [h6l h6r].
inversion h6r as [h7]. destruct h7 as [h7l h7r].
inversion h7l as [h8].
inversion h8 as [pr h8l | pr h8r].  subst.
pose proof (bag_set_unq_set_incl C) as h9.
assert (h10:Ensembles.In C (x', n)). auto with sets. 
apply (fp_functional h1 _ _ _ h10 h8l).
subst.
inversion h8r. subst.
simpl in h7r. omega. 
destruct (classic_dec (x = x')) as [h6 | h7].
subst. 
destruct (classic (Ensembles.In (dom_rel C) x')) as [h6 | h7].
exists (pred (fps_to_f C h1 0 x')).
red. split. split.
constructor. exists x'. constructor. simpl. split; auto with arith.
rewrite sub_add_comm. right. constructor.
intro h7. inversion h7.
pose proof (O_lt_pred_n_lt_n _ h5). omega.
constructor. simpl. split; auto with arith.
rewrite sub_add_comm. right. constructor.
intro h7. inversion h7. pose proof (O_lt_pred_n_lt_n _ h5). omega.
intros n h7. destruct h7 as [h7l h7r].
destruct h7r as [h7r]. simpl in h7r. rewrite sub_add_comm in h7r.
destruct h7r as [h7a h7b].
inversion h7a as [? h8 | ? h9]. subst.
destruct h8 as [h8l h8r].
contradict h8r. 
pose proof (fps_to_f_s_compat h1 0 _ h6) as h9.
pose proof (fp_functional h1 _ _ _ h8l h9). subst. constructor.
subst.
inversion h9. reflexivity.
intro h8. inversion h8.
pose proof (O_lt_pred_n_lt_n _ h5). omega.
pose proof (fps_to_f_def h1 0 _ h7) as h8.
rewrite h8 in h5. simpl in h5. omega.
exists (fps_to_f C h1 0 x'). red. split. split.
constructor. exists x'. constructor. simpl. split.
rewrite sub_add_comm. left. constructor.
apply fps_to_f_s_compat.
destruct h3 as [h3]. destruct h3 as [n h3].
inversion h3 as [h4]. destruct h4 as [h4l h4r].
constructor. exists n. assumption.
intro h8. inversion h8. subst. contradict h7. reflexivity.
intro h8. inversion h8.
pose proof (O_lt_pred_n_lt_n _ h5). omega.
destruct h3 as [h3]. destruct h3 as [n h3].
inversion h3 as [h4]. destruct h4 as [h4l h4r].
simpl in h4r.
pose proof (fp_in_dom _ _ _ h1  _ h4l) as h8.
simpl in h8.
pose proof (fps_to_f_s_compat h1 0 _ h8) as h9.
pose proof (fp_functional h1 _ _ _ h4l h9). subst.
assumption.
constructor. simpl. split.
constructor. left.
pose proof (bag_set_unq_set_incl C) as h8.
pose proof (dom_rel_incl _ _ h8) as h9.
assert (h10:Ensembles.In (dom_rel C) x'). auto with sets.
apply (fps_to_f_s_compat h1 0 _ h10).
intro h8.
inversion h8. subst. contradict h7. reflexivity.
destruct h3 as [h3]. destruct h3 as [n h3]. inversion h3 as [h4].
simpl in h4. destruct h4 as [h4l h4r].
pose proof (fp_in_dom _ _ _ h1 _ h4l) as h8. simpl in h8.
pose proof (fps_to_f_s_compat h1 0 _ h8) as h9.
pose proof (fp_functional h1 _ _ _ h4l h9). subst. assumption.
intros n h8. destruct h8 as [h8l h8r].
inversion h8r as [h9]. simpl in h9. destruct h9 as [h9l h9r].
inversion h9l as [h10]. clear h9l.
inversion h10 as  [? h11 | ? h12]. subst.
pose proof (bag_set_unq_set_incl C) as h12.
pose proof (dom_rel_incl _ _ h12) as h13.
assert (h14:Ensembles.In (dom_rel C) x'). auto with sets.
pose proof (fps_to_f_s_compat h1 0 _ h14) as h15.
apply (fp_functional h1 _ _ _ h15 h11). subst.
inversion h12. subst. contradict h7. reflexivity.
intros pr h2. split.
constructor. exists (snd pr). rewrite <- surjective_pairing.
assumption.
constructor. exists (fst pr). rewrite <- surjective_pairing.
assumption.
Qed.


Lemma nonzero_bag_self_fp_subtract_set : 
  forall {T:Type} {C:Ensemble (T*nat)} (pf:self_fp C)
         (x:T),
  nonzero (bag_self_fp_subtract_set pf x).
intros T C h1 x. red.
intros x' h2.
inversion h2 as [h3]. clear h2. simpl in h3.
destruct h3; omega.
Qed.

Definition bag_subtract 
           {T:Type} (B:Bag T) (x:T) :=
  bag_intro _ _ (finite_bag_self_fp_subtract_set (self_fp_bag_to_pair_set B) x (finite_bag_to_pair_set B)) (self_fp_bag_self_fp_subtract_set (self_fp_bag_to_pair_set B) x) (nonzero_bag_self_fp_subtract_set (self_fp_bag_to_pair_set B) x).

Lemma self_fp_bag_self_fp_subtract_set_fps_to_f :
  forall {T:Type} {C:Ensemble (T*nat)}
         (pf:self_fp C) (x:T), 
    fps_to_f _ (self_fp_bag_self_fp_subtract_set pf x) 0 x =
    pred (fps_to_f _ pf 0 x).
intros T C h1 x. generalize (self_fp_bag_self_fp_subtract_set h1 x). intro h0.
destruct (classic (Ensembles.In (dom_rel (bag_self_fp_subtract_set h1 x)) x)) as [h2 | h3].
destruct h2 as [h2]. destruct h2 as [n h2].
inversion h2 as [h3]. clear h2. destruct h3 as [h3l h3r].
inversion h3l as [h4 h5]. clear h3l.
assert (h6:~Ensembles.In C (x, n)).
  intro h6.
  pose proof (fp_in_dom _ _ _ h1 _ h6) as h7. simpl in h7.
  pose proof (fps_to_f_s_compat h1 0 _ h7) as h8.
  pose proof (fp_functional h1 _ _ _ h6 h8). subst.
  contradict h5.
  constructor. 
destruct (classic_dec (Ensembles.In (dom_rel C) x)) as [h7 | h8].
inversion h4 as [? h9 | ? h10]. contradiction. subst.
inversion h10 as [h11].  clear h10.
destruct h7 as [h7]. destruct h7 as [m h7].
pose proof (fp_in_dom _ _ _ h1 _ h7) as h12. simpl in h12.
pose proof (fps_to_f_s_compat h1 0 _ h12) as h13.
pose proof (fp_functional h1 _ _ _ h7 h13).  subst.
assert (h14:Ensembles.In (bag_self_fp_subtract_set h1 x) (x, pred (fps_to_f C h1 0 x))).
  constructor. split; auto. constructor. right. constructor. 
  assumption.
pose proof (fp_in_dom _ _ _ h0 _ h14) as h15. simpl in h15.
pose proof (fps_to_f_s_compat h0 0 _ h15) as h16.
apply (fp_functional h0 _ _ _ h16 h14).
pose proof (fps_to_f_def h1 0 _ h8) as h9. rewrite h9. 
assert (h10:~Ensembles.In (dom_rel (bag_self_fp_subtract_set h1 x)) x).  
  contradict h8.
  destruct h8 as [h8]. destruct h8 as [m h8]. 
  inversion h8 as [h10]. simpl in h10. destruct h10 as [h10l h10r].
  inversion h10l as [h11 h12]. clear h10l.
  inversion h11 as [? h13 | ? h14]. subst.
  contradict h12.
  pose proof (fp_in_dom _ _ _ h1 _ h13) as h14.
  simpl in h14.
  pose proof (fps_to_f_s_compat h1 0 _ h14) as h15.
  pose proof (fp_functional h1 _ _ _ h13 h15) as h16. subst. constructor. subst.
  inversion h14; subst.
  rewrite h9 in h10r. simpl in h10r. omega.
pose proof (fps_to_f_def h0 0 _ h10) as h11. rewrite h11.
simpl. reflexivity. 
pose proof (fps_to_f_def h0 0 _ h3) as h4. rewrite h4.
destruct (zerop (pred (fps_to_f C h1 0 x))) as [h5 | h6].
rewrite h5. reflexivity.
contradict h3.
constructor.
exists (pred (fps_to_f C h1 0 x)). constructor.
split; auto.
constructor. right. constructor.
intro h7.
inversion h7.
pose proof (O_lt_pred_n_lt_n _ h6). omega.
Qed.

Lemma self_fp_bag_self_fp_subtract_set_fps_to_f_neq :
  forall {T:Type} {C:Ensemble (T*nat)}
         (pf:self_fp C) (a x:T),
    a <> x ->
    fps_to_f _ (self_fp_bag_self_fp_subtract_set pf a) 0 x =
    fps_to_f _ pf 0 x.
intros T C h1 a x h2.
generalize (self_fp_bag_self_fp_subtract_set h1 a). intro h3.
destruct (classic (Ensembles.In (dom_rel (bag_self_fp_subtract_set h1 a)) x)) as [h4 | h5]. 
assert (h5:Ensembles.In (dom_rel C) x).
  destruct h4 as [h4].  destruct h4 as [n h4].  destruct h4 as [h4].
  simpl in h4. destruct h4 as [h4 h5]. 
  inversion h4 as [h6 h7]. clear h4.
  inversion h6 as [? h6l | ?r h6r]. subst. clear h6.
  constructor. exists n. assumption.
  inversion h6r. contradiction.
pose proof (fps_to_f_s_compat h1 0 _ h5) as h6.
pose proof (fps_to_f_s_compat h3 0 _ h4) as h7.
inversion h7 as [h8]. clear h7. simpl in h8.
destruct h8 as [h8l h8r].
inversion h8l as [h9 h10]; clear h8l.
inversion h9 as [? h11 | ? h12]; subst; clear h9.
apply (fp_functional h1 _ _ _ h11 h6).
inversion h12. contradiction.
rewrite fps_to_f_def; auto. 
destruct (zerop (fps_to_f C h1 0 x)) as [h6 | h7]. rewrite h6. reflexivity.
assert (h8:Ensembles.In (dom_rel C) x).
  apply NNPP.
  intro h8. pose proof (fps_to_f_def h1 0 _ h8). omega.
contradict h5.
constructor.
exists (fps_to_f C h1 0 x).
constructor. simpl. split; auto with arith.
constructor.
left.
apply fps_to_f_s_compat. assumption.
intro h9.
inversion h9. contradiction.
Qed.



Lemma bag_subtract_app_eq :
  forall {T:Type} (B:Bag T) (x:T),
    (bag_subtract B x) {->} x = pred (B {->} x).
intros T B x.
do 2 rewrite bag_to_fin_map_compat'.
pose proof (self_fp_bag_self_fp_subtract_set_fps_to_f (self_fp_bag_to_pair_set B) x) as h1.
simpl. 
rewrite <- h1.
f_equal.
apply proof_irrelevance.
Qed.

Lemma bag_subtract_app_neq :
  forall {T:Type} (B:Bag T) (a x:T),
    a <> x ->
    (bag_subtract B a) {->} x = B {->} x.
intros T B a x h1.
do 2 rewrite bag_to_fin_map_compat'.
pose proof (self_fp_bag_self_fp_subtract_set_fps_to_f_neq (self_fp_bag_to_pair_set B) a x h1) as h2.
simpl.
rewrite <- h2.
f_equal.
apply proof_irrelevance.
Qed.


Lemma bag_subtract_ninb1 : 
  forall {T:Type} (B:Bag T) (x:T),
    ~Inb1 B x ->
    bag_subtract B x = B.
intros T B x h1.
destruct B as [C h2 h3 h4].
unfold bag_subtract.
apply bag_intro_functional.
pose proof h1 as h1'.  
rewrite inb1_dom_rel_compat_iff in h1.
unfold bag_self_fp_subtract_set.
assert (h5:fps_to_f (bag_to_pair_set (bag_intro T C h2 h3 h4))
          (self_fp_bag_to_pair_set (bag_intro T C h2 h3 h4)) 0 x = 0).
  destruct (zerop (fps_to_f (bag_to_pair_set (bag_intro T C h2 h3 h4))
          (self_fp_bag_to_pair_set (bag_intro T C h2 h3 h4)) 0 x)) as [h6 | h7].
  assumption.
  assert (h8:Inb1 (bag_intro T C h2 h3 h4) x).
    red. rewrite bag_to_fin_map_compat'. auto with arith.
  contradiction.
rewrite h5.
simpl.
red in h4. pose proof (h4 x) as h4'.
rewrite (sub_add_compat_nin _ _ h4'). 
apply Extensionality_Ensembles. 
red; split; try apply bag_set_unq_set_incl; auto.
red.
intros pr h6.
constructor.
split; auto.
pose proof (h4 (fst pr)) as h4''.
destruct (zerop (snd pr)) as [h7 | h8].
rewrite <- h7 in h4''. rewrite <- surjective_pairing in h4''.
contradiction.
auto with arith.
Qed.

Lemma ninb1_bag_pair_subtract_app :
  forall {T:Type} (B:Bag T) (x:T),
    ~Inb1 (bag_pair_subtract B (x, (B {->} x))) x.
intros T B x h1.
apply inb1_inb2_compat in h1.
red in h1.
simpl in h1. destruct h1 as [? h1]. 
assert (h2:Inb1 (bag_pair_subtract B (x, B{->}x)) x).
  red. assumption.
pose proof (inb1_dom_rel_compat _ _ h2) as h3.
simpl in h3.
destruct h3 as [h3].
destruct h3 as [n h3].
assert (h4:Ensembles.In (bag_to_pair_set B) (x, n)).
  pose proof (incl_subtract (bag_to_pair_set B) (x, B{->}x)) as h4.
  auto with sets. 
pose proof (bag_in_compat _ _ h4) as h5.
simpl in h5. subst.
inversion h3 as [h5 h6].
contradict h6.
constructor.
Qed.


Lemma bag_subtract_bag_pair_subtract_eq : 
  forall {T:Type} (B:Bag T) (x:T),
    bag_subtract (bag_pair_subtract B (x, B{->}x)) x =
    bag_pair_subtract B (x, B{->}x).
intros T B x. 
apply bag_subtract_ninb1.
apply ninb1_bag_pair_subtract_app.
Qed.  

Lemma bag_pair_subtract_app_eq_O : 
  forall {T:Type} (B:Bag T) (x:T),
    bag_pair_subtract B (x, B{->}x) {->} x = 0.
intros T B x.
destruct (inb1_dec B x) as [h0 | h1].
destruct B as [S h2 h3 h4].
unfold bag_pair_subtract.
rewrite bag_to_fin_map_compat'. simpl. 
assert (h5:~Ensembles.In (dom_rel (Subtract S (x, bag_intro T S h2 h3 h4 {->}x))) x).
  intro h6.
  rewrite dom_rel_subtract_self_fp in h6; auto. simpl in h6.
  destruct h6 as [h6l h6r]. contradiction h6r. constructor.
  intro h7.
  pose proof (inb1_in_compat _ _ h0) as h8. simpl in h8.
  contradiction.
apply fps_to_f_def.
assumption.
assert (h2:~Inb1 (bag_pair_subtract B (x, B{->}x)) x).
  intro h2.
  pose proof (not_inb1 _ _ h1) as h3. 
  rewrite h3 in h2.
  rewrite bag_pair_subtract_O in h2.
  contradiction.
apply fin_map_app_def.
intro h3.
apply in_dom_rel_inb1_compat in h3.
contradiction.
Qed.



Lemma bag_pair_subtract_app_neq : 
  forall {T:Type} (B:Bag T) (a x:T) (n:nat),
    a <> x ->
    bag_pair_subtract B (a, n) {->} x = B {->} x.
intros T B a x n h1.
destruct (inb1_dec B x) as [h2 | h3].
pose proof (inb1_in_compat _ _ h2) as h3.
pose proof (bag_in_compat _ _ h3) as h4. 
assert (h5:Ensembles.In (bag_to_pair_set (bag_pair_subtract B (a, n))) (x, B {->} x)).
  simpl. constructor. assumption.
  intro h5. inversion h5. contradiction.
assert (h6:Ensembles.In (dom_rel (bag_to_pair_set (bag_pair_subtract B (a, n)))) x).
  constructor.
  exists (B{->}x). assumption.
pose proof (fps_to_f_s_compat (self_fp_bag_to_pair_set (bag_pair_subtract B (a, n))) 0 _ h6) as h8.
pose proof (bag_to_fin_map_compat' (bag_pair_subtract B (a, n)) x) as h9.
simpl in h8. simpl in h9. simpl in h8.
rewrite <- h9 in h8.
pose proof (incl_subtract (bag_to_pair_set B) (a, n)) as h10.
assert (h11:Ensembles.In (bag_to_pair_set B) (x, bag_pair_subtract B (a, n){->} x)). auto with sets.
apply (fp_functional (self_fp_bag_to_pair_set B) _ _ _ h11 h3).
pose proof (not_inb1 _ _ h3) as h4. rewrite h4.
assert (h5:~Inb1 (bag_pair_subtract B (a, n)) x).
  intro h5.
  pose proof (inb1_in_compat _ _ h5) as h6.
  simpl in h6.
inversion h6 as [h7 h8].
pose proof (bag_in_compat _ _ h7) as h9. simpl in h9.
rewrite h4 in h9.
rewrite h9 in h7.
pose proof (nonzero_bag_to_pair_set B x) as h10.
contradiction.
apply not_inb1.
assumption.
Qed.


Lemma bag_pair_add_undoes_bag_pair_subtract : 
  forall {T:Type} (B:Bag T) (pr:T*nat),
    bag_pair_add (bag_pair_subtract B pr) (fst pr, B{->}fst pr) = B.
intros T B pr.
apply bag_to_fin_map_inj. apply functional_extensionality.
intro x.
destruct (eq_dec x (fst pr)) as [h1 | h2]. subst.
pose proof (bag_pair_add_app_eq (bag_pair_subtract B pr) (fst pr, B{->}fst pr)) as h1.
simpl  in h1. simpl. rewrite h1.
reflexivity.
rewrite bag_pair_add_app_neq.
apply neq_sym in h2.
pose proof (bag_pair_subtract_app_neq B (fst pr) x (snd pr) h2) as h3.
rewrite (surjective_pairing pr).
assumption.
simpl.
assumption.
Qed.



Lemma  bag_subtract_undoes_bag_add : 
  forall {T:Type} (B:Bag T) (x:T),
    bag_subtract (bag_add B x) x = B.
intros T B x.
apply bag_to_pair_set_inj.
simpl.
apply Extensionality_Ensembles.
red. split.
red. intros pr h1.
destruct h1 as [h1]. destruct h1 as [h1l h1r].
destruct h1l as [h1l h2]. 
destruct h1l as [pr h3 | pr h4].
destruct h3 as [h3 h5].
destruct h3 as [pr h3l | pr h3r].
assumption.
destruct h3r; subst.
contradict h2.
rewrite <- bag_to_fin_map_compat'.
rewrite <- bag_add_app_eq.
rewrite bag_to_fin_map_compat'.
simpl.
constructor.
destruct h4; subst.
apply inb2_in_compat.
red. simpl. simpl in h1r. split; auto.
rewrite bag_to_fin_map_compat'.
apply S_inj.
assert (h3:(fps_to_f (bag_self_fp_add_set (self_fp_bag_to_pair_set B) x)
             (self_fp_bag_to_pair_set (bag_add B x)) 0 x) > 0).
  pose proof (O_lt_pred_n _ h1r).
  auto with arith. 
simpl in h3. 
rewrite <- (S_pred _ _ h3).
rewrite <- bag_to_fin_map_compat'.
rewrite <- bag_add_app_eq.
rewrite bag_to_fin_map_compat'.
reflexivity.
red.
intros pr h1.
destruct (eq_dec (fst pr) x) as [h2 | h3].
subst.
constructor. split.
constructor. right. 
pose proof (bag_to_fin_map_compat' (bag_add B (fst pr)) (fst pr)) as h2.
simpl in h2.
rewrite <- h2.
pose proof (bag_add_app_eq B (fst pr)) as h3.
simpl. simpl in h3.
rewrite h3.
simpl.
pose proof (bag_in_compat _ _ h1) as h4.
rewrite <- h4.
rewrite surjective_pairing. constructor.
intro h2. 
rewrite surjective_pairing in h2.
inversion h2 as [h3]. clear h2.
pose proof (bag_to_fin_map_compat' (bag_add B (fst pr)) (fst pr)) as h4.
simpl in h4.
rewrite <- h4 in h3.
pose proof (bag_in_compat _ _ h1) as h5.
simpl in h3.
pose proof (bag_add_app_eq B (fst pr)) as h6. simpl in h6.
rewrite h6 in h3.
rewrite h5 in h3.
omega.
pose proof (in_inb2_compat _ _ h1) as h2.
red in h2. destruct h2; assumption.
constructor. split.
constructor.
left.
constructor.
left.
assumption.
intro h4. rewrite surjective_pairing in h4.
inversion h4 as [h5]. symmetry in h5. contradiction.
intro h4.
rewrite surjective_pairing in h4.
inversion h4 as [h5]. symmetry in h5. contradiction.
pose proof (in_inb2_compat _ _ h1) as h2.
destruct h2 as [h2l h2r]. assumption.
Qed.

Lemma bag_add_undoes_bag_subtract_inb1 :
  forall {T:Type} (B:Bag T) (x:T),
    Inb1 B x ->
    bag_add (bag_subtract B x) x = B.
intros T B a h1.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro x.
destruct (eq_dec a x) as [h2 | h3].
subst.
rewrite (bag_add_app_eq (bag_subtract B x) x).
rewrite (bag_subtract_app_eq B x).
red in h1.
rewrite <- (S_pred _ _ h1).
reflexivity.
rewrite (bag_add_app_neq (bag_subtract B a)).
rewrite (bag_subtract_app_neq B a).
reflexivity.
assumption.
assumption.
Qed.

Lemma bag_add_adds_bag_subtract_ninb1 : 
  forall {T:Type} (B:Bag T) (x:T),
    ~Inb1 B x ->
    bag_add (bag_subtract B x) x = bag_add B x.
intros T B a h1.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro x.
rewrite (bag_subtract_ninb1).
reflexivity.
assumption.
Qed.

Lemma bag_subtract_inb1_inj : 
  forall {T:Type} (B B':Bag T) (x:T),
    Inb1 B x -> Inb1 B' x ->
    bag_subtract B x = bag_subtract B' x ->
    B = B'.
intros T B B' x hin hnin h1.
pose proof (f_equal (fun B=>bag_add B x) h1) as h3.
simpl in h3.
do 2 rewrite bag_add_undoes_bag_subtract_inb1 in h3; auto.
Qed.

Lemma bag_subtract_inb1_inj' : 
  forall {T:Type} (B B':Bag T) (x:T),
    (Inb1 B x <-> Inb1 B' x) ->
    bag_subtract B x = bag_subtract B' x ->
    B = B'.
intros T B B' x h0 h1.
pose proof (f_equal (fun B=>bag_add B x) h1) as h3.
simpl in h3.
destruct (inb1_dec B x) as [hin  | hnin]; destruct (inb1_dec B' x) as [hin' | hnin'].
do 2 rewrite bag_add_undoes_bag_subtract_inb1 in h3; auto.
rewrite h0 in hin. contradiction.
rewrite h0 in hnin. contradiction.
do 2 rewrite bag_add_adds_bag_subtract_ninb1 in h3; auto.
do 2 rewrite bag_subtract_ninb1 in h1; auto.
Qed.


Lemma bag_add_inj : 
  forall {T:Type} (B B':Bag T) (x:T),
    bag_add B x = bag_add B' x ->
    B = B'.
intros T B B' x h1.
pose proof (f_equal (fun B=>bag_subtract B x) h1) as h2.
simpl in h2.
do 2 rewrite bag_subtract_undoes_bag_add in h2.
assumption.
Qed.



Lemma bag_subtract_inclb : 
  forall {T:Type} (B:Bag T) (x:T),
    Inclb (bag_subtract B x) B.
intros T B a.
red.
intros x h1.
red. simpl. split.
red in h1.
destruct (eq_dec a x) as [h2 | h3].
subst.
pose proof (bag_subtract_app_eq B x) as h2. simpl in h2.
rewrite h2. simpl in h1. rewrite h2 in h1.
omega.
pose proof (bag_subtract_app_neq B _ _ h3) as h4.
simpl in h4. rewrite h4.
omega.
red in h1. auto with arith.
Qed.

Lemma pred_bag_ex :
  forall {T:Type} (B:Bag T),
    B <> empty_bag T ->
    exists (B':Bag T) (x:T),
      B = bag_add B' x /\
      Inclb B' B /\ B' <> B.
intros T B h1.
pose proof (non_empty_bag_inhabited _ h1) as h2.
destruct h2 as [x h2].
exists (bag_subtract B x). exists x.
rewrite bag_add_undoes_bag_subtract_inb1; auto.
split; auto.
split. red.
intros w h3.
destruct (eq_dec w x) as [h4 | h5].
subst.
pose proof (bag_subtract_app_eq B x) as h4.
rewrite h4.
red.
simpl; split. omega.
red in h3.
rewrite h4 in h3.
auto with arith.
apply neq_sym in h5.
pose proof (bag_subtract_app_neq B _ _ h5) as h6.
simpl in h6. simpl. rewrite h6.
red. simpl. split; auto.
red in h3. simpl in h6. simpl in h3. rewrite h6 in h3.
auto with arith.
intro h3.
pose proof (f_equal (fun B => B{->}x) h3) as h4.
simpl in h4.
pose proof (bag_subtract_app_eq B x) as h5. 
simpl in h5. simpl in h4. rewrite h5 in h4.
red in h2.
rewrite (S_pred _ _ h2) in h4.
simpl in h4.
omega.
Qed.


Fixpoint bag_subtract_n {T:Type} (B:Bag T) (x:T) (n:nat) :=
  match n with
    | O => B
    | S n => bag_subtract (bag_subtract_n B x n) x
  end.

Lemma bag_subtract_n_app_le : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    n > 0 ->
    bag_subtract_n B x n {->} x<= bag_subtract B x {->} x.
intros T B x n. revert T B x.
induction n as [|n h1]; simpl.
intros; omega.
intros T B x h2.
destruct (zerop n) as [h3 | h4].
subst. simpl. auto with arith.
specialize (h1 _ B x h4). 
pose proof (bag_subtract_app_eq (bag_subtract_n B x n) x) as h5.
simpl in h5.
rewrite h5.
apply le_S_n.
destruct (zerop (pred (bag_subtract_n B x n{->} x))) as [h6 | h7].
rewrite h6.
omega.
assert (h8:0 < bag_subtract_n B x n{->}x). omega.
rewrite <- (S_pred _ _ h8).
auto with arith.
Qed.

Lemma bag_subtract_n_ninb1 :
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    ~Inb1 B x -> bag_subtract_n B x n = B.
intros T B x n h1. revert B x h1.
induction n as [|n h1].
simpl. auto.
intros B x h2. simpl.
rewrite h1; auto.
apply bag_subtract_ninb1.
assumption.
Qed.

Lemma bag_subtract_n_app_1 : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    n > 0 ->
    B{->}x = 1 -> bag_subtract_n B x n = bag_subtract B x.
intros T B x n h0 h1. revert B x h0 h1.
induction n as [|n h1].
intros; omega.
 intros B x h0 h2. simpl.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro y.
destruct (zerop n) as [h3 | h4]. subst. simpl. reflexivity.
specialize (h1 B x h4 h2).
rewrite h1.
destruct (classic (x = y)) as [h5 | h6]. subst.
rewrite bag_subtract_app_eq. rewrite bag_subtract_app_eq.
rewrite h2. simpl. reflexivity.
rewrite bag_subtract_app_neq; auto.
Qed.


Lemma bag_subtract_n_app_neq : 
  forall {T:Type} (B:Bag T) (a x:T) (n:nat),
    a <> x ->
    (bag_subtract_n B a n) {->} x = B {->} x.
intros  T B a x n. revert T B a x.
induction n as [|n h1].
simpl. auto.
intros T B a x h2.
specialize (h1 T B a x h2).
simpl. 
pose proof (bag_subtract_app_neq (bag_subtract_n B a n) _ _ h2) as h3.
simpl in h3. rewrite h3.
assumption.
Qed.


Lemma bag_subtract_n_eq_mono : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    bag_subtract_n (bag_subtract B x) x n{->}x <=
    bag_subtract_n B x n{->}x.
intros T B x n. revert T B x.
induction n as [|n h1].
simpl. intros. pose proof (bag_subtract_app_eq B x) as h1.
simpl in h1. rewrite h1. omega.
intros T B x.
simpl.
pose proof (bag_subtract_app_eq (bag_subtract_n (bag_subtract B x) x n) x) as h2. simpl in h2. rewrite h2.
pose proof (bag_subtract_app_eq (bag_subtract_n B x n) x) as h3.
simpl in h3. rewrite h3.
apply le_pred.
apply h1.
Qed.

Lemma bag_subtract_n_eq_pred : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    pred (bag_subtract_n B x n{->}x) = bag_subtract_n (bag_subtract B x) x n {->} x.
intros T B x n. revert T B x.
induction n as [|n h1]; simpl.
intros; rewrite <- bag_subtract_app_eq.
reflexivity.
intros T B x.
pose proof (bag_subtract_app_eq (bag_subtract_n B x n) x) as h2.
simpl in h2.
rewrite h2.
pose proof (bag_subtract_app_eq (bag_subtract_n (bag_subtract B x) x n) x) as h3. simpl in h3.
rewrite h3.
f_equal.
apply h1.
Qed.

Lemma bag_subtract_n_lt_constraint : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    bag_subtract_n B x n{->}x > 0 -> n < B {->}x.
intros T B x n. revert T B x.
induction n as [|n h1]; simpl; auto with arith.
intros T B x h2.
pose proof (bag_subtract_app_eq (bag_subtract_n B x n) x) as h3.
simpl in h3. rewrite h3 in h2.
assert (h4:bag_subtract_n (bag_subtract B x) x n {->}x > 0).
  rewrite <- bag_subtract_n_eq_pred. assumption.
specialize (h1 _ _ _ h4).
pose proof (lt_n_S _ _ h1) as h5.
rewrite bag_subtract_app_eq in h5.
destruct (zerop (B{->}x)) as [h6 | h7].
rewrite h6 in h5. omega.
rewrite <- (S_pred _ _ h7) in h5.
assumption.
Qed.


Lemma bag_pair_subtract_ninb2_le_impl : 
  forall {T:Type} (B:Bag T) (x:T) (n:nat),
    ~Inb2_le B (x, n) -> n = 0 \/ ~Inb1 B x \/ 
                     bag_subtract_n B x n =
                     bag_pair_subtract B (x, B {->} x).
intros T B x n. revert B x.
induction n as [|n h1]. 
intros B x h2.
left. reflexivity. 
intros B x h2. right.
specialize (h1 (bag_subtract B x) x).
assert (h3:~Inb2_le (bag_subtract B x) (x, n)).
  intro h4. red in h4.
  simpl  in h4. destruct h4 as [h4l h4r]. 
  unfold Inb2_le in h2. simpl in h2. simpl in h4l. 
  pose proof (bag_subtract_app_eq B x) as h5. simpl in h5.
  rewrite h5 in h4l.
  omega.
specialize (h1 h3).
destruct h1 as [h1a | [h1b | h1c]]. subst.
left.
intro h4. contradict h2.
red. simpl. split; auto with arith.
destruct (inb1_dec B x) as [h4 | h5].
assert (h6:B{->}x = 1).
  unfold Inb1 in h1b. rewrite bag_subtract_app_eq in h1b.
  assert (h5:0 <= pred (B{->}x)). auto with arith.
  red in h4. omega.  
right. rewrite h6.
pose proof (bag_subtract_bag_pair_subtract_eq B x) as h10.
rewrite h6 in h10.
rewrite <- h10.
simpl.
f_equal.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro w.
destruct (eq_dec w x) as [h8 | h9].
subst.
destruct (zerop n) as [h8 | h9]. subst.
simpl. 
contradict h2. red. simpl. split; auto with arith.
rewrite bag_subtract_n_app_1; auto.
rewrite bag_subtract_app_eq. rewrite h6. simpl.
rewrite <- h6.
symmetry.
apply bag_pair_subtract_app_eq_O.
rewrite bag_pair_subtract_app_neq; auto.
rewrite bag_subtract_n_app_neq; auto.
left; auto.
right.
simpl.
simpl in h1c. simpl. 
destruct (zerop n) as [h4 | h5].
subst. simpl.
assert (h4:~Inb1 B x).
intro h4. contradict h2. red. simpl. split; auto.
rewrite bag_subtract_ninb1.
pose proof (not_inb1 _ _ h4) as h5. rewrite h5.
rewrite bag_pair_subtract_O.
reflexivity. assumption.
apply bag_to_fin_map_inj. apply functional_extensionality.
intro a. 
destruct (eq_dec a x) as [h6 | h7]. subst.
rewrite bag_pair_subtract_app_eq_O.
rewrite bag_subtract_app_eq.
pose proof (bag_subtract_app_eq B x) as h6. simpl in h6.
rewrite h6  in h1c.
pose proof (bag_pair_subtract_app_eq_O (bag_subtract B x) x) as h7.
rewrite bag_subtract_app_eq in h7.
rewrite <- h1c in h7. 
rewrite <- bag_subtract_n_eq_pred in h7.
destruct (pred_n_O_dec _ h7) as [h8 | h9]. assumption.
contradict h3.
red. simpl.  split; auto.
assert (h10:bag_subtract_n B x n{->}x > 0). omega.
pose proof (bag_subtract_n_lt_constraint _ _ _ h10).
pose proof (bag_subtract_app_eq B x) as h11. simpl in h11.
rewrite h11. omega.
rewrite bag_subtract_app_neq.
rewrite bag_pair_subtract_app_neq.
rewrite bag_subtract_n_app_neq.
reflexivity. apply neq_sym in h7.
assumption.  apply neq_sym in h7. assumption.
apply neq_sym in h7. assumption.
Qed.

Lemma bag_subtract_n_app_ninb1 : 
  forall {T:Type} (B:Bag T) (x:T), 
    ~Inb1 (bag_subtract_n B x (B{->}x)) x.
intros T B x h1.
red in h1. 
pose proof (bag_subtract_n_lt_constraint _ _ _ h1) as h2.
omega.
Qed.


Lemma bag_subtract_n_bag_pair_subtract_compat : 
  forall {T:Type} (B:Bag T) (x:T),
    bag_subtract_n B x (B{->}x) = bag_pair_subtract B (x, B{->}x).
intros T B a.
apply bag_to_fin_map_inj.
apply functional_extensionality.
intro x.
destruct (eq_dec a x) as [h1 | h2]. subst.
pose proof (bag_subtract_n_app_ninb1 B x) as h1.
pose proof (not_inb1 _ _ h1) as h2. rewrite h2.
rewrite bag_pair_subtract_app_eq_O.
reflexivity.
rewrite bag_subtract_n_app_neq.
rewrite bag_pair_subtract_app_neq.
reflexivity. assumption. assumption.
Qed.

End BagSubtract.


Lemma im_set_bag_ex : 
  forall {T U:Type} (A:Ensemble T) (pf:Finite A)
         (f:T->U),
    exists! (B:Bag U),
      (Im A f = bag_to_set B) /\
      (forall pr:U*nat, 
        Inb2 B pr ->
        (cardinal _ [x:T | Ensembles.In A x /\ f x = fst pr] (snd pr))).
intros  T U A h1 f.
induction h1 as [|A h2 h3 x h4].
exists (empty_bag U). red. split.
split.
rewrite empty_bag_empty_set_compat.
rewrite image_empty. reflexivity.
intros pr h2. contradict h2. intro h3. 
pose proof (inb2_impl_inb1 _ _ h3) as h4.
contradict h4. apply not_inb1_empty_bag.
intros B h1. 
destruct h1 as [h1l h1r].
rewrite image_empty in h1l.
symmetry. 
apply empty_set_empty_bag_compat; auto.
destruct h3 as [B h3].
red in h3. destruct h3 as [h3l h3r]. destruct h3l as [h3a h3b].
exists (bag_add B (f x)).
red. split. split.
simpl.
rewrite bag_to_set_add.
rewrite Im_add.
f_equal. assumption.
intros pr h5.
pose proof (bag_add_inv _ _ _ h5) as h6.
destruct h6 as [h6l h6r].
destruct h6l as [h7 | h8].
assert (hneq:fst pr <> f x). tauto.
specialize (h3b _ h7).
assert (h8: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = fst pr] =
            [x0 : T | Ensembles.In A x0 /\ f x0 = fst pr]).
  apply Extensionality_Ensembles.
  red. split. red. intros t h8.
  destruct h8 as [h8]. destruct h8 as [h8l h8r].
  destruct h8l as [t h9 | t h10].
  constructor. split; auto.
  destruct h10. subst. contradict h7. auto.
  red. intros t h8.
  destruct h8 as [h8]. destruct h8 as [h8l h8r].
  constructor. split; auto. left. assumption. 
rewrite h8.
assumption.
rewrite h8.
assert (h9:~ Inb2 B pr). tauto.
unfold Inb2 in h9.
apply not_and_or in h9.
destruct h9 as [h9l | h9r].
rewrite h8 in h9l.
rewrite surjective_pairing in h5.
rewrite h8 in h5.
red in h5. simpl in h5.
destruct h5 as [h5l h5r].
pose proof (bag_add_app_eq B (f x)) as h10.
simpl in h10. rewrite h10 in h5l. 
pose proof (f_equal pred h5l) as h11.
simpl in h11.
destruct (zerop (pred (snd pr))) as [h12 |h13].
rewrite h12 in h11.
assert (h13:snd pr = 1). omega. 
rewrite h13.
assert (h14: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = f x] =
             Singleton x).
  apply Extensionality_Ensembles.
  red. split. red.
  intros t h14.
  destruct h14 as [h14].
  destruct h14 as [h14l h14r].
  destruct h14l as [t h14a | t h14b].
  assert (h15:Ensembles.In (Im A f) (f t)). apply Im_intro with t.
    assumption.
    reflexivity.
  rewrite h3a in h15. unfold bag_to_set in h15.
  destruct h15 as [pr' h15l]. subst.
  pose proof (bag_in_compat _ _ h15l) as h16.
  pose proof (nonzero_bag_to_pair_set B) as h17.
  red in h17. 
  assert (h18:snd pr' <> 0).
    intro h18. specialize (h17 (fst pr')).
    rewrite surjective_pairing in h15l. rewrite h18 in h15l.
    contradiction.
    rewrite h16 in h18.
    rewrite <- H in h18.
    contradiction.
  assumption.
  red. intros x' h15. destruct h15. constructor.
    split; auto with sets.
rewrite h14.
apply card_sing.
rewrite <- h11 in h13.
assert (h14:Inb2 B ((f x), B{->}f x)).
  red. simpl. split; auto with arith.
specialize (h3b _ h14). simpl in h3b.
rewrite h11 in h3b.
pose proof h9l as h9l'. clear h9l.
assert (h9: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = fst pr] =
            Add  [x0 : T | Ensembles.In A x0 /\ f x0 = fst pr] x).
  apply Extensionality_Ensembles.
  red. split. red. intros t h9.
  destruct h9 as [h9]. destruct h9 as [h9l h9r]. destruct h9l as [t h9a |t h9b].
  constructor. constructor. split; auto.
  destruct h9b. right. constructor.
  red. intros t h9. 
  destruct h9 as [t h9l | t h9r].
  destruct h9l as [h9l]. destruct h9l as [h9a h9b].
  constructor. split. left. assumption. assumption.
  destruct h9r. constructor. split; auto. right. constructor.
rewrite h8 in h9.
rewrite h9.
rewrite (S_pred _ _ h5r).
constructor.
assumption. 
intro h15.
destruct h15 as [h15]. destruct h15; contradiction.
assert (h10r:snd pr = 0). omega. 
rewrite surjective_pairing in h5. rewrite h10r in h5.
red in h5. simpl in h5.
destruct h5; omega.
intros B' h5.  
destruct h5 as [h5l h5r].  
rewrite Im_add in h5l.
apply bag_to_fin_map_inj. apply functional_extensionality.
intro c.
destruct (eq_dec (f x) c) as [h6 | h7].
subst.
pose proof (Add_intro2 _ (Im A f) (f x)) as h6.
rewrite h5l in h6.
rewrite in_bag_to_set_iff in h6.
destruct (inb1_dec B (f x)) as [h7 | h8].
rewrite bag_add_app_eq.
pose proof (inb1_inb2_compat _ _ h6) as h8.
pose proof (inb1_inb2_compat _ _ h7) as h9.
specialize (h3b _ h9). simpl in h3b.
specialize (h5r _ h8). simpl in h5r.
clear h9.
assert (h9: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = f x] =
            Add  [x0 : T | Ensembles.In A x0 /\ f x0 = f x] x).
  apply Extensionality_Ensembles.
  red. split. red. intros t h9.
  destruct h9 as [h9]. destruct h9 as [h9l h9r]. destruct h9l as [t h9a |t h9b].
  constructor. constructor. split; auto.
  destruct h9b. right. constructor.
  red. intros t h9. 
  destruct h9 as [t h9l | t h9r].
  destruct h9l as [h9l]. destruct h9l as [h9a h9b].
  constructor. split. left. assumption. assumption.
  destruct h9r. constructor. split; auto. right. constructor.
rewrite h9 in h5r. 
assert (h10:~Ensembles.In [x0 : T | Ensembles.In A x0 /\ f x0 = f x] x).
  intro h11. destruct h11 as [h11]. destruct h11; contradiction.
pose proof (card_add_nin _ _ h3b _ h10) as h11.
eapply cardinal_is_functional. apply h11. apply h5r. reflexivity.
pose proof (not_inb1 _ _ h8) as h9.
rewrite bag_add_app_eq.
rewrite h9.
assert (h14: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = f x] =
             Singleton x).
  apply Extensionality_Ensembles. red. split.
  red. intros t h10.
  destruct h10 as [h10]. destruct h10 as [h10l h10r].
  destruct h10l as [t h10a |t h10b].
  assert (h11:Ensembles.In (Im A f) (f t)). apply Im_intro with t; auto.
  rewrite h10r in h11.
  rewrite h3a in h11.
  contradict h8.
  rewrite in_bag_to_set_iff in h11. assumption.
  assumption.
  red. intros t h10. destruct h10. constructor. split; try right; auto. constructor.
  pose proof (inb1_inb2_compat _ _ h6) as h15.
  specialize (h5r _ h15). simpl in h5r. rewrite h14 in h5r.
  pose proof (card_sing x) as h16.
  eapply cardinal_is_functional. apply h16. apply h5r. reflexivity.
rewrite bag_add_app_neq; auto.
destruct (inb1_dec B c) as [h8 | h9]; destruct (inb1_dec B' c) as [h10 | h11].
pose proof (inb1_inb2_compat _ _ h8) as h11.
pose proof (inb1_inb2_compat _ _ h10) as h12.
specialize (h3b _ h11). simpl in h3b.
specialize (h5r _ h12). simpl in h5r.
assert (h13:[x0 : T | Ensembles.In A x0 /\ f x0 = c] = 
            [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = c]).
  apply Extensionality_Ensembles. red. split.
  red. intros t h13.
  destruct h13 as [h13]. destruct h13 as [h13l h13r]. constructor. split; try left; auto.
  red. intros t h13. destruct h13 as [h13]. destruct h13 as [h13l h13r]. destruct h13l as [t h13a | t h13b]. constructor. split; auto.
  destruct h13b. contradiction.
rewrite <- h13 in h5r.
eapply cardinal_is_functional.
apply h3b. apply h5r. reflexivity.
rewrite h3a in h5l.
rewrite <- in_bag_to_set_iff in h8.
assert (h12:Ensembles.In (bag_to_set B') c). rewrite <- h5l. left. assumption.
rewrite in_bag_to_set_iff in h12. contradiction.
rewrite h3a in h5l.
apply inb1_dom_rel_compat in h10.
rewrite <- bag_to_set_dom_rel_compat in h10.
rewrite <- h5l in h10.
destruct h10 as [c h10l | c h10r].
rewrite in_bag_to_set_iff in h10l.
contradiction.
destruct h10r. contradict h7. reflexivity.
pose proof (not_inb1 _ _ h9) as h10.
pose proof (not_inb1 _ _ h11) as h12.
rewrite h10, h12.
reflexivity.
Qed.

Definition im_set_bag {T U:Type} (A:Ensemble T) (pf:Finite A) (f:T->U) : Bag U :=
  proj1_sig (constructive_definite_description _ (im_set_bag_ex _ pf f)).

Lemma im_set_bag_compat :
  forall {T U:Type} (A:Ensemble T) (pf:Finite A) (f:T->U),
    let B := (im_set_bag A pf f) in
    (Im A f = bag_to_set B) /\
    (forall pr:U*nat,
       Inb2 B pr ->
       (cardinal _ [x:T | Ensembles.In A x /\ f x = fst pr] (snd pr))).
intros T U A h1 f B. unfold B. unfold im_set_bag.
destruct constructive_definite_description as [h2 h3].
simpl. assumption.
Qed.


Lemma im_fin_map_bag_ex : 
  forall {T U:Type} {A:Ensemble T} {D:Ensemble U} {def:U}
         (F:Fin_map A D def),
    exists! (B:Bag U),
      (im_fin_map F = bag_to_set B) /\
      (forall pr:U*nat, 
        Inb2 B pr ->
        (cardinal _ [x:T | Ensembles.In A x /\ F |-> x = fst pr] (snd pr))).
intros T U A D def F.
pose proof (im_set_bag_ex A (fin_map_fin_dom F) (fin_map_app F)) as h1.
destruct h1 as [B h1]. red in h1. destruct h1 as [h1l h1r].
destruct h1l as [h1a h1b].
exists B. red. split. split.
unfold im_fin_map. assumption.
assumption.
assumption.
Qed.

Definition im_fin_map_bag {T U:Type} {A:Ensemble T} {D:Ensemble U} 
           {def:U} (F:Fin_map A D def) :=
  proj1_sig (constructive_definite_description _ (im_fin_map_bag_ex F)).

Lemma im_fin_map_bag_compat : 
  forall {T U:Type} {A:Ensemble T} {D:Ensemble U} {def:U}
         (F:Fin_map A D def),
    let B := (im_fin_map_bag F) in
      (im_fin_map F = bag_to_set B) /\
      (forall pr:U*nat, 
        Inb2 B pr ->
        (cardinal _ [x:T | Ensembles.In A x /\ F |-> x = fst pr] (snd pr))).
intros T U A D def F. unfold im_fin_map_bag.
destruct constructive_definite_description as [B h1]. simpl.
assumption.
Qed.  


Lemma im_set_bag_empty : 
  forall {T U:Type} (f:T->U),
    im_set_bag (Empty_set T) (Empty_is_finite T) f = empty_bag U.
intros T U f.
pose proof (im_set_bag_compat _ (Empty_is_finite T) f) as h1.
destruct h1 as [h1l h1r].
rewrite image_empty in h1l.
symmetry in h1l.
apply empty_set_empty_bag_compat in h1l.
assumption.
Qed.

Lemma im_set_bag_functional1 : 
  forall {T U:Type} (A A':Ensemble T) (pf:Finite A) (pf':Finite A') (f:T->U),
    A = A' -> im_set_bag A pf f = im_set_bag A' pf' f.
intros T U A A' h1 h2 f h3. subst.
assert (h1 = h2). apply proof_irrelevance. subst.
reflexivity.
Qed.

Lemma im_set_bag_functional1' : 
  forall {T U:Type} (A:Ensemble T) (pf:Finite A) (f f':T->U),
    f = f' -> im_set_bag A pf f = im_set_bag A pf f'.
intros; subst; reflexivity.
Qed.

Lemma im_set_bag_functional2 : 
  forall {T U:Type} (A A':Ensemble T) (pf:Finite A) (pf':Finite A') (f f':T->U),
    A = A' -> f = f' -> im_set_bag A pf f = im_set_bag A' pf' f'.
intros T U A A' h1 h2 f f' h3 h4. subst.
assert (h1 = h2). apply proof_irrelevance. subst.
reflexivity.
Qed.


Definition im_set_bag_in {T U:Type} (A:Ensemble T) (pfa:Finite A) 
           (P:T->Prop) (f:{x:T|P x}->U)
           (pfx:(forall x:T, Ensembles.In A x -> P x)) : 
  Bag U :=
  im_set_bag (Full_set {x:T|Ensembles.In A x}) (iff1 (finite_full_sig_iff A) pfa)
     (fun x:{x:T|Ensembles.In A x}=>f (exist _ (proj1_sig x) (pfx _ (proj2_sig x)))).
   


Lemma im_set_bag_in_empty :
  forall {T U:Type} (pfe:Finite (Empty_set _)) 
         (P:T->Prop) (f:{x:T|P x}->U)
         (pfx:(forall x:T, Ensembles.In (Empty_set _) x -> P x)),
    im_set_bag_in _ pfe _ f pfx = empty_bag _.
intros T U h1 P f h2.
unfold im_set_bag_in.
assert (h3:Full_set {x:T | Ensembles.In (Empty_set T) x} = Empty_set _). apply Extensionality_Ensembles; red; split; simpl; auto with sets. red. intro x. pose proof (proj2_sig x). contradiction. 
pose proof (subsetT_eq_compat _ _ _ _  (iff1 (finite_full_sig_iff (Empty_set T)) h1) (Empty_is_finite {x:T | Ensembles.In (Empty_set T) x}) h3) as h4.
dependent rewrite -> h4.
apply im_set_bag_empty.
Qed.

Lemma im_set_bag_add : 
  forall {T U:Type} (A:Ensemble T) (pf:Finite A) (x:T),
    ~Ensembles.In A x ->
    forall f:T->U,
      im_set_bag (Add A x) (Add_preserves_Finite _ _ x pf) f =
      bag_add (im_set_bag A pf f) (f x).
intros T U A h1 x h2 f.
pose proof (im_set_bag_compat _ (Add_preserves_Finite _ _ x h1) f) as h3.
destruct h3 as [h3l h3r].
apply bag_to_fin_map_inj. apply functional_extensionality.
intro y.
destruct (eq_dec y (f x)) as [h4 | h5]. subst.
rewrite bag_add_app_eq.
pose proof (Add_intro2 _ A x) as h4.
assert (h5:Ensembles.In (Im (Add A x) f) (f x)). apply Im_intro with x; auto.
rewrite h3l in h5.
rewrite in_bag_to_set_iff in h5.
pose proof (inb1_inb2_compat _ _ h5) as h6.
specialize (h3r _ h6). simpl in h3r.
pose proof (im_set_bag_compat _ h1 f) as h7.
destruct h7 as [h7l h7r].
destruct (inb1_dec (im_set_bag A h1 f) (f x)) as [h8 | h9].
pose proof (inb1_inb2_compat _ _ h8) as h10.
specialize (h7r _ h10). simpl in h7r.
pose proof (@card_add_nin).
assert (h11:~Ensembles.In [x0:T | Ensembles.In A x0 /\ f x0 = f x]x).
  intro h12. destruct h12 as [h12]. destruct h12; contradiction.
pose proof (card_add_nin _ _ h7r _ h11) as h12.
assert (h13:Add [x0 : T | Ensembles.In A x0 /\ f x0 = f x] x = 
             [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = f x]).
  apply Extensionality_Ensembles. red. split. red.
  intros a h14. constructor. destruct h14 as [a h14l | a h14r].
  destruct h14l as [h14l]. destruct h14l as [h14a h14b].
  split; auto. left. assumption.
  destruct h14r; subst. split; auto with sets.
  red. intros a h14.
  destruct h14 as [h14]. destruct h14 as [h14l h14r].
  destruct h14l as [a h14a | a h14b].  left. 
  constructor. constructor; auto. right. assumption.
rewrite h13 in h12.
eapply cardinal_is_functional; auto. apply h3r. apply h12.
pose proof (not_inb1 _ _ h9) as h10. rewrite h10.
assert (h11: cardinal _ [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = f x] 1).
  assert (h12:[x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = f x] =
              Singleton x).
    apply Extensionality_Ensembles.
    red. split.
    intros a h11. destruct h11 as [h11]. destruct h11 as [h11l h11r].
    destruct h11l as [a h11a | a h11b].
    assert (h12:Ensembles.In (Im A f) (f x)).
      rewrite <- h11r. apply Im_intro with a; auto.
    rewrite h7l in h12.
    rewrite in_bag_to_set_iff in h12. contradiction.
    assumption.
    red. intros a h11. destruct h11; subst.
    constructor. split; auto with sets.
  rewrite h12.
  apply card_sing.
eapply cardinal_is_functional; auto. apply h3r. apply h11.
apply neq_sym in h5.
rewrite bag_add_app_neq; auto.
destruct (inb1_dec (im_set_bag A h1 f) y) as [h6 | h7].
pose proof (inb1_inb2_compat _ _ h6) as h7.
pose proof (im_set_bag_compat A h1 f) as h8.
destruct h8 as [h8l h8r].
specialize (h8r _ h7). simpl in h8r.
assert (h9:Inb1 (im_set_bag (Add A x) (Add_preserves_Finite T A x h1) f) y).
  rewrite <- in_bag_to_set_iff.
  rewrite <- h3l.
  rewrite <- in_bag_to_set_iff in h6.
  rewrite <- h8l in h6.
  destruct h6 as [a h6 ? h9]. subst.
  apply Im_intro with a. left; auto. auto.
pose proof (inb1_inb2_compat _ _ h9) as h10.
specialize (h3r _ h10). simpl in h3r.
assert (h11: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = y] =
              [x0 : T | Ensembles.In A x0 /\ f x0 = y]).
  apply Extensionality_Ensembles.
  red. split.
  red. intros a h11. destruct h11 as [h11]. destruct h11 as [h11l h11r].
  destruct h11l as [a h11a | a h11b]. subst.
  constructor. split; auto.
  destruct h11b; subst. contradict h5. reflexivity.
  red. intros a h11. destruct h11 as [h11]. destruct h11 as [h11l h11r].
  subst. constructor. split; auto with sets.
rewrite h11 in h3r.
eapply cardinal_is_functional; auto.
apply h3r. apply h8r.
pose proof (not_inb1 _ _ h7) as h8.
rewrite h8. 
assert (hnin:~Inb1 (im_set_bag (Add A x) (Add_preserves_Finite T A x h1) f) y).
  intro h9.  
  pose proof (inb1_inb2_compat _ _ h9) as h10.
  specialize (h3r _ h10). simpl in h3r. 
assert (h11: [x0 : T | Ensembles.In (Add A x) x0 /\ f x0 = y] =
              [x0 : T | Ensembles.In A x0 /\ f x0 = y]).
  apply Extensionality_Ensembles.
  red. split.
  red. intros a h11. destruct h11 as [h11]. destruct h11 as [h11l h11r].
  destruct h11l as [a h11a | a h11b]. subst.
  constructor. split; auto.
  destruct h11b; subst. contradict h5. reflexivity.
  red. intros a h11. destruct h11 as [h11]. destruct h11 as [h11l h11r].
  subst. constructor. split; auto with sets.
rewrite h11 in h3r.
red in h9.
pose proof (card_gt_O_inh _ _ h9 h3r) as h12.
destruct h12 as [a h12].
destruct h12 as [h12]. destruct h12 as [h12l h12r].
assert (h13:Ensembles.In (Im A f) (f a)). apply Im_intro with a; auto.
pose proof (im_set_bag_compat A h1 f) as h14.
destruct h14 as [h14l h14r].
rewrite h14l in h13. 
rewrite in_bag_to_set_iff in h13.
subst.
contradiction.
pose proof (not_inb1 _ _ hnin) as h9.
rewrite h9.
reflexivity.
Qed.


Lemma card_bag_ex : 
  forall {T:Type} (B:Bag T),
    exists! n:nat,
      forall (l:list (T*nat)),
        NoDup l -> list_to_set l = bag_to_pair_set B ->
        n = plusl (map (@snd _ _) l).
intros T B.
pose proof (finite_bag_to_pair_set B) as h1.
pose proof (finite_set_list_no_dup _ h1) as h2.
destruct h2 as [l h2].
destruct h2 as [h2l h2r].
exists (plusl (map (snd (B:=nat)) l)).
red. split.
intros l' h3 h4.
rewrite h2l in h4.
apply plusl_nat_valued_fun_functional; auto.
intros n h3.
symmetry. symmetry in h2l. apply h3; auto.
Qed.

Definition card_bag {T:Type} (B:Bag T) :=
  proj1_sig (constructive_definite_description _ (card_bag_ex B)).

Lemma card_bag_compat :
  forall {T:Type} (B:Bag T) (l:list (T*nat)), NoDup l ->
         list_to_set l = bag_to_pair_set B ->
         (card_bag B) = plusl (map (@snd _ _) l).
intros. unfold card_bag. 
destruct constructive_definite_description as [? h2].
simpl.
apply h2; auto.
Qed.


(*The "power set" of a bag, except for the bag itself *)
Definition power_bag {T:Type} (B:Bag T) :=
  [B':Bag T | Inclb B' B].


(*maximum mulitiplicity of a bag *)
Definition max_mult {T:Type} (B:Bag T) (l:list nat) 
           (pf:(list_to_set l = Im (bag_to_set B) (fin_map_app (bag_to_fin_map B)))) : nat :=
  maxl l.

  
Lemma finite_power_bag : 
  forall {T:Type} (B:Bag T),
    Finite (power_bag B).
intros T B.
pose (bag_to_set B) as A.
pose proof (finite_bag_to_set B) as h1.
pose proof (finite_image _ _ (bag_to_set B) (fin_map_app (bag_to_fin_map B)) h1) as h2.
pose proof (finite_set_list _ h2) as h3.
destruct h3 as [l h3]. symmetry in h3.
pose (max_mult _ _ h3) as mm.
pose (seg_set_weak mm) as D.
pose proof (finite_seg_set_weak mm) as h4.
pose proof (cart_prod_fin _ _ h1 h4) as h5.
pose proof (power_set_finitet _ h5) as h6.
rewrite finite_full_sig_iff.
rewrite Finite_FiniteT_iff.
assert (h7:forall B':Bag T, Ensembles.In (power_bag B) B' -> 
                           Included (bag_to_pair_set B') (cart_prod (bag_to_set B) (seg_set_weak mm))).
  intros B' h7.
  destruct h7 as [h7].
  red. intros pr h8.
  red in h7.
  pose proof (in_inb2_compat _ _ h8) as h9.
  pose proof (inb2_impl_inb1 _ _ h9) as h10.
  specialize (h7 _ h10).
  pose proof (bag_in_compat _ _ h8) as h11.
  rewrite <- h11 in h7.
  constructor. split.
  red in h7. simpl in h7.
  destruct h7 as [h7l h7r].
  unfold bag_to_set.
  assert (h12:Inb1 B (fst pr)).
    red. omega.
  apply Im_intro with (fst pr, B{->}fst pr).
  apply inb1_in_compat. assumption. simpl. reflexivity.
  constructor. unfold mm. unfold max_mult.
  red in h7. simpl in h7. destruct h7 as [h7l h7r].
  assert (h12:In (B{->}fst pr) l).
    rewrite list_to_set_in_iff.
    rewrite h3.
    apply Im_intro with (fst pr). unfold bag_to_set.
    apply Im_intro with (fst pr, B{->}fst pr).
    apply inb1_in_compat. red. omega. simpl. reflexivity.
    reflexivity.
  pose proof (maxl_compat _ _ h12) as h13.
  omega.
pose (fun B':{x:Bag T | Ensembles.In (power_bag B) x} =>
        (exist (fun S => Included S (cart_prod (bag_to_set B) (seg_set_weak mm))) (bag_to_pair_set (proj1_sig B')) (h7 _ (proj2_sig B')))) as f.
apply (inj_finite _ _ f). assumption.
red.
intros B1 B2 h8.
destruct B1; destruct B2. unfold f in h8. simpl in h8.
pose proof (exist_injective _ _ _ _ _ h8) as h9.
pose proof (bag_to_pair_set_inj _ _ _ h9).
subst.
apply proj1_sig_injective. simpl.
reflexivity.
intros. apply classic.
Qed.


Lemma power_bag_empty : 
  forall (T:Type),
    power_bag (empty_bag T) = Singleton (empty_bag T) . 
intro T. unfold power_bag.
apply Extensionality_Ensembles; red; split.
red. intros B h1. 
destruct h1 as [h1].
red in h1.
destruct (eq_dec B (empty_bag T)) as [h2 | h3].
subst. constructor.
apply non_empty_bag_inhabited in h3.
destruct h3 as [x h3].
specialize (h1 _ h3).
red in h1. simpl in h1. destruct h1 as [h1l h1r].
pose proof (empty_bag_O x) as h4.
simpl in h4. simpl in h1l. rewrite h4 in h1l.
omega.
red. intros B h1.
destruct h1.
constructor.
apply inclb_refl.
Qed.



Lemma empty_bag_in_power_bag :
  forall {T:Type} (B:Bag T),
    Ensembles.In (power_bag B) (empty_bag T).
intros T B. constructor. red.
intros x h1.
pose proof (not_inb1_empty_bag _ x).
contradiction.
Qed.

Lemma full_bag_in_power_bag :
  forall {T:Type} (B:Bag T),
    Ensembles.In (power_bag B) B.
intros T B. constructor.
apply inclb_refl.
Qed.


Lemma bag_inductive_step_coarsen :
  forall {T:Type} (P:(Bag T)->Prop),
     (forall (B:Bag T) (x:T),
       P B -> P (bag_add B x)) ->
    (forall (B:Bag T) (pr:T*nat),
       B{->}fst pr <= snd pr ->
       P B -> P (bag_pair_add B pr)).
intros  T P h1 B pr h2. 
rewrite (surjective_pairing pr). 
rewrite bag_pair_add_bag_add_n_compat; auto.
simpl.
revert h1 h2.
induction (snd pr - B{->}fst pr) as [|n h1]. 
simpl. auto.
intros h2 h3 h4. simpl.
specialize (h1 h2). specialize (h1 h3 h4).
specialize (h2 (bag_add_n B (fst pr) n) (fst pr)).
specialize (h2 h1).
assumption.
Qed.



Lemma bag_induction :
  forall {T:Type} (P:(Bag T)->Prop),
    P (empty_bag T) ->
    (forall (B:Bag T) (x:T),
       P B -> P (bag_add B x)) ->
       forall B:(Bag T), P B.
intros T P h1 h2 B.   
pose proof (bag_inductive_step_coarsen _ h2) as h3. clear h2. 
pose proof (finite_bag_to_pair_set B) as h4. 
pose proof (finite_set_list_no_dup _ h4) as h5. 
destruct h5 as [l h5]. destruct h5 as [h5l h5r]. revert h5l h5r. clear h4. revert P h1 B h3.
induction l as [|pr l h6].
intros. simpl in h5l.
apply empty_pair_set_empty_bag_compat in h5l.
subst. assumption.
intros P h1 B h2 h3l h3r. 
pose proof (bag_to_pair_set_subtract_cons _ _ _ h3r h3l) as h4.
pose proof (no_dup_cons _ _ h3r) as h9.
specialize (h6 _ h1 _ h2 h4 h9). 
assert (h10:(bag_pair_subtract B pr){->}(fst pr) <=
          B{->}(fst pr)). 
  destruct (eq_dec (snd pr) (B{->}fst pr)) as [h10 |h11].
  rewrite (surjective_pairing pr). rewrite h10.
  rewrite bag_pair_subtract_app_eq_O. simpl. omega. 
  assert (h12:~Inb2 B pr).
    intro h13. red in h13. destruct h13 as [h13l h13r].
    symmetry in h13l. contradiction.
  pose proof (bag_pair_subtract_ninb2 _ _ h12) as h13.
  pose proof (bag_app_functional _ _ h13 (fst pr)).
  omega.
specialize (h2 _ (fst pr, B{->}fst pr) h10 h6).
pose proof (bag_pair_add_undoes_bag_pair_subtract B pr) as h11.
rewrite h11 in h2.
assumption.
Qed.



Lemma plus_bag_nat_ex : 
  forall (B:Bag nat),
  exists! n:nat, 
    forall l:list (nat*nat),
      NoDup l ->
      list_to_set l = bag_to_pair_set B ->
      (n = (plusl (map (fun pr=>(fst pr) * (snd pr)) l))).
intro B.
pose proof (finite_bag_to_pair_set B) as h1.
pose proof (finite_set_list_no_dup _ h1) as h2.
destruct h2 as [l h2].
destruct h2 as [h2l h2r].
exists (plusl (map (fun pr : nat * nat => fst pr * snd pr) l)).
red. split.
intros l' h3 h4.
rewrite h2l in h4.
erewrite plusl_nat_valued_fun_functional; auto. 
intros n h3.
symmetry. symmetry in h2l. apply h3; auto.
Qed.


Definition plus_bag_nat (B:Bag nat) 
  := proj1_sig (constructive_definite_description _ (plus_bag_nat_ex B)).

Lemma plus_bag_nat_compat : 
  forall (B:Bag nat) (l:list (nat*nat)),
    NoDup l -> list_to_set l = bag_to_pair_set B ->
    (plus_bag_nat B) = (plusl (map (fun pr=>(fst pr) * (snd pr)) l)).
intros.
unfold plus_bag_nat.
destruct constructive_definite_description as [n h4].
simpl. apply h4; auto.
Qed.

Lemma plus_bag_nat_empty : plus_bag_nat (empty_bag nat) = 0.
pose proof (empty_bag_empty_pair_set_compat nat) as h1. 
assert (h2:list_to_set (@nil (nat*nat)) = Empty_set _). auto.
assert (h3:NoDup (@nil (nat*nat))). constructor.
rewrite <- h1 in h2.
pose proof (plus_bag_nat_compat _ nil h3 h2) as h4.
simpl in h4.
assumption.
Qed.

Lemma plus_bag_nat_add :
  forall (B:Bag nat) (n:nat),
    plus_bag_nat (bag_add B n) =
    plus_bag_nat B + n.
intros B n. 
pose proof (plus_bag_nat_compat (bag_add B n)) as h3.
pose proof (plus_bag_nat_compat B) as h4.
pose proof (finite_bag_to_pair_set (bag_add B n)) as h5.
pose proof (finite_set_list_no_dup _ h5) as h6.
destruct h6 as [l h6].
destruct h6 as [h6l h6r].  symmetry in h6l.
specialize (h3 _ h6r h6l).
rewrite h3.  
pose proof (finite_bag_to_pair_set B) as h7.
pose proof (finite_set_list_no_dup _ h7) as h8.
destruct h8 as [l' h8r].
destruct h8r as [h8a h8b]. symmetry in h8a.
specialize (h4 _ h8b h8a).
rewrite h4.
destruct (inb1_dec B n) as [h9 | h10].
pose proof (inb1_inb2_compat _ _ h9) as h10.
pose proof (inb2_in_compat _ _ h10) as h11.
rewrite <- h8a in h11 at 1.
rewrite <- list_to_set_in_iff in h11. 
rewrite (plusl_map_remove l' _ _ h8b h11).  simpl. 
pose proof (bag_add_intro1 B n h9 n) as h12.
pose proof (inb1_inb2_compat _ _ h12) as h13.
pose proof (inb2_in_compat _ _ h13) as h14.
rewrite <- h6l in h14 at 1.
rewrite <- list_to_set_in_iff in h14.
rewrite (plusl_map_remove l _ _ h6r h14). simpl.
simpl. 
pose proof (bag_add_app_eq B n) as h15. simpl in h15. rewrite h15.
pose proof (no_dup_remove _ (n, (bag_add B n){->}n) h6r) as h16.
simpl in h16. rewrite h15 in h16.
pose proof (no_dup_remove _ (n, B{->}n) h8b) as h17. 
assert (h18:list_to_set (remove eq_dec (n, S (B{->}n)) l) = 
            list_to_set (remove eq_dec (n, B{->}n) l')).
  apply Extensionality_Ensembles.
  red. split.
  red. intros x h18. 
  rewrite <- list_to_set_in_iff. rewrite <- list_to_set_in_iff in h18.
  apply in_remove_inv in h18. destruct h18 as [h18l h18r].
  rewrite in_remove_iff.
  assert (hneq:fst x <> n).
    intro h19. subst.
    rewrite list_to_set_in_iff in h18l.
    rewrite h6l in h18l.
    apply in_inb2_compat in h18l.
    red in h18l. destruct h18l as [h18a h18b].
    rewrite bag_add_app_eq in h18a.
    assert (x = (fst x, S (B{->}fst x))).
      apply injective_projections; auto.
      contradiction.
 split.
  rewrite list_to_set_in_iff in h18l. 
  rewrite h6l in h18l.  
  apply in_inb2_compat in h18l. 
  apply bag_add_inv in h18l.
  destruct h18l as [h18a h18b].
  destruct h18a as [h19 | h20].
  apply inb2_in_compat in h19.
  rewrite <- h8a in h19.
  rewrite <- list_to_set_in_iff in h19. assumption. subst. 
  contradict hneq. reflexivity.
  intro h19. rewrite (surjective_pairing x) in h19.
  inversion h19. contradiction.
  red. intros x h18.
  rewrite <- list_to_set_in_iff. rewrite <- list_to_set_in_iff in h18.
  apply in_remove_inv in h18. destruct h18 as [h18l h18r].
  rewrite in_remove_iff.
  assert (hneq:fst x <> n).
    intro h19. subst.
    rewrite list_to_set_in_iff in h18l.
    rewrite h8a in h18l.
    apply in_inb2_compat in h18l.
    red in h18l. destruct h18l as [h18a h18b].
    assert (x = (fst x, (B{->}fst x))).
      apply injective_projections; auto.
      contradiction.
  split.    
  rewrite list_to_set_in_iff in h18l.  
  rewrite h8a in h18l.
  apply in_inb2_compat in h18l.
  pose proof (inb2_impl_inb1 _ _ h18l) as hinb1. 
  pose proof (bag_add_intro1 _ _ hinb1 n) as h19. 
  pose proof (inb1_inb2_compat _ _ h19) as h20.
  rewrite bag_add_app_neq in h20.
  pose proof (bag_add_inv _ _ _ h20) as h21. simpl in h21.
  destruct h21 as [h21a h21b].
  destruct h21a as [h21l | h21r].
  rewrite surjective_pairing in h18l.
  pose proof (inb2_inj _ _ _ _ h18l h21l ) as h22.
  rewrite <- h22 in h20.
  rewrite list_to_set_in_iff.
  rewrite h6l.
  apply inb2_in_compat. rewrite surjective_pairing.
  assumption. contradiction. apply neq_sym. assumption.
  intro h19. rewrite (surjective_pairing x) in h19. inversion h19.
  contradiction.
pose proof (plusl_nat_valued_fun_functional _ _ h16 h17 h18 (fun pr:nat*nat => fst pr * snd pr)) as h19.
rewrite h19.  ring.
pose proof (not_inb1 _ _ h10) as h11.
assert (h12:Inb2 (bag_add B n) (n, 1)).
  red. simpl. split.
  pose proof (bag_add_app_eq B n) as h13. simpl in h13. simpl.
  rewrite h13. rewrite h11. reflexivity. auto with arith.
apply inb2_in_compat in h12.
rewrite <- h6l in h12. rewrite <- list_to_set_in_iff in h12.
pose proof (plusl_map_remove _ (fun pr:nat*nat=>fst pr*snd pr) _ h6r h12) as h13.
rewrite h13. simpl.
pose proof (no_dup_remove _ (n, 1) h6r) as h14.
assert (h15:list_to_set (remove eq_dec (n, 1) l) = list_to_set l').
  apply Extensionality_Ensembles. red. split.
  red. intros x h15.
  rewrite <- list_to_set_in_iff in h15.
  rewrite in_remove_iff in h15.
  destruct h15 as [h15l h15r].
  rewrite h8a.
  rewrite list_to_set_in_iff in h15l. rewrite h6l in h15l.
  apply in_inb2_compat in h15l.
  assert (h16:fst x <> n).
    intro h17. subst.
    red in h15l. destruct h15l as [h15a h15b].
    rewrite bag_add_app_eq in h15a.
    rewrite h11 in h15a. rewrite h15a in h15r.
    contradict h15r. rewrite (surjective_pairing x) at 1. reflexivity.
  pose proof (bag_add_inv _ _ _ h15l) as h17.
  destruct h17 as [h17l h17r].
  destruct h17l as [h17a | h17b].
  apply inb2_in_compat; auto. contradiction.
  red. intros x h15.
  rewrite <- list_to_set_in_iff.
  rewrite in_remove_iff. split.
  rewrite h8a in h15.
  apply in_inb2_compat in h15.
  pose proof (inb2_impl_inb1 _ _ h15) as h16.
  pose proof (bag_add_intro1 B (fst x) h16 n) as h17.
  apply inb1_in_compat in h17.
  rewrite <- h6l in h17 at 1.
  rewrite list_to_set_in_iff.
  assert (h18:n <> fst x).
    intro h18. subst. contradiction.
  rewrite bag_add_app_neq in h17; auto.
  red in h15. destruct h15 as [h15l h15r].
  rewrite h15l in h17.
  rewrite surjective_pairing.
  assumption.
  intro h16.
  rewrite h8a in h15.
  apply in_inb2_compat in h15.
  apply inb2_impl_inb1 in h15.
  rewrite (surjective_pairing x) in h16.
  inversion h16. subst.
  contradiction.
pose proof (plusl_nat_valued_fun_functional _ _ h14 h8b h15 (fun pr:nat*nat => fst pr * snd pr)) as h16.
rewrite h16.
ring.
Qed.



Lemma plus_bag_nat_im_set_bag_add :
  forall {T:Type} (A:Ensemble T) (pf:Finite A) (x:T) (f:T->nat),
    ~Ensembles.In A x ->
    plus_bag_nat (im_set_bag (Add A x) (Add_preserves_Finite _ _ x pf) f) =
    plus_bag_nat (im_set_bag A pf f) + (f x).
intros T A h1 x f hnin. 
rewrite im_set_bag_add; auto.
apply plus_bag_nat_add.
Qed.




Lemma plus_bag_nat_im_set_bag_eq_f : 
  forall {T:Type} (A:Ensemble T) (pf:Finite A) (f:T->nat)
         (n:nat),
  (forall x:T, Ensembles.In A x -> f x = n) ->
  plus_bag_nat (im_set_bag A pf f) = (card_fun1 A) * n.
intros T A h1.
pose proof (finite_set_list_no_dup _ h1) as h2.
destruct h2 as [l h2].
revert h1 h2. revert A.
induction l as [|a l h1]; simpl.
intros A h1 h2 f n h3.
destruct h2 as [h2l h2r]. subst.
rewrite card_fun1_empty.
assert (h4:h1 = Empty_is_finite T). apply proof_irrelevance.
subst.
rewrite im_set_bag_empty.
rewrite plus_bag_nat_empty. auto with arith.
intros A h2 h3 f n h4.
destruct h3 as [h3l h3r].
pose proof (no_dup_cons_nin _ _ h3r) as h5.
pose proof (no_dup_cons _ _ h3r) as h6. subst.
rewrite list_to_set_in_iff in h5.
pose proof (list_to_set_finite l) as h7.
pose proof (plus_bag_nat_im_set_bag_add _ h7 _ f h5) as h8.
assert (h9:h2 = Add_preserves_Finite _ (list_to_set l) a h7). apply proof_irrelevance.
subst.
rewrite h8.
pose proof (h4 _ (Add_intro2 _ (list_to_set l) a)) as h9. rewrite h9.
assert (h10:forall x:T, Ensembles.In (list_to_set l) x -> f x = n).
  intros x h11.
  apply (h4 _ (Add_intro1 _ _ a _ h11)).
specialize (h1 _ h7 (conj (eq_refl _) h6) f n h10).
rewrite h1.
rewrite card_add_nin'; auto.
ring.
Qed.
 
