(* Copyright (C) 2014, Daniel Wyckoff*)
(*This file is part of BooleanAlgebrasIntro2.

BooleanAlgebrasIntro2 is free software: you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

BooleanAlgebrasIntro2 is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with BooleanAlgebrasIntro2.  If not, see <http://www.gnu.org/licenses/>.*)


Require Export BoolAlgBasics.
Require Import InfiniteOperations.
Require Import SetUtilities.
Require Import TypeUtilities.
Require Export Powerset.
Require Import ProofIrrelevanceFacts.
Require Import ProofIrrelevance.
Require Import Description.

Section power_set_sec.
Variable Xt : Type.
Definition X := Full_set Xt.
Definition PX := Power_set _ X.

(* PSA = Power Set Algebra *)
(*This definition just builds the Bconst component of the Boolean Algebra*)
Definition bc_psa := 
  Build_Bconst (Ensemble Xt) PX (@Union Xt) (@Intersection Xt) 
    X (Empty_set Xt) (@Ensembles.Complement Xt).

Lemma full_power_compat : Full_set (Ensemble Xt) = PX.
assert (h1:forall Y:(Ensemble Xt), Included Y X -> 
    In PX Y).
  unfold PX.
  apply Definition_of_Power_set.
assert (h2:forall Y:(Ensemble Xt), Included Y X).
  intro Y. unfold X.
  unfold Included.
  intros x h.
  apply (Full_intro Xt).
assert (h3:forall Y:Ensemble Xt, In PX Y).
  intro Y. apply h1.  apply h2.
apply Extensionality_Ensembles.
unfold Same_set.
split.
unfold Included.
intros S h4.
apply (h3 S).
unfold Included.
intros S h5.
apply Full_intro.
Qed.

Lemma full_power_compat' : BS bc_psa = Full_set (Btype bc_psa).
assert (h1: Btype bc_psa = Ensemble Xt).
unfold bc_psa. reflexivity. 
assert (h2: BS bc_psa = PX).
unfold bc_psa. reflexivity.
rewrite h2. 
assert (h3:Full_set (Btype bc_psa) = Full_set (Ensemble Xt)).
unfold bc_psa. 
reflexivity.
rewrite h3.
rewrite full_power_compat. reflexivity.
Qed.

Definition psa := Build_Bool_Alg bc_psa full_power_compat'
  (@assoc_sum_psa Xt) (@assoc_prod_psa Xt) (@comm_sum_psa Xt)
  (@comm_prod_psa Xt) (@abs_sum_psa Xt) (@abs_prod_psa Xt)
  (@dist_sum_psa Xt) (@dist_prod_psa Xt) (@comp_sum_psa Xt) (@comp_prod_psa Xt).

Lemma btype_bc_psa :  bt psa = Ensemble Xt.
simpl. reflexivity.
Qed.

Definition transfer_psa (P:bt psa) : Ensemble Xt :=
  transfer btype_bc_psa P.

End power_set_sec.



Section field_of_sets_sec.

Record Field_of_Sets :=
{ Xt:Type;
  F:(Ensemble (Ensemble Xt));
  non_empty_F : (exists S:(Ensemble Xt), In F S);
  Union_closed : forall S1 S2:(Ensemble Xt), 
    In F S1 -> In F S2 -> In F (Union S1 S2);
  Int_closed : forall S1 S2:(Ensemble Xt),
    In F S1 -> In F S2 -> In F (Intersection S1 S2);
  Comp_closed : forall S :(Ensemble Xt),
    In F S -> In F (Ensembles.Complement S);
  Ft := {S:(Ensemble Xt) | In F S};
  Union_fos := 
    fun (S1 S2:Ft) => (let S1' := proj1_sig S1 in let S2' := proj1_sig S2 in 
      exist (fun (S':(Ensemble Xt)) => (In F S'))
        (Union S1' S2') (Union_closed S1' S2' (proj2_sig S1) (proj2_sig S2)));
  Int_fos := 
    fun (S1 S2:Ft) => (let S1' := proj1_sig S1 in let S2' := proj1_sig S2 in 
      exist (fun (S':(Ensemble Xt)) => (In F S'))
        (Intersection S1' S2') (Int_closed S1' S2' (proj2_sig S1) (proj2_sig S2)));
  Comp_fos := 
    fun (S:Ft) => (let S' := proj1_sig S in  
      exist (fun (S'':(Ensemble Xt)) => (In F S''))
        (Ensembles.Complement S') (Comp_closed S' (proj2_sig S)))
      }.


Lemma f_fos_inj : 
  forall (fos fos':Field_of_Sets)
         (pf:Xt fos = Xt fos'),
    (@transfer_dep _ (fun T=>Ensemble (Ensemble T)) _ _ pf (F fos)) = F fos' -> fos = fos'.
intros fos fos' h1 h2.
destruct fos; destruct fos'.
simpl in h1. subst.
simpl in h2.
subst.
f_equal; apply proof_irrelevance.
Qed.


Variable fos:Field_of_Sets.

Lemma full_in_F : In (F fos) (Full_set (Xt fos)).
elim (non_empty_F fos).
intros S h1.
assert (h2: In (F fos) (Ensembles.Complement S)).
apply ((Comp_closed fos) S h1).
assert (h3: In (F fos) (Union S (Ensembles.Complement S))).
apply ((Union_closed fos) S (Ensembles.Complement S) h1 h2).
assert (h4: 
  (Union S (Ensembles.Complement S)) = 
    (Full_set (Xt fos))).
apply comp_sum_psa.
rewrite h4 in h3. assumption.
Qed.

Lemma empty_in_F : In (F fos) (Empty_set (Xt fos)).
elim (non_empty_F fos).
intros S h1.
assert (h2: In (F fos) (Ensembles.Complement S)).
apply ((Comp_closed fos) S h1).
assert (h3: In (F fos) (Intersection S (Ensembles.Complement S))).
apply ((Int_closed fos) S (Ensembles.Complement S) h1 h2).
assert (h4:  (Intersection  S (Ensembles.Complement S)) = 
    (Empty_set (Xt fos))).
apply comp_prod_psa.
rewrite h4 in h3. assumption.
Qed.

(*Check to see if I didn't define this elsewhere.*)
Lemma existTexist : forall (A:Type) (P:A->Prop) (x y:A) (p:(P x)) (q:(P y)),
  existT P x p = existT P y q -> exist P x p = exist P y q.
intros A P x y p q h1.
assert (h2: sig_of_sigT (existT P x p) = sig_of_sigT (existT P y q)).
rewrite h1. reflexivity.
apply h2.
Qed.

Lemma assoc_sum_fos : forall N M P:(Ft fos), (Union_fos fos N (Union_fos fos M P)) =
  (Union_fos fos (Union_fos fos N M) P).
intros N M P.
unfold Union_fos; simpl.
assert (h:Union (proj1_sig N) (Union (proj1_sig M) (proj1_sig P)) =
        Union (Union (proj1_sig N) (proj1_sig M)) (proj1_sig P)).
  apply assoc_sum_psa.
apply existTexist.
apply subsetT_eq_compat.
apply h.
Qed.

Lemma assoc_prod_fos : forall N M P:(Ft fos), (Int_fos fos N (Int_fos fos M P)) =
  (Int_fos fos (Int_fos fos N M) P).
intros N M P.
unfold Int_fos; simpl.
assert (h:(Intersection (proj1_sig N) 
  (Intersection (proj1_sig M) (proj1_sig P))) = 
  (Intersection (Intersection (proj1_sig N) (proj1_sig M))
     (proj1_sig P))).
  apply assoc_prod_psa.
apply existTexist.
apply subsetT_eq_compat.
apply h.
Qed.

Lemma comm_sum_fos : forall N M:(Ft fos), Union_fos fos N M = Union_fos fos M N.
intros N M.
unfold Union_fos; simpl.
assert (h: (Union (proj1_sig N) (proj1_sig M)) = 
           (Union (proj1_sig M) (proj1_sig N))).
  apply comm_sum_psa.
apply existTexist.
apply subsetT_eq_compat.
apply h.
Qed.

Lemma comm_prod_fos : forall N M:(Ft fos), Int_fos fos N M = Int_fos fos M N.
intros N M.
unfold Int_fos; simpl.
assert (h: (Intersection (proj1_sig N) (proj1_sig M)) = 
           (Intersection (proj1_sig M) (proj1_sig N))).
  apply comm_prod_psa.
apply existTexist.
apply subsetT_eq_compat.
apply h.
Qed.

Lemma abs_sum_fos  :
  forall N M:(Ft fos), Union_fos fos N (Int_fos fos N M) = N.
intros N M.
unfold Union_fos; unfold Int_fos; simpl.
assert (h1: (Union (proj1_sig N)
     (Intersection (proj1_sig N) (proj1_sig M))) = (proj1_sig N)).
  apply abs_sum_psa.
assert (h2: N = 
  exist (fun S' : Ensemble (Xt fos) => In (F fos) S')
    (proj1_sig N) (proj2_sig N)).
  destruct N; simpl.
  reflexivity.
rewrite h2 at 8.
apply existTexist.
apply subsetT_eq_compat.
apply h1.
Qed.

Lemma abs_prod_fos  :
  forall N M:(Ft fos), Int_fos fos N (Union_fos fos N M) = N.
intros N M.
unfold Union_fos; unfold Int_fos; simpl.
assert (h1: (Intersection (proj1_sig N)
     (Union (proj1_sig N) (proj1_sig M))) = (proj1_sig N)).
  apply abs_prod_psa.
assert (h2: N = 
  exist (fun S' : Ensemble (Xt fos) => In (F fos) S')
    (proj1_sig N) (proj2_sig N)).
  destruct N; simpl.
  reflexivity.
rewrite h2 at 8.
apply existTexist.
apply subsetT_eq_compat.
apply h1.
Qed.

Lemma dist_sum_fos :
  forall N M P:(Ft fos),
    Int_fos fos P (Union_fos fos N M) =
    Union_fos fos (Int_fos fos P N) (Int_fos fos P M).
intros N M P.
unfold Union_fos; unfold Int_fos; simpl.
assert (h: 
  (Intersection (proj1_sig P)
     (Union  (proj1_sig N) (proj1_sig M))) = 
  (Union (Intersection (proj1_sig P) (proj1_sig N))
     (Intersection (proj1_sig P) (proj1_sig M)))).
  apply dist_sum_psa.
apply existTexist.
apply subsetT_eq_compat.
apply h.
Qed.

Lemma dist_prod_fos :
  forall N M P:(Ft fos),
    Union_fos fos P (Int_fos fos N M) =
    Int_fos fos (Union_fos fos P N) (Union_fos fos P M).
intros N M P.
unfold Union_fos; unfold Int_fos; simpl.
assert (h: 
  (Union (proj1_sig P)
     (Intersection  (proj1_sig N) (proj1_sig M))) = 
  (Intersection (Union (proj1_sig P) (proj1_sig N))
     (Union (proj1_sig P) (proj1_sig M)))).
  apply dist_prod_psa.
apply existTexist.
apply subsetT_eq_compat.
apply h.
Qed.

Definition Full_fos := 
  exist (fun S':(Ensemble (Xt fos)) => (In (F fos) S')) (Full_set (Xt fos))
  full_in_F.

Lemma comp_sum_fos:
  forall N:(Ft fos), Union_fos fos N (Comp_fos fos N) = Full_fos.
intro N.
unfold Union_fos; unfold Comp_fos; unfold Full_fos; simpl.
assert (h1: 
  (Union (proj1_sig N) (Ensembles.Complement (proj1_sig N))) = 
    (Full_set (Xt fos))).
  apply comp_sum_psa.
apply existTexist.
apply subsetT_eq_compat.
assumption.
Qed.

Definition Empty_fos :=
  exist (fun S':(Ensemble (Xt fos)) => (In (F fos) S')) (Empty_set (Xt fos))
  empty_in_F.

Lemma comp_prod_fos:
  forall N:(Ft fos), Int_fos fos N (Comp_fos fos N) = Empty_fos.
intro N.
unfold Int_fos; unfold Comp_fos; unfold Empty_fos; simpl.
assert (h1: 
  (Intersection (proj1_sig N) (Ensembles.Complement (proj1_sig N))) = 
    (Empty_set (Xt fos))).
  apply comp_prod_psa.
apply existTexist.
apply subsetT_eq_compat.
assumption.
Qed.

Definition Bc_fos := Build_Bconst (Ft fos) (Full_set (Ft fos)) (Union_fos fos)
  (Int_fos fos) Full_fos Empty_fos (Comp_fos fos).

Lemma fos_full_compat : BS Bc_fos = Full_set (Btype Bc_fos).
unfold Bc_fos; simpl. reflexivity.
Qed.

Definition fos_ba := 
  Build_Bool_Alg Bc_fos fos_full_compat
    assoc_sum_fos assoc_prod_fos comm_sum_fos comm_prod_fos abs_sum_fos
    abs_prod_fos dist_sum_fos dist_prod_fos comp_sum_fos comp_prod_fos.

Lemma fos_psa_ex :
  forall (T:Type), exists! fos:Field_of_Sets, 
    (Xt fos) = T /\ (F fos) = power_set (Full_set (Xt fos)). 
intros T.
pose (power_set (Full_set T)) as F'.
assert (my_non_empty_F : exists S : Ensemble T, In F' S).
  exists (Empty_set _).
  constructor. auto with sets.

assert (my_Union_closed : forall S1 S2 : Ensemble T,
                   In F' S1 -> In F' S2 -> In F' (Union S1 S2)).
intros.
constructor. red. intros. constructor.

assert (my_Int_closed : forall S1 S2 : Ensemble T,
                 In F' S1 -> In F' S2 -> In F' (Intersection S1 S2)).
  intros. constructor. red. intros. constructor.
assert (my_Comp_closed : forall S : Ensemble T,
                  In F' S -> In F' (Ensembles.Complement S)).
  intros. constructor. red. intros. constructor.

pose (Build_Field_of_Sets T F' my_non_empty_F my_Union_closed my_Int_closed my_Comp_closed) as fos'.
exists fos'. simpl in fos'.
red; split; try split; simpl; try reflexivity.
intros B h1. 
destruct h1 as [h1 h2].
simpl.
assert (h3:T = Xt fos'). reflexivity. 
rewrite  h3 in h1.
symmetry in h1. 
assert (h5:@transfer_dep _ (fun T => Ensemble (Ensemble T)) (Xt fos') (Xt B) h1 (F fos') = F B). 
  simpl. unfold F'.    
  pose (fun t:{T:Type & (Ensemble (Ensemble T))} =>
          projT2 t = power_set (Full_set (projT1 t))) as P.
  pose proof (transfer_dep_prop _ _ h1 (F fos') P) as h4.
  unfold P in h4. simpl in h4.
  rewrite <- h2 in h4. unfold F' in h4.
  rewrite <- h4. reflexivity.
eapply f_fos_inj.
apply h5.
Qed.

Definition fos_psa (T:Type) := 
  proj1_sig (constructive_definite_description _ (fos_psa_ex T)).

Lemma fos_psa_compat : 
  forall (T:Type),
    let fos:=fos_psa T in
    (Xt fos) = T /\ (F fos) = power_set (Full_set (Xt fos)).
intros T Fos.
unfold Fos. unfold fos_psa.
destruct constructive_definite_description as [Fos' h1].
simpl.
assumption.
Qed.

Definition proj1_sig_fos_psa 
           {T:Type} (A:{S:Ensemble (Xt (fos_psa T)) |
                                 In (F (fos_psa T)) S}) : 
  Ensemble T.
destruct A as [A h1].
pose proof (fos_psa_compat T) as [h2 h3]. 
clear h1.
rewrite h2 in A.
refine A.
Defined.

Definition proj1_sig_fos_psa_compat : 
  forall {T:Type} (A:{S:Ensemble (Xt (fos_psa T)) |
                      In (F (fos_psa T)) S}),
    let A' := proj1_sig_fos_psa A in
    A' = transfer_dep   
                      match (fos_psa_compat T) with
                        | conj P _ => P 
                      end
                      (proj1_sig A). 
intros T A A'. unfold A'. 
symmetry. 
rewrite <- transfer_dep_eq_iff.
unfold proj1_sig_fos_psa.
destruct (fos_psa_compat T) as [h1 h2]. 
destruct A as [A h3].
simpl.
destruct h1.
simpl.
reflexivity.
Qed.

Lemma proj1_sig_fos_psa_compat' : 
  forall {T:Type} (A:{S:Ensemble (Xt (fos_psa T)) |
                      In (F (fos_psa T)) S}),
         existT Ensemble (Xt (fos_psa T)) (proj1_sig A) =
         existT Ensemble T (proj1_sig_fos_psa A).
intros T A.
pose proof (proj1_sig_fos_psa_compat A) as h1.
symmetry in h1.
rewrite <- transfer_dep_eq_iff in h1.
assumption.
Qed.


Definition exist_fos_psa 
           {T:Type} (A:Ensemble T) : 
  {S:Ensemble (Xt (fos_psa T)) |In (F (fos_psa T)) S}.
pose proof (fos_psa_compat T) as [h2 h3].
rewrite <- h2 in A.
pose proof (full_inclusion A) as h4.
rewrite h3.
assert (h5:In (power_set (Full_set (Xt (fos_psa T)))) A).
  constructor. red; intros; constructor.
refine (exist _ A h5).
Defined.

Lemma exist_fos_psa_compat : 
  forall {T:Type} (A:Ensemble T),
    let A' := exist_fos_psa A in
    proj1_sig A' = transfer_dep_r
                      match (fos_psa_compat T) with
                        | conj P _ => P 
                      end A.
intros T A A'.
destruct (fos_psa_compat T) as [h1 h2].
pose proof h1 as h1'. symmetry in h1'.
rewrite <- (transfer_dep_transfer_dep_r_compat h1' h1).
symmetry.
rewrite <- transfer_dep_eq_iff.
unfold A'.
unfold exist_fos_psa.
destruct (fos_psa_compat T) as [h3 h4]. unfold eq_rect_r.
rewrite h4. simpl.
rewrite h3. simpl.
reflexivity.
Qed.

Lemma exist_fos_psa_compat' : 
  forall {T:Type} (A:Ensemble T),
    existT Ensemble (Xt (fos_psa T)) (proj1_sig (exist_fos_psa A)) =
    existT Ensemble T A.
intros T A.
unfold exist_fos_psa.
destruct (fos_psa_compat T) as [h1 h2]. 
rewrite h2, h1.
unfold eq_rect_r.
rewrite <- eq_rect_eq.
simpl.
reflexivity.
Qed.

Lemma proj1_sig_exist_fos_psa_int : 
  forall {T:Type} (A B:Ensemble T),
    proj1_sig (exist_fos_psa (Intersection A B)) = 
    Intersection (proj1_sig (exist_fos_psa A))
                 (proj1_sig (exist_fos_psa B)).
intros T A B. 
unfold exist_fos_psa.
destruct (fos_psa_compat T) as [h1 h2].
unfold eq_rect_r.
rewrite h2.
do 3 rewrite <- eq_rect_eq.
simpl.
rewrite h1.
do 3 rewrite <- eq_rect_eq.
reflexivity.
Qed.

Lemma proj1_sig_exist_fos_psa_union : 
  forall {T:Type} (A B:Ensemble T),
    proj1_sig (exist_fos_psa (Union A B)) = 
    Union (proj1_sig (exist_fos_psa A))
                 (proj1_sig (exist_fos_psa B)).
intros T A B. 
unfold exist_fos_psa.
destruct (fos_psa_compat T) as [h1 h2].
unfold eq_rect_r.
rewrite h2.
do 3 rewrite <- eq_rect_eq.
simpl.
rewrite h1.
do 3 rewrite <- eq_rect_eq.
reflexivity.
Qed.

Lemma proj1_sig_exist_fos_psa_comp : 
  forall {T:Type} (A:Ensemble T),
    proj1_sig (exist_fos_psa (Ensembles.Complement A)) = 
    Ensembles.Complement (proj1_sig (exist_fos_psa A)).
intros T A.
unfold exist_fos_psa.
destruct (fos_psa_compat T) as [h1 h2].
unfold eq_rect_r.
rewrite h2.
do 2 rewrite <- eq_rect_eq.
simpl.
rewrite h1.
do 2 rewrite <- eq_rect_eq.
reflexivity.
Qed.


Lemma in_proj1_sig_exist_fos_psa_iff : 
  forall {T:Type} (A:Ensemble T) (x:Xt (fos_psa T)),
    In (proj1_sig (exist_fos_psa A)) 
       x <->
    In A (transfer (match (fos_psa_compat T)
           with | conj pf _ => pf end)
          x).
intros T A x.
destruct (fos_psa_compat T) as [h1l h1r].
rewrite <- (transfer_dep_undoes_transfer_dep_r h1l A) at 2.
rewrite <- transfer_in.
rewrite exist_fos_psa_compat.
destruct (fos_psa_compat T) as [h2 h3].
assert (h4:h2 = h1l). apply proof_irrelevance. rewrite h4.
reflexivity.
Qed.


Lemma in_proj1_sig_exist_fos_psa_iff_r : 
  forall {T:Type} (A:Ensemble T) (x:T),
    In (proj1_sig (exist_fos_psa A)) 
       (transfer_r 
          (match (fos_psa_compat T)
           with | conj pf _ => pf end)
          x) <->
    In A x.
intros T A x.
pose proof (transfer_in_r (match (fos_psa_compat T) with
                             |conj pf _ => pf end) A x) as h1.
rewrite h1.  
pose proof (exist_fos_psa_compat A) as h3.
rewrite h3.
reflexivity.
Qed.



End field_of_sets_sec.

Section finite_cofinite_sec.


Variable Xt : Type.
Definition F_fc := 
  [S:(Ensemble Xt) | Finite S \/ Finite (Ensembles.Complement S)].

Lemma Union_closed_fc : forall S1 S2:(Ensemble Xt), 
    In F_fc S1 -> In F_fc S2 -> In F_fc (Union S1 S2).
intros S1 S2 h1 h2. 
unfold F_fc. unfold F_fc in h1. unfold F_fc in h2.
inversion h1 as [h3].
inversion h2 as [h4].
destruct h3 as [h5 | h6].
  (*left*)
  destruct h4 as [h7 | h8].
    (*left*)
    constructor.  left.
    apply Union_preserves_Finite; assumption.
    (*right*)
    pose proof (comp_union _ S1 S2) as h9.
    assert (h10: Included (Intersection (Ensembles.Complement S1) (Ensembles.Complement S2))
      (Ensembles.Complement S2)).
      auto with sets.
    pose proof (Finite_downward_closed _ _ h8 _ h10) as h11.
    rewrite <- h9 in h11.
    constructor. right; assumption.
  (*right*)
  destruct h4 as [h12 | h13].
    (*left*)
    pose proof (comp_union _ S1 S2) as h14.
    assert (h15: Included (Intersection (Ensembles.Complement S1) (Ensembles.Complement S2))
      (Ensembles.Complement S1)).
      auto with sets.
    pose proof (Finite_downward_closed _ _ h6 _ h15) as h16.
    rewrite <- h14 in h16.
    constructor. right; assumption.
    (*right*)
    constructor.  right.
    pose proof (comp_union _ S1 S2) as h17.
    assert (h18: Included (Intersection (Ensembles.Complement S1) (Ensembles.Complement S2))
      (Ensembles.Complement S1)).
      auto with sets.
    rewrite <- h17 in h18.
    apply Finite_downward_closed with (Ensembles.Complement S1); 
      assumption.
Qed.

Lemma Comp_closed_fc : forall S :(Ensemble Xt),
    In F_fc S -> In F_fc (Ensembles.Complement S).
intros S h1.
unfold F_fc in h1.
inversion h1 as [h2].
destruct h2 as [h3 | h4].
(*left*)
unfold F_fc.
constructor. right.
pose proof (Complement_Complement _ S) as h5.
rewrite h5. assumption.
(*right*)
unfold F_fc.
constructor. left. assumption.
Qed.


Lemma Int_closed_fc : forall S1 S2:(Ensemble Xt),
    In F_fc S1 -> In F_fc S2 -> In F_fc (Intersection S1 S2).
intros S1 S2 h1 h2. 
pose proof (Complement_Complement _ S1) as h3.
pose proof (Complement_Complement _ S2) as h4.
rewrite <- h3. rewrite <- h4.
pose proof (comp_union _ 
  (Ensembles.Complement S1) (Ensembles.Complement S2)) as h5.
rewrite <- h5.
pose proof (Union_closed_fc 
  (Ensembles.Complement S1) (Ensembles.Complement S2)) as h6.
pose proof (Comp_closed_fc S1 h1) as h7.
pose proof (Comp_closed_fc S2 h2) as h8.
pose proof (h6 h7 h8) as h9.
apply Comp_closed_fc. assumption.
Qed.

Lemma non_empty_F_fc : (exists S:(Ensemble Xt), In F_fc S).
unfold F_fc.
exists (Empty_set Xt).
constructor.
left. constructor.
Qed.

Lemma full_in_F_fc : In F_fc (Full_set _).
pose proof non_empty_F_fc as h1.
elim h1. intros S h2.
pose proof (Comp_closed_fc _ h2) as h3.
pose proof (Union_closed_fc _ _ h2 h3) as h4.
pose proof (comp_sum_psa S) as h5.
rewrite <- h5; assumption.
Qed.

Lemma empty_in_F_fc : In F_fc (Empty_set _).
pose proof non_empty_F_fc as h1.
elim h1. intros S h2.
pose proof (Comp_closed_fc _ h2) as h3.
pose proof (Int_closed_fc _ _ h2 h3) as h4.
pose proof (comp_prod_psa S) as h5.
rewrite <- h5; assumption.
Qed.


Definition fin_cof_fos := Build_Field_of_Sets Xt F_fc non_empty_F_fc
  Union_closed_fc Int_closed_fc Comp_closed_fc.
Definition fin_cof_ba := fos_ba fin_cof_fos.
End finite_cofinite_sec.

Section Count_Cocount.
Variable Xt:Type.
Definition F_cc := 
  [S:(Ensemble Xt) | Countable S \/ Countable (Ensembles.Complement S)].

Lemma Union_closed_cc : forall S1 S2:(Ensemble Xt), 
    In F_cc S1 -> In F_cc S2 -> In F_cc (Union S1 S2).
intros S1 S2 h1 h2.
inversion h1 as [h3].
inversion h2 as [h4].
unfold F_cc.
constructor.
destruct h3 as [h5 | h6].
  (*left*)
  (*red in h5.*)
  destruct h4 as [h7|h8].
    (*left*)
    left. 
    apply countable_union2; trivial. 
    (*right*)
    right.
    pose proof (comp_union _ S1 S2) as h9.
    rewrite h9.
    apply (countable_downward_closed 
      (Intersection (Ensembles.Complement S1) (Ensembles.Complement S2))
      (Ensembles.Complement S2)).
    assumption. auto with sets.
  (*right*)
  destruct h4 as [h10|h11].
    (*left*)
    right.
    pose proof (comp_union _ S1 S2) as h12.
    rewrite h12.
    apply (countable_downward_closed
      (Intersection (Ensembles.Complement S1) (Ensembles.Complement S2))
      (Ensembles.Complement S1)).
    assumption. auto with sets.
    (*right*)
    right.
    pose proof (comp_union _ S1 S2) as h13.
    rewrite h13.
    apply countable_intersection; trivial.
Qed. 

Lemma Comp_closed_cc : forall S :(Ensemble Xt),
    In F_cc S -> In F_cc (Ensembles.Complement S).
unfold F_cc.
intros S h1.
inversion h1 as [h2].
constructor.
pose proof (Complement_Complement _ S) as h3.
rewrite h3.
tauto.
Qed.

Lemma Int_closed_cc : forall S1 S2:(Ensemble Xt),
    In F_cc S1 -> In F_cc S2 -> In F_cc (Intersection S1 S2).
intros S1 S2 h1 h2.
pose proof (Complement_Complement _ S1) as h3.
pose proof (Complement_Complement _ S2) as h4.
rewrite <- h3. rewrite <- h4.
pose proof (comp_union _ 
  (Ensembles.Complement S1) (Ensembles.Complement S2)) as h5.
rewrite <- h5.
pose proof (Union_closed_cc 
  (Ensembles.Complement S1) (Ensembles.Complement S2)) as h6.
pose proof (Comp_closed_cc S1 h1) as h7.
pose proof (Comp_closed_cc S2 h2) as h8.
pose proof (h6 h7 h8) as h9.
apply Comp_closed_cc. assumption.
Qed.

Lemma non_empty_F_cc : (exists S:(Ensemble Xt), In F_cc S).
unfold F_cc.
exists (Empty_set Xt).
constructor.
left.
apply Finite_impl_Countable.
constructor.
Qed.

Definition cnt_ccnt_fos := Build_Field_of_Sets Xt F_cc non_empty_F_cc
  Union_closed_cc Int_closed_cc Comp_closed_cc.
Definition cnt_ccnt_ba := fos_ba cnt_ccnt_fos.


End Count_Cocount.