/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _VARIABLES_TABLE_H_
#define _VARIABLES_TABLE_H_

#include <stdexcept>
#include <map>
#include <vector>
#include "numTypes.h"
#include "Utils.h"

namespace bmEval
{
  class storedValue
  {
    // An object of the class STOREDVALUE is a complex value
    // provided with a Boolean attribute specifying whether the value
    // is read-only or modifiable. If the STOREDVALUE associated to an
    // identifier is read-only, that identifier defines a constant,
    // otherwise a variable.
    
  public:
    // Constructor
    storedValue (cValue value = cValue(0, 0), bool isReadOnly = false) : m_value(value), m_isReadOnly (isReadOnly) {}

    // Copy-Constructor
    storedValue (const storedValue& obj) : m_value(obj.m_value), m_isReadOnly (obj.m_isReadOnly) {}

    // Safe assignment: the assignment operation is only
    // performed if the field M_ISREADONLY is
    // currently set to false.
    storedValue& operator= (const storedValue& obj) {
      if (&obj != this && !m_isReadOnly)
	{
	  m_value = obj.m_value;
	  m_isReadOnly = obj.m_isReadOnly;
	}
      return *this;
    }

    // Destructor
    ~storedValue() {
      m_value = cValue (0, 0);
      m_isReadOnly = false;
    }

    // Setting method for the value.
    // The field M_VALUE is set to the supplied NEWVALUE.
    void setValueTo (cValue newValue)
    {
      m_value = newValue;
    }
    
    // Setting method for the read-only flag.
    // The field M_ISREADONLY is set to the supplied NEWVALUE.
    void setReadOnlyFlagTo (bool newValue)
    {
      m_isReadOnly = newValue;
    }
 
    // Setting method for both M_VALUE and M_ISREADONLY.
    void setTo (cValue newValue, bool willBeReadOnly) {
      m_value = newValue;
      m_isReadOnly = willBeReadOnly;
    }
    
    // Getting methods for value and read-only flag.
    cValue getValue () const { return m_value; }
    bool isReadOnly () const { return m_isReadOnly; }
    
  private:
    cValue m_value;
    bool m_isReadOnly;
  };

  class variableDefinition {
    // An instance of the class VARIABLEDEFINITION consists of an identifier
    // (a string), of the complex value associated to the identifier, and 
    // of a Boolean attribute specifying whether this value is read-only or modifiable.
    // Value and boolean attribute form together the contents of
    // a VARIABLEDEFINITION. They are saved in the field
    // M_CONTENTS, which is an instance of the class STOREDVALUE.
    // Mind that, if the value associated to its identifier is read-only,
    // a VARIABLEDEFINITION defines a constant, otherwise a (proper) variable.    

    // The constructor of the class VARIABLEDEFINITION always checks if
    // the argument ID (providing the identifier of the new VARIABLEDEFINITION)
    // is conform to the rule which determines whether a string is a valid
    // identifier or not.
    // A valid identifier may consist of any combination of alphabetical letters,
    // underscores and digits, but it must always start with an alphabetical letter
    // or an underscore.
    // If the argument ID does not supply a valid identifier, the
    // constructor of the class VARIABLEDEFINITION will throw an
    // exception of type std::runtime_error.
  public: 
    variableDefinition (const std::string& id, cValue value = cValue(0, 0), bool isReadOnly = false):
      m_id(id), m_contents (value, isReadOnly) {
      checkIdValidity();
    }
      
    variableDefinition (const std::string& id, const storedValue& contents):
      m_id(id), m_contents(contents) {
      checkIdValidity();
    }
    // Default copy constructor and assignment operator are fine
    // Default destructor is fine

    // Getting methods for identifier and contents.
    std::string getId() const {
      return m_id;
    }

    storedValue getContents() const {
      return m_contents;
    }
    
  private:
    void checkIdValidity() {
      size_t len;
      if (!Utils::doesStartWithId (m_id, len)) {
	throw std::runtime_error (m_id + ": Invalid variable identifier");
      }
    }
    
    std::string m_id;
    storedValue m_contents;
  };

  class match {
    // Objects of the class MATCH are used within methods
    // of the class VARIABLESTABLE to select those variables
    // whose identifiers match a given prefix/suffix.
  public:
    // Two types of match are possible when comparing a
    // variable identifier against a given key: prefix match
    // or suffix match.
    enum type {
      PREFIX = 0,
      SUFFIX = 1
    };

    match (const std::string& tag, type mtype) : m_tag(tag), m_type(mtype) {}
    // Default copy constructor and destructor are fine for this class,
    // also default assignment operator is fine.
    bool operator() (const std::pair<std::string, storedValue>& vDef) const
    {
      if (m_type == PREFIX)
	return (vDef.first.size() >= m_tag.size() &&
		vDef.first.compare (0, m_tag.size(), m_tag) == 0);
      else
	return (vDef.first.size() >= m_tag.size() &&
		vDef.first.compare (vDef.first.size() - m_tag.size(), m_tag.size(),
				    m_tag) == 0);
    }

  private:
    std::string m_tag;
    type m_type;
  };
  
  class variablesTable
  {
    // An object of the class VARIABLESTABLE is roughly speaking a map
    // associating strings, the identifiers of the constants and
    // variables stored in the object, with their (complex) values.
    // These values are actually instances of the class STOREDVALUE, i.e.
    // they are provided with a Boolean attribute specifying
    // whether they are read-only, as it is the case for constants, or
    // modifiable, as in the case of proper variables.

    friend class basicCalculator;
  public:
    // This struct is only used in the method
    // getVariables() to return a vector with
    // the values of (some of) the variables
    // stored in the TABLE.
    struct entry {
      std::string m_id;
      cValue m_value;
      bool m_isReadOnly;
    };
    
    // Default constructors and assignment operator are fine

    // Destructor
    ~variablesTable ()
      {
	m_table.clear();
      }

    // Setting method.
    // If the ID of the given variable definition (VARDEF) is not present
    // in the TABLE, then a new variable according to the
    // provided definition is added to the TABLE and TRUE is returned. 
    // Otherwise, the variable with the ID of VARDEF, if it is 
    // not read-only, is set to the VALUE and READONLY flag
    // of VARDEF and TRUE is returned.
    // If the variable with the ID of VARDEF is read-only,
    // then it is not set, and the method returns FALSE.    
    bool setVariable (const variableDefinition& varDef)
    {
      return setVariable(varDef.getId(), varDef.getContents().getValue(), varDef.getContents().isReadOnly());
    }
    
    // Setting method. For every variable definition
    // from the vector DEFINITIONS (a variable definition
    // consists of a variable ID and a STOREDVALUE, i.e. a couple
    // VALUE + READONLY flag) the following operation is
    // performed: if the ID from the definition is not present
    // in the TABLE, then a new variable with this ID and with
    // the VALUE and READONLY flag requested by the definition is added to it. 
    // Otherwise, the variable with the speficied ID, if it is 
    // not read-only, is set to the requested VALUE and READONLY flag.
    void setVariables (const std::vector<variableDefinition>& definitions)
    {
      std::vector<variableDefinition>::const_iterator constIt;

      for (constIt = definitions.begin(); constIt != definitions.end(); ++constIt)
	{
	  m_table[constIt->getId()] = constIt->getContents();
	}
    }

    // If PREFIX does not start with an alphabetic character or underscore,
    // this function does nothing. Otherwise, for every variable definition
    // from the vector DEFINITIONS (a variable definition
    // consists here of a STOREDVALUE, i.e. a couple
    // VALUE + READONLY flag) the following operation is
    // performed: first an ID is constructed by concatenating PREFIX with the
    // index of the current definition (mind that the index of the first definition
    // in the vector is zero), then if the constructed ID is not present
    // in the TABLE, a new variable with this ID and with
    // the VALUE and READONLY flag requested by the definition is added to it. 
    // In case a variable with the constructed ID is already present in the TABLE,
    // this variable, if it is not read-only, is set to the requested
    // VALUE and READONLY flag from the current definition.
    void setVariables (const std::string& prefix, const std::vector<storedValue>& definitions)
    {
      size_t len;

      if ( (Utils::doesStartWithId (prefix, len)) )
	{
	  std::vector<storedValue>::size_type idx;
	  std::string id;
	  
	  for (idx = 0; idx < definitions.size(); idx++)
	    {
	      id = prefix + Utils::n2str (idx);
	      m_table[id] = definitions[idx];
	    }
	}
    }
    
    // Getting method. This function returns false if TABLE
    // contains no variable with the provided ID, true otherwise.
    // In the latter case, VALUE is set to the value, and
    // ISREADONLY to the read-only attribute of the
    // variable with the given ID. 
    bool isVariableDefined (const std::string& id, cValue& value,
			    bool& isReadOnly) const
    {
      bool rv;

      std::map<std::string, storedValue >::const_iterator constIt = 
	m_table.find(id);
      if ( (rv = (constIt != m_table.end())) )
	{
	  value = constIt->second.getValue();
	  isReadOnly = constIt->second.isReadOnly();
	}
      return rv;
    }

    // Erasing method. This function removes from TABLE the 
    // variable with the given ID. It returns 1 if such a variable
    // has been actually found and removed, 0 otherwise.
    size_t eraseVariable (const std::string& id)
    {
      return m_table.erase (id);
    }

    // This function removes all variables stored in TABLE.
    void clear ()
    {
      m_table.clear();
    }

    // This function returns the number of variables
    // stored in TABLE.
    size_t size () const
    {
      return m_table.size();
    }

    // If KEY is the empty string, this function will
    // return a vector with the list of all variables contained
    // in the TABLE.
    // If KEY is not the empty string and MTYPE is match::PREFIX (match::SUFFIX),
    // then this function will return in the form of a vector the list of
    // all variables from TABLE whose identifier starts (ends) with the
    // prefix (suffix) KEY.
    std::vector<entry>
    getVariables (const std::string& key, 
		  match::type mType = match::PREFIX) const;
    
    // If KEY is the empty string, this function will
    // print to OS the list of all variables contained in the TABLE.
    // If KEY is not the empty string and MTYPE is match::PREFIX (match::SUFFIX),
    // then this function will print to OS the list of
    // all variables from TABLE whose identifier starts (ends) with the
    // prefix (suffix) KEY.
    void listVariables (std::ostream& os, 
			const std::string& key, 
			match::type mType = match::PREFIX) const;

    // If KEY is the empty string, this function will
    // remove all variables stored in the TABLE.
    // If KEY is not the empty string and MTYPE is match::PREFIX (match::SUFFIX),
    // then this function will remove from TABLE all variables
    // whose identifier starts (ends) with the prefix (suffix) KEY.
    // The value returned is the number of variables actually removed.
    size_t eraseVariables (const std::string& key, 
			   match::type mType = match::PREFIX);

  private:
    static std::string fmtVar (const std::pair<std::string, 
			       storedValue>& variable);

    // Setting method. If the given variable ID is not present
    // in the TABLE, then a new variable with the supplied ID,
    // VALUE and READONLY flag is added to it and TRUE is returned. 
    // Otherwise, the variable with the speficied ID, if it is 
    // not read-only, is set to the given VALUE and READONLY flag
    // and TRUE is returned.
    // If the variable is read-only, then it is not set, and
    // the method returns FALSE.
    bool setVariable (const std::string& id, cValue value = 0, bool readOnly = false)
    {
      std::map<std::string, storedValue >::iterator constIt = m_table.find(id);
      if ( (constIt == m_table.end() || !constIt->second.isReadOnly()) )
	{
	  m_table[id].setTo(value, readOnly);
	  return true;
	}
      else
	return false;
    }

    std::map<std::string, storedValue > m_table;    // the actual TABLE
  };
} // end of namespace bmEval

#endif // _VARIABLES_TABLE_H_
