/*
   This file is part of the BasicMathEval Library - version 1.0
   Copyright (C)  2015, 2016    Ivano Primi ( ivprimi@libero.it )    

   The BasicMathEval Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The BasicMathEval library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this software.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _EVALUATOR_H_
#define _EVALUATOR_H_

#include <string>
#include <stack>
#include <queue>
#include <vector>
#include "evalError.h"
#include "parser.h"
#include "translator.h"
#include "variablesTable.h"
#include "basicCalculator.h"

namespace bmEval
{
  class evaluator
  {
  public:
    evaluator (bool allowForImplicitMultiplication = false, bool allowForComplexArithmetic = true) : 
      m_expr(""), m_exprSuccessfullyCompiled (false), 
      m_lastCompilationError("", "Empty expression", "Provide a valid mathematical expression", 0),
      m_listOfComputationalErrors(),
      m_vTable(), m_compiledExpr(), m_evaluationStack(), 
      m_parser(allowForImplicitMultiplication, allowForComplexArithmetic), 
      m_translator(), m_calculator (m_vTable, allowForComplexArithmetic) {}

    evaluator (const variablesTable& vTable, 
	       bool allowForImplicitMultiplication = false, bool allowForComplexArithmetic = true) : 
      m_expr(""), m_exprSuccessfullyCompiled (false), 
      m_lastCompilationError("", "Empty expression", "Provide a valid mathematical expression", 0),
      m_listOfComputationalErrors(),
      m_vTable(vTable), m_compiledExpr(), m_evaluationStack(),
      m_parser(allowForImplicitMultiplication, allowForComplexArithmetic), 
      m_translator(), m_calculator (m_vTable, allowForComplexArithmetic) {}
  
    evaluator (const evaluator& eval) :
      m_expr(eval.m_expr), m_exprSuccessfullyCompiled (eval.m_exprSuccessfullyCompiled),
      m_lastCompilationError(eval.m_lastCompilationError),
      m_listOfComputationalErrors(eval.m_listOfComputationalErrors),
      m_vTable(eval.m_vTable), m_compiledExpr(eval.m_compiledExpr), m_evaluationStack(),
      m_parser(eval.m_parser), m_translator(eval.m_translator),
      m_calculator(eval.m_calculator) {}

    evaluator& operator= (const evaluator& eval)
      {
	if (&eval != this)
	  {
	    m_expr = eval.m_expr;
	    m_exprSuccessfullyCompiled = eval.m_exprSuccessfullyCompiled;
	    m_lastCompilationError = eval.m_lastCompilationError;
	    m_listOfComputationalErrors = eval.m_listOfComputationalErrors;
	    m_vTable = eval.m_vTable;
	    m_compiledExpr = eval.m_compiledExpr;
	    purgeEvaluationStack();
	    m_parser = eval.m_parser;
	    m_translator = eval.m_translator;
	    m_calculator = eval.m_calculator;
	  }
	return *this;
      }
  
    ~evaluator ()
      {
	purgeEvaluationStack();
	purgeCompiledExpr();
      }

    variablesTable& internalVartable ()
      {
	return m_vTable;
      }

    // Set the internal parser such that it allows
    // for/forbids implicit multiplication.
    void allowForImplicitMultiplication (bool yesno) {
      m_parser.allowForImplicitMultiplication (yesno);
    }

    // Allow for/forbid complex numbers
    // while parsing and evaluating a
    // mathematical expression.
    void allowForComplexArithmetic (bool yesno) {
      m_parser.allowForComplexInput (yesno);
      m_calculator.allowForComplexArithmetic (yesno);
    }

    // If a NEWEXPRESSION is provided, this function sets the 
    // mathematical M_EXPRession to NEWEXPRESSION and then
    // evaluates it by recomputing (if necessary) and
    // evaluating the compiled expression.
    // If NEWEXPRESSION is not provided, then the compiled
    // expression M_COMPILEDEXPR is re-evaluated.
    // In both cases, the result of the evaluation is returned,
    // and VTABLE is used to look for the values of the variables
    // arising in the compiled expression.
    cValue evaluate (const char* newExpression 
		       = (const char*)0) 
    {
	m_listOfComputationalErrors.clear();
    	if ((newExpression) && m_expr.compare(newExpression) != 0)
	{
	  m_expr = newExpression;
	  try
	    {
	      compileExpression();
	    }
	  catch (const evalError& compilationError)
	    {
	      m_exprSuccessfullyCompiled = false;
	      m_lastCompilationError = compilationError;
	      throw;
	    }
	  m_exprSuccessfullyCompiled = true;
	}
      if ( (m_exprSuccessfullyCompiled) )
	{
	  return evaluateCompiledExpression();
	}
      else
	{
	  // re-throw the last compilation error 
	  throw m_lastCompilationError;
	}
    }

    const std::queue <mathToken>& latestCompiledExpr () const
    {
      return m_compiledExpr;
    }

    const std::vector <evalError>& listOfComputationalErrors () const
    {
      return m_listOfComputationalErrors;
    }

  private:

    void purgeCompiledExpr ()
    {
      while (!m_compiledExpr.empty())
	{
	  m_compiledExpr.pop();
	}
    }

    void purgeEvaluationStack()
    {
      while (!m_evaluationStack.empty())
	{
	  m_evaluationStack.pop();
	}
    }

    void compileExpression ();
    cValue evaluateCompiledExpression ();

    std::string m_expr;
    bool m_exprSuccessfullyCompiled;
    evalError m_lastCompilationError;
    std::vector<evalError> m_listOfComputationalErrors;
    variablesTable m_vTable;
    std::queue <mathToken> m_compiledExpr;
    std::stack <mathToken> m_evaluationStack;
    parser m_parser;
    translator m_translator;
    basicCalculator m_calculator;
  };
} // end of namespace bmEval

#endif // _EVALUATOR_H_
