/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include <bit/ibstream.h>
#include <sstream>
#include <iostream>
#include <iomanip>
#include <stdio.h>
#include <byteswap.h>

using namespace bit;
using namespace std;

void print_buf(const char* buf, int size);

void run_msb_octet_test(ibstream& b, int n, uint32_t e);
void run_lsb_octet_test(ibstream& b, int n, uint32_t e);
void run_msb_bit_test(ibstream& b, int n, uint32_t e);
void run_lsb_bit_test(ibstream& b, int n, uint32_t e);
template <typename T> void run_lsb_bit_var_test(ibstream& b, int n, T e);

int passed=0, failed=0;

int main() {
    stringstream s(ios_base::in|ios_base::out|ios_base::binary);
    ibstream b(s);
    string str;

    for (int k=0; k <4; k++)
        for (int i=1; i <=7; i+=2) {
            unsigned char x = i << 4;
            x |= i+1;
            s.put(x);
        }
    s.put(0xBB);
    s.put(0xBB);

    str = s.str();
    cout << "Buffer: ";
    print_buf(str.c_str(), str.length());
    cout << endl;

    /**************************************************************/
    cout << "MSB Octet Test\n";
    run_msb_octet_test(b, 3, 0x12345600);
    run_msb_octet_test(b, 1, 0x78000000);
    cout << endl;


    /**************************************************************/
    cout << "LSB Octet Test\n";
    run_lsb_octet_test(b, 3, 0x00123456);
    run_lsb_octet_test(b, 1, 0x00000078);
    cout << endl;

    /**************************************************************/
    cout << "MSB Bit Test\n";
    run_msb_bit_test(b, 4,  0x10000000);
    run_msb_bit_test(b, 16, 0x23450000);
    run_msb_bit_test(b, 12, 0x67800000);
    cout << endl;

    /**************************************************************/
    cout << "LSB Bit Test\n";
    run_lsb_bit_test(b, 4,  0x00000001);
    run_lsb_bit_test(b, 16, 0x00002345);
    run_lsb_bit_test(b, 12, 0x00000678);
    cout << endl;

    /**************************************************************/
    cout << "LSB Variable Bit Test\n";
    uint16_t x;
    uint32_t y;
    uint64_t z;
    int nx, ny, nz;

    b >> bits(2) >> x >> octets(3) >> y >> bits(3) >> z;
    cout << "\tx  value: " << x << "  result: " << ( (x==2)?"passed":"FAILED") << endl;
    (x==2) ? passed++: failed++;
    cout << "\ty  value: " << y << "  result: " << ( (y==7)?"passed":"FAILED") << endl;
    (y==7) ? passed++: failed++;
    cout << "\tz  value: " << z << "  result: " << ( (z==3)?"passed":"FAILED") << endl;
    (z==3) ? passed++: failed++;
    cout << endl;

    cout << endl << dec
        << "******************************************************************************" << endl
        << "Summary:  " << passed << " passed, " << failed << " failed" << endl;
    if (failed)
      cout << "WARNING: " << failed << " TESTS FAILED!!!" << endl << endl;
    cout << endl;

    return 0;
}

void print_buf(const char* buf, int size) {
  cout << "0x ";
  for (int i=0; i < size; i++)
    if (buf[i] & 0x80)
      printf("%01x%01x ", (unsigned int)buf[i]>>4&0x0f, (unsigned int)buf[i]&0x0f);
  else
    printf("%02x ", buf[i]);
}

void print_results(const char* label, int rn, uint32_t d, uint32_t e) {
  e = bswap_32(e); // switch expected value to big endian values
  cout << setfill('0') << "\t" << setw(6) << label << " read: " << dec << setw(2) << rn
      << "  val: 0x" << hex << setw(8) << bswap_32(d)
      << "  exp: 0x" << hex << setw(8) << bswap_32(e)
      << "  result: " << ( (d==e)?"passed":"FAILED") << endl;
  (d==e) ? passed++: failed++;
}

void run_msb_octet_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_octets( (unsigned char*) &d, n);
  print_results("octets", rn, d, e);
}

void run_lsb_octet_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_octets( (unsigned char*) &d, 4, n);
  print_results("octets", rn, d, e);
}

void run_msb_bit_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_bits( (unsigned char*) &d, n);
  print_results("bits", rn, d, e);
}

void run_lsb_bit_test(ibstream& b, int n, uint32_t e) {
  int rn;
  uint32_t d = 0x00;
  rn = b.read_bits( (unsigned char*) &d, 4, n);
  print_results("bits", rn, d, e);
}

