/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "buffertable.h"

#include <algorithm>
#include <sstream>

using namespace bitgtk;

BufferTable::BufferTable()
    : Gtk::HBox(), m_buffer(NULL), m_index(NULL), m_display_bit_width(0), m_table(NULL), m_highlighter_enabled(true), m_bg_color("black")

{

  m_tablebox.modify_bg(Gtk::STATE_NORMAL, m_bg_color);
  m_tablebox.set_border_width(3);
  create_table();
  m_tableboxouter.modify_bg(Gtk::STATE_NORMAL, m_bg_color);
  m_tableboxouter.add(m_tablebox);
  pack_start(m_tableboxouter);

  add_color_set("aliceblue");
  add_color_set("lightgreen");
  add_color_set("moccasin");
  add_color_set("thistle");
  add_color_set("beige");
  add_color_set("lightpink");
  add_color_set("lightyellow");
  add_color_set("orange");
  add_color_set("powderblue");
  add_color_set("plum");
  add_color_set("navajowhite");
}

BufferTable::~BufferTable()
{}

void BufferTable::set_buffer( bit::FieldBuffer& buffer )
{
  m_buffer = &buffer;
  m_index = &buffer.fields();
  on_index_changed();
  on_buffer_changed(*m_index);
  buffer.signal_field_changed().connect(sigc::mem_fun(*this, &BufferTable::on_buffer_field_changed));
  buffer.signal_fields_changed().connect(sigc::mem_fun(*this, &BufferTable::on_buffer_fields_changed));
}


void BufferTable::set_bit_width( size_t w )
{
  m_display_bit_width = w;
  on_index_changed();
}

void BufferTable::set_index( bit::Tuple& index )
{
  m_buffer = NULL;
  m_index = &index;
  on_index_changed();
}

void BufferTable::on_index_changed( )
{
  std::ostringstream text;
  size_t maximum;
  size_t rows, current_position, current_row, left, right, remaining_bits;
  Gtk::Label *pelement_label, *pbuffer_label;
  Gtk::EventBox *pelement_ebox, *pbuffer_ebox;
  Gtk::VBox* ppackbox;
  std::list<std::pair<Gdk::Color, Gdk::Color> >::iterator color = m_colors.begin();
  bit::Data data;
  std::string tooltip;

  // check to make sure we have set the index pointer
  if (m_index == NULL)
    return;

  // start with a fresh, clean table and destroy the old one if necessary
  create_table();
  m_tablebox.modify_bg(Gtk::STATE_NORMAL, m_bg_color);
  m_tableboxouter.modify_bg(Gtk::STATE_NORMAL, m_bg_color);

  // clear our pointer storage data structures
  m_fieldbox.clear();

  // find the maximum bit field size to display
  maximum = m_index->start() + m_index->bits();

  // calculate how many rows we will be displaying and adjust the table accordingly
  if (m_display_bit_width > 0)
      rows = maximum / m_display_bit_width;
  else
      rows = 1;

  // iterate through all the fields creating as many boxes as are necessary
  current_position = 0;
  current_row = 0;

  create_fieldboxes(*m_index);
  
  m_table->resize_children();
}

void BufferTable::create_fieldboxes(bit::TupleBase& fb) {
  size_t remaining_bits, current_row;
  size_t left, right;
  std::string name;
  static std::list<std::pair<Gdk::Color, Gdk::Color> >::iterator color = m_colors.begin();

  for (bit::TupleBase::iterator iter = fb.begin(); iter != fb.end(); iter++) {
    name = iter->full_name(iter->depth()-1);
    
      // start off with all the bits needing a display box
      remaining_bits = iter->bits();

      // find the proper row for this label to start on
      if (m_display_bit_width > 0)
        current_row = iter->start() / m_display_bit_width;

        // find the left and right attachment points of the hbox and attach it to table
      if (m_display_bit_width > 0)
      {
        left = iter->start() % m_display_bit_width;
        right = std::min(left + remaining_bits, m_display_bit_width);
      }
      else
      {
        left = iter->start();
        right = left + remaining_bits;
      }

      add_fieldbox(name, left, right, current_row, current_row+1, color->first, color->second);
      
      // calculate how many bits we have left in this field
      remaining_bits -= right - left;

      // if our display width is too small for the whole field we need more display
      // boxes for the remainder of the field
      // fortunately, they all start on the left boundary
      left = 0;

      // create the remaining row labels (if any bits remain)
      while (remaining_bits)
      {
        // move to the next row
        current_row++;

        // we know the left attachment point is 0, but still need to calculate the right
        right = std::min(left + remaining_bits, m_display_bit_width);
        
        add_fieldbox(name, left, right, current_row, current_row+1, color->first, color->second);

          // calculate how many bits we might still have left to add a display box for
        remaining_bits -= right - left;

      }
      
      // if we have a color palette move to the next color or start over
      if (m_colors.size() > 1)
      {
        color++;
        if (color == m_colors.end())
          color = m_colors.begin();
      }
  }
}

void BufferTable::add_fieldbox(std::string name, unsigned left, unsigned right, unsigned top, unsigned bottom, Gdk::Color bg, Gdk::Color hl) {
  std::ostringstream text;
  std::string tooltip;
  
  // create the hbox to contain the element and buffer labels and the backing
  // highlight box
  m_fieldbox[name].m_highlight_boxes.push_back( Gtk::manage(new Gtk::VBox()) );

  // create the element and buffer labels and pack them into event boxes
  // the event boxes are necessary to show the background color and to capture
  // events for future enhancements
  // also add them to the vectors
  m_fieldbox[name].m_field_labels.push_back( Gtk::manage(new Gtk::Label()) );
  m_fieldbox[name].m_field_event_boxes.push_back( Gtk::manage(new Gtk::EventBox()) );
  m_fieldbox[name].m_field_event_boxes.back()->add(*(m_fieldbox[name].m_field_labels.back()));

  m_fieldbox[name].m_value_labels.push_back( Gtk::manage(new Gtk::Label()) );
  m_fieldbox[name].m_value_event_boxes.push_back( Gtk::manage(new Gtk::EventBox()) );
  m_fieldbox[name].m_value_event_boxes.back()->add(*(m_fieldbox[name].m_value_labels.back()));

      // add the element and buffer event boxes to the hbox
  m_fieldbox[name].m_highlight_boxes.back()->pack_start(*(m_fieldbox[name].m_field_event_boxes.back()));
  m_fieldbox[name].m_highlight_boxes.back()->pack_start(*(m_fieldbox[name].m_value_event_boxes.back()));

      // add tooltips if there is a description
  if (m_index->operator[](name).description().size() > 0)
  {
    tooltip = name + "\n\n" + m_index->operator[](name).description();
    m_tooltips.set_tip(*(m_fieldbox[name].m_field_event_boxes.back()), tooltip);
    m_tooltips.set_tip(*(m_fieldbox[name].m_value_event_boxes.back()), tooltip);
  }

  m_fieldbox[name].m_field_event_boxes.back()->signal_enter_notify_event().connect(sigc::bind(sigc::mem_fun(*this, &BufferTable::on_enter), name));
  m_fieldbox[name].m_value_event_boxes.back()->signal_enter_notify_event().connect(sigc::bind(sigc::mem_fun(*this, &BufferTable::on_enter), name));
  m_fieldbox[name].m_field_event_boxes.back()->signal_leave_notify_event().connect(sigc::bind(sigc::mem_fun(*this, &BufferTable::on_leave), name));
  m_fieldbox[name].m_value_event_boxes.back()->signal_leave_notify_event().connect(sigc::bind(sigc::mem_fun(*this, &BufferTable::on_leave), name));

  m_fieldbox[name].m_field_event_boxes.back()->modify_bg(Gtk::STATE_NORMAL, bg);
  m_fieldbox[name].m_field_event_boxes.back()->modify_bg(Gtk::STATE_ACTIVE, hl);
  m_fieldbox[name].m_value_event_boxes.back()->modify_bg(Gtk::STATE_NORMAL, bg);
  m_fieldbox[name].m_value_event_boxes.back()->modify_bg(Gtk::STATE_ACTIVE, hl);

  m_table->attach(*(m_fieldbox[name].m_highlight_boxes.back()), left, right, top, bottom);

  // fill the element label text
  text << " " << name << " \n " << left << "-" << right-1 << " ";
  m_fieldbox[name].m_field_labels.back()->set_text(text.str());
  m_fieldbox[name].m_field_labels.back()->set_justify(Gtk::JUSTIFY_CENTER);
}

void BufferTable::on_buffer_changed( bit::TupleBase& fb )
{
  // we can't display buffer values if the buffer hasn't been set
  if (m_buffer == NULL)
    return;

  if (fb.size() > 1)
    for (size_t i = 0; i < fb.size(); i++)
      on_buffer_changed(fb[i]);
  else
    on_buffer_field_changed(fb.full_name(fb.depth()-1));
}

void BufferTable::on_buffer_fields_changed() {
  on_buffer_changed(*m_index);
}

void BufferTable::on_buffer_field_changed( std::string name )
{
  std::vector<Gtk::Label*>::iterator iterlabel;
  std::ostringstream text;
  bit::Data data;

  text.str("");
  data = m_buffer->operator[](name).data();
  std::string s = data.octet_string();
  text << " <i>0x" << s << "</i> ";
  for (iterlabel = m_fieldbox[name].m_value_labels.begin(); iterlabel != m_fieldbox[name].m_value_labels.end(); iterlabel++)
    {
      (*iterlabel)->set_use_markup();
      (*iterlabel)->set_markup(text.str());
    }


//   std::string name;
//   for (bit::TupleBase::iterator iter = m_buffer->fields().begin(); iter != m_buffer->fields().end(); iter++) {
//     name = iter->full_name(iter->depth()-1);
//     text.str("");
//     data = m_buffer->operator[](name).data();
//     std::string s = data.octet_string();
//     text << " <i>0x" << s << "</i> ";
//     for (iterlabel = m_fieldbox[name].m_value_labels.begin(); iterlabel != m_fieldbox[name].m_value_labels.end(); iterlabel++)
//     {
//       (*iterlabel)->set_use_markup();
//       (*iterlabel)->set_markup(text.str());
//     }
//   }
    
}

void BufferTable::create_table( )
{
  if (m_table != NULL)
    m_tablebox.remove();

  m_table = Gtk::manage(new Gtk::Table());
  m_table->set_homogeneous();
  m_table->set_spacings(3);
  m_tablebox.add(*m_table);
}

void BufferTable::enable_tooltips( bool b )
{
  if (b)
    m_tooltips.enable();
  else
    m_tooltips.disable();
  on_index_changed();
}

void BufferTable::enable_highlighter( bool b )
{
  m_highlighter_enabled = b;
  on_index_changed();
}

void BufferTable::set_bg_color( std::string s )
{
  m_bg_color.set(s);
  on_index_changed();
}

bool BufferTable::on_enter( GdkEventCrossing *, std::string name )
{
  if (! m_highlighter_enabled)
    return false;

  std::vector<Gtk::VBox*>::iterator i;
  for (i = m_fieldbox[name].m_highlight_boxes.begin(); i != m_fieldbox[name].m_highlight_boxes.end(); i++)
    (*i)->set_state(Gtk::STATE_ACTIVE);

  return true;
}

bool BufferTable::on_leave( GdkEventCrossing *, std::string name )
{
  if (! m_highlighter_enabled)
    return false;

  std::vector<Gtk::VBox*>::iterator i;
  for (i = m_fieldbox[name].m_highlight_boxes.begin(); i != m_fieldbox[name].m_highlight_boxes.end(); i++)
    (*i)->set_state(Gtk::STATE_NORMAL);

  return true;
}

void BufferTable::add_color_set( std::string normal_color, std::string highlight_color )
{
  std::pair<Gdk::Color, Gdk::Color> colorset;
  colorset.first.set(normal_color);
  colorset.second.set(highlight_color);
  m_colors.push_back(colorset);
}

void BufferTable::clear_color_sets( )
{
  m_colors.clear();
}

