/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "tuple.h"

#include <iostream>
#include <sstream>
#include <algorithm>

#include <ctype.h>

using namespace bit;

Tuple::Tuple(unsigned id, std::string name, size_t offset, std::string description, size_t size):
    TupleBase(id, offset, 0),
    m_size(size),
    m_description(description),
    m_is_vector(false)
{
  set_name(name);
}

Tuple::Tuple(TupleBase& parent, unsigned id, std::string name, size_t offset, std::string description, size_t size):
    TupleBase(parent, id, offset, 0),
    m_size(size),
    m_description(description),
    m_is_vector(false)
{
  set_name(name);
}

Tuple::Tuple(std::string filename):
    m_size(0),
    m_is_vector(false)
{
  load_xml(filename);
}

Tuple::~Tuple()
{}

Tuple& Tuple::operator=(const Tuple& other)
{
  m_parent = other.m_parent;
  m_id = other.m_id;
  m_offset = other.m_offset;
  m_bits = other.m_bits;
  m_name = other.m_name;
  m_description = other.m_description;
  m_size = other.m_size;

  m_fields.clear();
  m_namemap.clear();
  for (unsigned i = 0; i < other.m_fields.size(); i++)
    {
      try
        {
          sptrTupleBase sptr = other.m_fields[i];
          TupleBase* fb = sptr.get();
          Field* f1 = dynamic_cast<Field*>(fb);
          Field* f2 = new Field();
          *f2 = *f1;
          f2->set_parent(this);
          m_fields.push_back(sptrTupleBase(f2));
          m_namemap[f2->name()] = i;
        }
      catch (...)
        {
          Tuple* f1 = dynamic_cast<Tuple*>(other.m_fields[i].get());
          Tuple* f2 = new Tuple(*f1);
          f2->set_parent(this);
          m_fields.push_back(sptrTupleBase(f2));
          m_namemap[f2->name()] = i;
        }
    }
}

TupleBase& Tuple::operator[](size_t index)
{
  if (index < m_fields.size())
    return *(m_fields[index]);
  throw error::field::bad_subscript();
}

TupleBase& Tuple::operator[](std::string index)
{
  size_t split = index.find_first_of(".");
  if (split == std::string::npos )
    {
      NameMap::iterator i = m_namemap.find(index);
      if (i != m_namemap.end() )
        return *(m_fields[i->second]);
      throw error::field::bad_subscript();
    }
    else {
      std::string field, remainder;
      field = index.substr(0, split);
      remainder = index.substr(split+1, index.size());
      NameMap::iterator i = m_namemap.find(field);
      if (i != m_namemap.end() ) {
        TupleBase& fbr = *(m_fields[i->second]);
        return fbr[remainder];
      }
      throw error::field::bad_subscript();
    }
}

int Tuple::get_index( std::string name ) throw ()
{
  NameMap::iterator i = m_namemap.find(name);
  if ( i != m_namemap.end() )
    return i->second;
  return -1;
}

void Tuple::load_xml(std::string filename)
{
  xmlpp::Document* document;
  xmlpp::Element* element;
  size_t last_position = 0;

  try
    {
      xmlpp::DomParser parser(filename);
      parser.set_validate();
      parser.set_substitute_entities();
//       parser.parse_file(filename);
      document = parser.get_document();
      if (document != NULL) {
        element = document->get_root_node();
        if (element != NULL)
          read(*element, last_position);
      }
    }
  catch (const std::exception& ex)
    {
      std::cout << "xml loader exception: " << ex.what() << std::endl;
    }
}

size_t Tuple::read(xmlpp::Element& xml_element, size_t last_position)
{
  FieldContainer::iterator new_field_iter;
  xmlpp::Attribute* attribute_start;
  xmlpp::Attribute* attribute_name;
  xmlpp::Attribute* attribute_vector_size;
  bool start_set = false;

  m_size = 0;
  m_fields.clear();
  m_namemap.clear();

  if ( xml_element.get_name() != "tuple" )
    return last_position;

  attribute_name = xml_element.get_attribute("name");
  if (attribute_name != NULL)
    set_name(attribute_name->get_value());

  attribute_start = xml_element.get_attribute("start");
  if (attribute_start != NULL)
  {
    set_offset( atoi(attribute_start->get_value().c_str()) );
    last_position = 0;
    start_set = true;
  }

  attribute_vector_size = xml_element.get_attribute("vector_size");
  if (attribute_vector_size != NULL)
    set_size( atoi( attribute_vector_size->get_value().c_str() ) );


  if (!start_set)
  {
    set_offset(last_position);
    last_position = 0;
  }

  if (m_size > 1 )
    {
      Tuple fg(*this);
      Tuple *ptrfg;
      std::ostringstream sout;
      size_t bits;
      fg.read(xml_element, last_position);
      bits = fg.bits();
      for (size_t i=0; i < m_size; i++)
        {
          sout.str("");
          sout << fg.name() << "[" << i << "]";
          ptrfg = new Tuple();
          *ptrfg = fg;
          ptrfg->set_parent(this);
          ptrfg->set_name(sout.str());
          ptrfg->set_id(i);
          ptrfg->set_offset(bits*i);
          m_fields.push_back(sptrTupleBase(ptrfg));
          m_namemap[ptrfg->name()] = i;
        }
      last_position += bits * m_size;
      m_is_vector = true;
    }
  else
    {
      xmlpp::Node::NodeList children;
      xmlpp::Node::NodeList::iterator iter;
      xmlpp::Element* child;

      children = xml_element.get_children();
      for (iter = children.begin(); iter != children.end(); iter++) {
        try {
          child = dynamic_cast<xmlpp::Element*>(*iter);
        }
        catch (std::bad_cast ) {
          continue;
        }
        if (child == NULL)
          continue;
        if (child->get_name() == "description") {
          std::string description;
          if ( child->has_child_text() )
          {
            description = child->get_child_text()->get_content();
            while (isspace(description[description.size()-1]))
              description.erase(description.size()-1);
            set_description(description);
          }
          else
            set_description("");

        }
        else if (child->get_name() == "tuple") {
          sptrTupleBase sptr( new Tuple(*this) );
          Tuple& fg = *(dynamic_cast<Tuple*>(sptr.get()));

          last_position = fg.read(*child, last_position);

          sptr->set_id(m_fields.size());
          m_fields.push_back(sptr);
          m_namemap[fg.name()] = m_size++;
        }
        else if (child->get_name() == "field") {
          sptrTupleBase sptr( new Field(*this) );
          Field& f = *(dynamic_cast<Field*>(sptr.get()));

          last_position = f.read(*child, last_position);

          sptr->set_id(m_fields.size());
          m_fields.push_back(sptr);
          m_namemap[f.name()] = m_size++;
        }
        else if (child->get_name() == "skip") {
          int skip_start, skip_bits, skip_size=1;
          bool skip_start_set=false, skip_bits_set=false;
          xmlpp::Attribute* attribute_skip_start;
          xmlpp::Attribute* attribute_skip_bits;

          attribute_skip_start = child->get_attribute("start");
          attribute_skip_bits = child->get_attribute("bits");

          if (attribute_skip_start != NULL) {
            skip_start = atoi(attribute_skip_start->get_value().c_str());
            skip_start_set = true;
          }

          if (attribute_skip_bits != NULL) {
            skip_bits = atoi(attribute_skip_bits->get_value().c_str());
            skip_bits_set = true;
          }

          if (skip_start_set && (skip_start+skip_bits) >= 0)
            last_position = skip_start + skip_bits;
          else if (last_position + skip_bits >= 0)
            last_position += skip_bits;
        }
      }
    }

  return last_position;
}

std::string Tuple::name() const
  {
    return m_name;
  }


void Tuple::set_name(const std::string& value)
{
  if (m_name.find_first_of("[].") < m_name.size())
    throw error::name(value);
  m_name = value;
}


std::string Tuple::description() const
  {
    return m_description;
  }


void Tuple::set_description(const std::string& value)
{
  m_description = value;
}


size_t Tuple::bits( ) const
  {
    size_t bits = 0;
    for (unsigned i = 0; i < m_fields.size(); i++)
      bits = std::max(bits, m_fields[i]->offset() + m_fields[i]->bits());
    return bits;
  }

size_t Tuple::size() const
  {
    return m_size;
  }

void Tuple::set_size(const size_t& value)
{
  size_t v = value;
  if (v == 0)
    v = 1;
  m_size = v;
  m_fields.clear();
  /*  for (size_t i = 0; i < v; i++)
      m_elements.push_back( Element(*this, i, m_bits*i, m_bits) );*/
}

std::string Tuple::get_xml( )
{
  std::ostringstream sout;
  sout << "<tuple name=\"" << m_name << "\""
      << " start=\"" << m_offset << "\"";

  if (m_size > 1)
    sout << " vector_size=\"" << m_size << "\"";

  if (m_description.size() > 0)
    sout << "><description>" << m_description << "</description>";
  else
    sout << ">";
  if (m_is_vector) {
    Tuple& fg = dynamic_cast<Tuple&>(*(m_fields[0]));
    sout << fg.children_xml();
  }
  else
    sout << children_xml();
  sout << "</tuple>";
  return sout.str();
}

std::string Tuple::children_xml( )
{
  std::ostringstream sout;
  for (int i = 0; i < m_fields.size(); i++)
    sout << m_fields[i]->get_xml();
  return sout.str();
}
