/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef OBSTREAM_H
#define OBSTREAM_H

#include <ostream>
#include <cstring>
#include <byteswap.h>

#include <bit/bstream.h>

namespace bit {

/**
@author Rick L Vinyard Jr
 */
/**
 * Manage an output stream with facilities for bitwise insertion
 **/
class obstream: public bstream {
public:
    /**
    * Create an obstream without an output sink.
     */
    obstream();

    /**
      * Create an obstream from a standard output stream.
      * The output stream will be the ultimate destination of the data.
       */
    obstream(std::ostream& output);

    /**
     * Performs any stream cleanup including output of any remaining bits.
     */
    ~obstream();

    /**
     * MSB write of up to n octets from a memory buffer.
     * @return the number of octets written
     */
    size_t write_octets(unsigned char* buf, size_t octets) {
        return write_octets( (const unsigned char*) buf, octets);
    }
    size_t write_octets(const unsigned char* buf, size_t octets);

    /**
     * LSB write of up to n octets from a memory buffer of size m.
     * LSB, is not necessarily x86 little endian!
     * A write of 4 octets from a 6 octet buffer of 123456 results in 3456.
     * If n is greater than m the result will be zero padded.
     * @return the number of octets written.
     */
    size_t write_octets(unsigned char* buf,  size_t bufsize, size_t octets) {
        return write_octets( (const unsigned char*) buf, bufsize, octets);
    }
    size_t write_octets(const unsigned char* buf, size_t bufsize, size_t octets);

    /**
     * MSB bit write of up to n bits from a memory buffer.
     * @return the number of bits written
     */
    size_t write_bits(unsigned char* buf, size_t bits) {
        return write_bits( (const unsigned char*) buf, bits);
    }
    size_t write_bits(const unsigned char* buf, size_t bits);

    /**
     * LSB bit write of up to n bits from a memory buffer of size m octets
     * with zero padding.
     * No adjustment is made for x86 integer or any other architectures or types.
     * @return the number of bits written
     */
    size_t write_bits(unsigned char* buf, size_t bufsize, size_t bits) {
        return write_bits( (const unsigned char*) buf, bufsize, bits);
    }
    size_t write_bits(const unsigned char* buf, size_t bufsize, size_t bits);

    /**
     * LSB write of up to n octets from variable t with zero padding.
     * For architecture specific integer types the "right" thing is done to result
     * in a big-endian write.
     * The default is to write the entire variable into the stream.
     */
    template <typename T>
    size_t write_octets(T t, size_t n=sizeof(T));

    size_t write_octets(uint16_t t, size_t n=2);
    size_t write_octets(uint32_t t, size_t n=4);
    size_t write_octets(uint64_t t, size_t n=8);
    size_t write_octets(int16_t t, size_t n=2);
    size_t write_octets(int32_t t, size_t n=4);
    size_t write_octets(int64_t t, size_t n=8);


    /**
     * LSB write of up to n bits from variable t with zero padding.
     * For architecture specific integer types the "right" thing is done to result
     * in a big-endian write.
     * The default is to write the entire variable into the stream.
     */
    template <typename T>
    size_t write_bits(T t, size_t n=sizeof(T)*8);

    size_t write_bits(uint16_t t, size_t n=2);
    size_t write_bits(uint32_t t, size_t n=4);
    size_t write_bits(uint64_t t, size_t n=8);
    size_t write_bits(int16_t t, size_t n=2);
    size_t write_bits(int32_t t, size_t n=4);
    size_t write_bits(int64_t t, size_t n=8);

    /**
     * Flush any remaining bits into the output stream using a default MSB ordering
     * or LSB with one-octet width if parameter is false.
     **/
    void flush(bool msb=true);

    friend obstream& operator<<(obstream& s, const bits& b);
    friend obstream& operator<<(obstream& s, const octets& o);
    friend obstream& operator<<(obstream& s, const whole& w);
    template <typename T>
    friend obstream& operator<<(obstream& s, T& t);

    /**
     * Set the associated output stream to stream.
     * @param clearbits If true any cached bits will be cleared, and if false (default) cached bits remain
     * in the stream for the next write.
     */
    void attach_stream(std::ostream& stream, bool clearbits=false);

    /**
     * Remove the currently attached stream. Futher attempts to write will fail, including subsequent
     * attempts to flush any remaining bits. Therefore, if you're going to flush do it before this
     * function call or set flushbits to true.
     * @param clearbits If true any cached bits will be cleared, and if false (default) cached bits remain
     * in the stream for the next write.
     **/
    void detach_stream(bool clearbits=false);

protected:
    /** A pointer to the associated output stream. A reference is not used since
    * it is anticipated that future versions will provide a constructor that does not
    * require an associated stream.
     */
    std::ostream* m_output;

};

template <typename T>
inline        size_t obstream::write_octets(T t, size_t n) {
  if (m_output == NULL) return 0;
  if (n > sizeof(T))
        n = sizeof(T);
    write_octets( (unsigned char*) &t, n, sizeof(T));
}

template <typename T>
inline size_t obstream::write_bits(T t, size_t n) {
  if (m_output == NULL) return 0;
  if (n > sizeof(T) * 8 )
        n = sizeof(T) * 8;
    write_bits( (unsigned char*) &t, n, sizeof(T));
}

inline obstream& operator<<(obstream& s, const bits& b) {
    s.m_state = bstream::BITS;
    s.m_stateval = b.val;
    return s;
}

inline obstream& operator<<(obstream& s, const octets& o) {
    s.m_state = bstream::OCTETS;
    s.m_stateval = o.val;
    return s;
}

inline obstream& operator<<(obstream& s, const whole& w) {
    s.m_state = bstream::WHOLE;
    s.m_stateval = 0;
    return s;
}


template <typename T>
inline obstream& operator<<(obstream& s, T& t) {
    switch (s.m_state) {
    case bstream::WHOLE:
        s.write_octets(t);
        break;
    case bstream::BITS:
        s.write_bits(t, s.m_stateval);
        break;
    case bstream::OCTETS:
        s.write_octets(t, s.m_stateval);
        break;
    }
    return s;
}


}; // namespace bit


#endif
