/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "ibstream.h"
#include <cstring>

using namespace bit;


ibstream::ibstream( ): m_input(NULL)
{
}

ibstream::ibstream(std::istream& input):
m_input(&input) {}


ibstream::~ibstream() {}

size_t ibstream::read_octets(unsigned char* buf, size_t octets) {
    std::streamsize octets_read;
    unsigned char tempbits;
    char* p;
    unsigned char* q;

    if (!m_input) return 0;

    m_input->read( (char*)buf, octets);
    octets_read = m_input->gcount();

    // fast case is when there are no leftoverbits and we can just do octet reads
    if (m_numleftoverbits == 0) {
        return static_cast<size_t>(octets_read);
    } else {
        // we have m_numleftoverbits left from the previous read operation and need to shift these
        // bits onto the data and save the last m_numleftoverbits of data for the next read
        for (q=buf; q < buf+octets; q++) {
            tempbits = *q & m_masks[m_numleftoverbits];   // get the bits that will be lost in the shift
            m_leftoverbits <<= 8 - m_numleftoverbits;     // shift up the bits in m_leftoverbits for bitwise-or
            *q >>= m_numleftoverbits;                     //shift off the number of bits creating room for previous bits
            *q |= m_leftoverbits;                         // create new octet
            m_leftoverbits = tempbits;                    // store last unused bits for next octet or next read
        }
    }
    return static_cast<size_t>(octets_read);
}

size_t ibstream::read_octets(unsigned char* buf, size_t bufsize, size_t octets) {
    size_t oread;
    if (!m_input) return 0;

    if (octets > bufsize)                                 // check to make sure that we don't have parameters beyond buffer
        octets = bufsize;
    oread = read_octets(buf, octets);                     // read into buffer as MSB
    if (oread != bufsize) {                               // if they're equal we're done
        memmove(buf+bufsize-oread, buf, oread);           // move octets downward for LSB
        memset(buf, 0x00, bufsize-oread);                 // clear the upper octets
    }
    return oread;
}

size_t ibstream::read_bits(unsigned char* buf, size_t n) {
    size_t whole_octets, woctets_read, bits_read;
    short unsigned int last_bits, more_bits, shift_bits;
    unsigned char* last_octet;
    char read_octet;
    unsigned char tempbits;

    if (m_input == NULL && m_numleftoverbits == 0) return 0;

    // start by reading all whole octets
    // this results in a potentially new set of leftover bits
    // but does the right thing for the first n/8 octets
    whole_octets = n / 8;
    woctets_read = read_octets(buf, whole_octets);
    bits_read = woctets_read * 8;

    // now we need to read n%8 more bits into the last octet
    last_bits = n % 8;

    // we only need to continue if we have more bits to read
    if (last_bits) {
        // make sure we offset from the whole octets read, and not from what we set out to read initially
        // i.e. what if the actual octet read was less than asked for?
        last_octet = buf + woctets_read;

        // we have enough bits in leftoverbits to satisfy remainder
        // therefore we don't need to pull any more bytes off the stream
        if (last_bits <= m_numleftoverbits) {
            *last_octet = m_leftoverbits;                    // copy in temp bits
            *last_octet >>= m_numleftoverbits - last_bits;   // shift off unused leftoverbits since we'll only use the upper bits
            *last_octet <<= 8 - last_bits;                   // shift up into MSB position
            m_numleftoverbits -= last_bits;                  // subtract used bits
            m_leftoverbits &= m_masks[m_numleftoverbits];    // mask out used bits
            bits_read += last_bits;                          // add on these bits to read bit count
        }
        // we don't have enough so we'll take what we have and get one more octet
        else {
            more_bits = last_bits - m_numleftoverbits;         // how many more bits do we need?
            m_input->get
            (read_octet);                       // read in one more octet
            if (! m_input->fail()) {                      // if it's not 1 we didn't read anything
                *last_octet = m_leftoverbits;                  // copy in the temp bits that we have already
                m_leftoverbits = read_octet;                   // make a copy of read_octet and mask out the used bits
                m_numleftoverbits = 8 - more_bits;             // calculate number of unused bits from read_octet after this read
                m_leftoverbits &= m_masks[m_numleftoverbits];  // mask out bits used in this read
                *last_octet <<= more_bits;                     // shift up to make room for additional bits
                read_octet >>= 8 - more_bits;                  // shift off unused bits
                *last_octet |= read_octet;                     // bitwise or in the additional bits
                *last_octet <<= 8 - last_bits;                 // shift up into MSB position
                bits_read += more_bits;                        // tack on these bits to return value
            }
        }

        //?????*last_octet |= tempbits; // bitwise or in the last bits shifted off from the whole octet read

    }
    return bits_read;
}

size_t ibstream::read_bits(unsigned char* buf, size_t bufsize, size_t bits) {
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  size_t bread;
    if (bits > bufsize*8)                                 // check to make sure that we don't have parameters beyond buffer
        bits = bufsize*8;
    bread = read_bits(buf, bits);                     // read into buffer as MBS

    // at this point we have the bits in the buffer, but they're in MSB and need
    // to be shifted downward to LSB

    // start by shifting down the proper number of whole bytes
    size_t octsize = bread/8 + ((bread%8)?1:0); // calculate used size as whole octets plus a possible partial octet
    size_t offset = bufsize - octsize; // calculate offset
    if (offset) {
        memmove(buf+offset, buf, octsize);           // move octets downward for LSB
        memset(buf, 0x00, offset);                 // clear the upper octets
    }

    // now we have the buffer in the right position with respect to the octets, now shift downward for
    // the bits to be in little endian
    size_t bitshiftdown = 8 - bread%8; // the whole buffer needs to be downshifted by this many bits
    if (bitshiftdown < 8) { // if bread%8 == 0 then we don't need to do any shifting
        size_t bitshiftup = bread%8; // and for convenience we'll calulate the upshift necessary ahead of time
        unsigned char temp1 = 0, temp2 = 0;
        for( unsigned char* p=buf+offset; p<buf+bufsize; p++) {
            temp2 = *p; // make a copy because we're about to lose some bits
            *p >>= bitshiftdown; // shift off the bits destined for the next octets
            *p |= temp1; // bring in the bits from the previous octets
            temp1 = temp2 << bitshiftup; // and prepare temp1 for the next octet
        }
    }

    return bread;
}

size_t ibstream::read_octets(uint16_t& t, size_t n) {
  size_t retval;
  if (!m_input) return 0;

  if (n > 2)
        n = 2;
    retval = read_octets( (unsigned char*) &t, 2, n);
    t = bswap_16(t);
    return retval;
}

size_t ibstream::read_octets(uint32_t& t, size_t n) {
  size_t retval;
  if (!m_input) return 0;

  if (n > 4)
        n = 4;
  retval = read_octets( (unsigned char*) &t, 4, n);
    t = bswap_32(t);
    return retval;
}

size_t ibstream::read_octets(uint64_t& t, size_t n) {
  size_t retval;
  if (!m_input) return 0;

  if (n > 8)
        n = 8;
    retval = read_octets( (unsigned char*) &t, 8, n);
    t = bswap_64(t);
    return retval;
}


size_t ibstream::read_octets(int16_t& t, size_t n) {
  size_t retval;
  if (!m_input) return 0;

  if (n > 2)
        n = 2;
    retval = read_octets( (unsigned char*) &t, 2, n);
    t = bswap_16(t);
    return retval;
}

size_t ibstream::read_octets(int32_t& t, size_t n) {
  size_t retval;
  if (!m_input) return 0;

  if (n > 4)
        n = 4;
    retval = read_octets( (unsigned char*) &t, 4, n);
    t = bswap_32(t);
    return retval;
}

size_t ibstream::read_octets(int64_t& t, size_t n) {
  size_t retval;
  if (!m_input) return 0;

  if (n > 8)
        n = 8;
    retval = read_octets( (unsigned char*) &t, 8, n);
    t = bswap_64(t);
    return retval;
}

size_t ibstream::read_bits(uint16_t& t, size_t n) {
  size_t retval;
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > 16)
        n = 16;
    retval = read_bits( (unsigned char*) &t, 2, n);
    t = bswap_16(t);
    return retval;
}

size_t ibstream::read_bits(uint32_t& t, size_t n) {
  size_t retval;
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > 32)
        n = 32;
    retval = read_bits( (unsigned char*) &t, 4, n);
    t = bswap_32(t);
    return retval;
}

size_t ibstream::read_bits(uint64_t& t, size_t n) {
  size_t retval;
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > 64)
        n = 64;
    retval = read_bits( (unsigned char*) &t, 8, n);
    t = bswap_64(t);
    return retval;
}


size_t ibstream::read_bits(int16_t& t, size_t n) {
  size_t retval;
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > 16)
        n = 16;
    retval = read_bits( (unsigned char*) &t, 2, n);
    t = bswap_16(t);
    return retval;
}

size_t ibstream::read_bits(int32_t& t, size_t n) {
  size_t retval;
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > 32)
        n = 32;
    retval = read_bits( (unsigned char*) &t, 4, n);
    t = bswap_32(t);
    return retval;
}

size_t ibstream::read_bits(int64_t& t, size_t n) {
  size_t retval;
  if (m_input == NULL && m_numleftoverbits == 0) return 0;

  if (n > 64)
        n = 64;
    retval = read_bits( (unsigned char*) &t, 8, n);
    t = bswap_64(t);
    return retval;
}

void ibstream::attach_stream(std::istream& stream, bool clearbits) {
  m_input = &stream;
  if (clearbits) {
    m_numleftoverbits = 0;
    m_leftoverbits = 0x00;
    }
}

void ibstream::detach_stream(bool clearbits) {
  m_input = NULL;
  if (clearbits) {
    m_numleftoverbits = 0;
    m_leftoverbits = 0x00;
  }
}
