/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#include "field.h"

#include <sstream>
#include <ctype.h>

using namespace bit;

Field::Field(unsigned id, std::string n, size_t offset, size_t element_bits, std::string d, size_t size):
    TupleBase(id, offset, element_bits),
    m_description(d)
{
  set_size(size);
  set_name(n);
}

Field::Field(TupleBase& parent, unsigned id, std::string n, size_t offset, size_t element_bits, std::string d, size_t size):
    TupleBase(parent, id, offset, element_bits),
    m_description(d)
{
  set_size(size);
  set_name(n);
}

Field& Field::operator=(const Field& other)
{
  m_parent = other.m_parent;
  m_id = other.m_id;
  m_offset = other.m_offset;
  m_bits = other.m_bits;
  m_name = other.m_name;
  m_description = other.m_description;
  m_size = other.m_size;
  m_elements = other.m_elements;
  for (ElementContainer::iterator i = m_elements.begin(); i != m_elements.end(); i++)
    i->set_parent(this);
  return *this;
}

TupleBase& Field::operator[](size_t index)
{
  if (index >= m_size)
    throw error::field::bad_subscript();

  return m_elements[index];
}

TupleBase& Field::operator[](std::string index)
{
  std::istringstream sin(index);
  char c1, c2;
  size_t num_index;
  sin >> c1 >> num_index >> c2;
  if ( ! (c1 == '[' && c2 == ']') )
    throw error::field::bad_subscript();
  return operator[](num_index);
}

#include <iostream>
size_t Field::read( xmlpp::Element& xml_element, size_t last_position )
{
  xmlpp::Attribute* attribute_start;
  xmlpp::Attribute* attribute_bits;
  xmlpp::Attribute* attribute_name;
  xmlpp::Attribute* attribute_vector_size;
  xmlpp::Node::NodeList description_children;
  xmlpp::Node::NodeList::iterator iter;
  xmlpp::Element* xml_description;
  std::string description;
  bool start_set = false;

  m_name.clear();
  m_offset = 0;
  m_bits = 0;

  if ( xml_element.get_name() != "field" )
    return last_position;

  attribute_start = xml_element.get_attribute("start");
  if (attribute_start != NULL)
    {
      m_offset = atoi(attribute_start->get_value().c_str());
      start_set = true;
    }

  attribute_bits = xml_element.get_attribute("bits");
  if (attribute_bits != NULL)
    set_element_bits( atoi(attribute_bits->get_value().c_str()) );

  attribute_name = xml_element.get_attribute("name");
  if (attribute_name != NULL)
    set_name(attribute_name->get_value());

  attribute_vector_size = xml_element.get_attribute("vector_size");
  if (attribute_vector_size != NULL)
    set_size( atoi( attribute_vector_size->get_value().c_str() ) );

  description_children = xml_element.get_children("description");
  if (description_children.size() == 0)
    {
      set_description("");
    }
  else
    {
      xml_description = dynamic_cast<xmlpp::Element*>( description_children.front() );
      if ( xml_description && xml_description->has_child_text() )
        {
          description = xml_description->get_child_text()->get_content();
          while (isspace(description[description.size()-1]))
            description.erase(description.size()-1);
          set_description(description);
        }
    }

  if ( ! start_set )
    m_offset = last_position;

  return (m_offset + m_bits * m_size);
}

size_t Field::element_bits() const
  {
    return m_bits;
  }


void Field::set_element_bits(const size_t& value)
{
  m_bits = value;
  m_elements.clear();
  for (size_t i = 0; i < m_size; i++)
    m_elements.push_back( Element(*this, i, m_bits*i, m_bits) );
}


std::string Field::name() const
  {
    return m_name;
  }


void Field::set_name(const std::string& value) throw (error::name)
{
  if (m_name.find_first_of("[].") < m_name.size())
    throw error::name(value);
  m_name = value;
}

std::string Field::description() const
  {
    return m_description;
  }


void Field::set_description(const std::string& value)
{
  m_description = value;
}


void Field::set_name_without_bracket_check( std::string value )
{
  if (m_name.find_first_of(".") < m_name.size())
    throw error::name(value);
  m_name = value;
}

size_t Field::bits( ) const
  {
    return m_bits * m_size;
  }

size_t Field::size() const
  {
    return m_size;
  }

void Field::set_size(const size_t& value)
{
  size_t v = value;
  if (v == 0)
    v = 1;
  m_size = v;
  m_elements.clear();
  for (size_t i = 0; i < v; i++)
    m_elements.push_back( Element(*this, i, m_bits*i, m_bits) );
}

std::string bit::Field::get_xml( )
{
  std::ostringstream sout;
  sout << "<field name=\"" << m_name
  << "\" start=\"" << m_offset
  << "\" bits=\"" << m_bits
  << "\"";
  if (m_size > 1)
    sout << " vector_size=\"" << m_size << "\"";
  if (m_description.size() > 0)
    sout << "><description>" << m_description << "</description></field>";
  else
    sout << "/>";
  return sout.str();
}

