/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef BITERROR_H
#define BITERROR_H

/**
 * @defgroup errors
 */

#include <stdexcept>

namespace bit
  {

  /**
   * Base class for all bit library errors.
   * @author Rick L Vinyard Jr
   * @ingroup errors
   */
  class bit_error: public std::runtime_error
    {
    public:
      bit_error(const std::string s="bit: Unknown error."):
          std::runtime_error(s),
          m_error_string(s)
      {}

      ~bit_error() throw ()
      { }

      virtual const char * what () const throw ()
      {
        return m_error_string.c_str();
      }

    protected:
      std::string m_error_string;
    }
  ;

  namespace error
    {

    /**
    * @author Rick L Vinyard Jr
    * @ingroup errors
     */
    class name: public bit_error
      {
      public:
        name(): bit_error("bit: Name identifiers may not contain the following characters: .[]")
        { }
        name(const std::string s): bit_error(s)
        {
          m_error_string = "bit: The name \"" + s + "\" is invalid. Name identifiers may not contain the following characters: .[]";
        }
      }
    ;

    /**
     * @author Rick L Vinyard Jr
     * @ingroup errors
     */
    class no_fields: public bit_error
      {
      public:
        no_fields(): bit_error("bit:FieldBuffer: fields() method called, but this instance does not have any fields.")
        { }
      }
    ;

    /**
                 * @author Rick L Vinyard Jr
                 * @ingroup errors
     */
    class bad_subscript: public bit_error
      {
      public:
        bad_subscript(): bit_error("bit:Field: subscript requested larger than the number of elements.")
        { }
      }
    ;


  }

}

#include <bit/error_field.h>
#include <bit/error_fields.h>
#include <bit/error_indices.h>

#endif

