/***************************************************************************
 *   Copyright (C) 2001 by Rick L. Vinyard, Jr.                            *
 *   rvinyard@cs.nmsu.edu                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Lesser General Public License as        *
 *   published by the Free Software Foundation version 2.1.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the                 *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA              *
 ***************************************************************************/
#ifndef BIT_DATA_H
#define BIT_DATA_H

#include <boost/shared_array.hpp>

#include <sstream>
#include <iomanip>

namespace bit {

/**
 * This structure represents a data block allocated via a smart pointer.
 *
 * Additional it provides several typedefs that may be used to ensure type consistency both within
 * the conexus library, and with applications using the conexus library.
 *
 * @author Rick L Vinyard Jr
 */
struct Data
  {
    /**
    * @typedef Octet
    * Typedefs the concept of an octet as an unsigned 8 bit integer.
    */
    typedef uint8_t Octet;

    /**
     * @typedef sptr
     * Provides a convenience typedef of a smart pointer to data.
     * Hint: access externally as Data::sptr
     */
    typedef boost::shared_array<Octet> sptr;

    /**
     * Default constructor initializes data size to zero and relies on smart pointer constructor to
     * initialize pointer to NULL.
     */
    Data(): size(0)
    { }

    /**
     * Constructor which accepts a smart pointer to data and data size in octets.
     */
    Data(sptr d, size_t s): data(d), size(s)
    { }

    /**
     * Constructor which creates a data object of a given size in octets.
     */
    Data(size_t s): size(s)
    {
      data = sptr(new Octet[s]);
    }

    /**
     * Overloaded dereference operator provides direct access to actual data. If you want access
     * to the smart pointer, use the @em data member instead.
     */
    operator Octet*()
    {
      return data.get();
    }

    std::string octet_string(std::string separator="")
    {
      std::string s;
      size_t i;
      uint8_t* p;
      char hex[3];

      p = data.get();
      for (i=0; i < size; i++)
        {
          if (i != 0)
            s += separator;
          hex[0] = '0' + (*p >> 4);
          hex[1] = '0' + (*p & 0x0F);
          if (hex[0] > '9')
            hex[0] += 7;
          if (hex[1] > '9')
            hex[1] += 7;
          hex[2] = 0x00;
          s += hex;
          p++;
        }
      return s;
    }
    
    /**
     * Smart pointer to the underlying data.
     */
    sptr data;

    /**
     * Size of the data in bytes.
     */
    size_t size;
  };

/**
 * This class represents a constant data block allocated via a smart pointer, hence the name CData as a shortened
 * form of ConstantData.
 *
 * Additional it provides several typedefs that may be used to ensure type consistency both within
 * the conexus library, and with applications using the conexus library.
 * @author Rick L Vinyard Jr
 */
struct CData
  {
    /**
     * @typedef Octet
     * Typedefs the concept of an octet as an unsigned 8 bit integer.
     */
    typedef uint8_t Octet;

    /**
     * @typedef sptr
     * Provides a convenience typedef of a smart pointer to constant data.
     * Hint: access externally as CData::sptr
     */
    typedef boost::shared_array<const Octet> sptr;

    /**
     * Default constructor initializes data size to zero and relies on smart pointer constructor to
     * initialize pointer to NULL.
     */
    CData(): size(0)
    { }

    /**
     * Constructor which accepts a smart pointer to data and data size in bytes.
     */
    CData(const Octet* d, size_t s): data(d), size(s)
    { }

    /**
     * Overloaded dereference operator provides direct access to actual data. If you want access
     * to the smart pointer, use the @em data member instead.
     */
    operator const Octet*()
    {
      return data.get();
    }

    /**
     * Smart pointer to the underlying data.
     */
    sptr data;

    /**
     * Size of the data in bytes.
     */
    const size_t size;
  };

}

#endif
