-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

function math.clamp(a, b, c)
   if a < b then a = b end
   if a > c then a = c end

   return a
end

function math.permute(a, ...)
   local b = {}

   for i, j in ipairs {...} do
      b[i] = a[j]
   end

   return b
end

function math.add(a, ...)
   for _, b in ipairs {...} do
      a = {a[1] + b[1], a[2] + b[2], a[3] + b[3]}
   end

   return a
end

function math.subtract(a, ...)
   for _, b in ipairs {...} do
      a = {a[1] - b[1], a[2] - b[2], a[3] - b[3]}
   end

   return a
end

function math.multiply(a, ...)
   for _, b in ipairs {...} do
      a = {a[1] * b[1], a[2] * b[2], a[3] * b[3]}
   end

   return a
end

function math.divide(a, ...)
   for _, b in ipairs {...} do
      a = {a[1] / b[1], a[2] / b[2], a[3] / b[3]}
   end

   return a
end

function math.scale(a, b)
   return {a[1] * b, a[2] * b, a[3] * b}
end

function math.dot(a, b)
   return a[1] * b[1] + a[2] * b[2] + a[3] * b[3]
end

function math.cross(a, b)
   return {
      a[2] * b[3] - a[3] * b[2],
      a[3] * b[1] - a[1] * b[3],
      a[1] * b[2] - a[2] * b[1]
   }
end

function math.length2(a)
   return math.dot (a, a)
end

function math.length(a)
   return math.sqrt (math.length2 (a))
end

function math.distance2(a, b)
   return math.length2(math.subtract(a, b))
end

function math.distance(a, b)
   return math.sqrt(math.distance2(a, b))
end

function math.normalize (a)
   local l = math.length(a)

   return l > 0 and math.scale (a, 1 / l) or {0, 0, 0}
end

function math.project (a, n)
   local dot

   dot = a[1] * n[1] + a[2] * n[2] + a[3] * n[3]

   return {
      a[1] - dot * n[1],
      a[2] - dot * n[2],
      a[3] - dot * n[3]
   }
end

function math.row (T, n)
   return {
      T[3 * n + 1],
      T[3 * n + 2],
      T[3 * n + 3]
   }
end

function math.column (T, n)
   return {
      T[n],
      T[3 + n],
      T[6 + n]
   }
end

function math.rotate (v, u, theta)
   -- Use Rodrigues' rotation formula to rotate
   -- v by theta radians around u

   return math.add (
      math.scale (v, math.cos(theta)),
      math.scale (math.cross (v, u), math.sin (theta)),
      math.scale (u, math.dot (v, u) * (1 - math.cos (theta)))
   )
end
