-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "transforms"
require "transitions"
require "moremath"
require "billiard"
require "moremeshes"
require "resources"
require (options.toon and "toon" or "shading")

require "textures"
require "accoustics"
require "switches"

local function respotball (ball, offset)
   if not offset then 
      offset = 0.25 * billiards.tablewidth
   end

   ball.position = {offset, 0, billiards.ballradius}
   ball.velocity = {0, 0, 0}
   ball.spin = {0, 0, 0}

   for _, other in pairs (bodies.balls) do
      local p = ball.position
      local q = other.position
      local r = billiards.ballradius

      if math.distance (p, q) <= 2 * r and other ~= ball then
	 if options.pool then
	    respotball (ball, q[1] - math.sqrt (4 * r * r - q[2] * q[2]) - 1e-3)
	 else
	    respotball (ball, offset > 0 and -offset or 0)
	 end
	 
	 break
      end
   end
end

meshes.ball = function ()
   local mesh = meshes.sphere (billiards.ballradius, 32, 16)
		 
   for i = 7, #mesh.vertices, 8 do
      mesh.vertices[i] = mesh.vertices[i] * 4 - 1.5
      mesh.vertices[i + 1] = mesh.vertices[i + 1] * 2 - 0.5
   end
   
   return meshes.static(mesh)
end

if options.eightball then
   textures.decals = {
      {0.97, 0.97, 0.82},
      resources.patched "scripts/imagery/diffuse/one.lc",
      resources.patched "scripts/imagery/diffuse/two.lc",
      resources.patched "scripts/imagery/diffuse/three.lc",
      resources.patched "scripts/imagery/diffuse/four.lc",
      resources.patched "scripts/imagery/diffuse/five.lc",
      resources.patched "scripts/imagery/diffuse/six.lc",
      resources.patched "scripts/imagery/diffuse/seven.lc",
      resources.patched "scripts/imagery/diffuse/eight.lc",
      resources.patched "scripts/imagery/diffuse/nine.lc",
      resources.patched "scripts/imagery/diffuse/ten.lc",
      resources.patched "scripts/imagery/diffuse/eleven.lc",
      resources.patched "scripts/imagery/diffuse/twelve.lc",
      resources.patched "scripts/imagery/diffuse/thirteen.lc",
      resources.patched "scripts/imagery/diffuse/fourteen.lc",
      resources.patched "scripts/imagery/diffuse/fifteen.lc",
   }
elseif options.nineball then
   textures.decals = {
      {0.97, 0.97, 0.82},
      resources.patched "scripts/imagery/diffuse/one.lc",
      resources.patched "scripts/imagery/diffuse/two.lc",
      resources.patched "scripts/imagery/diffuse/three.lc",
      resources.patched "scripts/imagery/diffuse/four.lc",
      resources.patched "scripts/imagery/diffuse/five.lc",
      resources.patched "scripts/imagery/diffuse/six.lc",
      resources.patched "scripts/imagery/diffuse/seven.lc",
      resources.patched "scripts/imagery/diffuse/eight.lc",
      resources.patched "scripts/imagery/diffuse/nine.lc",
   }
else
   textures.decals = {
      {0.97, 0.97, 0.82},
      {0.92, 0.66, 0.18},
      {0.4, 0.12, 0.1},
   }
end

math.randomseed (os.clock())

-- Create the balls.

bodies.balls = {}

for i, decal in ipairs (textures.decals) do
   bodies.balls[i] = bodies.ball {
      position = billiards.opening[i],

      radius = billiards.ballradius,
      mass = physics.spheremass (3 * billiards.ballmass /
				 (4 * math.pi *
				  billiards.ballradius^3),
			         billiards.ballradius),

      lethargy = {0.01, 0.01, 10, 0},

      isball = true,

      shading = switches.button {
	 surface = options.toon and toon.cel {
	    color = decal,
	    mesh = meshes.ball()
	 } or shading.fresnel {
	    diffuse = decal,
	    specular = {0.7, 0.7, 0.7},
	    parameter = {96, 0.1},
      
	    mesh = meshes.ball()
	 },

	 selected = frames.event {
	    outline = shapes.halo {
	       color = {0.7, 0.6, 0.1},
	       opacity = 0.55,
	       width = 5.0,
	       geometry = meshes.ball()
	    },

	    buttonpress = function (self, button)
	       local grandparent = self.parent.parent
	     
	       if button == bindings.move and
		  not bodies.observer.isaiming then
		  grandparent.home = grandparent.position
	       end

	       if button == bindings.ready then
		  if grandparent == bodies.cueball then
		     if bodies.observer.islooking and not
			(bodies.cueball.ispocketed or
			 bodies.cueball.isout) then
			bodies.observer.isaiming = true
		     elseif bodies.observer.isaiming then
			bodies.observer.islooking = true
		     end
		  elseif bodies.observer.islooking then
		     bodies.cueball = grandparent
		     bodies.observer.longitude = bodies.cueball.position[1]
		     bodies.observer.latitude = bodies.cueball.position[2]
		  end
	       end
	    end,

	    buttonrelease = function (self, button)
	       local grandparent = self.parent.parent

	       if grandparent.home then
		  grandparent.home = nil
	       end

	       if bodies.observer.wasaiming then
		  bodies.observer.wasaiming = false
		  bodies.observer.isaiming = true

		  bodies.observer.azimuth = bodies.observer.oldazimuth
		  bodies.observer.elevation = bodies.observer.oldelevation
		  bodies.observer.longitude = bodies.cueball.position[1]
		  bodies.observer.latitude = bodies.cueball.position[2]
	       end
	    end,

	    motion = function (self, button, x, y, dx, dy)
	       local grandparent = self.parent.parent
	     
	       if grandparent.home then
		  local w, h, u, v

		  w = billiards.tablewidth / 2 - billiards.ballradius - 1e-3
		  h = billiards.tableheight / 2 - billiards.ballradius - 1e-3

		  u = transforms.fromnode(bodies.observer.eye, {0, 1, 0})
		  v = transforms.fromnode(bodies.observer.eye, {0, 0, -1})

		  u = math.normalize(math.project(u, {0, 0, 1}))
		  v = math.normalize(math.project(v, {0, 0, 1}))

		  grandparent.home = {
		     math.clamp(grandparent.home[1] +
				0.003 * (dx * u[1] - dy * v[1]),
			        -w, w),
		     math.clamp(grandparent.home[2] +
				0.003 * (dx * u[2] - dy * v[2]),
			        -h, h),
		     billiards.ballradius
		  }

		  bodies.observer.longitude = grandparent.home[1]
		  bodies.observer.latitude = grandparent.home[2]
	       end
	    end,
	 }
      },

      shadow = options.toon and frames.gimbal {
	 surface = toon.flat {
	    color = {0, 0, 0},

	    toon.shadow {
	       position = {0, 0, -billiards.ballradius + 0.001},
	       mesh = meshes.ball()
	    }
	 }
      },
   }
end

for _, ball in pairs (bodies.balls) do
   physics.addforce (ball, {5 * math.random(), 5 * math.random(), 0})
end 

bodies.cueball = bodies.balls[1]

-- Add them to the scene.

graph.gear = bodies.system {

   ambience = not options.toon and shading.ambient {
      intensity = resources.clamped(options.pool and
				    "scripts/imagery/light/poolambience.lc" or
				    "scripts/imagery/light/caromambience.lc")
   },

   prepare = function (self)
		local isstable = true
		
		-- Check whether the shot is finished. A ball is
		-- stable if it is either resting or off the table
		
		for _, ball in pairs (bodies.balls) do
		   if not (ball.isout or
			   ball.isresting or
			   ball.ispocketed) then
		      isstable = false
		   end
		end

		-- Finished?

		if not isstable and
		   not bodies.observer.iswaiting then
		   bodies.observer.iswaiting = true
		end

		if bodies.observer.iswaiting and isstable then
		   bodies.observer.islooking = true
		end
	     end,

   unpack (bodies.balls)
}

-- Now for the simulation and control logic.

billiards.looking.newshot = function ()
   graph.transition = transitions.fade {
      duration = 0.7
   }

   -- Replace any balls that have fallen
   -- off the table to their opening spots

   for _, ball in pairs (bodies.balls) do
      if ball.isout then
	 respotball (ball)
      end
   end

   -- Set-up the camera.

   bodies.observer.radius = 1.5

   if bodies.cueball.ispocketed then
      bodies.observer.azimuth = math.rad(0)
      bodies.observer.elevation = math.rad(80)
      bodies.observer.longitude = 0.5 * billiards.tablewidth + 0.4
      bodies.observer.latitude = 0
   else
      bodies.observer.longitude = bodies.cueball.position[1]
      bodies.observer.latitude = bodies.cueball.position[2]
   end
end

billiards.aiming.reset = function ()
   graph.transition = transitions.fade {
      duration = 0.7
   }

   bodies.cue.elevation = 0
   bodies.cue.sidespin = 0
   bodies.cue.follow = 0

   bodies.observer.radius = 0.4
   bodies.observer.elevation = math.pi / 2 -
			       bodies.cue.elevation -
			       math.rad (10)
end

billiards.turning.aim = function ()
   bodies.cue.azimuth = bodies.observer.azimuth
end

billiards.striking.zoom = function ()
   bodies.observer.radius = 1.5
end

graph.observer.controls.selected.keypress = function (self, key)
   if key == 'q' then
      common.iterate = false
   elseif key == 'd' then
      for _, ball in ipairs(bodies.balls) do
	 if ball.position[3] > billiards.ballradius / 2 then
	    local s = math.random() - 0.5
	    local t = math.random() - 0.5

	    physics.wake(ball)
	    ball.position = {s / math.abs(s) * billiards.tablewidth/2,
			     t / math.abs(t) * billiards.tableheight/2,
			     0.1}
	    break
	 end
      end
   elseif key == 'f' then
      for _, ball in ipairs(bodies.balls) do
	 if ball.position[1] < 0 then
	    respotball (ball)
	    break
	 end
      end
   end
end
