/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <AL/al.h>
#include <AL/alc.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glx.h>
#include <time.h>

#include "node.h"

Node *list;

static void callhooks (lua_State *L, void *key, int reference)
{
    if (reference != LUA_REFNIL) {
	lua_rawgeti(L, LUA_REGISTRYINDEX, reference);
	
	if (lua_isfunction (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushlightuserdata (L, key);
	    lua_gettable (L, -2);
	    lua_replace (L, -2);
	    
	    lua_call(L, 1, 0);
	} else if (lua_istable (L, -1)) {
	    lua_getfield (L, LUA_REGISTRYINDEX, "userdata");
	    lua_pushnil(L);
	
	    while(lua_next(L, -3)) {
		lua_pushlightuserdata (L, key);
		lua_gettable (L, -4);
		lua_call (L, 1, 0);
	    }
	    
	    lua_pop (L, 2);
	} else {
	    lua_pop (L, 1);
	}
    }
}

@implementation Node

-(Node *) init
{
    int i, j;
    
    [super init];
    
    self->link = LUA_REFNIL;
    self->unlink = LUA_REFNIL;
    self->transform = LUA_REFNIL;
    self->step = LUA_REFNIL;
    self->traverse = LUA_REFNIL;
    self->prepare = LUA_REFNIL;
    self->cleanup = LUA_REFNIL;
    
    self->linked = 0;

    self->next = NULL;

    for (i = 0 ; i < 3 ; i += 1) {
	for (j = 0 ; j < 3 ; j += 1) {
	    self->orientation[i * 3 + j] = i == j ? 1 : 0;
	    self->rotation[i * 3 + j] = i == j ? 1 : 0;
	}

	self->position[i] = 0;
	self->translation[i] = 0;
    }

    return self;
}

-(int) linked
{
    return self->linked;
}

-(void) setPositionTo: (GLfloat *)new
{
    int i;

    for (i = 0 ; i < 3 ; i += 1) {
	self->position[i] = new[i];
    }
}

-(void) setOrientationTo: (GLfloat *)new
{
    int i;

    for (i = 0 ; i < 9 ; i += 1) {
	self->orientation[i] = new[i];
    }
}

-(GLfloat *) position
{
    return self->position;
}

-(GLfloat *) orientation
{
    return self->orientation;
}

-(GLfloat *) translation
{
    return self->translation;
}

-(GLfloat *) rotation
{
    return self->rotation;
}

-(GLfloat *) homogenous
{
    return self->homogenous;
}

-(void) toggle
{
    id child;

    self->linked = !self->linked;

    if (linked) {
	callhooks (_L, self, self->link);
    } else {
	callhooks (_L, self, self->unlink);
    }
    
    /* Recurse on the node's children. */
    
    for(child = [self children] ; child ; child = [child sister]) {
        [child toggle];
    }
}

-(void) stepBy: (double) h
{
    id child;

    callhooks (_L, self, self->step);

    for(child = [self children] ; child ; child = [child sister]) {
	[child stepBy: h];
    }
}

-(void) transform
{
    id child;
    
    GLfloat *r, *R, *p, *W;
    int i, j;

    /* Call the node's transform hook before
       doing anything else. */
    
    callhooks (_L, self, self->transform);

    p = [self position];
    W = [self orientation];
    
    if ([self parent]) {

	r = [[self parent] translation];
	R = [[self parent] rotation];

	for (i = 0 ; i < 3 ; i += 1) {
	    for (j = 0 ; j < 3 ; j += 1) {
		self->rotation[i * 3 + j] = R[i * 3] * W[j] +
		                            R[i * 3 + 1] * W[j + 3] +
		                            R[i * 3 + 2] * W[j + 6];
	    }

	    self->translation[i] = R[i * 3] * p[0] +
		                   R[i * 3 + 1] * p[1] +
		                   R[i * 3 + 2] * p[2] +
		                   r[i]; 
	}
    } else {

	/* Just return the position and orientation. */
	
	for (i = 0 ; i < 3 ; i += 1) {
	    self->translation[i] = self->position[i];
	}

	for (i = 0 ; i < 9 ; i += 1) {
	    self->rotation[i] = self->orientation[i];
	}
    }

    /* Calculate the homogenous transform matrix. */
    
    self->homogenous[0] = self->rotation[0];
    self->homogenous[1] = self->rotation[3];
    self->homogenous[2] = self->rotation[6];
    self->homogenous[3] = 0;
    
    self->homogenous[4] = self->rotation[1];
    self->homogenous[5] = self->rotation[4];
    self->homogenous[6] = self->rotation[7];
    self->homogenous[7] = 0;
    
    self->homogenous[8] = self->rotation[2];
    self->homogenous[9] = self->rotation[5];
    self->homogenous[10] = self->rotation[8];
    self->homogenous[11] = 0;

    self->homogenous[12] = self->translation[0];
    self->homogenous[13] = self->translation[1];
    self->homogenous[14] = self->translation[2];
    self->homogenous[15] = 1;
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child transform];
    }
}

-(void) prepare
{
    id child;
    
    callhooks (_L, self, self->prepare);

/*     glPolygonMode (GL_FRONT_AND_BACK, GL_LINE); */
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child prepare];
    }
}

-(void) deferTraversal
{
    Node *previous, *node;

    /* Sort the node into the list. */
    
    for (previous = NULL, node = list;
	 node && node->translation[2] >= self->translation[2];
	 previous = node, node = node->next);

    if (node == list) {
	self->next = list;
	list = self;
    } else {
	self->next = node;
	previous->next = self;
    }
}

-(void) traverseDeferred
{
}

-(void) traverse
{
    id child;
    Node *node;

    callhooks (_L, self, self->traverse);

    for(child = [self children] ; child ; child = [child sister]) {
	[child traverse];
    }

    /* Once the tree is traversed traverse the deffered nodes. */
    
    if (!self->up) {
	for (node = list ; node ; node = node->next) {
	    [node traverseDeferred];
	}
	
	list = NULL;
    }
}

-(void) cleanup
{
    id child;
    
    callhooks (_L, self, self->cleanup);
    
    for(child = [self children] ; child ; child = [child sister]) {
	[child cleanup];
    }
}

-(void) get
{
    const char *k;
    int i;

    k = lua_tostring(_L, 2);

    if (!xstrcmp(k, "position")) {
        lua_newtable (_L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber (_L, self->position[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "orientation")) {
        lua_newtable (_L);
        
        for(i = 0; i < 9; i += 1) {
            lua_pushnumber (_L, self->orientation[i]);
            lua_rawseti (_L, -2, i + 1);
        }
    } else if (!xstrcmp(k, "parent")) {
	lua_getmetatable (_L, 1);
	lua_pushstring (_L, "__parent");
	lua_gettable (_L, -2);
    } else if (!xstrcmp(k, "link")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->link);
    } else if (!xstrcmp(k, "unlink")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->unlink);
    } else if (!xstrcmp(k, "transform")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->transform);
    } else if (!xstrcmp(k, "step")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->step);
    } else if (!xstrcmp(k, "traverse")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->traverse);
    } else if (!xstrcmp(k, "prepare")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->prepare);
    } else if (!xstrcmp(k, "cleanup")) {
        lua_rawgeti (_L, LUA_REGISTRYINDEX, self->cleanup);
    } else  {
	lua_getmetatable (_L, 1);
	lua_replace (_L, 1);
	lua_gettable (_L, 1);
    }
}

-(void) set
{    
    const char *k;
    int i;

    k = lua_tostring (_L, 2);

    if (!xstrcmp(k, "position")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->position[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "orientation")) {
        if(lua_istable (_L, 3)) {
            for(i = 0 ; i < 9 ; i += 1) {
                lua_rawgeti (_L, 3, i + 1);
                self->orientation[i] = lua_tonumber (_L, -1);
                
                lua_pop (_L, 1);
            }
        }
    } else if (!xstrcmp(k, "parent")) {
	if (lua_isuserdata (_L, 3)) {
	    lua_pushlightuserdata(_L, (void *)lua_topointer (_L, 1));
	    lua_pushvalue (_L, 1);
	    lua_settable (_L, 3);
	} else {
	    if (luaL_getmetafield(_L, 1, "__parent")) {
		luaL_getmetafield(_L, 1, "__key");
		lua_pushnil (_L);
		lua_settable (_L, -3);
	    }
	}
    } else if (!xstrcmp(k, "unlink")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->unlink);
        self->unlink = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "link")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->link);
        self->link = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "transform")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->transform);
        self->transform = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "step")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->step);
        self->step = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "traverse")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->traverse);
        self->traverse = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "prepare")) {
        luaL_unref (_L, LUA_REGISTRYINDEX, self->prepare);
        self->prepare = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else if (!xstrcmp(k, "cleanup")) {
	luaL_unref (_L, LUA_REGISTRYINDEX, self->cleanup);
        self->cleanup = luaL_ref (_L, LUA_REGISTRYINDEX);
    } else {
	Node *child;
	const char *name;

        /* Check if there's a node linked with
	   the same key and if so free it. */

	lua_getmetatable (_L, 1);
	lua_pushvalue (_L, 2);
	lua_gettable (_L, -2);
        
	if(lua_isuserdata (_L, -1)) {
	    child = *(Node **)lua_touserdata (_L, -1);
            
	    if ([child isKindOf: [Node class]]) {
		if([child linked]) {
		    [child toggle];
		}
		
		[self renounce: child];
		[child adoptedBy: nil];
	    }

	    lua_getmetatable (_L, -1);
	    lua_pushnil (_L);
	    lua_setfield (_L, -2, "__parent");
	    lua_pushnil (_L);
	    lua_setfield (_L, -2, "__key");
	    lua_pop (_L, 1);
	}
	
	lua_pop (_L, 1);
	
	/* Link the new node. */
	
	if(lua_isuserdata (_L, 3)) {
	    child = *(Node **)lua_touserdata (_L, 3);

	    if ([child isKindOf: [Node class]]) {
		/* Unlink the child if it already has a parent... */

		if (luaL_getmetafield (_L, 3, "__parent")) {
		    luaL_getmetafield (_L, 3, "__key");
		    lua_pushnil (_L);
		    lua_settable (_L, -3);
		    lua_pop (_L, 1);
		}

		/* ...and link to the new parent. */

		lua_getmetatable (_L, 3);
		lua_pushvalue (_L, 1);
		lua_setfield (_L, -2, "__parent");
		lua_pushvalue (_L, 2);
		lua_setfield (_L, -2, "__key");
		lua_pop (_L, 1);
	    
		name = lua_tostring (_L, 2);

		[self adopt: child named: name];
		[child adoptedBy: self];
	
		if([self linked]) {
		    [child toggle];
		}
	    }
	}
        
	/* Update the children table. */

	lua_replace (_L, 1);
	lua_settable (_L, 1);
    }
}

@end
