/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glext.h>
#include "texture.h"
#include "fixed.h"
#include "light.h"
#include "ambient.h"
#include "fog.h"

@implementation Fixed

+(void) load
{
    GLfloat black[4] = {0, 0, 0, 1};

    glLightModelfv (GL_LIGHT_MODEL_AMBIENT, black);
    glLightfv (GL_LIGHT0, GL_DIFFUSE, black);
    glLightfv (GL_LIGHT0, GL_SPECULAR, black);
}

-(Fixed *)init
{
    [super init];

    self->diffuse.texture = nil;
    self->diffuse.values[0] = 0;
    self->diffuse.values[1] = 0;
    self->diffuse.values[2] = 0;
     
    self->specular.texture = nil;
    self->specular.values[0] = 0;
    self->specular.values[1] = 0;
    self->specular.values[2] = 0;

    self->parametric.texture = nil;
    self->parametric.values[0] = 0;
    self->parametric.values[1] = 0;
    self->parametric.values[2] = 0;

    return self;
}
    
-(void)traverse
{
    id parent, child;
    int n = 0, m = 0, s = 0;

    GLfloat whiteColor[4] = {1, 1, 1, 1};
    GLfloat diffuseColor[4] = {self->diffuse.values[0],
			       self->diffuse.values[1],
			       self->diffuse.values[2],
			       1};

    GLfloat specularColor[4] = {self->specular.values[0],
				self->specular.values[1],
				self->specular.values[2],
				1};

    GLfloat shininess = self->parametric.values[0];
		       
    glUseProgramObjectARB(0);
    glActiveTexture (GL_TEXTURE0);

    glEnable (GL_DEPTH_TEST);
    glEnable (GL_TEXTURE_2D);
    glEnable (GL_CULL_FACE);
    glEnable (GL_LIGHTING);

    if (!self->diffuse.texture) {
	glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, diffuseColor);
	glBindTexture(GL_TEXTURE_2D, 0);
    } else {
	glMaterialfv (GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, whiteColor);
	glBindTexture(GL_TEXTURE_2D, [self->diffuse.texture index]);
    }

    glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, specularColor);
    glMaterialf (GL_FRONT_AND_BACK, GL_SHININESS, shininess);

    for (n = 0 ; n < GL_MAX_LIGHTS ; n += 1) {
	glDisable (GL_LIGHT0 + n);
    }

    for (n = 0, m = 0, parent = [self parent];
	 parent;
	 parent = [parent parent]) {
	for (child = [parent children] ; child ; child = [child sister]) {
	    if ([child isMemberOf: [Light class]]) {
		GLfloat origin[4] = {[child translation][0],
				     [child translation][1],
				     [child translation][2],
					     1};
		
		GLfloat intensityColor[4] = {[child intensityConstant][0],
					     [child intensityConstant][1],
					     [child intensityConstant][2],
					     1};

		glLightfv (GL_LIGHT0 + n, GL_POSITION, origin);
		glLightfv (GL_LIGHT0 + n, GL_DIFFUSE, intensityColor);
		glLightfv (GL_LIGHT0 + n, GL_SPECULAR, intensityColor);

		glLightf (GL_LIGHT0 + n, GL_CONSTANT_ATTENUATION,
			  [child attenuation][0]);
		glLightf (GL_LIGHT0 + n, GL_LINEAR_ATTENUATION,
			  [child attenuation][1]);
		glLightf (GL_LIGHT0 + n, GL_QUADRATIC_ATTENUATION,
			  [child attenuation][2]);

		glEnable (GL_LIGHT0 + n);

		n += 1;
	    } else if ([child isMemberOf: [Ambient class]] && m == 0) {
		GLfloat intensityColor[4] = {[child intensityConstant][0],
					     [child intensityConstant][1],
					     [child intensityConstant][2],
					     1};
		
		glLightfv (GL_LIGHT0 + m, GL_AMBIENT, intensityColor);

		m += 1;
	    } else if ([child isMemberOf: [Fog class]] && s == 0) {
		GLfloat fogColor[4] = {[child colorConstant][0],
				       [child colorConstant][1],
				       [child colorConstant][2],
				       1};
		
		glFogf (GL_FOG_MODE, GL_EXP);
		glFogf (GL_FOG_DENSITY, [child densityConstant][0]);
		glFogfv (GL_FOG_COLOR, fogColor);
		    
		s += 1;
	    }
	}
    }

    glPopMatrix();

    if (s > 0) {
	glEnable(GL_FOG);
    }
    
    if (self->traverse != LUA_REFNIL) {
	lua_rawgeti (_L, LUA_REGISTRYINDEX, self->traverse);
	lua_call (_L, 0, 0);
    }

    for(child = [self children] ; child ; child = [child sister]) {
	[child traverse];
    }

    glDisable (GL_LIGHTING);
    glDisable (GL_DEPTH_TEST);
    glDisable (GL_CULL_FACE);
    glDisable (GL_TEXTURE_2D);
    glDisable (GL_FOG);
}

@end
