/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "node.h"

static int translate(lua_State *L)
{
    double a[3], b[3];
    int i;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TTABLE);

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 1, i + 1);
	a[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }
    
    lua_newtable(L);
    
    for(i = 0 ; i < 3 ; i += 1) {
	lua_pushnumber(L, a[i] + b[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int rotate(lua_State *L)
{
    double v[3], T[9];
    int i;

    luaL_checktype (L, 1, LUA_TTABLE);
    luaL_checktype (L, 2, LUA_TTABLE);

    for (i = 0 ; i < 9 ; i += 1) {
	lua_rawgeti (L, 1, i + 1);
	T[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	v[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    lua_newtable(L);
        
    lua_pushnumber(L, T[0] * v[0] + T[1] * v[1] + T[2] * v[2]);
    lua_rawseti(L, -2, 1);
        
    lua_pushnumber(L, T[3] * v[0] + T[4] * v[1] + T[5] * v[2]);
    lua_rawseti(L, -2, 2);
        
    lua_pushnumber(L, T[6] * v[0] + T[7] * v[1] + T[8] * v[2]);
    lua_rawseti(L, -2, 3);

    return 1;
}

static int rotation(lua_State *L)
{
    double theta, a[3], rotation[16];
    int i;

    luaL_checknumber (L, 1);
    luaL_checktype (L, 2, LUA_TTABLE);

    theta = lua_tonumber(L, 1);

    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	a[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    /* We let the GL calculate the inverse
       rotation to save us a transposition. */
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glRotated (-theta, a[0], a[1], a[2]);
    glGetDoublev(GL_MODELVIEW_MATRIX, rotation);
    glPopMatrix();

    lua_newtable(L);
    
    for(i = 0 ; i < 9 ; i += 1) {
	lua_pushnumber(L, rotation[i + (i / 3)]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int inverse(lua_State *L)
{
    int i, j;

    luaL_checktype (L, 1, LUA_TTABLE);

    lua_newtable (L);
    
    for (j = 0 ; j < 3 ; j += 1) {
	for (i = 0 ; i < 3 ; i += 1) {
	    lua_rawgeti(L, 1, j * 3 + i + 1);
	    lua_rawseti(L, 2, i * 3 + j + 1);
	}
    }

    return 1;
}

static int euler(lua_State *L)
{
    double x, y, z, rotation[16];
    int i;

    x = luaL_checknumber (L, 1);
    y = luaL_checknumber (L, 2);
    z = luaL_checknumber (L, 3);
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glRotated (-z, 0, 0, 1);
    glRotated (-y, 0, 1, 0);
    glRotated (-x, 1, 0, 0);
    glGetDoublev(GL_MODELVIEW_MATRIX, rotation);
    glPopMatrix();

    lua_newtable(L);
    
    for(i = 0 ; i < 9 ; i += 1) {
	lua_pushnumber(L, rotation[i + (i / 3)]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int relue(lua_State *L)
{
    double x, y, z, rotation[16];
    int i;

    x = luaL_checknumber (L, 1);
    y = luaL_checknumber (L, 2);
    z = luaL_checknumber (L, 3);
    
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glRotated (-x, 1, 0, 0);
    glRotated (-y, 0, 1, 0);
    glRotated (-z, 0, 0, 1);
    glGetDoublev(GL_MODELVIEW_MATRIX, rotation);
    glPopMatrix();

    lua_newtable(L);
    
    for(i = 0 ; i < 9 ; i += 1) {
	lua_pushnumber(L, rotation[i + (i / 3)]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int fromnode(lua_State *L)
{
    id N;
    float *R, b[3], w[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    R = [N orientation];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    w[0] = R[0] * b[0] + R[1] * b[1] + R[2] * b[2];
    w[1] = R[3] * b[0] + R[4] * b[1] + R[5] * b[2];
    w[2] = R[6] * b[0] + R[7] * b[1] + R[8] * b[2];
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int tonode(lua_State *L)
{
    id N;
    float *R, b[3], w[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    R = [N orientation];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    w[0] = R[0] * b[0] + R[3] * b[1] + R[6] * b[2];
    w[1] = R[1] * b[0] + R[4] * b[1] + R[7] * b[2];
    w[2] = R[2] * b[0] + R[5] * b[1] + R[8] * b[2];

    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int unproject(lua_State *L)
{
    GLdouble x, y, z, s, t, p, M[16], P[16];
    GLint v[4];

    luaL_checktype (L, 1, LUA_TTABLE);

    glMatrixMode (GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();
    glGetDoublev (GL_MODELVIEW_MATRIX, M);
    glPopMatrix();
    
    glGetDoublev (GL_PROJECTION_MATRIX, P);
    glGetIntegerv (GL_VIEWPORT, v);

    lua_rawgeti (L, 1, 1);
    x = lua_tonumber (L, -1);
    lua_rawgeti (L, 1, 2);
    y = lua_tonumber (L, -1);
    lua_rawgeti (L, 1, 3);
    z = lua_tonumber (L, -1);

    gluUnProject (x, y, z, M, P, v, &s, &t, &p);

    lua_newtable (L);
    lua_pushnumber (L, s);
    lua_rawseti (L, -2, 1);
    lua_pushnumber (L, t);
    lua_rawseti (L, -2, 2);
    lua_pushnumber (L, p);
    lua_rawseti (L, -2, 3);

    return 1;
}

int luaopen_transforms (lua_State *L)
{
    const luaL_Reg transforms[] = {
	{"rotation", rotation},
	{"euler", euler},
	{"relue", relue},
	{"inverse", inverse},
	{"rotate", rotate},
	{"translate", translate},
	{"fromnode", fromnode},
	{"tonode", tonode},
	{"unproject", unproject},
	{NULL, NULL}
    };

    luaL_register (L, "transforms", transforms);

    return 0;
}
