/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glext.h>
#include "texture.h"
#include "specular.h"
#include "light.h"

static const GLchar *header =
"uniform sampler2D specularSampler;			  \n"
"uniform vec3 specularConstant;				  \n"
"						   	  \n"
"vec3 specularSum(vec2 texcoord)		   	  \n"
"{						   	  \n"    
"    return specularConstant + 			   	  \n"
"           vec3(texture2D(specularSampler, texcoord));   \n"
"}						  	  \n"    
"						   	  \n"
"#line 1						  \n";

@implementation Specular

-(Specular *)init
{
    self->specular.texture = nil;
    self->specular.values[0] = 0;
    self->specular.values[1] = 0;
    self->specular.values[2] = 0;

    self->specular.constant = glGetUniformLocationARB (self->program,
						    "specularConstant");    
    self->specular.sampler = glGetUniformLocationARB (self->program,
						   "specularSampler");
    [super init];
   
    return self;
}

-(void)attachVertexSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);

    [super attachVertexSource: (const GLchar *)newsource];
}

-(void)attachFragmentSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);
    
    [super attachFragmentSource: (const GLchar *)newsource];
}

-(void) get: (lua_State *)L
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "specular")) {
	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_gettable (L, 1);
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{    
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "specular")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->specular.values[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
	    
	    self->specular.texture = nil;
        } else if(lua_isuserdata (L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                self->specular.values[i] = 0;
            }
	    
	    self->specular.texture = *(id *)lua_touserdata (L, 3);
	}

	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_settable (L, 1);
    } else {
	[super set: L];
    }
}

-(void)traverse: (lua_State *)L
{
    GLint unit;
    
    glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
    glActiveTexture (unit + 1);

    if(self->specular.texture) {
	glBindTexture(GL_TEXTURE_2D, [self->specular.texture index]);
    } else {
	glBindTexture(GL_TEXTURE_2D, 0);
    }

    glUniform3fvARB (self->specular.constant, 1, self->specular.values);
    glUniform1iARB (self->specular.sampler, unit + 1 - GL_TEXTURE0);
	    
    [super traverse: L];
}

@end
