/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctype.h>
#include <string.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include "system.h"
#include "body.h"
#include "ball.h"
#include "billiard.h"
#include "box.h"
#include "plane.h"
#include "capsule.h"
#include "environment.h"
#include "ray.h"
#include "polyhedron.h"
#include "heightfield.h"

#include "slider.h"
#include "hinge.h"
#include "universal.h"
#include "spherical.h"

static char *decapitalize (char *s)
{
    s[0] = tolower(s[0]);

    return s;
}

static int node_tostring(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    lua_pushstring(L, [N name]);
   
    return 1;
}

static int node_index(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N get: L];
    
    return 1;
}

static int node_newindex(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N set: L];

    return 0;
}

static int node_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);
    
    [N free];

    return 0;
}

static int constructnode (lua_State *L)
{
    Class class;
    id object;
    dGeomID geom;

    lua_pushvalue (L, lua_upvalueindex (1));
    class = (Class)lua_touserdata(L, -1);
    lua_pop(L, 1);

    object = [[class alloc] init];
    
    if ([object isKindOf: [Body class]]) { 
	geom = [object geom];
    } else {
	geom = NULL;
    }
    
    /* Create the userdata... */
    
    *(id *)lua_newuserdata(L, sizeof(id)) = object;
    
    lua_newtable (L);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, (lua_CFunction)node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, (lua_CFunction)node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, (lua_CFunction)node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, (lua_CFunction)node_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 1)) {
	lua_pushnil(L);
	
	while(lua_next(L, 1)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 2);
	}
    }

    if(geom) {
	lua_getfield(L, LUA_REGISTRYINDEX, "geoms");
	
	if(lua_isnil(L, -1)){
	    lua_pop (L, 1);
	    
	    lua_newtable (L);
	    lua_newtable (L);

	    lua_pushstring (L, "__mode");
	    lua_pushstring (L, "v");
	    lua_settable (L, -3);
	    lua_setmetatable (L, -2);
	    
	    lua_pushvalue (L, -1);
	    lua_setfield(L, LUA_REGISTRYINDEX, "geoms");
	}

	lua_pushlightuserdata (L, geom);
	lua_pushvalue (L, 2);
	lua_settable (L, -3);
	lua_pop(L, 1);
    }

    return 1;
}

static int polyhedron_len(lua_State *L)
{
    id M;

    M = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [M vertices]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [M indices] / 3);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int polyhedron_call (lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */
	
    *(id *)lua_newuserdata(L, sizeof(id)) = [[N copy] init];
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, polyhedron_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, node_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    return 1;
}

static int polyhedron_tostring(lua_State *L)
{
    id M;

    M = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [M name]);
    lua_pushstring(L, " foundry");
    lua_concat(L, 2);
    
    return 1;
}

static int polyhedron_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    [N freeData];
    [N free];

    return 0;
}

static int constructpolyhedron(lua_State *L)
{
    id M;
    float *vertices;
    int *indices;
    int i, size[2];
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "vertices");
    lua_gettable(L, 1);
    
    /* Get the vertices. */

    if (lua_type(L, -1) == LUA_TTABLE) {
	vertices = (float *)malloc(3 * size[0] * sizeof(float));

	for(i = 0 ; i < 3 * size[0] ; i += 1) {
	    lua_rawgeti(L, -1, i + 1);
	
	    vertices[i] = (float)lua_tonumber(L, -1);
	
	    lua_pop(L, 1);
	}
    } else {
	vertices = (float *)lua_tostring(L, -1);
    }

    lua_pushstring(L, "indices");
    lua_gettable(L, 1);

    /* And the indices. */

    if (lua_type(L, -1) == LUA_TTABLE) {
	indices = (int *)malloc(size[1] * sizeof(int));

	for(i = 0 ; i < size[1] ; i += 1) {
	    lua_rawgeti(L, -1, i + 1);
	
	    indices[i] = (int)lua_tonumber(L, -1);
	
	    lua_pop(L, 1);
	}
    } else {
	indices = (int *)lua_tostring(L, -1);
    }
    
    /* Create and initialize the polyhedron userdata. */
    
    M = [[Polyhedron alloc] initWithVertices: vertices
	                    andIndices: indices
	                    ofSize: size];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = M;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, polyhedron_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, polyhedron_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, polyhedron_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);

    if (lua_type(L, -3) == LUA_TTABLE) {
	free(vertices);
    }
    
    if (lua_type(L, -2) == LUA_TTABLE) {
	free(indices);
    }

    return 1;
}

static int heightfield_len(lua_State *L)
{
    id T;

    T = *(id *)lua_touserdata(L, 1);

    lua_newtable(L);
    
    lua_pushnumber(L, [T width]);
    lua_rawseti(L, -2, 1);

    lua_pushnumber(L, [T height]);
    lua_rawseti(L, -2, 2);

    return 1;
}

static int heightfield_call (lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    /* Create the userdata... */
	
    *(id *)lua_newuserdata(L, sizeof(id)) = [[N copy] init];
    
    lua_newtable (L);
    lua_pushstring(L, "__len");
    lua_pushcfunction(L, heightfield_len);
    lua_settable(L, -3);
    lua_pushstring(L, "__index");
    lua_pushcfunction(L, node_index);
    lua_settable(L, -3);
    lua_pushstring(L, "__newindex");
    lua_pushcfunction(L, node_newindex);
    lua_settable(L, -3);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, node_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, node_gc);
    lua_settable(L, -3);
    lua_pushstring(L, "__reference");
    lua_pushvalue(L, 1);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    /* ...and initialize it. */

    if(lua_istable(L, 2)) {
	lua_pushnil(L);
	
	while(lua_next(L, 2)) {
	    lua_pushvalue(L, -2);
	    lua_insert(L, -2);
	    lua_settable(L, 3);
	}
    }

    return 1;
}

static int heightfield_tostring(lua_State *L)
{
    id M;

    M = *(id *)lua_touserdata(L, 1);

    lua_pushstring(L, [M name]);
    lua_pushstring(L, " foundry");
    lua_concat(L, 2);
    
    return 1;
}

static int heightfield_gc(lua_State *L)
{
    id N;

    N = *(id *)lua_touserdata(L, 1);

    [N freeData];
    [N free];

    return 0;
}

int constructheightfield(lua_State *L)
{
    id T;
    float *pixels;
    int i, size[2];
    float area[2], scale, offset, thickness;
    
    luaL_checktype(L, 1, LUA_TTABLE);

    lua_pushstring(L, "size");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            size[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "area");
    lua_gettable(L, 1);

    if(lua_istable(L, -1)) {
        for(i = 0 ; i < 2 ; i += 1) {
            lua_rawgeti(L, -1, i + 1);
            area[i] = lua_tonumber(L, -1);
                
            lua_pop(L, 1);
        }
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "scale");
    lua_gettable(L, 1);

    if (lua_isnumber (L, -1)) {
	scale = lua_tonumber (L, -1);
    } else {
	scale = 1;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "offset");
    lua_gettable(L, 1);

    if (lua_isnumber (L, -1)) {
	offset = lua_tonumber (L, -1);
    } else {
	offset = 0;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "thickness");
    lua_gettable(L, 1);

    if (lua_isnumber (L, -1)) {
	thickness = lua_tonumber (L, -1);
    } else {
	thickness = 0;
    }
    
    lua_pop(L, 1);

    lua_pushstring(L, "pixels");
    lua_gettable(L, 1);
    
    if (lua_type(L, -1) == LUA_TTABLE) {
	pixels = (float *)malloc(size[0] * size[1] * sizeof(float));

	for(i = 0 ; i < size[0] * size[1] ; i += 1) {
	    lua_rawgeti(L, -1, i + 1);
	    pixels[i] = (float)(lua_tonumber(L, -1));
	    lua_pop(L, 1);
	}
    } else {
	pixels = (float *)lua_tostring(L, -1);
    }

    /* Create and initialize the heightfield userdata. */

    T = [[Heightfield alloc] initWithFloats: pixels
                                     ofSize: size
		                   spanning: area
		                   scaledBy: scale
		                   offsetBy: offset
	                     andThickenedBy: thickness];
    
    *(id *)lua_newuserdata(L, sizeof(id)) = T;

    lua_newtable(L);
    lua_pushstring(L, "__tostring");
    lua_pushcfunction(L, heightfield_tostring);
    lua_settable(L, -3);
    lua_pushstring(L, "__call");
    lua_pushcfunction(L, heightfield_call);
    lua_settable(L, -3);
    lua_pushstring(L, "__gc");
    lua_pushcfunction(L, heightfield_gc);
    lua_settable(L, -3);
    lua_setmetatable(L, -2);
    
    if (lua_type(L, -2) == LUA_TTABLE) {
	free(pixels);
    }

    return 1;
}

int luaopen_simulator (lua_State *L)
{
    int i;
    
    Class bodies[] = {
	[System class], [Body class], [Ball class], [Box class],
	[Environment class], [Capsule class], [Billiard class],
	[Ray class], [Plane class]
    };	
    
    Class joints[] = {
	[Slider class], [Universal class], [Hinge class], [Spherical class]
    };	

    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(bodies) / sizeof(bodies[0]) ; i += 1) {
	lua_pushlightuserdata (L, bodies[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([bodies[i] name])));
    }

    lua_pushcfunction (L, constructpolyhedron);
    lua_setfield(L, -2, decapitalize(strdupa([Polyhedron name])));

    lua_pushcfunction (L, constructheightfield);
    lua_setfield(L, -2, decapitalize(strdupa([Heightfield name])));
    
    lua_setglobal (L, "bodies");

    lua_newtable (L);
    
    for (i = 0 ; i < sizeof(joints) / sizeof(joints[0]) ; i += 1) {
	lua_pushlightuserdata (L, joints[i]);
	lua_pushcclosure (L, constructnode, 1);
	lua_setfield(L, -2, decapitalize(strdupa([joints[i] name])));
    }

    lua_setglobal (L, "joints");

    return 0;
}
