/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <lua.h>
#include <lauxlib.h>
#include <ode/ode.h>

#include "body.h"
#include "system.h"
#include "polyhedron.h"

extern void addContactJoint (Body *a, Body *b, dVector3 pos, dVector3 normal,
			     dReal depth, dReal mu, dReal bounce);

static int addforce (lua_State *L)
{
    Body *N;
    double F[3], p[3];
    int i;

    N = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2)) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    F[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}

	if (lua_istable (L, 3)) {
	    for(i = 0; i < 3; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		p[i] = lua_tonumber (L, -1);

		lua_pop(L, 1);
	    }
	
	    dBodyAddForceAtPos ([N body],
				F[0], F[1], F[2],
				p[0], p[1], p[2]);
	} else {
	    dBodyAddForce ([N body], F[0], F[1], F[2]);
	}

	dBodyEnable([N body]);
    }

    return 0;
}

static int addrelativeforce (lua_State *L)
{
    Body *N;
    double F[3], p[3];
    int i;

    N = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2)) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    F[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}

	if (lua_istable (L, 3)) {
	    for(i = 0; i < 3; i += 1) {
		lua_rawgeti(L, 3, i + 1);
		p[i] = lua_tonumber (L, -1);

		lua_pop(L, 1);
	    }
	
	    dBodyAddRelForceAtRelPos ([N body],
				      F[0], F[1], F[2],
				      p[0], p[1], p[2]);
	} else {
	    dBodyAddRelForce ([N body], F[0], F[1], F[2]);
	}

	dBodyEnable([N body]);
    }

    return 0;
}

static int addtorque (lua_State *L)
{
    Body *N;
    double T[3];
    int i;

    N = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2)) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    T[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}
	
	dBodyAddTorque ([N body], T[0], T[1], T[2]);

	dBodyEnable([N body]);
    }

    return 0;
}

static int addrelativetorque (lua_State *L)
{
    Body *N;
    double T[3];
    int i;

    N = *(Body **)lua_touserdata (L, 1);

    if (lua_istable (L, 2)) {
	for(i = 0; i < 3; i += 1) {
	    lua_rawgeti(L, 2, i + 1);
	    T[i] = lua_tonumber (L, -1);

	    lua_pop(L, 1);
	}
	
	dBodyAddRelTorque ([N body], T[0], T[1], T[2]);

	dBodyEnable([N body]);
    }

    return 0;
}

static void pushmass (lua_State *L, dMass *M)
{
    int i, j;
    
    lua_newtable(L);

    lua_pushnumber(L, M->mass);
    lua_rawseti(L, -2, 1);

    lua_newtable(L);

    for(i = 0 ; i < 3 ; i += 1) {
	lua_pushnumber(L, M->c[i]);
	lua_rawseti(L, -2, i + 1);
    }

    lua_rawseti(L, -2, 2);

    lua_newtable(L);

    for(j = 0 ; j < 3 ; j += 1) {
	for(i = 0 ; i < 3 ; i += 1) {
	    lua_pushnumber(L, M->I[j * 4 + i]);
	    lua_rawseti(L, -2, j * 3 + i + 1);
	}
    }

    lua_rawseti(L, -2, 3);
}

static int spheremass (lua_State *L)
{
    dMass M;
    dReal rho, r;

    rho = (dReal)luaL_checknumber (L, 1);
    r = (dReal)luaL_checknumber (L, 2);

    dMassSetSphere (&M, rho, r);
    pushmass (L, &M);

    return 1;
}

static int capsulemass (lua_State *L)
{
    dMass M;
    dReal rho, l, r;

    rho = (dReal)luaL_checknumber (L, 1);
    r = (dReal)luaL_checknumber (L, 2);
    l = (dReal)luaL_checknumber (L, 3);

    dMassSetCapsule (&M, rho, 3, r, l);
    
    pushmass (L, &M);

    return 1;
}

static int polyhedronmass (lua_State *L)
{
    id polyhedron;
    dGeomID geom;
    dMass M;
    dReal rho;

    polyhedron = *(id *)lua_touserdata (L, 1);
    rho = (dReal)luaL_checknumber (L, 2);
    
    geom = dCreateTriMesh (NULL, [polyhedron data], NULL, NULL, NULL);
    dMassSetTrimesh (&M, rho, geom);
    dGeomDestroy (geom);

    printf ("center of mass is at (%f, %f, %f)\n", M.c[0], M.c[1], M.c[2]);
    
    M.c[0] = 0;
    M.c[1] = 0;
    M.c[2] = 0;

    pushmass (L, &M);

    return 1;
}

static int spring (lua_State *L)
{
    dReal k_d, k_s, h;

    k_s = (dReal)luaL_checknumber (L, 1);
    k_d = (dReal)luaL_checknumber (L, 2);

    lua_getglobal (L, "simulator");
    lua_getfield (L, -1, "stepsize");
    
    h = (dReal)lua_tonumber (L, -1);

    lua_newtable (L);
    lua_pushnumber (L, 1.0 / (h * k_s + k_d));
    lua_rawseti (L, -2, 1);
    lua_pushnumber (L, h * k_s / (h * k_s + k_d));
    lua_rawseti (L, -2, 2);

    return 1;
}

static int collide (lua_State *L)
{
    dContact contact;
    dJointID j;
    Body *a, *b;
    dReal mu, depth, bounce;
    dVector3 pos, normal;
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TUSERDATA);
    luaL_checktype (L, 3, LUA_TTABLE);
    luaL_checktype (L, 4, LUA_TTABLE);

    depth = (dReal)luaL_checknumber (L, 5);
    mu = (dReal)luaL_checknumber (L, 6);
    bounce = (dReal)luaL_checknumber (L, 7);

    for(i = 0; i < 3; i += 1) {
	lua_rawgeti(L, 3, i + 1);
	pos[i] = lua_tonumber (L, -1);

	lua_rawgeti(L, 4, i + 1);
	normal[i] = lua_tonumber (L, -1);
	
	lua_pop(L, 2);
    }

    a = *(Body **)lua_touserdata (L, 1);
    b = *(Body **)lua_touserdata (L, 2);

/*     { */
/* 	dReal *u; */

/* 	u = dBodyGetLinearVel ([a body]); */
/* 	printf ("v_a = %f, %f, %f\n", u[0], u[1], u[2]); */
	
/* 	u = dBodyGetLinearVel ([b body]); */
/* 	printf ("v_b = %f, %f, %f\n", u[0], u[1], u[2]); */
/*     } */
    
    contact.surface.mode = dContactBounce | dContactApprox1 | dContactFDir1;
    contact.surface.mu = mu;
    contact.surface.bounce = bounce;
    contact.surface.bounce_vel = 0.01;

    for (i = 0 ; i < 3; i += 1) {
	contact.geom.pos[i] = pos[i];
	contact.geom.normal[i] = normal[i];
    }

    dNormalize3 (contact.geom.normal);

    printf ("%f, %f, %f\n",
	    contact.geom.normal[0],
	    contact.geom.normal[1],
	    contact.geom.normal[2]);
    {
	dVector3 u, v, delta;
	dReal *r, *n, ndotdelta;

	r = pos;
	n = normal;
		
	dSetZero (u, 3);
	dSetZero (v, 3);
		
	if ([a body]) {
	    dBodyGetPointVel ([a body], r[0], r[1], r[2], u);
	}
		    
	if ([b body]) {
	    dBodyGetPointVel ([b body], r[0], r[1], r[2], v);
	}

	delta[0] = u[0] - v[0];
	delta[1] = u[1] - v[1];
	delta[2] = u[2] - v[2];
		
	ndotdelta = dDOT(n, delta);
	contact.fdir1[0] = delta[0] - ndotdelta * n[0]; 
	contact.fdir1[1] = delta[1] - ndotdelta * n[1]; 
	contact.fdir1[2] = delta[2] - ndotdelta * n[2];
    }
    
    contact.geom.depth = depth;
    contact.geom.g1 = [a geom];
    contact.geom.g2 = [b geom];

    j = dJointCreateContact ([[a parent] world],
			     [[a parent] group],
			     &contact);

    dJointAttach (j, [a body], [b body]);

    return 0;
}

static int joined (lua_State *L)
{
    Body *a, *b;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TUSERDATA);

    a = *(Body **)lua_touserdata (L, 1);
    b = *(Body **)lua_touserdata (L, 2);

    if ([a body] && [b body]) {
	lua_pushboolean (L, dAreConnected([a body], [b body]));
    } else {
	lua_pushboolean (L, 0);
    }
    
    return 1;
}

static int pointfrombody(lua_State *L)
{
    id N;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    body = [N body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetRelPointPos (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int vectorfrombody(lua_State *L)
{
    id N;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    body = [N body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyVectorToWorld (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int pointtobody(lua_State *L)
{
    id N;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    body = [N body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetPosRelPoint (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int vectortobody(lua_State *L)
{
    id N;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    body = [N body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyVectorFromWorld (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

static int pointvelocity(lua_State *L)
{
    id N;
    dBodyID body;
    dVector3 w;
    dReal b[3];
    int i;

    luaL_checktype (L, 1, LUA_TUSERDATA);
    luaL_checktype (L, 2, LUA_TTABLE);

    N = *(id *)lua_touserdata (L, 1);
    body = [N body];
    
    for (i = 0 ; i < 3 ; i += 1) {
	lua_rawgeti (L, 2, i + 1);
	b[i] = lua_tonumber(L, -1);
	lua_pop(L, 1);
    }

    dBodyGetPointVel (body, b[0], b[1], b[2], w);
    
    lua_newtable(L);
    
    for(i = 0; i < 3; i += 1) {
	lua_pushnumber(L, w[i]);
	lua_rawseti(L, -2, i + 1);
    }

    return 1;
}

int luaopen_physics (lua_State *L)
{
    const luaL_Reg physics[] = {
	{"addforce", addforce},
	{"addtorque", addtorque},
	{"addrelativeforce", addrelativeforce},
	{"addrelativetorque", addrelativetorque},
	{"spheremass", spheremass},
	{"capsulemass", capsulemass},
	{"polyhedronmass", polyhedronmass},
	{"spring", spring},
	{"joined", joined},
	{"pointfrombody", pointfrombody},
	{"pointtobody", pointtobody},
	{"vectorfrombody", vectorfrombody},
	{"vectortobody", vectortobody},
	{"pointvelocity", pointvelocity},
	{"collide", collide},
	{NULL, NULL}
    };

    luaL_register (L, "physics", physics);

    return 0;
}
