/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdlib.h>
#include <lua.h>
#include <lauxlib.h>
#include <GL/gl.h>
#include <GL/glext.h>
#include "texture.h"
#include "diffuse.h"
#include "light.h"

static const GLchar *header =
"uniform sampler2D diffuseSampler;			 \n"
"uniform vec3 diffuseConstant;				 \n"
"						   	 \n"
"vec3 diffuseSum(vec2 texcoord)			   	 \n"
"{						   	 \n"    
"    return diffuseConstant + 			   	 \n"
"           vec3(texture2D(diffuseSampler, texcoord));   \n"
"}						  	 \n"    
"						   	 \n"
"#line 1						 \n";

@implementation Diffuse

-(Diffuse *)init
{
    self->diffuse.texture = nil;
    self->diffuse.values[0] = 0;
    self->diffuse.values[1] = 0;
    self->diffuse.values[2] = 0;

    self->diffuse.constant = glGetUniformLocationARB (self->program,
						   "diffuseConstant");
    self->diffuse.sampler = glGetUniformLocationARB (self->program,
						  "diffuseSampler");
    
    [super init];

    return self;
}

-(void)attachVertexSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);

    [super attachVertexSource: (const GLchar *)newsource];
}

-(void)attachFragmentSource: (const GLchar *)source
{
    GLchar *newsource;

    newsource = alloca(strlen(header) + strlen(source) + 2);

    strcpy (newsource, header);
    strcat (newsource, source);
    
    [super attachFragmentSource: (const GLchar *)newsource];
}

-(void) get: (lua_State *)L
{
    const char *k;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "diffuse")) {
	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_gettable (L, 1);
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{    
    const char *k;
    int i;

    k = lua_tostring(L, 2);

    if (!strcmp(k, "diffuse")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->diffuse.values[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
	    
	    self->diffuse.texture = nil;
        } else if(lua_isuserdata (L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                self->diffuse.values[i] = 0;
            }
	    
	    self->diffuse.texture = *(id *)lua_touserdata (L, 3);
	}

	lua_getmetatable (L, 1);
	lua_replace (L, 1);
	lua_settable (L, 1);
    } else {
	[super set: L];
    }
}

-(void)traverse: (lua_State *)L
{
    GLint unit;
    
    glGetIntegerv(GL_ACTIVE_TEXTURE, &unit);
    glActiveTexture (unit + 1);

    if(self->diffuse.texture) {
	glBindTexture(GL_TEXTURE_2D, [self->diffuse.texture index]);
    } else {
	glBindTexture(GL_TEXTURE_2D, 0);
    }

    glUniform3fvARB (self->diffuse.constant, 1, self->diffuse.values);
    glUniform1iARB (self->diffuse.sampler, unit + 1 - GL_TEXTURE0);
    
    [super traverse: L];
}

@end
