/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "system.h"
#include "body.h"

@implementation Body

-(Body *) init
{
    [super init];

    self->body = dBodyCreate ([self world]);
    self->geom = NULL;

    dBodyDisable ([self body]);
    
    return self;
}

-(void) free
{
    if (self->geom) {
	dGeomDestroy (self->geom);
    }

    dBodyDestroy (self->body);
}

-(dSpaceID) space
{
    return [[self parent] space];
}

-(dBodyID) body
{
    return self->body;
}

-(dGeomID) geom
{
    return self->geom;
}

-(void) addBody: (id) node
{
    [[self parent] addBody: node];
}

-(void) removeBody: (id) node
{
    [[self parent] removeBody: node];
}

-(void) toggle: (lua_State *)L
{
    if (!linked) {
	dBodyEnable ([self body]);
	[[self parent] addBody: self];
    } else {
	dBodyDisable ([self body]);
	[[self parent] removeBody: self];
    }

    [super toggle: L];
}

-(void) transform: (lua_State *)L
{
    const dReal *r, *R, *drdt, *dRdt;
    int i, j;
    
    r = dBodyGetPosition (self->body);
    R = dBodyGetRotation (self->body);
    drdt = dBodyGetLinearVel (self->body);
    dRdt = dBodyGetAngularVel (self->body);
    
    for(i = 0 ; i < 3 ; i += 1) {
	self->position[i] = r[i];
	self->velocity[i] = drdt[i];
	self->spin[i] = dRdt[i];
    }
	    
    for(i = 0 ; i < 3 ; i += 1) {
	for(j = 0 ; j < 3 ; j += 1) {
	    self->orientation[i * 3 + j] = R[i * 4 + j];
	}
    }

    [super transform: L];
}

-(void) get: (lua_State *)L
{
    const char *k;
    int i, j;
    
    k = lua_tostring(L, -1);

    if (!strcmp(k, "velocity")) {
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, self->velocity[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "spin")) {
        lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, self->spin[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "mass")) {
        lua_newtable(L);

        lua_pushnumber(L, self->mass.mass);
        lua_rawseti(L, -2, 1);

        lua_newtable(L);

        for(i = 0 ; i < 3 ; i += 1) {
            lua_pushnumber(L, self->mass.c[i]);
            lua_rawseti(L, -2, i + 1);
        }

        lua_rawseti(L, -2, 2);

        lua_newtable(L);
	
	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		lua_pushnumber (L, self->mass.I[i * 4 + j]);
		lua_rawseti(L, -2, i * 3 + j + 1);
	    }
	}

        lua_rawseti(L, -2, 3);
    } else if (!strcmp(k, "classification")) {
	unsigned int i, mask;
	
	mask = dGeomGetCategoryBits ([self geom]);
	lua_newtable (L);
	
	for (i = 0 ; i < 32 ; i += 1) {
	    if (mask & ((unsigned int)0x1 << i)) {
		int n;

		n = lua_objlen (L, -1);
		lua_pushnumber (L, i + 1);
		lua_rawseti(L, -2, n + 1);
	    }
	}

	if (lua_objlen(L, -1) == 1) {
	    lua_rawgeti (L, -1, 1);
	}
    } else if (!strcmp(k, "position")) {
	const dReal *r;
	
	[super set: L];

	r = dBodyGetPosition (self->body);

	lua_newtable(L);
        
        for(i = 0; i < 3; i += 1) {
            lua_pushnumber(L, r[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "orientation")) {
	const dReal *R;
	int i, j;
	    
	[super set: L];
	    
	R = dBodyGetRotation (self->body);

        lua_newtable(L);
	
	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		lua_pushnumber (L, R[i * 4 + j]);
		lua_rawseti(L, -2, i * 3 + j + 1);
	    }
	}
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{
    const char *k;
    int i, j;

    k = lua_tostring(L, -2);

    if (!strcmp(k, "velocity")) {
	if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->velocity[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }

	dBodySetLinearVel (self->body,
			   self->velocity[0],
			   self->velocity[1],
			   self->velocity[2]);
    } else if (!strcmp(k, "spin")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 3 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->spin[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }


	dBodySetAngularVel (self->body,
			    self->spin[0],
			    self->spin[1],
			    self->spin[2]);
    } else if (!strcmp(k, "mass")) {
	dMassSetZero(&self->mass);
	
        if(lua_istable(L, 3)) {
	    lua_rawgeti(L, 3, 1);
	    
	    self->mass.mass = lua_tonumber(L, -1);

	    lua_pop(L, 1);
	
	    lua_rawgeti(L, 3, 2);

	    for(i = 0 ; i < 3 ; i += 1) {
		lua_rawgeti(L, -1, i + 1);
		
		self->mass.c[i] = lua_tonumber(L, -1);
		
		lua_pop(L, 1);
	    }
	
	    lua_pop(L, 1);

	    lua_rawgeti(L, 3, 3);
	
	    for(i = 0 ; i < 3 ; i += 1) {
		for(j = 0 ; j < 3 ; j += 1) {
		    lua_rawgeti(L, -1, i * 3 + j + 1);

		    self->mass.I[i * 4 + j] = lua_tonumber(L, -1);

		    lua_pop(L, 1);
		}
	    }

	    lua_pop(L, 1);
	}

	dBodySetMass(self->body, &self->mass);
    } else if (!strcmp(k, "classification")) {
	unsigned int i, n, mask;

	if (lua_istable (L, 3)) {
	    mask = 0;
	    n = lua_objlen (L, 3);
	    
	    for (i = 0 ; i < n ; i += 1) {
		lua_rawgeti (L, 3, i + 1);
		mask |= (unsigned int)0x1 << ((int)lua_tonumber (L, -1) - 1);
		lua_pop (L, 1);
	    }
	} else if (lua_isnumber (L, 3)) {
	    mask = 1 << ((int)lua_tonumber (L, 3) - 1);
	} else {
	    mask = 0;
	}
	    
	dGeomSetCategoryBits ([self geom], mask);
	dGeomSetCollideBits ([self geom], mask);
    } else if (!strcmp(k, "position")) {
	[super set: L];

	dBodySetPosition (self->body,
			  self->position[0],
			  self->position[1],
			  self->position[2]);

	if (self->geom &&
	    dGeomGetClass(self->geom) != dPlaneClass) {
	    dGeomSetPosition (self->geom,
			      self->position[0],
			      self->position[1],
			      self->position[2]);
	}
    } else if (!strcmp(k, "orientation")) {
	dMatrix3 R;
	int i, j;
	    
	[super set: L];

	for(i = 0 ; i < 3 ; i += 1) {
	    for(j = 0 ; j < 3 ; j += 1) {
		R[i * 4 + j] = self->orientation[i * 3 + j];
	    }
	}
	    
	dBodySetRotation (self->body, R);
	    
	if (self->geom &&
	    dGeomGetClass(self->geom) != dPlaneClass) {
	    dGeomSetRotation (self->geom, R);
	}
    } else {
	[super set: L];
    }
}

@end
