/* Copyright (C) 2008 Papavasileiou Dimitris                             
 *                                                                      
 * This program is free software: you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation, either version 3 of the License, or    
 * (at your option) any later version.                                  
 *                                                                      
 * This program is distributed in the hope that it will be useful,      
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
 * GNU General Public License for more details.                         
 *                                                                      
 * You should have received a copy of the GNU General Public License    
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ode/ode.h>
#include <lua.h>
#include <lualib.h>
#include <lauxlib.h>

#include "billiard.h"

@implementation Billiard

-(Billiard *) init
{
    [super init];

    self->friction[0] = 0;
    self->friction[1] = 0;

    return self;
}

-(void) toggle: (lua_State *)L
{
    [super toggle: L];

    if (linked) {
	self->motors[0] = dJointCreateLMotor ([self world], 0);
	dJointSetLMotorNumAxes (self->motors[0], 1);
	dJointSetLMotorParam (self->motors[0], dParamVel, 0);
	
	self->motors[1] = dJointCreateAMotor ([self world], 0);
	dJointSetAMotorNumAxes (self->motors[1], 1);
	dJointSetAMotorParam (self->motors[1], dParamVel, 0);
	
	self->motors[2] = dJointCreateAMotor ([self world], 0);
	dJointSetAMotorNumAxes (self->motors[2], 1);
	dJointSetAMotorAxis (self->motors[2], 0, 0, 0, 0, 1);
	dJointSetAMotorParam (self->motors[2], dParamVel, 0);
	
	dBodySetFiniteRotationMode (self->body, 1);
    }
}

-(void) get: (lua_State *)L
{
    const char *k;
    int i;
    
    k = lua_tostring(L, -1);

    if (!strcmp(k, "friction")) {
        lua_newtable(L);
        
        for(i = 0; i < 2; i += 1) {
            lua_pushnumber(L, self->friction[i]);
            lua_rawseti(L, -2, i + 1);
        }
    } else if (!strcmp(k, "resting")) {
	lua_pushboolean(L, self->state == 0 ? 1 : 0);
    } else if (!strcmp(k, "rolling")) {
	lua_pushboolean(L, self->state == 1 || self->state == 4 ? 1 : 0);
    } else if (!strcmp(k, "sliding")) {
	lua_pushboolean(L, self->state == 2 ? 1 : 0);
    } else if (!strcmp(k, "spinning")) {
	lua_pushboolean(L, self->state == 3 || self->state == 4 ? 1 : 0);
    } else if (!strcmp(k, "flying")) {
	lua_pushboolean(L, self->state == 5 ? 1 : 0);
    } else {
	[super get: L];
    }
}

-(void) set: (lua_State *)L
{
    const char *k;
    int i;
    
    k = lua_tostring(L, -2);

    if (!strcmp(k, "friction")) {
        if(lua_istable(L, 3)) {
            for(i = 0 ; i < 2 ; i += 1) {
                lua_rawgeti(L, 3, i + 1);
                self->friction[i] = lua_tonumber(L, -1);
                
                lua_pop(L, 1);
            }
        }
    } else {
	[super set: L];
    }
}

-(void) transform: (lua_State *)L
{
    dReal v_c[3], l, m, absg;
    dVector3 g;
    const dReal *w, *v, threshold = 1e-4;
    
    if (dBodyGetNumJoints(self->body) < 1) {
	if (self->state != 5) {
	    dJointAttach (self->motors[0], NULL, NULL);
	    dJointAttach (self->motors[1], NULL, NULL);
	    dJointAttach (self->motors[2], NULL, NULL);
	}
	    
	self->state = 5;
    } else {
	dWorldGetGravity ([self world], g);

	absg = fabs(g[2]);
	v = dBodyGetLinearVel (self->body);
	w = dBodyGetAngularVel (self->body);
    
	/* The velocity of the contact point:
	   -     -   -      -
	   v_c = v - w x (-rk). */
	
	v_c[0] = v[0] - w[1] * self->radius;
	v_c[1] = v[1] + w[0] * self->radius;
	v_c[2] = v[2];

	l = dDot (v_c, v_c, 3);
	m = dDot (v, v, 3);

	if (m > threshold && l > threshold) {
	    if (self->state != 2) {
		dJointAttach (self->motors[0], NULL, NULL);
 		dJointAttach (self->motors[1], NULL, NULL);
		dJointAttach (self->motors[2], NULL, NULL);
	    }

	    self->state = 2;
	} else if (m > threshold && l < threshold) {
	    dReal F, tau, x, y, z;
	    dReal T;
		
	    F = self->friction[0] * self->mass.mass * absg;
	    tau = self->friction[0] * absg * self->mass.I[0] / self->radius;
	    T = self->friction[1] * self->mass.mass * absg;
	    
	    x = v[0] / m;
	    y = v[1] / m;
	    z = v[2] / m;

	    if (w[2] * w[2] < threshold) {
		if (self->state != 1) {
		    /* Add rolling friction only. */

		    dJointSetLMotorAxis (self->motors[0], 0, 0, x, y, 0);
		    dJointSetLMotorParam (self->motors[0], dParamFMax, F);
		    
		    dJointSetAMotorAxis (self->motors[1], 0, 0, -y, x, 0);
		    dJointSetAMotorParam (self->motors[1], dParamFMax, tau);

		    dJointAttach (self->motors[0], self->body, NULL);
		    dJointAttach (self->motors[1], self->body, NULL);
		    dJointAttach (self->motors[2], NULL, NULL);
		}
		    
		self->state = 1;
	    } else {
		if (self->state != 4) {
		    /* Add rolling and spinning friction. */

		    dJointSetLMotorAxis (self->motors[0], 0, 0, x, y, 0);
		    dJointSetLMotorParam (self->motors[0], dParamFMax, F);
		    
		    dJointSetAMotorAxis (self->motors[1], 0, 0, -y, x, 0);
		    dJointSetAMotorParam (self->motors[1], dParamFMax, tau);

		    dJointSetAMotorParam (self->motors[2], dParamFMax, T);

		    dJointAttach (self->motors[0], self->body, NULL);
		    dJointAttach (self->motors[1], self->body, NULL);
		    dJointAttach (self->motors[2], self->body, NULL);
		}
	    
		self->state = 4;
	    }
	} else if (m < threshold && w[2] * w[2] > threshold) {
	    if (self->state != 3) {
		dReal tau = self->friction[1] * self->mass.mass * absg;
		
		/* Add spinning friction only. */
		
		dJointSetAMotorParam (self->motors[2], dParamFMax, tau);

		dJointAttach (self->motors[0], NULL, NULL);
		dJointAttach (self->motors[1], NULL, NULL);
		dJointAttach (self->motors[2], self->body, NULL);
	    }

	    self->state = 3;
	} else {
	    if (self->state != 0) {
		dJointAttach (self->motors[0], NULL, NULL);
		dJointAttach (self->motors[1], NULL, NULL);
		dJointAttach (self->motors[2], NULL, NULL);

		dBodySetLinearVel (self->body, 0, 0, 0);
		dBodySetAngularVel (self->body, 0, 0, 0);
	    }
	    
	    self->state = 0;
	}
    }
    
    [super transform: L];
}
@end
