-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.


local actions
local allowedcounts = billiards.allowedcounts

local function callhooks (hooks)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
	 hook ()
      end
   elseif type (hooks) == "function" then
      hook ()
   end
end

local function resetball (ball, place)
   place[3] = billiards.ballradius

   ball.position = place
   ball.velocity = {0, 0, 0}
   ball.spin = {0, 0, 0}
   ball.counts = 0
end

local function scoredcounts()
   local first = nil
   local n = 0

   -- Check for fouls first.

   for _, ball in pairs(balls) do
      if ball.position[3] < 0 then
	 -- A ball bounced off the floor.

	 event[1] = faces.small "Balls outside." {
	    opacity = 3
	 }
	    
	 return 0
      elseif (billiards.balkline or billiards.straight) and
             allowedcounts < 0 then
	 -- Both object balls remained in the balk space.
 
	 event[1] = faces.small "Remaining in there." {
	    opacity = 3
	 }
	 
	 return 0
      end
   end

   for _, object in ipairs(cueball.bounces) do
      if tostring(object) == "Capsule" then
	 -- We've hit a cushion so count it.

	 n = n + 1
      elseif tostring(object) == "Billiard" then
	 -- We've hit a ball.  If this is the
	 -- first one, make a note of it, otherwise
	 -- score a point if the required cushion
	 -- count has been reached.

	 if first == nil then
	    first = object
	 elseif object ~= first then
	    if n >= billiards.cushions or not billiards.cushion then
	       event[1] = faces.small (string.format("%s scores a count.",
						     cueball.name)) {
		  opacity = 3
	       }

	       return 1
	    else
	       if n == 0 then
		  event[1] = faces.small "No cushion." {
		     opacity = 3
		  }
	       elseif n == 1 then
		  event[1] = faces.small "One cushion." {
		     opacity = 3
		  }
	       elseif n == 2 then
		  event[1] = faces.small "Two cushions." {
		     opacity = 3
		  }
	       else
		  event[1] = faces.small (string.format("%d cushions.", n)) {
		     opacity = 3
		  }
	       end

	       return 0
	    end
	 end
      end
   end

   if first == nil then
      event[1] = faces.small "No object ball." {
	 opacity = 3
      }
   else
      event[1] = faces.small "One object ball." {
	 opacity = 3
      }
   end

   return 0
end

local function newgame()
   -- Rack'em up.

   resetball (cueball, {-0.25 * billiards.tablewidth, -0.1524})
   resetball (objectball, {-0.25 * billiards.tablewidth, 0})
   resetball (redball, {0.25 * billiards.tablewidth, 0})
	 
   callhooks (billiards.newgame)
end

local function newturn()
   -- Swap the cueballs.
   
   if billiards.players > 1 then
      cueball = cueball == balls[1] and balls[2] or balls[1]
      objectball = otherball == balls[1] and balls[2] or balls[1]
   end
   
   callhooks (billiards.newturn)
end

local function zoom (dx, dy)
   -- Zoom in on the cue ball.

   eye.radius = eye.radius - billiards.linear * dy
end

local function rotate (dx, dy)
   eye.heading = eye.heading - billiards.angular * dx
   eye.elevation = eye.elevation - billiards.angular * dy

   cue.heading  = eye.heading
end

local function adjustelevation(dx, dy)
   -- Raise or lower the cue.

   cue.elevation = cue.elevation - 0.5 * dy
end

local function adjustenglish(dx, dy)
   -- Translate the cue to apply english.

   cue.english = {
      cue.english[1] + 0.0002 * dx,
      cue.english[2] - 0.0002 * dy
   }
end

local function strike (dx, dy)
   hand.motor = {billiards.stroke * dy, billiards.cueforce}
end

billiards.newshot.game = function ()
			    cue.heading = eye.heading
			    cue.elevation = 0
			 end

billiards.looking.game = function ()
			    billiards.angular = billiards.oldangular or
				                billiards.angular

			    eye.radius = 1.5
			      
			    actions = {
			       [billiards.rotate] = rotate,
			       [billiards.zoom] = zoom
			    }
			 end

billiards.aiming.game = function ()
			   billiards.oldangular = billiards.angular
			   billiards.angular = 0.05 * billiards.angular

			   eye.radius = 1
			   eye.elevation = 85
			   
			   actions = {
			      [billiards.rotate] = rotate, 
			      [billiards.strike] = strike, 
			      [billiards.offset] = adjustenglish,
			      [billiards.elevate] = adjustelevation
			   }
			end

billiards.waiting.game = function ()
			    billiards.angular = billiards.oldangular or
			                        billiards.angular

			    eye.radius = 2.5
			    eye.elevation = 40

			    actions = {
			       [billiards.rotate] = rotate,
			       [billiards.zoom] = zoom
			    }
			 end

billiards.finished.game = function ()
   local result, spacecrossed

   -- Update the number of allowed counts remaining.

   for _, ball in pairs {objectball, redball} do
      for _, bounce in pairs (ball.bounces) do
	 if tostring(bounce) == "Box" then
	    spacecrossed = true
	 end
      end
   end

   if spacecrossed == true or
      objectball.space == 9 or 
      objectball.space ~= redball.space then
      allowedcounts = billiards.allowedcounts
   else
      allowedcounts = allowedcounts - 1
   end
   
   -- Calculate the counts the shot earns.
   
   result = scoredcounts()
   
   -- If the shot was successful update the score,
   -- otherwise change the turn.

   if result > 0 then
      cueball.counts = cueball.counts + result

      if cueball.counts == billiards.distance then
	 cueball.games = cueball.games + 1

	 newgame ()
      end
   else
      allowedcounts = billiards.allowedcounts
      
      newturn()
   end
end

graphics.buttonpress.game = function (button)
   local x_0 = graphics.pointer[1]
   local y_0 = graphics.pointer[2]

   -- For the first three buttons
   -- assign the corresponding action
   -- to the motion event.

   if actions[button] then
      graphics.motion.game = function(x, y)
				     if actions[button] then
					actions[button] (x - x_0, y - y_0)
				     end
					
				     x_0, y_0 = x, y
				  end
   end
end

graphics.buttonrelease.game = function (button)
   graphics.motion.game = nil
end

graphics.keyrelease.game = function (key)
   if key == billiards.overhead then
      eye.elevation, eye.heading, eye.radius =     
	 eye.oldelevation, eye.oldheading, eye.oldradius
   elseif key == billiards.elevate or
          key == billiards.rotate or 
          key == billiards.zoom or 
          key == billiards.strike or 
          key == billiards.offset then
	  graphics.buttonrelease.game(key)
   end
end

graphics.keypress.game = function (key)
   if key == billiards.quit then
      graph.iterate = false
   elseif key == billiards.overhead then
      -- Calculate the closest "horizontal" orientation.

      local theta = math.floor((eye.heading - 90) / 180 + 0.5) * 180 + 90

      -- Calculate the radius necessary to fit the whole
      -- table horizontally and vertically.

      local rho = (0.25 + 0.5 * billiards.tablewidth +
		   math.abs(torso.position[1])) /
		  graphics.perspective[2] * graphics.perspective[5]

      local Rho = (0.25 + 0.5 * billiards.tableheight +
		   math.abs(torso.position[2])) /
		  graphics.perspective[4] * graphics.perspective[5]

      -- Finally save the current orientation and reset it.

      eye.oldelevation, eye.oldheading, eye.oldradius = 
	 eye.elevation, eye.heading, eye.radius

      eye.elevation, eye.heading, eye.radius = 0, theta, math.max(rho, Rho)
   elseif key == billiards.pause then
      if simulator.timescale > 0 then
	 simulator.oldtimescale = simulator.timescale
	 simulator.timescale = 0
      else
	 simulator.timescale = simulator.oldtimescale
      end
   elseif key == billiards.slower then
      simulator.timescale = 2 * simulator.timescale
   elseif key == billiards.faster then
      simulator.timescale = 0.5 * simulator.timescale
   elseif key == billiards.elevate or
          key == billiards.rotate or 
          key == billiards.zoom or 
          key == billiards.strike or 
          key == billiards.offset then
	  graphics.buttonpress.game(key)
   else
      print (string.format("(%s)", key))
   end
end

newgame()
