-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "simulator"
require "transforms"
require "physics"
require "frames"
require "shapes"
require "shading"
require "widgets"
require "sound"

require "solids"
require "textures"
require "waveforms"

local function readfile (name)
   print ("  " .. name)
   return dofile(name)
end

local metatable = getmetatable (graphics)
local oldindex = metatable.__index
local oldnewindex = metatable.__newindex
local field = 0

metatable.__index = function (table, key)
		       if key == "field" then
			  local x = math.abs(graphics.perspective[4])
			  local y = math.abs(graphics.perspective[5])

			  return 2 * math.deg(math.atan2(x, y))
		       elseif key == "range" then
			  return {graphics.perspective[5],
				  graphics.perspective[6]}
		       else
			  return oldindex(table, key)
		       end
		    end

metatable.__newindex = function (table, key, value)
			  if key == "range" then
			     local tantheta = math.tan(math.rad(graphics.field))
			     local z = value
			     local a = graphics.window[1] / 
				       graphics.window[2]
		      
			     graphics.perspective = {-a * z[1] * tantheta,
						     a * z[1] * tantheta,
						     -z[1] * tantheta,
						     z[1] * tantheta,
						     z[1], z[2]}
			  elseif key == "field" then
			     local tantheta = math.tan(math.rad(0.5 * value))
			     local z = graphics.range
			     local a = graphics.window[1] / 
				       graphics.window[2]
			     
			     field = value
			     graphics.perspective = {-a * z[1] * tantheta,
						     a * z[1] * tantheta,
						     -z[1] * tantheta,
						     z[1] * tantheta,
						     z[1], z[2]}
			  else
			     oldnewindex(table, key, value)
			  end
		       end

-- These values are fixed.

graphics.title = "Billiards"
graphics.hide = false
graphics.perspective = {-1, 1, -1, 1, 0, 1}
graphics.range = {0.01, 20}

graphics.buttonpress = {}
graphics.buttonrelease = {}
graphics.keypress = {}
graphics.keyrelease = {}
graphics.motion = {}

simulator.surfacelayer = 1e-4
simulator.tolerance = {1e-7, 0.2}

-- Load the default configuration.

billiards = {}
billiards.waiting = {}
billiards.looking = {}
billiards.aiming = {}
billiards.finished = {}
billiards.ballcollision = {}
billiards.cushioncollision = {}
billiards.cuecollision = {}
billiards.newshot = {}
billiards.newgame = {}
billiards.newturn = {}
billiards.reorienting = {}
billiards.adjusting = {}

dofile "scripts/defaults.lua"

-- Load per-user configuration.

print "Reading local user configuration."

local rcscript = loadfile (os.getenv ("HOME") .. "/.billiards")

if rcscript then
   print ("  " .. os.getenv ("HOME") .. "/.billiards")
   rcscript ()
else
   local defaultsfile, rcfile, error

   print "Creating user configuration with default values."

   -- If none exists create one by copying the faults.

   defaultsfile = io.open ("scripts/defaults.lua", "r")
   rcfile, error = io.open (os.getenv ("HOME") .. "/.billiards", "w")

   if defaultsfile and rcfile then
      local defaults

      defaults = defaultsfile:read "*a"
      rcfile:write (defaults)
   else
      print (error)
   end

   io.close (defaultsfile)
   io.close (rcfile)
end


-- Override the configured values based on
-- the options specified on the command line.

for option, _ in pairs(options) do
   if option == "practice" then
      billiards.players = 1
   elseif option == "twoplayer" then
      billiards.players = 2
   elseif option == "training" then
      billiards.drawtrajectories = true
      billiards.drawvectors = true
      billiards.drawcenterline = true
      billiards.drawsidelines = true
   elseif option == "straight" then
      billiards.straight = true
      billiards.balkline = false
      billiards.cushion = false
   elseif option == "onecushion" then
      billiards.straight = false
      billiards.balkline = false
      billiards.cushion = true
      billiards.cushions = 1
   elseif option == "threecushion" then
      billiards.straight = false
      billiards.balkline = false
      billiards.cushion = true
      billiards.cushions = 3
   elseif option == "fixed" then
      graphics.fixed = true
   else
      local distance, spacing, strokes, rcfile, first, second

      _, _, distance = string.find (option, "^(%d*)countgame$")
      _, _, spacing, strokes = string.find (option, "^(%d*)%.(%d*)balkline$")
      _, _, first, second = string.find (option, "^(%w*),?(%w*)$");

      rcfile = loadfile (option)

      if distance then
	 distance = tonumber (distance)

	 if distance < 1 then distance = 1 end

	 billiards.distance = distance
      elseif spacing or strokes then
	 spacing = tonumber (spacing)
	 strokes = tonumber (strokes)

	 if spacing < 0 then spacing = 0 end
	 if spacing > 71 then spacing = 71 end
	 if strokes < 0 then strokes = 0 end

	 billiards.straight = false
	 billiards.balkline = true
	 billiards.cushion = false
	 
	 billiards.balkspacing = 0.01 * spacing
	 billiards.allowedcounts = strokes
      elseif rcfile then
	 print ("  " .. option)
	 rcfile()
      elseif first or second then
	 if first and #first > 0 then
	    billiards.firstname = first
	 end
	 
	 if second and #second > 0 then
	    billiards.secondname = second
	 end
      else
	 print (string.format("Ignoring unintelligible option `%s'", option))
      end
   end
end

print "\nLoading the http server pages."

-- Load the http server pages.

readfile "scripts/pages/stylesheet.lua"
readfile "scripts/pages/logo.lua"
readfile "scripts/pages/default.lua"
readfile "scripts/pages/newgame.lua"
readfile "scripts/pages/system.lua"
readfile "scripts/pages/screenshot.lua"
readfile "scripts/pages/writeoptions.lua"

dofile "scripts/reset.lua"
