-- Copyright (C) 2008 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.


local chi, psi, alpha, beta = 0, 0, 0, 0
local theta_0, theta, phi, rho = 0, 0, 0, 0

local function callhooks (hooks, ...)
   if type (hooks) == "table" then
      for _, hook in pairs(hooks) do
	 if ... and type(...) == "table" then
	    hook (unpack(...))
	 else
	    hook (...)
	 end
      end
   elseif type (hooks) == "function" then
      if ... and type(...) == "table" then
	 hooks(unpack(...))
      else
	 hook (...)
      end
   end
end

local brake = frames.custom {
   transform = function()
		 hand.motor = {0, billiards.cueforce}
	      end
}

local awaitshot = frames.custom {
   prepare = function ()
		local stable = true

		-- Make the cue disapper.

		if graph.observer.gear.cue ~= nil and
		   not physics.joined (cue, cueball) then
 		   graph.observer.gear.cue = nil
		   graph.brake = nil
		end
		
		-- Check whether the shot is finished. A ball is
		-- stable if it is either resting or off the table
		
		for _, ball in pairs (balls) do
		   if ball.position[3] > 0 and not ball.resting then
		      stable = false
		   end
		end
		
		if stable then
		   -- Finish.

		   graph.awaitshot = nil

		   waiting = false
		   finished = true

		   callhooks (billiards.finished)
		end
	     end
}

local function replaceball(ball, spot)
   if not ball then
      return nil
   end
   
   -- Check whether there are any other
   -- balls in the neighborhood.

   for _, other in pairs (balls) do
      local d

      d = math.sqrt ((spot[1] - other.position[1]) ^ 2 + 
		     (spot[2] - other.position[2]) ^ 2)

      if d < 2 * billiards.ballradius then
	 return false
      end
   end

   -- If the area is clear place then ball.

   ball.position = {spot[1], spot[2], billiards.ballradius}
   ball.velocity = {0, 0, 0}
   ball.spin = {0, 0, 0}

   return true
end

local function replaceballatdiamond (ball, spot)
   local x, y

   x = math.min (math.max (spot[1], 0.1), 7.9)
   y = math.min (math.max (spot[2], 0.1), 3.9)

   replaceball (ball, {(x / 8 - 0.5) * billiards.tablewidth,
		       (y / 4 - 0.5) * billiards.tableheight})
end

local function adjusteye(radius, heading, elevation)
   local linear, angular

   rho = tonumber(radius) or rho
   theta, phi = tonumber(heading) or theta, tonumber(elevation) or phi

   if rho > 10 then rho = 10 end
   if rho < 0 then rho = 0 end

   -- The joints become unstable when setting
   -- their stops at +/- 180 degrees.

   if theta < -179 then theta = -179 end
   if theta > 179 then theta = 179 end

   if phi > 90 then phi = 90 end
   if phi < 0 then phi = 0 end

   linear = physics.spring(500, 150)
   angular = physics.spring(500, 150)

   -- Adjust the camera rig.

   neck.stops = {{-rho, -rho}, linear, 0.1}
   waist.stops = {
      {{math.rad(phi), math.rad(phi)}, angular, 0.1},
      {{math.rad(theta), math.rad(theta)}, angular, 0.1}
   }

   callhooks(billiards.reorienting)
end

local function adjustcue(heading, elevation, horizontal, vertical)
   local c, d, r, l
   local n, t, t_0, psi_0
   local h, w, h, h_0, rcosc

   chi, psi = tonumber(heading) or chi, tonumber(elevation) or psi
   alpha, beta = tonumber(horizontal) or alpha, tonumber(vertical) or beta

   if psi < 0 then psi = 0 end
   if psi > 90 then psi = 90 end

   if alpha > billiards.ballradius then alpha = billiards.ballradius end
   if alpha < -billiards.ballradius then alpha = -billiards.ballradius end

   if beta > billiards.ballradius then beta = billiards.ballradius end
   if beta < -billiards.ballradius then beta = -billiards.ballradius end

   -- Setup the cue rig.

   cue.orientation = transforms.relue (90 - psi, 0, chi + 90)
   cue.velocity = {0, 0, 0}
   cue.spin = {0, 0, 0}

   c = transforms.fromnode (cue, {alpha, beta,
				  0.3 * billiards.cuelength +
				  billiards.ballradius + 0.01})

   cue.position = {cueball.position[1] + c[1],
		   cueball.position[2] + c[2],
		   cueball.position[3] + c[3]}

   hand.axis = transforms.fromnode (cue, {0, 0, 1})
   hand.bodies = {cue, nil}
 
   -- Correct the elevation to clear the rails if necessary.

   w = billiards.tablewidth
   h = billiards.tableheight
   h_0 = billiards.cushionheight

   n = hand.axis
   d = transforms.fromnode (cue, {alpha, beta, 0})
   c = transforms.translate(cueball.position, d)

   t = {
      (-c[1] - 0.5 * w) / n[1],
      (-c[1] + 0.5 * w) / n[1], 
      (-c[2] - 0.5 * h) / n[2],
      (-c[2] + 0.5 * h) / n[2]
   }

   for i = 1, 4 do
      if t[i] < 0 then
	 t[i] = math.huge
      end
   end

   t_0 = math.min (t[1], t[2], t[3], t[4], billiards.cuelength)
   psi_0 = math.deg(math.atan2(h_0 + 0.01 - c[3], t_0))

   -- Take neighbouring balls into account as well.

   r = billiards.ballradius
   l = billiards.cuelength
   
   for _, ball in ipairs(balls) do
      c = ball.position
      d = physics.pointfrombody (cue, {0, 0, -0.3 * 1.44})

      t_0 = n[1] * (c[1] - d[1]) + n[2] * (c[2] - d[2])
      s_0 = -n[1] * (c[2] - d[2]) + n[2] * (c[1] - d[1])

      if t_0 >= 0 and t_0 < l then
	 h = math.sqrt (r^2 - s_0^2 + 4e-4) + r - d[3]
	 psi_0 = math.max (psi_0, math.deg(math.atan2(h, t_0)))
      end
   end

   if psi_0 > psi then
      adjustcue(chi, psi_0, alpha, beta)
   end

   callhooks(billiards.adjusting)
end

local function look()
   -- Enter look mode

   graph.awaitshot = nil
   graph.observer.gear.cue = nil

   looking = true
   aiming = false
   waiting = false
   finished = false

   callhooks (billiards.looking)
end

local function aim()
   -- Enter aim mode

   graph.awaitshot = nil
   graph.brake = brake

   graph.observer.gear.cue = cue

--    adjustcue(chi, psi, alpha, beta)

   aiming = true
   looking = false

   callhooks (billiards.aiming)
end

local function wait()
   -- Enter wait mode

   graph.awaitshot = awaitshot

   waiting = true
   aiming = false

   callhooks (billiards.waiting)
end

local function newshot ()
   chi, psi, alpha, beta = 0, 90, 0, 0 * -0.01
   theta, phi, rho = 0, 60, 1.5

   -- Replace any balls that have fallen
   -- off the table to their opening spots

   for _, ball in pairs (balls) do
      ball.bounces = {}

      if ball.position[3] < 0 then
	 for _, spot in pairs {{0.72, 0},
			       {-0.72, 0},
			       {0, 0}} do
	    if replaceball (ball, spot) then
	       break
	    end
	 end	       
      end
   end

   -- Orient the viewer towards the midpoint between
   -- the other two balls.

   if #balls >= 3 then
      local dx = cueball.position[1] - redball.position[1]
      local dy = cueball.position[2] - redball.position[2]

      local abstheta_0 = math.deg(math.acos(dx / math.sqrt(dx * dx + dy * dy)))
      
      if dy < 0 then
	 theta_0 = -abstheta_0
      else
	 theta_0 = abstheta_0
      end
   else
      theta_0 = 0
   end

   -- Set up the camera rig accordingly.

   torso.position = cueball.position
   eye.position = {cueball.position[1],
		   cueball.position[2],
		   2 * billiards.ballradius + cueball.position[3]}

   eye.orientation = transforms.euler (0, 0, theta_0)

   waist.anchor = cueball.position
   waist.axes = {
      transforms.fromnode (eye, {0, 1, 0}),
      transforms.fromnode (eye, {0, 0, 1}),
   };

   neck.axis = {0, 0, 1}

   callhooks (billiards.newshot)
   look()
end

function simulator.collision (a, b)
   if a == cue or b == cue then
      if a == cueball or b == cueball then
	 -- Release the grip on the cue to
	 -- allow it to rebound of the ball

	 hand.bodies = {nil, nil}

	 -- Then wait for the shot to finish.
	 
	 callhooks(billiards.cuecollision)
	 wait()

-- 	 print(math.deg(math.atan (2.5 * l *
-- 				   math.sqrt(1 - l ^ 2) /
-- 				(1 + 75 +
-- 				 2.5 * l ^ 2))))
	 
	 return billiards.strikingfriction,
	 billiards.strikingrestitution,
	 1
      end

      return 0, 0, 0
   end

   -- A ball is entering a balk space.  Make a note
   -- of it but also make sure we don't get bothered
   -- again until it has left.

   if a ~= bed and tostring(a) == "Box" then
      local classification = {1, 2, 3, 4, 5, 6, 7, 8, 9,
			      10, 11, 12, 13, 14, 15, 16}

      table.remove (classification, a.classification)

      b.classification = classification
      b.space = a.classification

      return 0, 0, 0
   elseif b ~= bed and tostring(b) == "Box" then
      local classification = {1, 2, 3, 4, 5, 6, 7, 8, 9,
			      10, 11, 12, 13, 14, 15, 16}

      table.remove (classification, b.classification)

      a.classification = classification
      a.space = b.classification

      return 0, 0, 0
   end

   -- Keep a record of the cushions and
   -- other balls our cueball collides
   -- with.

   for i, ball in pairs(balls) do
      if a == ball and b ~= bed then
	 table.insert(ball.bounces, b)
      elseif b == ball and a ~= bed then
	 table.insert(ball.bounces, a)
      end
   end

   if tostring(a) == "Billiard" and tostring(b) == "Billiard" then
      local mu_0, mu_1, v_a, v_b, p, V, mu

      -- The coefficient of friction for
      -- ball-ball collisions is highly
      -- dependent on velocity.
      
      mu_0 = billiards.slowfriction
      mu_1 = billiards.fastfriction
      p = {0.5 * (a.position[1] + b.position[1]),
	   0.5 * (a.position[2] + b.position[2]),
	   0.5 * (a.position[3] + b.position[3])}
      
      v_a = physics.pointvelocity(a, p)
      v_b = physics.pointvelocity(b, p)

      V = math.sqrt((v_a[1] - v_b[1]) ^ 2 +
		 (v_a[2] - v_b[2]) ^ 2 +
	      (v_a[3] - v_b[3]) ^ 2)
      mu = mu_1 + (mu_0 - mu_1) * math.exp(-0.77 * V)

      callhooks(billiards.ballcollision)

      return mu,
      billiards.collidingrestitution,
      1
   elseif tostring(a) == "Capsule" or tostring(b) == "Capsule" then
      callhooks (billiards.cushioncollision)

      -- Nothing fancy for ball-cushion 
      -- collisions for now.
	  
      -- print (math.deg(math.atan(math.abs(cueball.position[2]) /
      -- 		       (0.5 * billiards.tablewidth))))
      -- os.exit()

      return billiards.bouncingfriction,
      billiards.bouncingrestitution,
      1
   else
      -- If we've come this far the ball
      -- is just sliding (or jumping) on
      -- the cloth.

      return billiards.slidingfriction,
      billiards.jumpingrestitution,
      1
   end
end

graphics.keypress.billiards = function (key)
   if key == billiards.ready then
      if looking then
	 aim()
      elseif aiming then
	 look()
      elseif finished then
	 newshot()
      end
   elseif key == "u" then
      simulator.timescale = 1
   elseif key == "return" then
      aim()
      cue.english = {0, -0.4 * 0.0305}
      hand.motor = {-25, 10*billiards.cueforce}
--       simulator.timescale = 0.125
   end
end

-- billiards.cuecollision.foo = function()
-- 				simulator.timescale = 0.125
-- 			     end

-- Extend the eye and cue nodes.

local cuemeta = getmetatable(cue)
local cueindex = cuemeta.__index
local cuenewindex = cuemeta.__newindex

local eyemeta = getmetatable(eye)
local eyeindex = eyemeta.__index
local eyenewindex = eyemeta.__newindex

eyemeta.__index = function(table, key)
		     if key == "heading" then
			return theta_0 + theta
		     elseif key == "elevation" then
			return phi
		     elseif key == "radius" then
			return rho
		     else
			return eyeindex(table, key)
		     end
		  end

eyemeta.__newindex = function(table, key, value)
			if key == "heading" then
			   value = tonumber(value) and value - theta_0
			   adjusteye(nil, value, nil)
			elseif key == "elevation" then
			   adjusteye(nil, nil, value)
			elseif key == "radius" then
			   adjusteye(value, nil, nil)
			else
			   eyenewindex(table, key, value)
			end
		     end

cuemeta.__index = function(table, key)
		     if key == "english" then
			return {alpha, beta}
		     elseif key == "heading" then
			return chi
		     elseif key == "elevation" then
			return psi
		     else
			return cueindex(table, key)
		     end
		  end

cuemeta.__newindex = function(table, key, value)
			if key == "english" then
			   adjustcue(chi, psi, unpack(value))
			elseif key == "heading" then
			   adjustcue(value, psi, alpha, beta)
			elseif key == "elevation" then
			   adjustcue(chi, value, alpha, beta)
			else
			   cuenewindex(table, key, value)
			end
		     end

newshot()