/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet.support.io;

import java.nio.*;
import java.nio.channels.*;
import java.io.*;
import freenet.transport.*;
import freenet.*;
import freenet.support.*;

public final class NIOInputStream extends InputStream implements NIOReader {
	private volatile boolean dead = false;
	private volatile boolean alreadyClosedLink = false;
	private boolean disabledInSelector = false;
	private volatile boolean registered = false;
	private ByteBuffer accumulator;
	private SelectableChannel chan;
	private ReadSelectorLoop rsl;
	private NIOReader nextReader = null;
	private boolean logDEBUG;
	private tcpConnection conn;
	
	//profiling
	//WARNING:remove before release
	public volatile static int instances=0;
	private static final Object profLock = new Object();
	public Object regLock, unregLock;
	
	public NIOInputStream (ByteBuffer buf, SelectableChannel chan, 
						   tcpConnection conn) {
		this.accumulator = buf;
		this.chan = chan;
		regLock= new Object();
		unregLock = new Object();
		this.conn = conn;
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		if(logDEBUG)
			Core.logger.log(this, "Created "+this+" for "+conn+" on "+chan+
							" with "+buf, Logger.DEBUG);
		//profiling
		//WARNING:remove before release
		synchronized(profLock){
			instances++;
		}
	}
	
	public boolean isRegistered() {
		return registered;
	}
	
	public NIOInputStream(SelectableChannel chan, tcpConnection conn) {
		this(ByteBuffer.allocateDirect(64*1024),
			chan, conn);
		accumulator.limit(0).position(0);
	}
	
	public final boolean markSupported(){return false;}
	public void mark(int r){}
	public void reset() {}
	/**
	 * this constructor will create internal buffer
	 */
	
	
	public void configRSL(ReadSelectorLoop rsl) {
		this.rsl = rsl;
	}
	
	public void registered(){
		registered = true;
		if(logDEBUG) Core.logger.log(this, "Registered "+this, Logger.DEBUG);
		synchronized(regLock) {
			regLock.notifyAll();
		}
	}
	
	public void unregistered(){
		registered = false;
		if(logDEBUG) Core.logger.log(this, "Unregistered "+this, 
									 new Exception("debug"), Logger.DEBUG);
		if(nextReader != null) {
			if(logDEBUG) Core.logger.log(this, "nextReader not null: "+this+":"+
										 nextReader+":"+alreadyClosedLink, 
										 Logger.DEBUG);
			if(!alreadyClosedLink) {
				nextReader.configRSL(rsl);
				rsl.register(chan, nextReader);
				rsl.scheduleMaintenance(nextReader);
			} else {
				nextReader.closed();
			}
			nextReader = null; // we won't get any traffic from RSL now
		}
		synchronized(unregLock) {
			unregLock.notifyAll();
		}
	}
	
	public void setNextReader(NIOReader r) {
		nextReader = r;
	}
	
	public int process(ByteBuffer b) {
		if(b == null) return 1;
		else 
			synchronized(accumulator) {
				accumulator.notifyAll();
				// FIXME: hardcoded, but should be ok for inet
				if(b.capacity() - b.limit() > 2048)
					return 1;
				else {
					disabledInSelector = true;
					return 0;
				}
			}
	}
	
	
	
	public ByteBuffer getBuf() {return accumulator;}
	
	public int available() {
		synchronized(accumulator) {
			return accumulator.remaining();
		}
	}
	private String accumStatus() {
		return "accumulator:"+accumulator.position()+"/"+accumulator.limit()+
			"/"+accumulator.capacity()+"/"+toString();
	}
	public long skip(long n) throws IOException {
		synchronized(accumulator) {
		if(logDEBUG)
			Core.logger.log(this, "Trying to skip "+n+" bytes on "+
							this+": "+
							accumStatus(), Logger.DEBUG);
		while(true) {
		    if(accumulator.remaining() >= 1) {
			int got = accumulator.remaining();
			if(n < got) got = (int)n;
			accumulator.position(got);
			accumulator.compact();
			accumulator.limit(accumulator.position());
			accumulator.position(0);
			if(disabledInSelector && 
			   ((accumulator.capacity() - accumulator.limit()) > 4096))
				reregister(); // FIXME: hardcoded
			if(logDEBUG)
				Core.logger.log(this, "Skipped "+got+"/"+n+" bytes, "+
								accumStatus()+": "+this, Logger.DEBUG);
			return got;
		    } else {
				if(alreadyClosedLink) return -1;
			// Uh oh...
			try {
				if (alreadyClosedLink) return -1;
			    long now = System.currentTimeMillis();
			    accumulator.wait(5*60*1000);
			    if (System.currentTimeMillis() - now >= 5*60*1000) {
			    	Core.logger.log(this, "waited more than 5 minutes in NIOIS.skip() "+conn+":"+this+" - closing",Logger.MINOR);
					close();
					return -1;
			    }
			} catch (InterruptedException e) {};
		    }
		}
		}
	}
	
	
	public int read(byte[] b) throws IOException {
	    synchronized(accumulator) {
		if(logDEBUG)
			Core.logger.log(this, "Trying to read "+b.length+" bytes on "+
							this+": "+
							accumStatus(), Logger.DEBUG);
		while(true) {
		    if(accumulator.remaining() >= 1) {
			int got = accumulator.remaining();
			if(b.length < got) got = b.length;
			accumulator.get(b);
			accumulator.compact();
			accumulator.limit(accumulator.position());
			accumulator.position(0);
			if(disabledInSelector && 
			   ((accumulator.capacity() - accumulator.limit()) > 4096))
				reregister(); // FIXME: hardcoded
			if(logDEBUG)
				Core.logger.log(this, "Read "+got+"/"+b.length+" bytes, "+
								accumStatus()+": "+this,
								Logger.DEBUG);
			return got;
		    } else {
				// Uh oh...
				if(alreadyClosedLink) return -1;
				try {
					long now = System.currentTimeMillis();
					if (alreadyClosedLink) return -1;
					accumulator.wait(5*60*1000);
					if (System.currentTimeMillis() - now >= 5*60*1000) {
						Core.logger.log(this, "waited more than 5 minutes in NIOIS.read(byte[]) " + conn +":"+this+"- closing",Logger.MINOR);
						close();
						return -1;
					}
				} catch (InterruptedException e) {};
		    }
		}
	    }
	}
	
	public int read (byte[] b, int off, int len) throws IOException {
	    synchronized(accumulator) {
		if(logDEBUG)
			Core.logger.log(this, "Trying to read "+len+" bytes on "+
							this+": "+
							accumStatus(), Logger.DEBUG);
		while(true) {
		    if(accumulator.remaining() >= 1) {
			int got = accumulator.remaining();
			if(len < got) got = len;
			accumulator.get(b, off, got);
			accumulator.compact();
			accumulator.limit(accumulator.position());
			accumulator.position(0);
			if(disabledInSelector && 
			   ((accumulator.capacity() - accumulator.limit()) > 4096))
				reregister(); // FIXME: hardcoded
			if(logDEBUG)
				Core.logger.log(this, "Read "+got+"/"+len+" bytes, "+
								accumStatus()+": "+this,
								Logger.DEBUG);
			return got;
		    } else {
			// Uh oh...
			if(alreadyClosedLink) return -1;
			try {
			    long now = System.currentTimeMillis();
				if (alreadyClosedLink) return -1;
			    accumulator.wait(5*60*1000);
			    if (System.currentTimeMillis() - now >= 5*60*1000) {
			    	Core.logger.log(this, "waited more than 5 minutes in NIOIS.read(byte[],int,int) "+conn+":"+this+" - closing",Logger.MINOR);
				close();
				return -1;
			    }
			} catch (InterruptedException e) {};
		    }
		}
	    }
	}
	
	public int read() throws IOException {
	    synchronized(accumulator) {
		if(logDEBUG)
			Core.logger.log(this, "Trying to read 1 bytes on "+
							this+": "+accumStatus(),
							new Exception("debug"), Logger.DEBUG);
		while(true) {
		    if(accumulator.remaining() >= 1) {
			int x = (int)accumulator.get();
			accumulator.compact();
			accumulator.limit(accumulator.position());
			accumulator.position(0);
			if(disabledInSelector && 
			   ((accumulator.capacity() - accumulator.limit()) > 4096))
				reregister(); // FIXME: hardcoded
			if(logDEBUG)
				Core.logger.log(this, "Read 1 bytes, "+
								accumStatus()+": "+this,
								Logger.DEBUG);
			return (x & 0xff);
		    } else {
				// Uh oh...
				if(alreadyClosedLink) {
					if(logDEBUG)
						Core.logger.log(this, "Link already closed ("+this+")",
										Logger.DEBUG);
					return -1;
				} else {
					try {
						if(logDEBUG)
							Core.logger.log(this, "Waiting for more data ("+
											this+")", Logger.DEBUG);
						long now = System.currentTimeMillis();
						if (alreadyClosedLink) return -1;
						accumulator.wait(5*60*1000);
						if (System.currentTimeMillis() - now >= 5*60*1000) {
							Core.logger.log(this, "waited more than 5 minutes in NIOIS.read() "+conn+":"+this+"- closing",Logger.MINOR);
							close();
							return -1;
						}
					} catch (InterruptedException e) {};
				}
			}
	    }
		}
	}
	/*
	public void discontinue() {
		Core.logger.log(this, "Discontinuing read input stream from "+
						this+": "+accumStatus(),
						Logger.DEBUG);
	    close();
	}*/
	
	public void close() {
		if(logDEBUG)
			Core.logger.log(this, "Closing read input stream from "+
							this+": "+accumStatus(),
							Logger.DEBUG);
		if(alreadyClosedLink) return;
		if(dead) return;
		synchronized(accumulator) {
			dead = true;
			accumulator.notifyAll(); //notify if other threads are waiting
			if(conn.isClosed()) return; 
			// don't block if already closed or if called from tcpConn
			rsl.queueClose((SocketChannel)chan, this);
			//this was a thinko - queueClose used to set alreadyClosedLink
			while(!alreadyClosedLink) {
				try {
					long now = System.currentTimeMillis();
					accumulator.wait(5*60*1000);
					if (System.currentTimeMillis() -now >=5*60*1000)
						Core.logger.log(this,"waited 5 minutes in NIOIS.close()???",Logger.ERROR);
						//and maybe throw?
				} catch (InterruptedException e) {};
			}
		}
	    //doneMovingTrailingFields = true;
// 		if(disabledInSelector && 
// 		   ((accumulator.capacity() - accumulator.limit()) > 4096))
// 	    	reregister(); // FIXME: hardcoded
// 	    if(rsl != null) {
// 		rsl.scheduleMaintenance(this);
// 	    } else {
// 		throw new IllegalStateException("Do not know my ReadSelectorLoop !");
// 	    }
	}
	
	private void reregister() {
		if(logDEBUG)
			Core.logger.log(this, "Reregistering "+this, Logger.DEBUG);
		disabledInSelector = false;
		try {
			rsl.register(chan, this);
		} catch (IllegalBlockingModeException e) {
			Core.logger.log(this, "Cannot reregister "+this+
							", due to exception", e, Logger.ERROR);
		}
	}
	
	public void queuedClose() {
		//set this again in case called from elsewhere
		dead=true;
		nextReader = null;
	}
	
	public void closed() {
		synchronized(accumulator) {
			alreadyClosedLink = true;
			accumulator.notifyAll();
		}
		if(nextReader != null) nextReader.closed();
	}
	
	public boolean alreadyClosedLink() {
		return alreadyClosedLink;
	}
	
	public boolean isClosed() {
		return alreadyClosedLink || dead;
	}
	
	public boolean shouldThrottle() {
		return conn.shouldThrottle();
	}
	
	public boolean countAsThrottled() {
		return conn.countAsThrottled();
	}
	
	//profiling
	//WARNING:remove before release
	protected void finalize() {
		nextReader = null;
		accumulator = null;
		chan = null;
		rsl = null;
		conn = null;
		synchronized(profLock) {
			instances--;
		}
	}
}
