package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.node.states.data.*;
import freenet.node.ds.KeyCollisionException;
import freenet.message.*;
import freenet.support.Logger;
import java.util.Vector;
import java.io.IOException;

/**
 * The state of an insert after both DataInsert and Accepted.
 * Data is being transferred upstream, but we may still be
 * waiting for an InsertReply or DataReply.  Since we already
 * have the DataInsert, if we restart we'll go back to
 * TransferInsertPending, not InsertPending.
 */
public class TransferInsert extends InsertPending {

    // message queue
    private Vector mq = new Vector();

    TransferInsert(InsertPending ancestor) {
        super(ancestor);
        dim = ancestor.dim;
        dimRecvTime   = ancestor.dimRecvTime;
        checkTime     = ancestor.checkTime;
        receivingData = ancestor.receivingData;
        dataReceived  = ancestor.dataReceived;
        sendingData   = ancestor.sendingData;
        accepted = ancestor.accepted;
        approved = ancestor.approved;
    }

    public final String getName() {
        return "Transferring Insert";
    }   

    private final State transition(State s, boolean doQueue) throws StateTransition {
        NodeMessageObject[] q = new NodeMessageObject[mq.size()];
        mq.copyInto(q);
        throw new StateTransition(s, q, doQueue);
    }



    //=== message handling =====================================================

    public State receivedMessage(Node n, QueryAborted qf) throws StateException {
        if (!fromOrigPeer(qf)) {
            throw new BadStateException("QueryAborted from the wrong peer! for "+this);
        }
        receivingData.cancel();
        sendingData.abort(Presentation.CB_ABORTED);
        mq.addElement(qf);
        return transition(new TransferInsertPending(this), true);
    }

    // this is delicate, because we want to remain in TransferInsert
    // if possible
    public State receivedMessage(Node n, QueryRestarted qr) throws StateException {
        try {
            super.receivedQueryRestarted(n, qr);
        }
        catch (RequestAbortException rae) {
            // going to RequestDone with SendFailedException
            receivingData.cancel();
            sendingData.abort(Presentation.CB_ABORTED);
            queryAborted(n);
            return transition(rae.state, false);  // drop queue
        }
        return this;
    }

    // must have timed out waiting for InsertReply/DataReply
    public State receivedMessage(Node n, RequestInitiator ri) throws StateException {
	if(this.ri == null || this.ri != ri) {
	    throw new BadStateException("Not my request initiator: "+ri+" for "+this);
	}
        sendingData.abort(Presentation.CB_ABORTED);
        try {
            super.receivedRequestInitiator(n, ri);
        }
        catch (EndOfRouteException e) {
            return super.publicEndRoute(n);
        }
        catch (RequestAbortException rae) {
            cancelNoInsert();
            // this is going to RequestDone with no route found
            return rae.state;
        }
        return transition(new TransferInsertPending(this), false);
    }

    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRejected from the wrong peer! for "+
					this);
        }
        sendingData.abort(Presentation.CB_ABORTED);
        mq.addElement(qr);
        return transition(new TransferInsertPending(this), true);
    }

    public State receivedMessage(Node n, DataReply dr) throws StateException {
        if (!fromLastPeer(dr)) {
            throw new BadStateException("DataReply from the wrong peer! for "+this);
        }
        sendingData.abort(Presentation.CB_ABORTED);
        mq.addElement(dr);
        return transition(new TransferInsertPending(this), true);
    }
    
    public State receivedMessage(Node n, InsertReply ir) throws StateException {
        if (!fromLastPeer(ir)) {
            throw new BadStateException("InsertReply from wrong peer! for "+this);
        }
	if(logDEBUG) n.logger.log(this, "Got "+ir+" for "+this, Logger.DEBUG);
        if (approved) {
            // this could belong after a restart..
            mq.addElement(ir);
	    if(logDEBUG) n.logger.log(this, "Already approved, ignoring for "+this,
				      Logger.DEBUG);
            return this;
        }
	if(logDEBUG) n.logger.log(this, "Approving... "+this, Logger.DEBUG);
        approved = true;
        checkTime = System.currentTimeMillis();
        cancelRestart();
	if(logDEBUG) n.logger.log(this, "Cancelled restart for "+this, Logger.DEBUG);
        try {
            insertReply(n);
        }
        catch (RequestAbortException rae) {
            // send failed..
            sendingData.abort(Presentation.CB_ABORTED);  // QueryAborted already sent
            transition(rae.state, false);
        }
        // keep on trucking..
        return checkTransition(n);
    }

    public State receivedMessage(Node n, Accepted a) {
        // we don't care
        return this;
    }
    
    public State receivedMessage(Node n, StoreData sd) throws StateException {
        super.receivedStoreData(n, sd);
        return this;
    }

    public State receivedMessage(Node n, InsertRequest ir) {
        super.receivedRequest(n, ir);
        return this;
    }
    
    private State checkTransition(Node n) throws StateTransition {
	if (!approved) return this;
	
	if (dataSent == null || 
	    dataSent.getCB() != Presentation.CB_OK) {
	    logFailure(n); // it's only a routing failure on sending
	    return this;
	}
	
	if (dataReceived == null ||
	    dataReceived.getCB() != Presentation.CB_OK) {
	    return this;
	}
	
        try {
            receivingData.commit();  // make the key available
	    logSuccess(n);
        } catch (KeyCollisionException e) {
            // this is a little bit of a hack.  we jump into a
            // DataPending state and then handle a restart which
            // makes us check for the data in the store again
            n.logger.log(this, "Going to DataPending after key collision for "+this,
                         Logger.MINOR);
            scheduleRestart(n, 0);
	    logSuccess(n);
            transition(new DataPending(this), false);
        } catch (IOException e) {
            fail(n, "Cache failed");
            n.logger.log(this, "Cache failed on commit for "+this, e, Logger.ERROR);
	    logSuccess(n); // not node's fault
            transition(new RequestDone(this), false);
        }
	
        NoStoreData nosd = new NoStoreData(this);
        long t = Math.max(checkTime, dimRecvTime)
                 + Core.storeDataTime(hopsToLive, dim.length());
        n.ticker().addAbs(t, nosd);
        
        if (storeData != null) mq.addElement(storeData);
        return transition(new AwaitingStoreData(this, nosd, true), true);
    }
    
    
    public State receivedMessage(Node n, DataReceived dr) throws StateException {
        if (receivingData != dr.source()) {
            throw new BadStateException("Not my DataReceived: "+dr+" for "+this);
        }
        dataReceived = dr;
        int cb = dr.getCB();
        switch (cb) {
            
	case Presentation.CB_OK:
	    n.logger.log(this, "Data received successfully! for "+this,
			 n.logger.MINOR);
	    break;
	    
	case Presentation.CB_CACHE_FAILED:
	    n.logger.log(this, "Cache failed while receiving data! for "+this,
			 Logger.ERROR);
	    fail(n, "Cache failed");
	    break;
	    
	case Presentation.CB_BAD_DATA:
	    fail(n, "You sent bad data! for "+this);
	case Presentation.CB_ABORTED:
	case Presentation.CB_RECV_CONN_DIED:
	    break;
	    
	default:
	    fail(n, "You sent "+Presentation.getCBdescription(cb));
	    n.logger.log(this,
			 "Failed to receive insert data with CB "+Presentation.getCBdescription(cb)
			 +", for "+this,
			 Logger.MINOR);
        }
	
        return checkTransition(n);
    }
    
    public State receivedMessage(Node n, DataSent ds) throws StateException {
        if (sendingData != ds.source()) {
            throw new BadStateException("Not my DataSent: "+ds);
        }
        dataSent = ds;
        int cb = ds.getCB();
        switch (cb) {
            
	case Presentation.CB_OK:
	    n.logger.log(this, "Insert transferred successfully! for "+this,
			 Logger.MINOR);
	    break;
	    
	case Presentation.CB_CACHE_FAILED:
	    n.logger.log(this, "Transfer of insert failed, cache broken! for "+this, 
			 Logger.ERROR);
	    queryAborted(n);
	    transition(new RequestDone(this), false);
	    
	case Presentation.CB_SEND_CONN_DIED:
	    n.logger.log(this,
			 "Send died while transferring insert: CB "+
			 Presentation.getCBdescription(cb)+", for "+this,
			 Logger.MINOR);
	    scheduleRestart(n, 0);
	    transition(new TransferInsertPending(this), false);
	    
	default:
	    n.logger.log(this,
			 "Failed to send insert data with CB "+
			 Presentation.getCBdescription(cb)+", for"+this,
			 Logger.MINOR);
	    queryAborted(n);
	    transition(new RequestDone(this), false);
        }
	
        return checkTransition(n);
    }
}



