package freenet.node.states.announcing;

import freenet.node.states.announcement.*;
import freenet.*;
import freenet.node.*;
import freenet.message.*;
import freenet.support.KeyList;
import freenet.support.Logger;
import freenet.crypt.SHA1;
import java.io.*;

/**
 * This phase initiates the second phase of the announcement when
 * we get the AnnouncementReply message back.
 *
 * @author oskar
 */

public class ExecuteAnnouncement extends AnnouncingState {

    private NoReply nr;
    private boolean accepted = false;

    ExecuteAnnouncement(SendAnnouncement sa, NoReply nr) {
        super(sa);
        this.nr = nr;
    }

    public String getName() {
        return "Execute My Node Announcement";
    }

    public boolean receives(MessageObject mo) {
        if (mo instanceof NoReply)
            return nr == mo;
        else if (mo instanceof Message)
            return target.equalsIdent(((Message) mo).peerIdentity());
        else
            return false;
    }

    public State receivedMessage(Node n, AnnouncementFailed af) {
        nr.cancel();
        if (af.reason() == af.KNOWN_ANNOUNCEE) {
            n.logger.log(this,
                         "Announcement failed: "+af.reasonName(),
                         Logger.NORMAL);
            signalFailed(n, true, af.reasonName());
	    return null;
	    // don't reduce the HTL
        }
        n.logger.log(this, "Announcement attempt failed: "
                           +af.reasonName(), Logger.NORMAL);

        signalFailed(n, false, af.reasonName());
	return null;
    }

    public State receivedMessage(Node n, QueryRejected mo) {
        nr.cancel();
        n.logger.log(this, "Announcement attempt failed: "+mo, Logger.MINOR);

        signalFailed(n, false, mo.toString());

	return null;
	// No reason to reduce HTL, target was overloaded
    }

    public State receivedMessage(Node n, NoReply nr) throws BadStateException {
        if (nr != this.nr)
            throw new BadStateException("Not my NoReply");
        n.logger.log(this,
            "Announcement attempt failed, no reply from: "+target,
            Logger.MINOR);

        signalFailed(n, false, nr.toString());

	return null;
    }

    public State receivedMessage(Node n, Accepted a) throws BadStateException {
        if (accepted)
            throw new BadStateException("Received a second accepted");
        nr.cancel();
        accepted = true;
        nr = new NoReply(id);
        // We give it a lot time, since noreply is terminal
        n.schedule(3 * getTime(hopsToLive), nr);
        return this;
    }

    public State receivedMessage(Node n, QueryRestarted a) {
        nr.cancel();
        nr = new NoReply(id);
        n.schedule(3 * getTime(hopsToLive), nr);
        return this;
    }

    public State receivedMessage(Node n, AnnouncementReply ar) {
        nr.cancel();

        byte[] resultVal = ar.getReturnVal();
        if (resultVal.length < myVal.length) {
            // nullpad
            byte[] tmp = new byte[myVal.length];
            System.arraycopy(resultVal, 0, tmp, 0, resultVal.length);
            resultVal = tmp;
        }

        for (int i = 0 ; i < myVal.length ; i++) {
            resultVal[i] ^= myVal[i];
        }
        
        KeyList kl = new KeyList(new byte[][] {myVal});
        String sign = n.sign(resultVal).writeAsField();

        //System.err.println("LALA TOTAL: " + 
        //                   freenet.support.Fields.bytesToHex(resultVal));

        AnnouncementExecute ae = new AnnouncementExecute(id, kl, sign);
        OutputStream out;
        try {
            out = n.sendMessage(ae, target);
        } catch (CommunicationException e) {
            n.logger.log(this, "Sending AnnouncementExecute failed",
                         e, Logger.MINOR);
	    // Error sending
            signalFailed(n, false, e.toString());
	    return null;
        }
        try {
            kl.writeTo(out);
        } catch (IOException e) {
            n.logger.log(this, "Appending my value to AnnouncementExecute failed",
                         e, Logger.MINOR);
            signalFailed(n, false, e.toString());
	    return null;
        } finally {
            try {
                out.close();
            } catch (IOException e) {}
        }
        
        NoComplete nc = new NoComplete(id);
        n.schedule(3 * getTime(hopsToLive), nc);
        return new CompleteAnnouncement(this, new Key(resultVal), nc);
    }
}

