package freenet.node.states.FNP;

import freenet.*;
import freenet.node.*;
import freenet.node.states.request.RequestAbortException;
import freenet.node.states.request.RequestDone;
import freenet.message.*;
import freenet.support.Logger;
import freenet.support.Fields;

public abstract class NewRequest extends State {

    protected Peer origRec;
    
    // Chance to decrease HTL p:
    //      p = 1 - e^(k*HTL^2)     where k == HTL_FACTOR
    //
    // a value of -1.5 gives:
    //          p
    //        ---
    // HTL 1  .78
    // HTL 2  .99
    //
    public static final double HTL_FACTOR = -1.5;
    
    protected NewRequest(long id) {
        super(id);
    }
    
    protected void genReceived(Node n, Request mo) 
                                throws RequestAbortException {

	// Augmented because is on the routingTime path
	
	long startTime = System.currentTimeMillis();
	boolean shouldLog = Core.logger.shouldLog(Core.logger.DEBUG);
	
        origRec = n.getPeer(mo.getSource());
	long time1 = System.currentTimeMillis();
	
	logTime(1, time1-startTime, shouldLog);
	
        if (origRec == null) {
            n.logger.log(this,
                "No shared session and presentation with: "+mo.getSource(),
                Logger.MINOR);
	    // FIXME: add to loadStats here? if not, document why not
            throw new RequestAbortException(null);
        }
	
        try {
            // enforce version aloofness
            String vers = mo.getSource().getVersion();
            if (vers != null && !Version.checkGoodVersion(vers)) {
                String reason = Version.explainBadVersion(vers);
                n.logger.log(this,
                    "Rejecting query from host of type "+vers+": "+reason,
                    Logger.MINOR);
		n.sendMessage(new QueryRejected(id, mo.hopsToLive,
						reason, mo.otherFields), 
			      origRec);
		n.loadStats.receivedQuery(false);
                throw new RequestAbortException(null);
            }

	    long time3 = System.currentTimeMillis();
	    logTime(2, time3-time1, shouldLog);
	    
            // Any request that gets past the node version check.
            n.diagnostics.occurrenceCounting("inboundAggregateRequests", 1);

	    long time4 = System.currentTimeMillis();
	    logTime(3, time4-time3, shouldLog);
	    
            // decrement HTL 
            int htl = (int) mo.hopsToLive;
            if (htl < 1)
                htl = 1;
            else if (htl > Node.maxHopsToLive)
                htl = Node.maxHopsToLive;

	    long time5 = System.currentTimeMillis();
	    logTime(4, time5-time4, shouldLog);
	    
            if (Math.exp(HTL_FACTOR*htl*htl) < 1 - n.randSource.nextDouble()){
                //n.logger.log(this, "Decrementing HTL",
                //             n.logger.DEBUG);
                --htl;
            } else {
                //n.logger.log(this, "Not decrementing HTL",
                //             n.logger.DEBUG);
            }
	    
	    long time6 = System.currentTimeMillis();
	    logTime(5, time6-time5, shouldLog);
	    
	    // Enforce request rate limiting
            if (!n.acceptRequest(mo.searchKey, mo.source.peerAddress())) {
                String reason = "Node overloaded";
                n.logger.log(this,
                    "Rejecting query, rate limit exceeded.",
                    Logger.DEBUGGING);
		n.sendMessage(new QueryRejected(id, htl, 1, reason, 
						// ^--- attenuate routing.
						mo.otherFields), origRec);
                n.loadStats.receivedQuery(false);
                throw new RequestAbortException(null);
            }

	    long time7 = System.currentTimeMillis();
	    
	    logTime(6, time7-time6, shouldLog);
	    
	    mo.hopsToLive = htl; // <- actually, I think Tavin recreates
                                 //    the messages now anyways.

            // Any request that gets past the rate limiting.
            n.diagnostics.occurrenceCounting("inboundAggregateRequestsHandled",
                                             1);
            n.loadStats.receivedQuery(true);
            
            // reply with Accepted A.S.A.P.
            long time = System.currentTimeMillis();
	    if(shouldLog)
		Core.logger.log(this, "Time so far for "+Fields.longToHex(id)+
				" genReceived: "+(time - startTime), Logger.DEBUG);
	    if(shouldLog)
		Core.logger.log(this, "Chain "+Fields.longToHex(id)+" sending Accepted "+
				"at "+(time - mo.stateTime) + " millis after stateTime ("+
				(time - mo.getReceivedTime())+")", Logger.DEBUG);
            n.sendMessage(new Accepted(id), origRec);
            // discount the seconding time from the routing time measurement
            // this needs some cleaning...
	    long acceptTime = System.currentTimeMillis() - time;
	    if(shouldLog)
		Core.logger.log(this, "Chain "+Fields.longToHex(id)+" took "+
				acceptTime+" millis to send Accepted",
				Logger.DEBUG);
            mo.setReceivedTime(acceptTime + mo.getReceivedTime());
            
        } catch (CommunicationException e) {
            n.logger.log(this, "Failed to send initial response to Request.", 
                         e, Logger.MINOR);
            throw new RequestAbortException(null);
        }
    }

    private void logTime(int num, long time, boolean shouldLog) {
	if(shouldLog) {
	    Core.logger.log(this, "logTime: "+Fields.longToHex(id)+":"+num+". "+
			    time+" ms.", time>500 ? Core.logger.MINOR : 
			    Core.logger.DEBUG);
	}
    }
    
    /**
     * Requests in state new do nothing when lost.
     */
    public final void lost(Node n) {}
}



