package freenet.node.rt;

import freenet.Core;
import freenet.Identity;
import freenet.node.NodeReference;
import freenet.node.BadReferenceException;
import freenet.fs.dir.*;
import freenet.support.*;
import java.util.Enumeration;
import java.util.NoSuchElementException;
import java.io.IOException;


/**
 * Uses a pair of DataObjectStores to hold node refs and node properties
 * needed to implement RoutingStore.
 * @author tavin
 */
public class DataObjectRoutingStore implements RoutingStore, Checkpointed {
    
    final DataObjectStore rtNodes, rtProps;

    private int count = 0;  // number of nodes


    public DataObjectRoutingStore(DataObjectStore rtNodes,
                                  DataObjectStore rtProps) {
        this.rtNodes = rtNodes;
        this.rtProps = rtProps;
	removeOrphans();
        // initial count: number of node file entries
        Enumeration rme = rtNodes.keys(true);
        while (rme.hasMoreElements()) {
            ++count;
            rme.nextElement();
        }
    }

    private final void removeOrphans() {
	// Must not have orphaned nodes or even worse orphaned props
	for(Enumeration e=rtNodes.keys(true);e.hasMoreElements();) {
	    FileNumber fn = (FileNumber)(e.nextElement());
	    Enumeration props = rtProps.keys(new PrefixFilePattern(fn, true));
	    if(props == null || !props.hasMoreElements()) {
		Core.logger.log(this, "Removing orphaned node "+fn.toString(),
				Core.logger.NORMAL);
		rtNodes.remove(fn);
	    }
	}
	for(Enumeration e=rtProps.keys(true);e.hasMoreElements();) {
	    FileNumber fn = (FileNumber)(e.nextElement());
	    boolean gotIt = false;
	    Enumeration ee=rtNodes.keys(fn, false);
	    if(ee == null || !ee.hasMoreElements()) {
		if(Core.logger.shouldLog(Core.logger.DEBUG))
		    Core.logger.log(this, "No nodes found looking for property...",
				    Core.logger.DEBUG);
	    } else {
		FileNumber f = (FileNumber)ee.nextElement();
		FilePattern p = new PrefixFilePattern(f, true);
		if(p.matches(fn)) {
		    gotIt = true;
		} else {
		    if(Core.logger.shouldLog(Core.logger.DEBUG)) {
			Core.logger.log(this, "Pattern does not match: "+
					f.toString()+" should be prefix of "+
					fn.toString(), Core.logger.DEBUG);
		    }
		}
	    }
	    if(!gotIt) {
		Core.logger.log(this, "Removing orphaned property "+fn.toString(), 
				Core.logger.NORMAL);
		rtProps.remove(fn);
	    }
	}
    }

    public final String getCheckpointName() {
        return "Saving routing table changes.";
    }

    public final long nextCheckpoint() {
        return System.currentTimeMillis() + 1000 * 300;  // 5 minutes from now
    }

    public synchronized final void checkpoint() {
	// There might be a remove or something going on in parallel
	// We need to serialize out a consistent state
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, "DataObjectRoutingStore checkpointing",
			    Core.logger.DEBUG);
        try {
            rtNodes.flush();
            rtProps.flush();
        }
        catch (IOException e) {
            // hmz..
            Core.logger.log(this, "I/O error flushing routing table data",
                            e, Core.logger.ERROR);
        }
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, "DataObjectRoutingStore checkpointed",
			    Core.logger.DEBUG);
    }
    
    // Don't need to synchronize
    public final int size() {
        return count;
    }
    

    public boolean remove(Identity ident) {
        FileNumber fn = new FileNumber(ident.fingerprint());
	Core.logger.log(this, "Removing identity "+ident.fingerprint(),
			Core.logger.DEBUG);
	return remove(fn, ident.fingerprint().toString());
    }

    protected synchronized boolean remove(FileNumber fn, String log) {
        if (rtNodes.remove(fn)) {
            Core.logger.log(this, "Removed from rtNodes", 
                            Core.logger.DEBUG);
            --count;
            
            // clear out properties entries
            Enumeration pk = rtProps.keys(new PrefixFilePattern(fn, true));
            while (pk.hasMoreElements()) {
                fn = (FileNumber) pk.nextElement();
		Core.logger.log(this, "Removing "+fn+" from rtProps",
				Core.logger.DEBUG);
                rtProps.remove(fn);
		Core.logger.log(this, "Removed "+fn+" from rtProps",
				Core.logger.DEBUG);
            }
            

            Core.logger.log(this, "Removed identity "+log,
                            Core.logger.DEBUG);
            return true;
        }
	Core.logger.log(this, "Failed removing identity "+log,
			Core.logger.DEBUG);
        return false;
    }

    // Do need to synchronize - underlying is not threadsafe
    public synchronized final boolean contains(Identity ident) {
        return rtNodes.contains(new FileNumber(ident.fingerprint()));
    }

    
    public synchronized final Enumeration elements() {
        return new RoutingMemoryEnumeration(rtNodes.keys(true));
    }

    private final class RoutingMemoryEnumeration implements Enumeration {
        
        private final Enumeration keys;
        private Object next;
        
        RoutingMemoryEnumeration(Enumeration keys) {
            this.keys = keys;
            next = step();
        }

        private Object step() {
            while (keys.hasMoreElements()) {
                FileNumber fn = (FileNumber) keys.nextElement();
                Object o = getNode(fn);
                if (o != null)
                    return o;
            }
            return null;
        }

        public final boolean hasMoreElements() {
            return next != null;
        }

        public final Object nextElement() {
            if (next == null) throw new NoSuchElementException();
            try {
                return next;
            }
            finally {
                next = step();
            }
        }
    }
    
    

    private RoutingMemory getNode(FileNumber fn) {
        try {
	    synchronized(this) { // rtNodes not threadsafe
		return (DataObjectRoutingMemory) rtNodes.get(fn);
	    }
        }
        catch (DataObjectUnloadedException dop) {
            try {
                return new DataObjectRoutingMemory(this, dop);
            }
            catch (BadReferenceException e) {
                Core.logger.log(this, "bad reference while resolving: "+fn,
                                e, Core.logger.ERROR);
		remove(fn, "FileNumber: "+fn.toString());
                return null;
            }
            catch (IOException e) {
                Core.logger.log(this, "I/O error while resolving: "+fn,
                                e, Core.logger.ERROR);
		remove(fn, "FileNumber: "+fn.toString());
                return null;
            }
        }
    }

    public final RoutingMemory getNode(Identity ident) {
        return getNode(new FileNumber(ident.fingerprint()));
    }
    
    
    public RoutingMemory putNode(NodeReference nr) {
        Core.logger.log(this, "Adding node to DataObjectRoutingStore", Core.logger.DEBUG);
        Identity ident = nr.getIdentity();
        FileNumber fn = new FileNumber(ident.fingerprint());

        boolean extant = rtNodes.contains(fn);

	DataObjectRoutingMemory mem;

        synchronized(this) {
	    mem = (DataObjectRoutingMemory) getNode(fn);
	    
	    if (mem == null)
		mem = new DataObjectRoutingMemory(this, nr);
	    else if (mem.noderef == null || nr.supersedes(mem.noderef))
		mem.noderef = nr;
	    
	    if(nr.noPhysical()) {
		Core.logger.log(this, "Not adding node to rtNodes: "+nr, Logger.DEBUG);
		return null;
	    }
	    
	    Core.logger.log(this, "About to add node to rtNodes: "+nr, Logger.DEBUG);
	    rtNodes.set(fn, mem);
	    Core.logger.log(this, "Added node to rtNodes: "+nr, Logger.DEBUG);
	    
	    if (!extant)
		++count;
	}
	
	Core.logger.log(this, "Added node to DataObjectRoutingStore: "+nr, 
			Logger.DEBUG);
	
        return mem;
    }
}



