/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet.node.http.infolets;
import freenet.node.http.MultipleFileInfolet;
import freenet.node.http.SimpleAdvanced_ModeUtils;
import freenet.node.Node;
import freenet.support.graph.*;
import freenet.support.Logger;
import freenet.support.Fields;
import freenet.support.servlet.HtmlTemplate;
import freenet.node.rt.RTDiagSnapshot;
import freenet.Key;
import freenet.client.http.NodeStatusServlet;
import freenet.support.StringMap;
import freenet.Core;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import javax.servlet.http.*;

/** 
 * Routing Table Histogram Infolet
 *
 * @author   amphibian
 * @created  April 4, 2003
 */
public class RTInfolet extends MultipleFileInfolet {
	
	Node n;
	NodeStatusServlet status;
	HtmlTemplate boxTemplate;
	
	public String longName() {
		return "Routing Table";
	}
	
	public String shortName() {
		return "rthist";
	}
	
	public boolean visibleFor(HttpServletRequest req){
		return SimpleAdvanced_ModeUtils.isAdvancedMode(req);
	}

	public void init(Node n) {
		this.n = n;
		this.status = new NodeStatusServlet();
		status.init(n);
		try {
			boxTemplate = HtmlTemplate.createTemplate("aqua/titleBox.tpl");
		} catch (IOException e) {
			Core.logger.log(this, "IOException creating template!", e,
							Logger.ERROR);
			boxTemplate = null;
		}
	}
	
	public void toHtml(PrintWriter pw) {
		toHtml(pw, null);
	}
	
	public synchronized void toHtml(PrintWriter pw, HttpServletRequest req) {
		if(boxTemplate == null) {
			try {
				boxTemplate = HtmlTemplate.createTemplate("aqua/titleBox.tpl");
			} catch (IOException e) {
				Core.logger.log(this, "IOException creating template!", e,
								Logger.ERROR);
				boxTemplate = null;
				e.printStackTrace(pw);
				return;
			}
		}
		StringWriter ssw = new StringWriter(500);
		PrintWriter sw = new PrintWriter(ssw);
		Dimensions dim = new Dimensions();
		if(req != null) dim.parseParams(req);
		boolean b = (req.getRequestURI().indexOf("rthist/") == -1);
		sw.println("<p>This is a representation of the keys in your node's routing table designed to show the degree of specialization.  Initially keys will be evenly distributed however as your node adapts to the network you will notice areas of specialization which will show up as dark strips. You can zoom the graph by clicking on it.</p>");
		float zf = dim.zoomFactor();
		sw.println("Zoom factor: "+zf);
		if(zf-1.0F > 1.0F/(dim.length*dim.length))
			sw.println(" (<a href=\""+(b?"rthist/":"")+dim.zoomOut().encode()+
					   "\">Zoom Out</a>)<br>");
		else sw.println("<br>");
		sw.println("<img src=\""+(b?"rthist/":"")+"rthist.bmp"+dim.encode()+
				   "\" usemap=\"#mymap\" width=\""+dim.length+"\" height=\""+
				   dim.heightPerSection * dim.sections+"\"><br>");
		sw.println("<map name=\"mymap\">");
		for(int x=0;x<16;x++) {
			sw.println("<area href=\"rthist"+(b?"/":"")+
					   (dim.getSubsection(x, 16).encode())+
					   "\" shape=\"rect\" coords=\""+x*(dim.length/16)+",0,"+
					   (x+1)*(dim.length/16)+","+dim.heightPerSection+"\">");
		}
		sw.println("</map>");
		sw.println("<table border=\"0\"><tr><td>Graph starts at key prefix</td><td>"+
				   paddedFormat(dim.start*2)+"</td></tr>");
		sw.println("<tr><td>Graph ends at key prefix</td><td>"+
				   paddedFormat(dim.end*2)+"</td></tr></table>");
		RTDiagSnapshot snapshot = n.rt.getSnapshot();
		StringMap tableData = snapshot.tableData();
		status.makePerTableData(snapshot, tableData, sw);
		sw.println("<a href=\"/servlet/nodestatus/nodestatus.html\">" /* FIXME! */
					+ "Click here</a> for more details.");
		sw.println("<hr>");
		sw.println("<p>Use the form below to get a more or less detailed "+
				   "graphical representation of the density of keys in the "+
				   "routing table.</p>");
		sw.println("<form method=\"get\" action=\"rthist"+(b?"/":"")+"\">");
		sw.println("<table><tr><td>Length:</td><td><input name=\"length\" value=\"640\">"+
				   "</td></tr>");
		sw.println("<tr><td>Row Height:</td><td><input name=\"height\" value=\"100\">"+
				   "</td></tr>");
		sw.println("<tr><td>Number of Rows:</td><td><input name=\"sections\" value=\"1\">"+
				   "</td></tr>");
                sw.println("</table><input type=\"submit\" value=\"Execute\"></form>");
		boxTemplate.set("TITLE", "Routing Table");
		boxTemplate.set("CONTENT", ssw.toString());
		boxTemplate.toHtml(pw);
	}

	class Dimensions {
		int length = 480;
		int heightPerSection = 100;
		int sections = 1;
		long start = 0;
		long end = (1L<<31);
		
		float zoomFactor() {
			return (float) (((double)(1L << 31)) / ((double)(end - start)));
		}
		
		Dimensions getSubsection(int x, int total) {
			Dimensions d = new Dimensions();
			d.length = length;
			d.heightPerSection = heightPerSection;
			d.sections = sections;
			long len = end - start;
			long center = start + (((end - start) * x) / total);
			long eachside = ((len >> 1) * 2)/3;
			if(center < (eachside)) center = eachside;
			if(center > ((1L << 31) - eachside)) center = (1L << 31) - eachside;
			d.start = center - eachside;
			d.end = center + eachside;
			return d;
		}
		
		Dimensions zoomOut() {
			Dimensions d = new Dimensions();
			d.length = length;
			d.heightPerSection = heightPerSection;
			d.sections = sections;
			long len = end - start;
			long center = (start + end) >> 1;
			long eachside = ((len >> 1) * 3)/2;
			if(eachside > (1L << 30)) eachside = (1L << 30);
			if(center < (eachside)) center = eachside;
			if(center > ((1L << 31) - eachside)) center = (1L << 31) - eachside;
			d.start = center - eachside;
			d.end = center + eachside;
			return d;
		}
		
		String encode() {
			return "?length="+length+"&height="+heightPerSection+
				"&sections="+sections+"&end="+Fields.longToHex(end)+
				"&start="+Fields.longToHex(start);
		}
		
		void parseParams(HttpServletRequest req) {
			String lengthAsString = req.getParameter("length");
			if(lengthAsString != null && !lengthAsString.equals("")) {
				try {
					length = Integer.parseInt(lengthAsString);
				} catch (NumberFormatException e) {};
			}
			String heightAsString = req.getParameter("height");
			if(heightAsString != null && !heightAsString.equals("")) {
				try {
					heightPerSection = Integer.parseInt(heightAsString);
				} catch (NumberFormatException e) {};
			}
			String sectionsAsString = req.getParameter("sections");
			if(sectionsAsString != null && !sectionsAsString.equals("")) {
				try {
					sections = Integer.parseInt(sectionsAsString);
				} catch (NumberFormatException e) {};
			}
			String startAsString = req.getParameter("start");
			if(startAsString != null && !startAsString.equals("")) {
				try {
					start = Fields.hexToLong(startAsString);
				} catch (NumberFormatException e) {};
			}
			String endAsString = req.getParameter("end");
			if(endAsString != null && !endAsString.equals("")) {
				try {
					end = Fields.hexToLong(endAsString);
				} catch (NumberFormatException e) {};
			}
		}
	}

	public boolean write(String file, HttpServletRequest req, HttpServletResponse resp)
		throws java.io.IOException {
		if(file.equals("rthist.bmp")) {
			Dimensions dim = new Dimensions();
			if(req != null) dim.parseParams(req);
			int height = dim.sections * dim.heightPerSection;
			int[][] density = new int[dim.length][dim.sections];
			Key[] keys = n.rt.getSnapshot().keys();
			for(int i=0;i<keys.length;i++) {
				byte[] b = keys[i].getVal();
				if(b.length < 4) continue;
				long x = ((((long)b[0]) & 0xff) << 24) + ((((long)b[1]) & 0xff) << 16) +
					((((long)b[2]) & 0xff) << 8) + (((long)b[3]) & 0xff);
				x >>= 1;
				if(x < dim.start || x > dim.end) {
					continue;
				}
				x = ((x - dim.start) << 31) / (dim.end - dim.start);
				x = (x * dim.length * dim.sections) >> 31;
				int section = (int)(x / dim.length);
				x = x % dim.length;
				density[(int)x][section]++;
			}
			Bitmap bmp = new Bitmap(dim.length, height);
			bmp.setGreyscalePalette();
			bmp.setPenColor(new Color(0,0,0));
			int max = 0;
			for(int x=0; x<dim.length; x++) {
				for(int y=0; y<dim.sections; y++) {
					int c = density[x][y];
					if(max < c) max = c;
				}
			}
			int mul = 1;
			if(max > 0 && max < 256) {
				mul = 256/max;
			}
			for(int x=0; x<dim.length; x++) {
				for(int y=0; y<dim.sections; y++) {
					int c = density[x][y];
					if(c > 0) {
						if(c > 255) c = 255;
						c *= mul;
						c = 255 - c;
						bmp.setPenColor(new Color(c, c, c));
						for(int h = 0; h < dim.heightPerSection; h++) {
							bmp.setPixel((int)x, h+(y*dim.heightPerSection));
						}
					}
				}
			}
			BitmapEncoder enc = new DibEncoder();
			enc.setBitmap(bmp);
			resp.setContentType(enc.getMimeType());
			OutputStream os = resp.getOutputStream();
			enc.encode(os);
			os.close();
			return true;
		}
		return false;
	}
	
	String paddedFormat(long x) {
		if(x >= (2L << 32)) return "End of Keyspace";
		else if (x <= 0) return "Start of Keyspace";
		else {
			String s = Fields.longToHex(x);
			while(s.length() < 8) s += '0';
			return s;
		}
	}
}
