package freenet.fs.dir;

import freenet.Core;
import freenet.fs.*;
import freenet.fs.FileSystem;
import freenet.fs.acct.*;
import freenet.fs.acct.sys.AccountingTreeCheck;
import freenet.fs.acct.fsck.FaultAnalysis;
import freenet.support.*;
import freenet.support.BinaryTree.Node;
import freenet.crypt.Digest;
import java.io.*;
import java.text.DateFormat;
import java.util.Date;
import java.util.Vector;
import java.util.Enumeration;

/**
 * An implementation of the Directory that uses freenet.fs.Filesystem
 * as the backing store.  Space is reserved within the FileSystem
 * to contain accounting tables that hold the directory data.
 *
 * @see freenet.fs.FileSystem
 * @see freenet.fs.acct.*
 * @see freenet.fs.acct.sys.*
 * @author tavin
 */
public class FSDirectory implements Directory, FSDirectoryConst {

    public static void dump(FSDirectory dir, PrintWriter pw) {
        synchronized (dir.semaphore()) {
        
            pw.println("Version: "+FSDirectoryRoot.VERSION);
            pw.println("Free space: "+dir.available());
            pw.println("Dirty: "+dir.dirty());
            pw.println("Accounting ranges: "
                       + (dir.acctRanges == null
                          ? "none" : Fragment.rangeList(dir.acctRanges)));
            pw.println();
        
            pw.println("Files");
            pw.println("-----");
            dumpFiles(dir, pw);

            pw.println("LRU Files (oldest last)");
            pw.println("-----------------------");
            dumpLRUFiles(dir, pw);
        }
    }

    public static void dumpFiles(FSDirectory dir, PrintWriter pw) {
        synchronized (dir.semaphore()) {
            Enumeration keys = dir.keys(true);
            while (keys.hasMoreElements()) {
                FileNumber fn = (FileNumber) keys.nextElement();
                Buffer buffer = dir.fetch(fn);
                try {
                    pw.println(fn + " @ " + Fragment.rangeList(buffer.ticket().ranges));
                }
                finally {
                    buffer.release();
                }
            }
            pw.println();
        }
    }

    public long countKeys() {
	throw new UnsupportedOperationException();
    }
    
    public static void dumpLRUFiles(FSDirectory dir, PrintWriter pw) {
        synchronized (dir.semaphore()) {
            DateFormat df = DateFormat.getDateTimeInstance();
            Enumeration keys = dir.lruKeys(false);
            while (keys.hasMoreElements()) {
                FileNumber fn = (FileNumber) keys.nextElement();
                Buffer buffer = dir.fetch(fn);
                try {
                    pw.println(fn + " @ " + df.format(new Date(buffer.ticket().timestamp)));
                }
                finally {
                    buffer.release();
                }
            }
            pw.println();
        }
    }

    
    

    
                            
    private final FileSystem fs;
    

    private final Digest ctx;
    
    private final int rootBlockSize,
                      acctBlockSize;
    
    // incremented each time the root blocks are updated
    private long root_no;

    
    // the fragments occupied by the accounting tables
    Fragment[] acctRanges = null;
    
                            
    private final SingleAccountingProcess proc;
    
    private final FragmentManager fragmentMgr;
    private final TicketManager ticketMgr;

    private final KeyMap keyMap;
    private final LRUMap lruMap;

    private boolean dirty = false;

    
    /**
     * @param fs    the file-system to contain the directory accounting data
     * @param ctx   checksum algorithm
     * @param rootBlockSize     size of each of the two root blocks
     * @param acctBlockSize     size of each accounting block
     * @param blockCacheSize    number of accounting blocks to cache
     * @param ticketCacheSize   number of live tickets to cache
     */
    public FSDirectory(FileSystem fs, Digest ctx,
                       int rootBlockSize, int acctBlockSize,
                       int blockCacheSize, int ticketCacheSize) throws IOException {
        
        this.fs = fs;
        this.ctx = ctx;
        this.rootBlockSize = rootBlockSize;
        this.acctBlockSize = acctBlockSize;

        if (fs.size() <= 2 * rootBlockSize)
            throw new IOException("file-system impossibly small..");

        // examine the directory root

        FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx, rootBlockSize);
        this.root_no = root.getNextRootNumber();
        this.acctRanges = root.getRanges();

        // create accounting structures

        AccountingTable acct = new AccountingTable(fs, acctRanges, ctx, acctBlockSize);

        Cache cache = new LRUCache(blockCacheSize);

        AccountingInitializer acctInit = new AccountingInitializer(acct);
        proc = new SingleAccountingProcess(acctInit);
        
        TicketMap liveTicketMap, mainTicketMap;
        FragmentSizeMap fsizeMap;
        FragmentPositionMap fposMap;
    
        liveTicketMap = new TicketMap(proc.share(LIVE_TICKET_MAP), true);
        mainTicketMap = new TicketMap(proc.share(MAIN_TICKET_MAP), false);
        fsizeMap = new FragmentSizeMap(proc.share(FRAGMENT_SIZE_MAP), cache);
        fposMap = new FragmentPositionMap(proc.share(FRAGMENT_POSITION_MAP), cache);

        this.keyMap = new KeyMap(proc.share(KEY_MAP), cache);
        this.lruMap = new LRUMap(proc.share(LRU_MAP), cache);

        // initialize all accounting structures

        SharedAccountingInitializer shInit = new SharedAccountingInitializer();
        shInit.add(LIVE_TICKET_MAP, liveTicketMap);
        shInit.add(MAIN_TICKET_MAP, mainTicketMap);
        shInit.add(FRAGMENT_SIZE_MAP, fsizeMap);
        shInit.add(FRAGMENT_POSITION_MAP, fposMap);
        shInit.add(KEY_MAP, keyMap);
        shInit.add(LRU_MAP, lruMap);
        acctInit.initialize(shInit);

        // set aside initial boundary markers and free space if necessary

        if (fposMap.count() == 0) {
            fposMap.allocate(new Fragment(fs.size(), Long.MAX_VALUE - 1), -1);
            fposMap.allocate(new Fragment(0, 2 * rootBlockSize - 1), -1);
            fsizeMap.put(new Fragment(2 * rootBlockSize, fs.size() - 1));
            dirty = true;
        }

        // otherwise check for changed FS size
        
        else {
            Node n = fposMap.treeMax();
            Fragment f = (Fragment) n.getObject();
            if (fs.size() < f.getLowerBound()) {
                throw new IOException("cannot adapt to decreased file-system size");
            }
            if (fs.size() > f.getLowerBound()) {
                Node n2 = fposMap.treePredecessor(n);
                Fragment f2 = (Fragment) n2.getObject();
                if (f.getLowerBound() - f2.getUpperBound() > 1) {
                    // first clear out the now too-small entry in the size map
                    fsizeMap.remove(new Fragment(f2.getUpperBound() + 1,
                                                 f.getLowerBound() - 1));
                }
                fsizeMap.put(new Fragment(f2.getUpperBound() + 1, fs.size() - 1));
                fposMap.treeRemove(n);
                fposMap.allocate(new Fragment(fs.size(), Long.MAX_VALUE - 1), -1);
                dirty = true;
            }
        }

        // finish setting up structs..
        
        fragmentMgr = new FragmentManager(fsizeMap, fposMap);
        
        ticketMgr = new TicketManager(this, ticketCacheSize,
                                      liveTicketMap, mainTicketMap);
        
        // finally, clean out uncommitted files

        Vector v = new Vector();
        synchronized (v) {
            Walk ltw = liveTicketMap.tickets();
            Ticket ticket;
            while (null != (ticket = (Ticket) ltw.getNext())) {
                if (ticket.timestamp == -1) {
                    v.addElement(ticket);
                    fragmentMgr.free(ticket.ranges);
                }
            }
            if (v.size() > 0) {
                Enumeration lte = v.elements();
                while (lte.hasMoreElements())
                    liveTicketMap.treeRemove((Ticket) lte.nextElement());
                dirty = true;
            }
        }
    }
    

    /**
     * @return  an Object to synchronize on when doing
     *          multiple directory operations
     */
    public final Object semaphore() {
        return this;
    }

    /**
     * When files are committed or deleted, the directory
     * needs to be flushed and is "dirty."
     * @return  true, if the directory is dirty
     */
    public final boolean dirty() {
        return dirty;
    }


    /**
     * A utilitarian convenience..
     */
    public synchronized void freeze() throws IOException {
        proc.freeze();
        dirty = false;
    }
    

    private final Vector flushVec = new Vector();

    /**
     * @return  0 if the tables were flushed,
     *          otherwise the minimum size free Fragment
     *          needed to do the flush
     */
    public synchronized long flush() throws IOException {
        
        if (!dirty)
            return 0;
        
        if (proc.getFreeCount() < 0) {
            AccountingTable acct = proc.getAccountingTable();
            synchronized (flushVec) {
                try {
                    Fragment[] ranges = acct.ranges();
                    for (int i=0; i<ranges.length; ++i)
                        flushVec.addElement(ranges[i]);
    
                    int blockDelta = 0;
                    try {
                        do {
                            int target = (int) Math.max(10, -2 * proc.getFreeCount());
                            long width = target * acct.getBlockWidth();
                            Fragment newFrag = fragmentMgr.allocate(width, width, -1);
                            if (newFrag == null)
                                return width;
                            blockDelta += target;
                            flushVec.addElement(newFrag);
                        }
                        while (blockDelta + proc.getFreeCount() < 0);
                    }
                    finally {
                        if (blockDelta > 0) {
                            ranges = new Fragment[flushVec.size()];
                            flushVec.copyInto(ranges);
                            acct = new AccountingTable(acct, ranges);
                            proc.setAccountingTable(acct);
                            FSDirectoryRoot root = new FSDirectoryRoot(fs, ctx,
                                                                       rootBlockSize,
                                                                       root_no,
                                                                       ranges);
                            this.root_no = root.getNextRootNumber();
                            this.acctRanges = root.getRanges();
                        }
                    }
                }
                finally {
                    flushVec.removeAllElements();
                }
            }
        }
        
        proc.flush();
        dirty = false;
        return 0;
    }


    // promote/demote are only entered from TicketManager code
    // that is called from another synchronized method in this class
    
    synchronized final void promote(long timestamp, FileNumber fn) {
        //Core.logger.log(this, "promoting: "+fn+" @ "+timestamp, Core.logger.DEBUG);
        lruMap.remove(timestamp, fn);
    }

    synchronized final void demote(long timestamp, FileNumber fn) {
        //Core.logger.log(this, "demoting: "+fn+" @ "+timestamp, Core.logger.DEBUG);
        lruMap.insert(timestamp, fn);
    }


    /**
     * @return  the amount of free space
     */
    public synchronized final long available() {
        return fragmentMgr.available();
    }


    /**
     * IMPORTANT:  Stay synchronized while iterating over the enum.
     */
    public synchronized final Enumeration keys(boolean ascending) {
        return new WalkEnumeration(keyMap.keys(ascending));
    }

    /**
     * IMPORTANT:  Stay synchronized while iterating over the enum.
     */
    public synchronized final Enumeration keys(FilePattern pat) {
        Walk w = keyMap.keys(pat.key(), true, pat.ascending());
        return new WalkEnumeration(FileNumber.filter(pat, w));
    }
        
    /**
     * IMPORTANT:  Stay synchronized while iterating over the enum.
     */
    public synchronized final Enumeration lruKeys(boolean ascending) {
        return new WalkEnumeration(lruMap.keys(ascending));
    }
    
    
    /**
     * Deletes a file.
     * @return  true if there was a matching file
     */
    public synchronized final boolean delete(FileNumber fn, boolean keepIfUsed) {
	if(keepIfUsed) throw new UnsupportedOperationException(); // this class is only here for back compatibility while upgrading an old store - FIXME: support it if we ever use this for anything else
        long ticketID = keyMap.remove(fn);
        if (ticketID == -1)
            return false;
        
        Ticket ticket = ticketMgr.delete(ticketID);
        if (ticket == null)
            throw new DirectoryException("ticket not found: #"
                                         + ticketID + " / " + fn);
        
        if (ticket.timestamp != -1)
            lruMap.remove(ticket.timestamp, ticket.fn);
        fragmentMgr.free(ticket.ranges);
        
        dirty = true;
        return true;
    }
    
    public boolean demote(FileNumber fn) {
	throw new UnsupportedOperationException();
    }
    
    /**
     * @return  true, if there is a matching file
     */
    public synchronized final boolean contains(FileNumber fn) {
        return -1 != keyMap.get(fn);
    }


    /**
     * Opens an existing file.
     * @throws DirectoryException
     *         if there is a ticket ID for the given file-number,
     *         but the corresponding ticket can't be found
     */
    public synchronized final Buffer fetch(FileNumber fn) {
        long ticketID = keyMap.get(fn);
        //Core.logger.log(this, "fetch: "+fn+" / "+ticketID, Core.logger.DEBUG);
        if (ticketID == -1)
            return null;
        TicketLock lock = ticketMgr.lock(ticketID);
        if (lock == null)
            throw new DirectoryException("ticket not found: #"
                                         + ticketID + " / " + fn);
        return new BufferImpl(lock);
    }

    
    /**
     * Allocates and opens a new file.
     * @return  the created Buffer, or null if there was insufficient storage
     */
    public synchronized final Buffer store(long size, FileNumber fn) {
        if (size > available()) {
            return null;
        }
        long ticketID = ticketMgr.getNextID();
        Fragment[] ranges = fragmentMgr.allocate(size, ticketID);
        // what would happen if somehow the returned ranges
        // didn't total the requested size?  it's supposed
        // to be impossible of course...
        return new BufferImpl(ticketMgr.create(ticketID, ranges, fn));
    }

    public final freenet.support.KeyHistogram getHistogram() {
	return null;
    }

    public final freenet.support.KeySizeHistogram getSizeHistogram() {
	return null;
    }


    
    /**
     * Buffer implementation.
     */
    private final class BufferImpl implements Buffer {
        
        private final TicketLock ticketLock;
        private final long length;
        private boolean failed = false;
        private boolean released = false;
        
        BufferImpl(TicketLock ticketLock) {
            this.ticketLock = ticketLock;
            this.length = Fragment.rangeLength(ticket().ranges);
            //Core.logger.log(FSDirectory.this,
            //                "granting buffer: "+this,
            //                new Exception("code-path to buffer grant.."),
            //                Core.logger.DEBUG);
        }

        public final String toString() {
            return ticket().toString();
        }
        
        
        public final Ticket ticket() {
            return ticketLock.ticket();
        }

        public final void touch() {
            synchronized (FSDirectory.this) {
                ticketLock.touch();
            }
        }
        
        public final void commit() {
            synchronized (FSDirectory.this) {
                if (ticket().ticketID != keyMap.put(ticket().fn, ticket().ticketID))
                    throw new DirectoryException("key collision");
                dirty = true;
                ticketLock.commit();
            }
        }
        
        public final void release() {
            if (!released) {
                //Core.logger.log(FSDirectory.this,
                //                "releasing buffer: "+this,
                //                Core.logger.DEBUG);
                released = true;
                synchronized (FSDirectory.this) {
                    ticketLock.release();
                    if (ticket().users == 0 && ticket().timestamp == -1) {
                        ticketMgr.delete(ticket().ticketID);
                        fragmentMgr.free(ticket().ranges);
                    }
                }
            }
        }

        protected final void finalize() {
            release();
        }

        public final long length() {
            return length;
        }

        public final boolean failed() {
            return failed;
        }


        public final InputStream getInputStream() throws IOException {
            if (failed)
                throw new BufferException("buffer failed");
            return new SafeInputStream(ReadLock.getInputStream(fs, ticket().ranges));
        }

        private final class SafeInputStream extends FilterInputStream {
            private SafeInputStream(InputStream in) {
                super(in);
            }
            public final int read() throws IOException {
                if (failed) throw new BufferException("buffer failed");
                return in.read();
            }
            public final int read(byte[] buf, int off, int len) throws IOException {
                if (failed) throw new BufferException("buffer failed");
                return in.read(buf, off, len);
            }
            public final int available() throws IOException {
                if (failed) throw new BufferException("buffer failed");
                return in.available();
            }
            public final long skip(long n) throws IOException {
                if (failed) throw new BufferException("buffer failed");
                return in.skip(n);
            }
        }
    

        public final OutputStream getOutputStream() throws IOException {
            if (failed)
                throw new BufferException("buffer failed");
            return new SafeOutputStream(WriteLock.getOutputStream(fs, ticket().ranges));
        }
        
        private final class SafeOutputStream extends FilterOutputStream {
            private SafeOutputStream(OutputStream out) {
                super(out);
            }
            public void write(int b) throws IOException {
                boolean worked = false;
                try {
                    out.write(b);
                    worked = true;
                }
                finally {
                    if (!worked)
                        failed = true;
                }
            }
            public void write(byte[] buf, int off, int len) throws IOException {
                boolean worked = false;
                try {
                    out.write(buf, off, len);
                    worked = true;
                }
                finally {
                    if (!worked)
                        failed = true;
                }
            }
            public void flush() throws IOException {
                boolean worked = false;
                try {
                    out.flush();
                    worked = true;
                }
                finally {
                    if (!worked)
                        failed = true;
                }
            }
            public void close() throws IOException {
                boolean worked = false;
                try {
                    out.close();
                    worked = true;
                }
                finally {
                    if (!worked)
                        failed = true;
                }
            }
        }
    }

}



