package freenet.client.http;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.DateFormat;
import javax.servlet.*;
import javax.servlet.http.*;

import freenet.ContactCounter;
import freenet.support.Logger;
import freenet.ConnectionHandlerComparator;
import freenet.Core;
import freenet.Key;
import freenet.node.Node;
import freenet.node.NodeReference;
import freenet.node.rt.RTDiagSnapshot;
import freenet.node.LoadStats;
import freenet.diagnostics.Diagnostics;
import freenet.diagnostics.DiagnosticsFormat;
import freenet.diagnostics.HtmlDiagnosticsFormat;
import freenet.diagnostics.HtmlIndexFormat;
import freenet.diagnostics.RowDiagnosticsFormat;
import freenet.diagnostics.FieldSetFormat;
import freenet.diagnostics.GraphRange;
import freenet.diagnostics.GraphRangeDiagnosticsFormat;
import freenet.diagnostics.GraphHtmlDiagnosticsFormat;
import freenet.diagnostics.GraphDiagnosticsFormat;
import freenet.node.rt.RoutingTable;
import freenet.node.ds.FSDataStore;
import freenet.support.servlet.http.HttpServletResponseImpl;
import freenet.support.io.WriteOutputStream;
import freenet.support.URLEncodedFormatException;
import freenet.support.sort.*;
import freenet.support.StringMap;
import freenet.support.KeyHistogram;
import freenet.support.KeySizeHistogram;
import freenet.support.graph.BitmapEncoder;

/*
  This code is distributed under the GNU Public Licence (GPL)
  version 2.  See http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Servlet to display and allow downloading
 * of node references as the node is running.
 * <p>
 * Example freenet.conf segment to run this servlet:
 * <pre>
 * nodestatus.class=freenet.client.http.NodeStatusServlet
 * # Change port number if you like.
 * nodestatus.port=8889
 * # Make sure that the servlet is listed in the services line.
 * services=fproxy,nodestatus
 * </pre>
 * <p>
 * @author giannij
 **/
public class NodeStatusServlet extends HttpServlet  {
    public void init() {
        ServletContext context = getServletContext();
        node = (Node)context.getAttribute("freenet.node.Node");
	init(node);
    }
    
    public void init(Node n) {
	node = n;
        if (node != null) {
            rt = node.rt;
            diagnostics = node.diagnostics;
            inboundContacts = node.inboundContacts;
            outboundContacts = node.outboundContacts;
            inboundRequests = node.inboundRequests;
            outboundRequests = node.outboundRequests;
            loadStats = node.loadStats;
        }
    }

    private final static String MSG_NO_REQUEST_DIST =
        "# Data for the inbound request distribution isn't being logged. \n" +
        "#  To enable logging set: \n" +
        "#   logInboundRequestDist=true \n" +
        "# in your freenet.conf / freenet.ini file.";
    
    private final static String MSG_NO_INBOUND_INSERT_DIST =
	 "# Data for the inbound insert request distribution isn't being logged. \n" +
	 "#  To enable logging set: \n" +
	 "#   logInboundInsertRequestDist=true \n" +
	 "# in your freenet.conf / freenet.ini file.";

    private final static String MSG_NO_INSERT_SUCCESS_DIST =
	"# Data for the distribution of externally originated successfully inserted keys isn't being logged\n" +
	"#  To enable logging set: \n" +
	"#   logSuccessfulInsertRequestDist=true \n" +
	"# in your freenet.conf / freenet.ini file.";

    private final static String MSG_OOPS =
        "# Coding error!";

    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
	throws ServletException, IOException
    {
        if ((rt == null)) {
            sendError(resp, HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
                      "Couldn't access the freenet.node.Node instance.  " +
                      "This servlet must be run in the same JVM as the node.");
            return;
        }

        try {
            String uri = freenet.support.URLDecoder.decode(req.getRequestURI());
            String baseURL = req.getContextPath() + req.getServletPath();
            if ((baseURL.length() > 0) && (!baseURL.endsWith("/"))) {
                baseURL += "/";
            } 

	    // FIXME: this is a monster. Most of the histograms could be parameterised and code reduced significantly
	    
            if (uri.endsWith("/loadStats.txt")) {
                sendLoadStats(resp);
                return;
            }
            
            if (uri.endsWith("/inboundContacts.txt")) {
                sendPerHostStats(resp, "inboundContacts");
                return;
            }

            if (uri.endsWith("/outboundContacts.txt")) {
                sendPerHostStats(resp, "outboundContacts");
                return;
            }

            if (uri.endsWith("/inboundRequests.txt")) {
                sendPerHostStats(resp, "inboundRequests");
                return;
            }

            if (uri.endsWith("/outboundRequests.txt")) {
                sendPerHostStats(resp, "outboundRequests");
                return;
            }
	    
	    if (uri.endsWith("version_histogram.txt")) {
		sendVersionHistogram(resp, false);
		return;
	    }
	    
	    if (uri.endsWith("version_data.txt")) {
		sendVersionHistogram(resp, true);
		return;
	    }
	    
            if (uri.endsWith("key_histogram.txt")) {
                sendRTHistogram(resp, false, false);
                return;
            }
	    
	    if (uri.endsWith("key_histogram_detail.txt")) {
		sendRTHistogram(resp, false, true);
		return;
	    }

            if (uri.endsWith("ds_histogram.txt")) {
                sendDSHistogram(resp, false, false);
                return;
            }

	    if (uri.endsWith("ds_histogram_detail.txt")) {
		sendDSHistogram(resp, false, true);
		return;
	    }
	    
	    if (uri.endsWith("ds_size_histogram.txt")) {
		sendDSSizeHistogram(resp, false);
		return;
	    };

            if (uri.endsWith("inbound_request_histogram.txt")) {
                int[] bins = null;
                if (node.requestDataDistribution != null) {
                    bins = node.requestDataDistribution.getBins();
                }
                sendKeyHistogram(resp, false, bins, 
                                 "Histogram of requested keys.",
                                 "This count has nothing to do with keys in your datastore",
                                 MSG_NO_REQUEST_DIST, "keys", false);
                return;
            }

	    if (uri.endsWith("inbound_insert_histogram.txt")) {
		int[] bins = null;
		if (node.requestInsertDistribution != null) {
		    bins = node.requestInsertDistribution.getBins();
		}
		sendKeyHistogram(resp, false, bins,
				 "Histogram of insert attempted keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_INBOUND_INSERT_DIST, "keys", false);
		return;
	    }
	    
	    if (uri.endsWith("inbound_request_histogram_detail.txt")) {
		int[] bins = null;
		if (node.requestDataDistribution != null) {
		    bins = node.requestDataDistribution.getBiggerBins();
		}
		sendKeyHistogram(resp, false, bins,
				 "Histogram of requested keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_REQUEST_DIST, "keys", false);
		return;
	    }
	    
	    if (uri.endsWith("inbound_insert_histogram_detail.txt")) {
		int[] bins = null;
		if (node.requestInsertDistribution != null) {
		    bins = node.requestInsertDistribution.getBiggerBins();
		}
		sendKeyHistogram(resp, false, bins,
				 "Histogram of attempted insert keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_INBOUND_INSERT_DIST, "keys", false);
		return;
	    }
	    
	    if (uri.endsWith("inbound_success_histogram.txt")) {
	       int [] bins = null;
	       if (node.successDataDistribution != null) {
		   bins = node.successDataDistribution.getBins();
	       }
	       sendKeyHistogram(resp, false, bins,
			        "Histogram of successful externally requested keys.",
				"This count has nothing to do with keys in your datastore",
				MSG_NO_REQUEST_DIST, "keys", false);
	       return;
	    }

	    if (uri.endsWith("inbound_insert_success_histogram.txt")) {
	       int [] bins = null;
	       if (node.successInsertDistribution != null) {
		   bins = node.successInsertDistribution.getBins();
	       }
	       sendKeyHistogram(resp, false, bins,
			        "Histogram of externally originated successfully inserted keys.",
				"This count has nothing to do with keys in your datastore",
				MSG_NO_INSERT_SUCCESS_DIST, "keys", false);
	       return;
	    }
	    
	    if (uri.endsWith("inbound_success_histogram_detail.txt")) {
	       int [] bins = null;
	       if (node.successDataDistribution != null) {
		   bins = node.successDataDistribution.getBiggerBins();
	       }
	       sendKeyHistogram(resp, false, bins,
			        "Histogram of successfully requested keys.",
				"This count has nothing to do with keys in your datastore",
				MSG_NO_REQUEST_DIST, "keys", false);
	       return;
	    }

	    if (uri.endsWith("inbound_insert_success_histogram_detail.txt")) {
	 	int [] bins = null;
		if (node.successInsertDistribution != null) {
		    bins = node.successInsertDistribution.getBiggerBins();
		}
		sendKeyHistogram(resp, false, bins,
				 "Histogram of externally originated successfully inserted keys.",
				 "This count has nothing to do with the keys in your datastore",
				 MSG_NO_INSERT_SUCCESS_DIST, "keys", false);
		return;
	    }

            if (uri.endsWith("key_histogram_data.txt")) {
                sendRTHistogram(resp, true, false);
                return;
            }

	    if (uri.endsWith("key_histogram_data_detail.txt")) {
		sendRTHistogram(resp, true, true);
		return;
	    }
	    
            if (uri.endsWith("ds_histogram_data.txt")) {
                sendDSHistogram(resp, true, false);
                return;
            }

	    if (uri.endsWith("ds_histogram_data_detail.txt")) {
		sendDSHistogram(resp, true, true);
		return;
	    }
	    
	    if (uri.endsWith("ds_size_histogram_data.txt")) {
		sendDSSizeHistogram(resp, true);
		return;
	    }

            if (uri.endsWith("inbound_request_histogram_data.txt")) {
                int[] bins = null;
                if (node.requestDataDistribution != null) {
                    bins = node.requestDataDistribution.getBins();
                }

                sendKeyHistogram(resp, true, bins,  
                                 "Histogram of requested keys.",
                                 "This count has nothing to do with keys in your datastore",
                                 MSG_NO_REQUEST_DIST, "keys", false); 

                return;
            }

	    if (uri.endsWith("inbound_insert_histogram_data.txt")) {
		int [] bins = null;
		if (node.requestInsertDistribution != null) {
		    bins = node.requestInsertDistribution.getBins();
		}

		sendKeyHistogram(resp, true, bins,
				 "Histogram of attempted insert keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_INBOUND_INSERT_DIST, "keys", false);

		return;
	    }
	    
	    if (uri.endsWith("inbound_request_histogram_data_detail.txt")) {
		int[] bins = null;
		if (node.requestDataDistribution != null) {
		    bins = node.requestDataDistribution.getBiggerBins();
		}

		sendKeyHistogram(resp, true, bins,
				 "Histogram of requested keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_REQUEST_DIST, "keys", false);
		return;
	    }
	    
	    if (uri.endsWith("inbound_insert_histogram_data_detail.txt")) {
		int[] bins = null;
		if (node.requestInsertDistribution != null) {
		    bins = node.requestDataDistribution.getBiggerBins();
		}

		sendKeyHistogram(resp, true, bins,
				 "Histogram of attempted insert keys.",
				 "This count has nothing to do with the keys in your datastore.",
				 MSG_NO_INBOUND_INSERT_DIST, "keys", false);
		return;
	    }
	    
	    if (uri.endsWith("inbound_success_histogram_data.txt")) {
		int[] bins = null;
		if (node.successDataDistribution != null) {
		    bins = node.successDataDistribution.getBins();
		}
		
		sendKeyHistogram(resp, true, bins,
				 "Histogram of successfully requested keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_REQUEST_DIST, "keys", false);
		return;
	    }

	    if (uri.endsWith("inbound_insert_success_histogram_data.txt")) {
		int[] bins = null;
		if (node.successInsertDistribution != null) {
		    bins = node.successInsertDistribution.getBins();
		}

		sendKeyHistogram(resp, true, bins,
				 "Histogram of externally originated successfully inserted keys",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_INSERT_SUCCESS_DIST, "keys", false);
		return;
	    }

	    if (uri.endsWith("inbound_success_histogram_data_detail.txt")) {
		int [] bins = null;
		if (node.successDataDistribution != null) {
		    bins = node.successDataDistribution.getBiggerBins();
		}

		sendKeyHistogram(resp, true, bins,
				 "Histogram of successfully requested keys.",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_REQUEST_DIST, "keys", false);
		return;
	    }

	    if (uri.endsWith("inbound_insert_success_histogram_data_detail.txt")) {
		int [] bins = null;
		if (node.successInsertDistribution != null) {
		    bins = node.successInsertDistribution.getBiggerBins();
		}

		sendKeyHistogram(resp, true, bins,
				 "Histogram of externally originated successfully inserted keys",
				 "This count has nothing to do with keys in your datastore",
				 MSG_NO_INSERT_SUCCESS_DIST, "keys", false);
		return;
	    }

	    if (uri.endsWith("psuccess_data.txt")) {
		sendPSuccessList(resp, false, false);
		return;
	    }
	    
	    if (uri.endsWith("psuccess_data_detail.txt")) {
		sendPSuccessList(resp, true, false);
	    	return;
	    }
	    
	    if (uri.endsWith("psuccess_insert_data.txt")) {
		sendPSuccessList(resp, false, true);
		return;
	    }

	    if (uri.endsWith("psuccess_insert_data_detail.txt")) {
		sendPSuccessList(resp, true, true);
		return;
	    }

            if (uri.endsWith("noderefs.txt")) {
                sendRefList(req, resp);
            }
            else if (uri.endsWith("nodestatus.html")) {
                sendStatusPage(resp);
            }
            else if (uri.endsWith("tickerContents.html")) {
                sendTickerContents(resp);
            }
            else if (uri.endsWith("ocmContents.html")) {
                sendOcmContents(resp,req);
            } else if (uri.endsWith("diagnostics/index.html")) {
                sendDiagnosticsIndex(resp);
            }
            else if (uri.indexOf(baseURL + "diagnostics/graphs") != -1) {
                sendGraphData(req, resp);
            }
            else if (uri.indexOf(baseURL + "diagnostics/") != -1) {
                // REDFLAG: clean up handling of bad urls
                int pos = uri.indexOf(baseURL + "diagnostics/") +
                                      (baseURL + "diagnostics/").length();
                int endName = uri.indexOf("/", pos);
                String varName = uri.substring(pos, endName);
                String period = uri.substring(endName + 1);
                sendVarData(resp, varName, period);
            }
            else {
                sendIndexPage(resp, baseURL);
            }
        }
        catch (URLEncodedFormatException uee) {
            // hmmm... underwhelming
            throw new IOException(uee.toString());
        }
    }

    private void reportNodeStats(PrintWriter pw) throws IOException {
        if (node == null) {
            return;
        }

        pw.println("<li> Uptime: &nbsp; " + getUptime() + " <br> "); 
	if (diagnostics != null)
            pw.println("<li> Current routingTime:  " + (long)diagnostics.getValue("routingTime", Diagnostics.MINUTE, Diagnostics.MEAN_VALUE ) + "ms. <br> ");

        // Thread load
        int jobs = node.activeJobs();

        // It's not just thread based.  There's also a hard
        // rate limit.
        boolean rejectingRequests = node.rejectingRequests();
        boolean rejectingConnections = node.rejectingConnections();

        if (jobs > -1) {
            String color = "black";
            String comment = "";
            if (rejectingConnections) {
                color = "red"; 
                comment = " &nbsp; <b> [Rejecting incoming connections and requests!] </b> ";
            }
            else if (rejectingRequests) {
                color = "red"; 
                comment = " &nbsp; <b> [QueryRejecting all incoming requests!] </b> ";
            }

            String msg = Integer.toString(jobs);

            int maxthreads = node.getThreadFactory().maximumThreads();
            if (maxthreads > 0) {
                float threadPoolLoad = (((float)jobs) / maxthreads) * 100.0f;
                msg += " &nbsp; &nbsp; (" +
                        Float.toString(threadPoolLoad) + "%) ";
            }

            msg += " <font color=\"" + color + "\"> " +
                comment + " </font>  <br>";
            
            pw.println("<li> Active pooled jobs:  " + msg); 

            if (rejectingConnections || rejectingRequests) {
                pw.println("It's normal for the node to sometimes reject connections or requests");
                pw.println("for a limited period.  If you're seeing rejections continuously the node");
                pw.println("is overloaded or something is wrong (i.e. a bug).");
            }

        }
	float f = node.estimatedLoad();
	pw.println("<li> Current estimated load:  " + f*100 + "%. <br> ");

        pw.println("<p>");
    }


    private final static String appendIntervalToMsg(long count, 
                                                    String msg,
                                                    String singular, String plural ) {

        if (count == 1) {
            return msg + " " + Long.toString(count) + " " + singular;
        }
        else {
            return msg + " " + Long.toString(count) + " " + plural;
        }
    }

    private String getUptime() {

        long deltat = (System.currentTimeMillis() - Node.startupTimeMs) / 1000;

        long days = deltat / 86400l;

        deltat -= days * 86400l;

        long hours = deltat / 3600l;

        deltat -= hours * 3600l;

        long minutes = deltat/ 60l;

        String msg = null;
        msg = appendIntervalToMsg(days, "", "day, &nbsp; ", "days, &nbsp;");
        msg = appendIntervalToMsg(hours, msg, "hour, &nbsp; ", "hours, &nbsp; ");
        msg = appendIntervalToMsg(minutes, msg, "minute", "minutes");
        
        if (msg.equals("")) {
            return " &nbsp; < 1 minute ";
        }

        return msg;
    }

    private void sendIndexPage(HttpServletResponse resp, String baseURL) throws IOException {
	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/html");
        PrintWriter pw = resp.getWriter();

        pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
        pw.println("<html>");
	pw.println("<head>");
	pw.println("<title>");
	pw.println("Node Status Info");
	pw.println("</title>");
	pw.println("</head>");
        pw.println("<body>");
	pw.println("<h1>Node Status Info</h1>");

        pw.println("<ul>");

        // Knows to send list items.
        reportNodeStats(pw);

        pw.println("    <li> <a href=\"" + baseURL +"nodestatus.html\"> Node Reference Status </a><p>");
        pw.println("    <li> Histograms ");
        pw.println("        <ul>");

        pw.println("            <li> <a href=\"" + baseURL + "key_histogram.txt\">" +
                   " Histogram of the keys in the node's routing table. </a>" +
                   "<a href=\"" + baseURL + "key_histogram_detail.txt\">(detail)</a><br>");
        pw.println("            <a href= \"" + baseURL + "key_histogram_data.txt\">(flat ascii)</a>" +
                   "<a href=\"" + baseURL + "key_histogram_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "inbound_request_histogram.txt\">" +
                   " Histogram of inbound request search keys. </a>" +
                   "<a href=\"" + baseURL + "inbound_request_histogram_detail.txt\">(detail)</a><br>");
        pw.println("            <a href= \"" + baseURL + "inbound_request_histogram_data.txt\">(flat ascii)</a>" +
                   "<a href=\"" + baseURL + "inbound_request_histogram_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL +"inbound_insert_histogram.txt\">" +
                   " Histogram of inbound insert search keys. </a>" +
                   "<a href=\"" + baseURL + "inbound_insert_histogram_detail.txt\">(detail)</a><br>");
        pw.println("            <a href= \"" + baseURL + "inbound_insert_histogram_data.txt\">(flat ascii)</a>" +
                   "<a href=\"" + baseURL + "inbound_insert_histogram_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "ds_histogram.txt\">" +
                   " Histogram of keys in the local DataStore. </a>" +
		   "<a href=\"" + baseURL + "ds_histogram_detail.txt\">(detail)</a><br>");
        pw.println("            <a href= \"" + baseURL + "ds_histogram_data.txt\">(flat ascii)</a>" +
                   "<a href=\"" + baseURL + "ds_histogram_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "ds_size_histogram.txt\">" +
                   " Histogram of sizes of keys in local DataStore. </a><br>");
        pw.println("<a href= \"" + baseURL + "ds_size_histogram_data.txt\">(flat ascii)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "inbound_success_histogram.txt\">" +
                   " Histogram of successful inbound request search keys. </a>" +
                   "<a href=\"" + baseURL + "inbound_success_histogram_detail.txt\">(detail)</a><br>");
        pw.println("            <a href=\"" + baseURL + "inbound_success_histogram_data.txt\">(flat ascii)</a>" +
		   "<a href=\"" + baseURL + "inbound_success_histogram_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "inbound_insert_success_histogram.txt\">" +
                   " Histogram of successful inbound insert requests. </a>" +
                   "<a href=\"" + baseURL + "inbound_insert_success_histogram_detail.txt\">(detail)</a><br>");
        pw.println("            <a href=\"" + baseURL + "inbound_insert_success_histogram_data.txt\">(flat ascii)</a>" +
		   "<a href=\"" + baseURL + "inbound_insert_success_histogram_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "psuccess_data.txt\">" +
                   " Probability of success of an incoming request. </a>"+
		   "<a href=\"" + baseURL + "psuccess_data_detail.txt\">(detail)</a><br>");

        pw.println("            <li> <a href=\"" + baseURL + "psuccess_insert_data.txt\">" +
                   " Probability of success of an incoming insert. </a>"+
		   "<a href=\"" + baseURL + "psuccess_insert_data_detail.txt\">(detail)</a><br>");
	pw.println("            <li> <a href=\"" + baseURL + "version_histogram.txt\">" +
		   " Histogram of the versions of nodes in the routing table. </a>" +
		   "            <a href=\"" + baseURL + "version_data.txt\">(flat ascii)</a><br>");
        pw.println("        </ul> <p>");
        pw.println("    <li> <a href=\"" + baseURL +"diagnostics/index.html\"> Diagnostics Values </a> <p>");
        pw.println("    <li> <a href=\"" + baseURL +"inboundContacts.txt\"> Inbound Contact Attempts </a> <br>");
        pw.println("         This shows the per host statistics for inbound connections to the");
        pw.println("         node's FNP interface. <p>");
        pw.println("    <li> <a href=\"" + baseURL +"outboundContacts.txt\"> Outbound Contact Attempts </a> <br>");
        pw.println("         This shows the per host statistics for outbound FNP connections that");
        pw.println("         the node attempts to make to other nodes. There are a couple of things");
        pw.println("         to keep in mind when comparing these numbers to the connection ");
        pw.println("         statistics under <a href=\"" + baseURL +"nodestatus.html\"> Node Reference Status </a>.");
        pw.println("         <p> First these statistics are for <em>physical connections</em> attempts");
        pw.println("         in contrast to the NRS numbers which include contacts made over cached");
        pw.println("         connections.");
        pw.println("         <p> Second, the NRS statistics are only for outbound routing attempts");
        pw.println("         whereas these numbers include all outbound connections.");
        pw.println("         <p>");

        pw.println("    <li> <a href=\"" + baseURL +"inboundRequests.txt\"> Inbound Requests </a> <br>");
        pw.println("         This shows per host aggregated (DataRequest, ");
        pw.println("         InsertRequest, AnnouncementRequest) inbound requests received on the node's");
        pw.println("         FNP interface.");
        pw.println("<p>");

        pw.println("    <li> <a href=\"" + baseURL +"outboundRequests.txt\"> Outbound Requests </a> <br>");
        pw.println("         This shows per host aggregated (DataRequest, ");
        pw.println("         InsertRequest, AnnouncementRequest) requests that your node sent to");
        pw.println("         other nodes.");
        pw.println("<p>");
        pw.println("    <li> <a href=\"" + baseURL +"tickerContents.html\"> Ticker Contents </a> <br>");
        pw.println("         This shows events scheduled on the nodes ticker.");
        pw.println("<p>");
        pw.println("    <li> <a href=\"" + baseURL +"ocmContents.html\"> Connection Manager Contents </a> <br>");
        pw.println("         This shows the connections cached in the nodes Open Connection Manager.");

        pw.println("<p>");
        pw.println("    <li> <a href=\"" + baseURL +"loadStats.txt\"> Global network load stats </a> <br>");
        pw.println("         This shows hourly inbound request rates from other nodes.  ");
        pw.println("         For now it's for informational purposes only.  Eventually these data may be");
        pw.println("         used to implement intra-node load balancing.");
        pw.println("<p>");
        pw.println("</ul>");
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer();
    }

    private final static String drawLine(int freq) {
        String ret = "";
        for (int i = 0; i < freq; i++) {
            ret += "=";
        }
        return ret;
    }

    private final static int[] fillBins(Key[] keys, boolean detail) {
        // Thelema, I changed this back because it makes it hard to see at a glance
        // what's going on. --gj
	// One *can't* see at a glance what's going on.  
	// 16 buckets makes the histogram *very* deceptive, 
	// as the groups you're comparing over are too big
	int[] bins;
	if(detail)
	{
            bins = new int[256];
	}
	else
	{
	    bins = new int[16];
	};
        int i = 0;
        for (i = 0; i < keys.length; i++) {
	    int binNumber;

	    if(detail)
	    {
                // most significant byte.
                binNumber = (((int)keys[i].getVal()[0]) & 0xff);
	    }
	    else
	    {
                // most significant nibble
                binNumber = (((int)keys[i].getVal()[0]) & 0xff)  >>> ((byte)4);            
	    }
            bins[binNumber]++;
        }
        return bins;
    }


    private String peakValue(int[] bins, int index, float mean, String[] names) {
        // hmmm... Allow edges to count as peaks?
        int nextCount = 0;
        int prevCount = 0;

        if (index > 0) {
            prevCount = bins[index - 1];
        }

        if (index < bins.length - 1) {
            nextCount = bins[index + 1];
        }

        if ((bins[index] > prevCount) &&
            (bins[index] > nextCount)) {
            return (names != null ? names[index] : Integer.toString(index, 16))
		+ " --> (" + (((float)bins[index]) / mean) + ")\n";
        }
	
        return null;
    }


    private void sendStats(PrintWriter pw, int[] bins, String[] names) {
        int max = 0;
        int sum = 0;
        int i = 0;
        for (i = 0; i < bins.length; i++) {
            if (bins[i] > max) {
                max = bins[i];
            }
            sum += bins[i];
        }
        
        float mean = ((float)sum) / bins.length;

	if(names != null) {
	    pw.println("mean: " + mean);
	    pw.println("");
	}
	
        if (mean < 1.0f) {
            return;
        }

        String text = "";
        for (i = 0; i < bins.length; i++) {
            String peakValue = peakValue(bins, i, mean, names);
            if (peakValue != null) {
                text += peakValue; 
            }
        }

        if (!text.equals("")) {
            pw.println("peaks (count/mean)");
            pw.println(text);
        }
    }

    private void sendRTHistogram(HttpServletResponse resp, boolean justData, boolean detail) throws IOException {
        final Key[] keys = rt.getSnapshot().keys();
        int[] bins = fillBins(keys, detail);
        sendKeyHistogram(resp, justData, bins, 
                         "Histogram of keys in in fred's Routing table",
                         "This count has nothing to do with keys in your datastore, " +
                         "these keys are used for routing",
                         MSG_OOPS /* bins should always be non-null */,
			 "keys", false);
                         
    }
    
    private void sendVersionHistogram(HttpServletResponse resp, boolean justData)
	throws IOException {
	RTDiagSnapshot status = rt.getSnapshot();
	final StringMap[] refData = status.refData();
	TreeMap map = new TreeMap();
	int longestString = 0;
	int shortestString = Integer.MAX_VALUE;
	if (refData == null || refData.length < 1) {
	    // Do nothing
	} else {
	    for(int j = 0; j < refData.length; j++) {
		String s = (String)(refData[j].value("Node Version"));
		if(s != null) {
		    if(s.length() > longestString) longestString = s.length();
		    if(s.length() < shortestString) shortestString = s.length();
		    Integer i = (Integer) (map.get(s));
		    if(i != null) {
			i = new Integer(i.intValue() + 1);
		    } else i = new Integer(1);
		    map.put(s, i);
		}
	    }
	}
	Set s = map.keySet();
	String[] names = (String[])s.toArray(new String[s.size()]);
	int[] bins = new int[map.size()];
	for(int i=0;i<map.size();i++) {
	    bins[i] = ((Integer)(map.get(names[i]))).intValue();
	}
	if(shortestString != longestString) {
	    for(int x=0;x<names.length;x++) {
		String st = names[x];
		while(st.length() < longestString) st += ' ';
		names[x] = st;
	    }
	}
	sendKeyHistogram(resp, justData, bins, names,
			 "Histogram of node versions in fred's Routing table",
			 "", MSG_OOPS, "nodes", false);
    }
    
    private void sendDSHistogram(HttpServletResponse resp, boolean justData, boolean detail) throws IOException {

        FSDataStore ds = (FSDataStore) node.ds;
        KeyHistogram histogram = ds.getHistogram();;

        sendKeyHistogram(resp, justData, detail ? histogram.getBiggerBins() : histogram.getBins(), 
                         "Histogram of keys in in fred's data store",
                         "These are the keys to the data in your node's " +
                         "local cache (DataStore)",
                         MSG_OOPS /* bins should always be non-null */,
			 "keys", false);
                         
    }

    private void sendDSSizeHistogram(HttpServletResponse resp, boolean justData) throws IOException {

	FSDataStore ds = (FSDataStore) node.ds;
	KeySizeHistogram histogram = ds.getSizeHistogram();

	sendKeyHistogram(resp, justData, histogram.getBins(),
		   "Histogram of sizes of keys in fred's data store",
		   "These are the numbers of keys in your DataStore" +
		   " of (roughly) each size",
		   MSG_OOPS /* bins should always be non-null */, "keys",
		   true);
    }

    private void sendPSuccessList(HttpServletResponse resp, boolean detail, boolean inserts) throws IOException
    {
	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/plain");
	PrintWriter pw = resp.getWriter();
	pw.println("Probability of success of an incoming request");
	if(inserts) {
	    boolean b = false;
	    if(node.successInsertDistribution == null) {
		pw.println(MSG_NO_INSERT_SUCCESS_DIST);
		b = true;
	    }
	    if(node.requestInsertDistribution == null) {
		pw.println(MSG_NO_INBOUND_INSERT_DIST);
		b = true;
	    }
	    if(b) {
		resp.flushBuffer();
		return;
	    }
	}
	DateFormat df = DateFormat.getDateTimeInstance();
	String date = df.format(new Date(System.currentTimeMillis()));
	pw.println(date);
	float pMax = 0;
	int iMax = node.binLength(detail);
	for (int i = 0; i < iMax; i++) {
	    float p = node.pSuccess(i, detail, inserts);
	    if(Float.isNaN(p))
	    {
		pw.println(Integer.toString(i, 16) + " | -");
	    }
	    else
	    {
	    	pw.println(Integer.toString(i, 16) + " | " + p);
	    }
	    if(p > pMax) pMax = p;
	};
	pw.println("");
	pw.println("Max: " + pMax);
	int x = node.binMostSuccessful(detail, inserts);
	if(x != -1) pw.println("Most successful: " + Integer.toString(x,16));
	resp.flushBuffer();
    }
    
    private void sendKeyHistogram(HttpServletResponse resp, boolean justData, 
				  int[] bins, String title, String comment,
                                  String complaint, String commodity,
				  boolean exponential) throws IOException {
	sendKeyHistogram(resp, justData, bins, null, title, comment, complaint,
			 commodity, exponential);
    }

    private void sendKeyHistogram(HttpServletResponse resp, boolean justData, 
				  int[] bins, String[] names,
                                  String title, String comment,
                                  String complaint, String commodity,
				  boolean exponential) throws IOException {
	
        if (bins == null) {
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType("text/plain");
            PrintWriter pw = resp.getWriter();
            pw.println(complaint);
            resp.flushBuffer(); 
            return;
        }

        int maximum = 0;
        int count = 0;
        int i = 0;
        for (i = 0; i < bins.length; i++) {
            count += bins[i];
            if (maximum < bins[i]) {
                maximum = bins[i];
            }
        }

        double scale = 1.0f;
        if (maximum > 64) {
            // REDFLAG: test
            // Scale factor for ASCII limits line lengths to 64
            scale = 64.0f / maximum; 
        }

        DateFormat df = DateFormat.getDateTimeInstance();
        String date = df.format(new Date(System.currentTimeMillis()));

        resp.setStatus(HttpServletResponse.SC_OK);
  	resp.setContentType("text/plain");
  	PrintWriter pw = resp.getWriter();
        
        if (justData) {
            pw.println("# " + title);
            pw.println("# " + date);
            pw.println("# " + commodity + ": " + count);

        }
        else {
            pw.println(title);
	    pw.println(comment);
            pw.println(date);
            pw.println(commodity + ": " + count);
            pw.println("scale factor: " + scale + " (This is used to keep lines < 64 characters)");
            pw.println("");
        }
        for (i = 0; i < bins.length; i++) {
	    String line;
	    if(names != null) {
		line = names[i];
	    } else if(exponential) {
		line = shortPowerString(10+i);
		if(i == (bins.length-1)) {
		    line += "+";
		};
	    } else {
		line = Integer.toString(i, 16);
	    };
            if (justData) {
		line += "\t" + bins[i];
            } else {
		if(!exponential || (i != (bins.length-1))) line += " ";
		while(line.length()<5) { line = " " + line; };
                line += "|" + drawLine(((int)(bins[i] * scale)));
            }
	    pw.println(line);
        }
	
        pw.println("");
	
        if (!justData) {
            sendStats(pw, bins, names);
        }
	
        resp.flushBuffer(); 
    }

    private String shortPowerString(int power)
    {
	int x = power % 10;
	if(x==10) x = 0;
	String ret = Integer.toString(1 << x);
	char SI[] = {'k','M','G','T','P','E'};
	if(power >= 10)
	{
		ret += SI[(power/10)-1];
	};
	return ret;
    };
    
    private void sendStatusPage(HttpServletResponse resp) throws IOException {
        long now = System.currentTimeMillis();
        DateFormat df = DateFormat.getDateTimeInstance();
        String date = df.format(new Date(now));

        RTDiagSnapshot status = rt.getSnapshot();
        final StringMap tableData = status.tableData();
        final StringMap[] refData = status.refData();
        String typeName = null;
        
        if (tableData != null) {
            typeName = (String)tableData.value("Implementation");
        }

  	resp.setStatus(HttpServletResponse.SC_OK);
  	resp.setContentType("text/html");
  	PrintWriter pw = resp.getWriter();

        pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
        pw.println("<html>");
  	pw.println("<head>");
  	pw.println("<title>");
  	pw.println("Routing Table status: " + date);
  	pw.println("</title>");
  	pw.println("</head>");
        pw.println("<body>");
  	
  	pw.println("<h1>Routing Table status: " + date + " </h1>");
        pw.println("<p>");

        makePerTableData(status, tableData, pw);

        makeRefDownloadForm(refData, pw);

	pw.println("<p><div align=\"center\"><img src=\"/servlet/nodeinfo/internal/rthist/rthist.bmp?length=800&height=50\"></div></p>");
	
        pw.println("<p>");

  	pw.println("<table border>"); 
        
        makeTableHeader(status, pw);

        makeRefRowEntries(refData, typeName, pw);
        
        pw.println("</table>");
        
	pw.println("<table>");
	String[] colors = { "blue", "green", "", "red" };
	String[] meanings = { "Fetching ARK", "OK", "Not tried yet", 
			      "Failing" };
	for(int x=0; x<colors.length;x++) {
	    StringBuffer s = new StringBuffer("<tr><td>");
	    if(!colors[x].equals(""))
		s.append("<font color=\"").append(colors[x]).append("\">");
	    if(colors[x].equals(""))
		s.append("default");
	    else
		s.append(colors[x]);
	    if(!colors[x].equals(""))
		s.append("</font>");
	    s.append("<td><b>").append(meanings[x]).append("</b></td></tr>");
	    pw.println(new String(s));
	}

	pw.println("</table>");

  	pw.println("</body>");
  	pw.println("</html>");

  	resp.flushBuffer();
    }



    public static void makePerTableData(RTDiagSnapshot status, StringMap data, 
					PrintWriter pw) {

        final String[] keys = data.keys();
        final Object[] values = data.values();
        if ((keys == null) && (values == null)) {
            return;
        }
        pw.println("<table border=\"0\">");
        for (int i=0; i < keys.length; i++) {
            pw.println("<tr><td>" + keys[i] + "</td><td>" + values[i] + "</td></tr>");
        }
	pw.println("</table>");
    }
    
    private static final String PROP_CP = "Contact Probablitity";
    private static final String PROP_CONNECTIONS = "Successful Connections";
    private static final String PROP_NODEREF = "NodeReference";

    private void makeRefDownloadForm(StringMap[] refs, PrintWriter pw) {

        if ((refs == null) || (refs.length < 1)) {
            return;
        }
        
        boolean hasCPs = refs[0].value(PROP_CP) != null;
        boolean hasConns = refs[0].value(PROP_CONNECTIONS) != null;

        if (!(hasCPs || hasConns)) {
            return;
        }

        pw.println("<form action=\"noderefs.txt\" method=\"Get\">");

        if (hasCPs) {
            pw.println("<b>Minimum contact probability:</b> ");
            pw.println(" <input size=\"5\" name=\"minCP\" value=\"0.0\"> <br>");
        }

        if (hasConns) {
            pw.println("<b>Minimum successful connections:</b> ");
            pw.println(" <input size=\"5\" name=\"minConnections\" value=\"1\"> <p> ");
        }

        pw.println(" <input type=\"submit\" value=\"Download References\">");
        pw.println("</form>");
    }

    private void makeRefRowEntries(StringMap[] refs, 
                                   String typeName, PrintWriter pw) {
        
        if ((refs == null) || (refs.length < 1)) {
            return;
        }

        for (int i = 0; i < refs.length; i++) {
            final String[] keys = refs[i].keys();
            final Object[] values = formatRef(typeName, refs[i].values());
            pw.println("<tr>");
            for (int j = 0; j < values.length; j++) {
                // Skip noderef values since we can't render them.
                if (keys[j].equals("NodeReference")) {
                    continue;
                }
		if (keys[j].equals("ARK URL")) {
		    continue;
		}
                pw.println("    <td> " + values[j] + " </td>");

            }
            pw.println("</tr>");
        }
    }

    private void makeTableHeader(RTDiagSnapshot status, PrintWriter pw) {
        final StringMap[] refs = status.refData();
        
        if ((refs == null) || (refs.length < 1)) {
            return;
        }

        pw.println("<tr>");
        final String[] keys = refs[0].keys();

        if (keys == null) {
            pw.println("</tr>");
            return;
        }

        for (int j = 0; j < keys.length; j++) {
            // Skip noderef values since we can't render them.
            if (keys[j].equals("NodeReference")) {
                continue;
            }
	    if (keys[j].equals("ARK URL")) {
		continue;
	    }
            pw.println("    <td> <b> " + keys[j] + " </b> </td>");
        }
        pw.println("</tr>");
    }

    private void sendRefList(HttpServletRequest req, HttpServletResponse resp) 
        throws IOException {
        float minCP = (float)0.0;
        try {
            String minCPAsString = req.getParameter("minCP");
            if (minCPAsString != null) {
                minCPAsString = freenet.support.URLDecoder.decode(minCPAsString);
                minCP = Float.valueOf(minCPAsString).floatValue();
            }
        }
        catch (NumberFormatException nfe) {
        }
        catch (URLEncodedFormatException uefe) {
        }

        if (minCP > 1.0 ) {
            minCP = (float)1.0;
        }
        if (minCP < 0.0) {
            minCP = (float)0.0;
        }

        int minConnections = 0;
        try {
            String minConnectionsAsString = req.getParameter("minConnections");
            if (minConnectionsAsString != null) {
                minConnectionsAsString = freenet.support.URLDecoder.decode(minConnectionsAsString);
                minConnections = Integer.parseInt(minConnectionsAsString);
            }
        }
        catch (NumberFormatException nfe) {
        }
        catch (URLEncodedFormatException uefe) {
        }
                
        if (minConnections < 0) {
            minConnections = 0;
        }

        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/plain");

	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
        final StringMap[] refs = rt.getSnapshot().refData();
        if ((refs == null) || (refs.length < 1)) {
            resp.getWriter().println(""); // so that document is not empty.
            resp.flushBuffer();
	    if(logDEBUG)
		Core.logger.log(this, "sendRefList returning empty because no refs",
				Logger.DEBUG);
            return;
        }
	
        // REDFLAG: doc key / type assumptions

        boolean hasCPs = (refs[0].value(PROP_CP) != null) && 
            (refs[0].value(PROP_CP) instanceof Float);
        boolean hasConns = (refs[0].value(PROP_CONNECTIONS) != null) &&
            (refs[0].value(PROP_CONNECTIONS) instanceof Integer);
        
        WriteOutputStream out = null;
        out = new WriteOutputStream(resp.getOutputStream());
        
        int count = 0;
        for (int i = 0; i < refs.length; i++) {
            if (hasCPs) {
                float cp =  ((Float)refs[i].value(PROP_CP)).floatValue();
                if (cp < minCP) {
		    if(logDEBUG)
			Core.logger.log(this, "sendRefList skipping node "+i+" of "+
					refs.length+" because CP = "+cp,
					Logger.DEBUG);
                    continue;
                }
            }

            if (hasConns) {
                int conns =  ((Integer)refs[i].value(PROP_CONNECTIONS)).intValue();
                if (conns < minConnections) {
		    if(logDEBUG)
			Core.logger.log(this, "sendRefList skipping node "+i+" of "+
					refs.length+" because conns = "+conns,
					Logger.DEBUG);
		    continue;
                }
            }
            
            NodeReference ref = (NodeReference)refs[i].value(PROP_NODEREF);
	    if(ref.noPhysical()) {
		if(logDEBUG)
		    Core.logger.log(this, "sendRefList skipping node "+i+" of "+
				    refs.length+" because noPhysical", Logger.DEBUG);
		continue; // not much use without a physical addr
	    }
            ref.getFieldSet().writeFields(out);
            count++;
        }
    
        // Don't send an empty document
        // error in browser.
        if (count == 0) {
            out.write('\n');
        }

        resp.flushBuffer(); 
    }

    // REDFLAG: C&P from fproxy
    private void sendError(HttpServletResponse resp, int status,
                             String detailMessage)
        throws IOException {

        // get status string
        String statusString = status + " " +
            HttpServletResponseImpl.getNameForStatus(status);

        // show it
	if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(this, "Sending HTTP error: " + statusString,
			    Logger.DEBUGGING);
        PrintWriter pw = resp.getWriter();
        resp.setStatus(status);
        resp.setContentType("text/html");
        pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
        pw.println("<html>");
        pw.println("<head><title>" + statusString + "</title></head>");
        pw.println("<body bgcolor=\"#ffffff\">");
        pw.println("<h1>" + statusString + "</h1>");
        pw.println("<p>" + detailMessage);
        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer(); 
    }

    ///////////////////////////////////////////////////////////
    // Support for printing contents of ticker and ocm.


    private final void sendTickerContents(HttpServletResponse resp) throws IOException {
        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Freenet Node Ticker Contents");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        node.ticker().writeEventsHtml(pw);
        pw.println("</body>");
        pw.println("</html>");
    }

    private final void sendOcmContents(HttpServletResponse resp,HttpServletRequest req) throws IOException {
        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Open ConnectionManager Contents");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        node.connections.writeHtmlContents(pw, req);
        pw.println("</body>");
        pw.println("</html>");
    }

    ////////////////////////////////////////////////////////////
    // Support for sending Diagnostics.

    private final void sendVarData(HttpServletResponse resp, 
                                   String varName, 
                                   String period) throws IOException {
        PrintWriter pw = resp.getWriter();

        DiagnosticsFormat format;
        boolean html = false;
        if (period.equalsIgnoreCase("occurrences")) {
            html = true;
            resp.setContentType("text/html");
            format =  new HtmlDiagnosticsFormat(-1);
        } else if (period.equalsIgnoreCase("raw")) {
            resp.setContentType("text/plain");
            format = new RowDiagnosticsFormat();
        } else if (period.startsWith("raw")) {
            resp.setContentType("text/plain");
            if (period.substring(3).equalsIgnoreCase("occurences"))
                format = new RowDiagnosticsFormat(-1);
            else
                format = 
                    new RowDiagnosticsFormat(Diagnostics.getPeriod(period.substring(3)));

        } else if (period.equalsIgnoreCase("fieldset")) {
            resp.setContentType("text/plain");
            format = new FieldSetFormat();
        } else {
            try {
                resp.setContentType("text/html");
                html = true;
                format = 
                    new HtmlDiagnosticsFormat(Diagnostics.getPeriod(period));
            } catch (IllegalArgumentException e) {
                sendError(resp, 404, "Unknown period type given.");
                return;
            }
        }

        try {
            if (html) {
                pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
                pw.println("<html><head><title>"+varName+"("+period+
			   ")</title></head><body>");
            }
            pw.println(diagnostics.writeVar(varName, format));
            if (html)
                pw.println("</body></html>");
        } catch (NoSuchElementException e) {
            sendError(resp, 404, "No such diagnostics field");
            return;
        }

        resp.flushBuffer();
    }
   
    private final void sendGraphData(HttpServletRequest req,
                                     HttpServletResponse resp) {
        try {
            String varName = req.getParameter("var");
            
            if (varName == null)
            {
                // variable is mandatory, bounce our confused user to the diagnostics
                // index so they can pick one
                //
                sendDiagnosticsIndex(resp);
                return;
            }
            
            GraphRange gr;
            
            String period = req.getParameter("period");
            
            if (period == null)
                period = "minute";
                
            final int p = period.equalsIgnoreCase("occurrences") ? -1 : Diagnostics.getPeriod(period);
                
            int type;
            
            try {
                type = Integer.parseInt(req.getParameter("type"));
            } catch (NumberFormatException e) {
                type = 0; // 0 == guess
            }
                    
            try {
                // try getting the range out of the query string
                gr = new GraphRange(req.getParameter("range"));
            } catch (IllegalArgumentException e) {
                // failing that, parse it from the data we're formatting
                GraphRangeDiagnosticsFormat grdf = 
                    new GraphRangeDiagnosticsFormat(p, type);
                                            
                diagnostics.writeVar(varName, grdf);
                
                gr = grdf.getRange();
            }
            
            type = gr.getType();
            
            String ctype = req.getParameter("content-type");
            
            if (ctype != null)
                try {
                    ctype = freenet.support.URLDecoder.decode(ctype);
                } catch (URLEncodedFormatException e) {
                    ctype = null;
                }
                
            BitmapEncoder be = BitmapEncoder.createByMimeType(ctype);
            
            if (be == null)
            {       
                // default to html
                PrintWriter pw = resp.getWriter();
                
                resp.setContentType("text/html");
                
                String itype = req.getParameter("image-type");
                // todo: get default from config
                if (itype == null) {itype = "image/x-ms-bmp"; }
               
                pw.println(diagnostics.writeVar(varName, new GraphHtmlDiagnosticsFormat(p, type, gr, itype)));
            } else {
                // output the image
                OutputStream os = resp.getOutputStream();
                
                resp.setContentType(be.getMimeType());
                
                diagnostics.writeVar(varName, new GraphDiagnosticsFormat(p, be, os, type, gr));
            }
            
            resp.flushBuffer();                           
        } catch (Throwable t) {
            Core.logger.log(this, "Grapher threw.", t, Logger.ERROR);
        }
    }


    private final void sendDiagnosticsIndex(HttpServletResponse resp)
        throws IOException {

        PrintWriter pw = resp.getWriter();
        resp.setContentType("text/html");

        pw.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">");
        pw.println("<html>");
        pw.println("<head>");
        pw.println("<title>");
        pw.println("Freenet Node Diagnostics Variables");
        pw.println("</title>");
        pw.println("</head>");
        pw.println("<body>");
        pw.println();
        pw.println();

        DiagnosticsFormat indexFormat = new HtmlIndexFormat();
        Core.diagnostics.writeVars(pw, indexFormat);

        pw.println("</body>");
        pw.println("</html>");
        resp.flushBuffer();
    }

    private final void sendLoadStats(HttpServletResponse resp)
        throws IOException {

	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/plain");
        loadStats.dump(resp.getWriter());
        resp.flushBuffer();
    }


    private final int getActive(ContactCounter.Record[] contacts) {
        int activeCount = 0;
        for (int i = 0; i < contacts.length; i++) {
            activeCount += contacts[i].activeContacts;
        }
        return activeCount;
    }

    private final void sendPerHostStats(HttpServletResponse resp, String kind)
        throws IOException {

	resp.setStatus(HttpServletResponse.SC_OK);
	resp.setContentType("text/plain");
        PrintWriter pw = resp.getWriter();
        ContactCounter.Record[] contacts = null;
        if (kind.equals("inboundContacts")) {
            if (inboundContacts != null) {
                contacts = inboundContacts.getSnapshot();
                pw.println("# unique contacts: " + contacts.length + 
                           ", active connections: " + getActive(contacts));
                
                pw.println("# format: <contact attempts> <successful> <active connections> <address>");
                pw.println("#");
            }
            else {
                pw.println("# Inbound contacts are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logInboundContacts=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }
        else if (kind.equals("outboundContacts")) {
            if (outboundContacts != null) {
                contacts = outboundContacts.getSnapshot();
                pw.println("# unique contacts: " + contacts.length + 
                           ", live connections: " + getActive(contacts));
                
                pw.println("# format: <contact attempts> <successful> <live connections> <address>");
                pw.println("#");
            } 
            else {
                pw.println("# Outbound contacts are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logOutboundContacts=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }
        else if (kind.equals("outboundRequests")) {
            if (outboundRequests != null) {
                contacts = outboundRequests.getSnapshot();
                pw.println("# unique hosts: " + contacts.length); 
                pw.println("# format: <requests> <address>");
                pw.println("#");
            }
            else {
                pw.println("# Outbound requests are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logOutboundRequests=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }
        else {
            if (inboundRequests != null) {
                contacts = inboundRequests.getSnapshot();
                pw.println("# unique hosts: " + contacts.length); 
                pw.println("# format: <requests> <requests accepted> <successful requests> <address>");
                pw.println("#");
            }
            else {
                pw.println("# Inbound requests are not being logged.");
                pw.println("# To enable logging set:");
                pw.println("#   logInboundRequests=true");
                pw.println("# in your freenet.conf / freenet.ini file.");
                resp.flushBuffer();
                return;
            }
        }

        if (contacts != null) {
            // Sort by count.
            HeapSorter.heapSort(new ArraySorter(contacts));
            
            for (int i = 0; i < contacts.length; i++) {
                if (kind.equals("outboundRequests")) {
                    pw.println( contacts[i].totalContacts + "\t" +
                                contacts[i].addr);
                   
                }
                else {
                    pw.println( contacts[i].totalContacts + "\t" +
                                contacts[i].successes + "\t" +
                                contacts[i].activeContacts + "\t"+
                                contacts[i].addr );
                }
            }
        }

        resp.flushBuffer();
    }

    ////////////////////////////////////////////////////////////
    // Hooks to provide nicer formatting for RoutingTable
    // implementations that we know about.
    private static final Long ZERO = new Long(0);
    private static final Boolean TRUE = new Boolean(true);
   
    Object[] formatRef(String rtType, Object[] refValues) {
        if (rtType == null) {
            return refValues;
        }
        
        if (rtType.equals("freenet.node.rt.CPAlgoRoutingTable")) {
            long attempts = ((Long)refValues[3]).longValue();
            long successful = ((Long)refValues[4]).longValue();
            if (attempts > 0 && successful > 0) {
                // percent successful
                refValues[4] = refValues[4].toString() + " &nbsp; &nbsp; ("
                    + (long) (100 * successful / (double)attempts) + "%)";
            }
            
	    boolean failing = true;
            if (refValues[2].equals(ZERO)) {
                refValues[2] = "none";
		failing = false;
            }

	    long ver = ((Long)refValues[9]).longValue();
	    NodeReference ref = (NodeReference)refValues[6];

	    String s = " <font color=\"";
	    String v = "";

	    if (refValues[10].equals(TRUE))
		v = "blue";  // Fetching ARK
	    else if (failing)
		v = "red";   // Failing
	    else if (!refValues[4].equals(ZERO))
		v = "green"; // Working fine

	    if(refValues[10].equals(TRUE)) {
		try {
		    if(ref.getARKURI(ver+1) == null) {
			refValues[10] = "broken"; // it will fail soon enough
		    } else {
			refValues[10] = "<a href=\"/"+
			    ref.getARKURI(ver+1).toString(false)+
			    "\">yes</a>";
		    }
		} catch (freenet.KeyException e) {
		    Core.logger.log(this, "NodeStatusServlet got "+e, e,
				    Logger.ERROR);
		    refValues[10] = "really broken!";
		}
	    } else {
		refValues[10] = "no";
	    }
	    
	    // Black means not connected yet 
	    
	    if(!v.equals(""))
		refValues[1] = s + v + "\"> " + refValues[1] + "</font> ";
	    
	    long lastTry = ((Long)refValues[5]).longValue();
            if (lastTry <= 0 || lastTry >= (1000*1000*1000)) {
		if(lastTry > 0)
		    Core.logger.log(this, "lastTry has ridiculous value "+
				    lastTry+" in formatRef", 
				    new Exception("debug"), Logger.NORMAL);
                refValues[5] = "never";
            } else {
                refValues[5] = refValues[5].toString() + " secs. ago";
            }
            
            // clean up version
            String verString = (String)refValues[7];
            int pos = verString.lastIndexOf(",");
            if (pos > -1 && pos < verString.length() - 1) {
                refValues[7] = verString.substring(pos + 1);
            }

	    if(refValues[11] != null && ver != 0) {
		refValues[9] = "<a href=\"/"+refValues[11]+"\">" + refValues[9] + "</a>";
	    }
	}
        
	final int y = 11;
	Object[] o = new Object[y];
	for(int x=0;x<y;x++)
	    o[x] = refValues[x];
	
	return o;
    }
    
    ////////////////////////////////////////////////////////////
    Node node = null;
    RoutingTable rt = null;
    Diagnostics diagnostics = null;
    ContactCounter inboundContacts = null;
    ContactCounter outboundContacts = null;
    ContactCounter inboundRequests = null;
    ContactCounter outboundRequests = null;
    KeyHistogram requestDataDistribution = null;
    LoadStats loadStats = null;
}






