package freenet;

import freenet.interfaces.Interface;
import freenet.interfaces.NIOInterface;
import freenet.transport.*;
import freenet.support.*;
import freenet.thread.*;
import freenet.session.*;
import freenet.crypt.*;
import freenet.config.*;
import freenet.diagnostics.*;
import java.util.*;
import java.io.*;


/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * This is a Wrapper object that contains the components of a Node in the
 * Adaptive Network.  It uses a Network object to communicate with other
 * Nodes in the Network.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 * @author <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 * @author oskar
 */

public class Core {

    public static final Config config = new Config();
    
    static {
        config.addOption("authTimeout",            1, 30000,  3100); // 30 sec
        config.addOption("connectionTimeout",      1, 600000, 3150); // 10 min
        config.addOption("hopTimeExpected",        1, 4000,   3200); // 4 sec
        config.addOption("hopTimeDeviation",       1, 7000,   3201); // 7 sec
        config.addOption("maximumThreads",         1, 120,    3250);
        config.addOption("blockSize",              1, 4096,   3300); // 4k
        config.addOption("streamBufferSize",       1, 16384,  3350); // 16 k
        config.addOption("maximumPadding",         1, 65536,  3400); // 64 k

        // authTimeout
        config.setExpert ("authTimeout", true);
        config.argDesc   ("authTimeout", "<millis>");
        config.shortDesc ("authTimeout", "timeout for crypto setup");
        config.longDesc  ("authTimeout",
            "How long to wait for authentication before giving up (in milliseconds)"
        );
        
        // connectionTimeout
        config.setExpert ("connectionTimeout", true);
        config.argDesc   ("connectionTimeout", "<millis>");
        config.shortDesc ("connectionTimeout", "timeout of idle connections.");
        config.longDesc  ("connectionTimeout",
            "How long to listen on an inactive connection before closing",
            "(if reply address is known)"                                        
        );
        
        // hopTimeExpected
        config.setExpert ("hopTimeExpected", true);
        config.argDesc   ("hopTimeExpected", "<millis>");
        config.shortDesc ("hopTimeExpected", "average time for each hop in routing");
        config.longDesc  ("hopTimeExpected",
            "The expected time it takes a Freenet node to pass a message.",
            "Used to calculate timeout values for requests."               
        );
        
        // hopTimeDeviation
        config.setExpert ("hopTimeDeviation", true);
        config.argDesc   ("hopTimeDeviation", "<millis>");
        config.shortDesc ("hopTimeDeviation", "std. devn. for hopTimeExpected");
        config.longDesc  ("hopTimeDeviation",
            "The expected standard deviation in hopTimeExpected."
        );
        
        // maximumThreads
        config.setExpert ("maximumThreads", true);
        config.argDesc   ("maximumThreads", "<integer>");
        config.shortDesc ("maximumThreads", "max. no. of threads in the pool");
        config.longDesc  ("maximumThreads",
            "Should we use thread management?  If this number is defined and non-zero,",
            "this specifies the max number of threads in the pool.  If this is overrun",
            "connections will be rejected and events won't execute on time."
        );
        
        // blockSize
        config.setExpert ("blockSize", true);
        config.argDesc   ("blockSize", "<bytes>");
        config.shortDesc ("blockSize", "size of byte blocks when copying data");
        config.longDesc  ("blockSize",
            "What size should the blocks have when moving data?"
        );

        // streamBufferSize
        config.setExpert ("streamBufferSize",true);
        config.argDesc   ("streamBufferSize","<bytes>");
        config.shortDesc ("streamBufferSize",
                          "The default size of stream buffers.");
        
        // maximumPadding
        config.setExpert("maximumPadding",true);
        config.argDesc("maximumPadding","<bytes>");
        config.shortDesc("maximumPadding","The max. bytes between messages");
        config.longDesc("maximumPadding",
           "The maximum number of bytes of padding to allow between messages",
           "and in Void messages.");

    }

    
    //=== static members and static initialization =============================

    /** whether classwide init() has occurred */
    static private Irreversible initialized = new Irreversible(false);

    /** Time to wait on authentication reads */
    public static int authTimeout;
    
    /** The number of milliseconds to leave a silent connection open */ 
    public static int connectionTimeout;
    
    /** The Expected per Node time of a hop */
    public static int hopTimeExpected;
    
    /** The expected standard deviation from the per hop time */
    public static int hopTimeDeviation;
    
    /** The size used for memory buffers when copying data */
    public static int blockSize;
    
    /** The size used for stream buffers */
    public static int streamBufferSize = 16384;

    /** The maximum padding between messages */
    public static int maxPadding;

    /** The central PRNG */
    public static RandomSource randSource;
    static {
        randSource =
            new Yarrow((new File("/dev/urandom").exists()
                        ? "/dev/urandom" : "prng.seed"), "SHA1", "Rijndael");
    }
    
    /** The object that logs events */
    public static Logger logger = new Logger();

    public static PrintStream logStream = System.out;
    
    protected ListenSelectorLoop interfaceLoop;
    
    /** The diagnostics module */
    public static Diagnostics diagnostics = new VoidDiagnostics();
                                            // avoid NPEs in client code
                                            // running in another JVM
    /** Autopoll jobs for diagnostics */
    public static AutoPoll autoPoll = new AutoPoll(diagnostics, logger);

    /** Per host inbound contact monitoring **/
    public static ContactCounter inboundContacts = null;

    /** Per host inbound request monitoring. Note this includes all request types. **/
    public static ContactCounter inboundRequests = null;

    /** Per host outbound contact monitoring **/
    public static ContactCounter outboundContacts = null;

    /** Per host outbound request monitoring. Note this includes all request types. **/
    public static ContactCounter outboundRequests = null;

    /** Distribution of inbound DataRequests over the keyspace **/
    public static KeyHistogram requestDataDistribution = null;

    /** Distribution of inbound InsertRequests over the keyspace **/
    public static KeyHistogram requestInsertDistribution = null;

    /** Distribution of successful inbound DataRequests over the keyspace **/
    public static KeyHistogram successDataDistribution = null;

    /** Distribution of successful inbound InsertRequests over the keyspace **/
    public static KeyHistogram successInsertDistribution = null;

    /**
     * Returns the probability of success of a request that would go into
     * a given bin for stats purposes
     */ /* fixme: can we optimize these somehow? */
    public static float pSuccess(int bin, boolean detail, boolean inserts)
    {
	int x = binRequest(bin, detail, inserts);
	int y = binSuccess(bin, detail, inserts);
	//if(x == 0) return 0;
	if (x <= (int)((requestDataDistribution.getTotal() / (binLength(detail)*3) + 1) ) )
	{
	    return Float.NaN;
	}
	return ((float)y / (float)x);
    }

    public static int binRequest(int x, boolean detail, boolean inserts)
    {
	return bin(x, false, detail, inserts);
    }

    public static int binSuccess(int x, boolean detail, boolean inserts)
    {
	return bin(x, true, detail, inserts);
    }

    public static int bin(int x, boolean success, boolean detail, boolean inserts)
    {
	if(success)
	{
	   if(detail) {
	      return inserts ? successInsertDistribution.getBiggerBin(x) :
		     successDataDistribution.getBiggerBin(x);
	   } else {
	      return inserts ? successInsertDistribution.getBin(x) :
		     successDataDistribution.getBin(x);
	   }
	} else {
	   if(detail) {
	      return inserts ? requestInsertDistribution.getBiggerBin(x) :
		     requestDataDistribution.getBiggerBin(x);
	   } else {
	      return inserts ? requestInsertDistribution.getBin(x) :
		     requestDataDistribution.getBin(x);
	   }
	}
    }
    
    public static int binLength(boolean detail)
    {
	return detail ? requestDataDistribution.lengthBigger() :
	       requestDataDistribution.length();
    }
    
    /** Returns the most successful bin
     */
    public static int binMostSuccessful(boolean detail, boolean inserts)
    {
	int ret = -1;
	float pHighest = 0;
	int iMax = binLength(detail);
	for(int i=0;i<iMax;i++)
	{
	    int x = binRequest(i, detail, inserts);
	    if(x>0)
	    {
	       float p = pSuccess(i, detail, inserts);
	       if(p>pHighest || 
		  (p == pHighest && ret != -1 && 
		   binRequest(i, detail, inserts) >
		    binRequest(ret, detail, inserts)))
	       {
		  pHighest = p;
		  ret = i;
	       };
	    };
	}
	return ret;
    }
    
    /**
     * Sets the logging object to be used for logging messages
     * @param log a Logger object that will log messages to an output stream.
     */
    public static void setLogger(Logger l) {
        logger = l;
    }

    /** @return  the global logger
      * I put this here for plugins to use .. if we redesign things
      * this might be better than having them all accessing Core.logger.
      */
    public static Logger getLogger() {
        return logger;
    }

    /**
     * Sets the core properties of Freenet.
     * @param params a Params object containing any command line or
     *                 config file settings. 
     * @throws CoreException  if Core was already initialized
     */
    public static void init(Params params) throws CoreException {
        try {
            initialized.change();
        } catch (IrreversibleException e) {
            throw new CoreException("already initialized");
        }
        authTimeout       = params.getInt("authTimeout");
        connectionTimeout = params.getInt("connectionTimeout");
        hopTimeExpected   = params.getInt("hopTimeExpected");
        hopTimeDeviation  = params.getInt("hopTimeDeviation");
        blockSize         = params.getInt("blockSize");
        streamBufferSize  = params.getInt("streamBufferSize");
        maxPadding        = params.getInt("maximumPadding");
    }


    //=== instance members =====================================================

    /** The node's cryptographic identity */
    public final Identity identity;
    /** And private key */
    public final Authentity privateKey;

    /** Registry of supported transports */
    public TransportHandler transports;
    /** Registry of supported session (link) layers */
    public SessionHandler sessions;
    /** Registry of supported protocols */
    public PresentationHandler presentations;

    /** Whether this Core has begun operating.
      * The timer, connections, and interfaces
      * are set when the Core is begun.
      */
    private Irreversible begun = new Irreversible(false);

    /** The Ticker to schedule execution of MessageObjects with */
    private Ticker timer;

    /** Caches active connections (AFH) */
    public OpenConnectionManager connections;

    /** The interfaces this Core is listening on */
    public NIOInterface[] interfaces;

    /** The threads running the interfaces */
    //private Thread[] interfaceThreads;
    Thread interfaceLoopThread;

    private Object waitForBegin = new Object();

    /**
     * Create a new Freenet Core.
     * @param privateKey  The node's private key!
     * @param identity    And public key.
     * @param th  A TransportHandler registering the available transports.
     * @param sh  A SessionHandler registering the available sessions.
     * @param ph  A PresentationHandler registering the available protocols.
     */
    public Core(Authentity privateKey, Identity identity, 
                TransportHandler th, SessionHandler sh,
                PresentationHandler ph) {

        this.privateKey = privateKey;
        this.identity = identity;
        this.transports = th;
        this.sessions = sh;
        this.presentations = ph;

        logger.log(this, this.toString() + " (build "+Version.buildNumber+")",
                   Logger.MINOR);
    }

    /** @return  something for the logs.. */
    public String toString() {
        return "Freenet Core: " + Fields.bytesToHex(identity.fingerprint());
    }

    /**
     * Adopts the ticker for scheduling MessageObjects and starts
     * listening on the given interfaces.  Returns immediately.
     *
     * @see setDaemon(boolean daemon), join()
     *
     * @param tg      The ThreadGroup the ticker and interface
     *                threads will belong to.
     * @param t       The ticker for scheduling of MOs.
     * @param ocm     The OpenConnectionManager to use.
     * @param inter   The interfaces to listen on.  May be null
     *                (transient, firewalled node).
     * @param daemon  The daemon-ness of the interface threads.
     * @throws CoreException  if enough threads couldn't be obtained
     *                        to run the ticker and all interfaces,
     *                        or if this Core was already begun once
     */
    public void begin(ThreadGroup tg, Ticker t,
                      OpenConnectionManager ocm, NIOInterface[] inter,
                      boolean daemon) throws CoreException {
        try {
            begun.change();
        } catch (IrreversibleException e) {
            throw new CoreException("already begun");
        }
        
        timer = t;
        connections = ocm;
        interfaces  = (inter == null ? new NIOInterface[0] : inter);
        
        logger.log(this, "Starting ticker..", Logger.NORMAL);
        Thread ticker = new Thread(tg, timer, "Ticker");
        if (ticker == null)
            throw new CoreException("ran out of threads");
        ticker.setDaemon(true);
	ticker.setPriority(Thread.MAX_PRIORITY);
        ticker.start();
        /*
        logger.log(this, "Starting interfaces..", Logger.NORMAL); 
        interfaceThreads = new Thread[interfaces.length];
        for (int i=0; i<interfaces.length; ++i) {
            interfaceThreads[i] = new Thread(tg, interfaces[i],
                                      interfaces[i].toString());
            if (interfaceThreads[i] == null)
                throw new CoreException("ran out of threads");
        }
        for (int i=0; i<interfaceThreads.length; ++i) {
            interfaceThreads[i].setDaemon(daemon);
            interfaceThreads[i].start();
        }*/

	try{
	
	interfaceLoop = new ListenSelectorLoop();
	interfaceLoopThread = new Thread (interfaceLoop," interface thread");
	
        logger.log(this, "Starting interfaces..", Logger.NORMAL); 
        for (int i = 0;i < interfaces.length; i++) {
		interfaces[i].register(interfaceLoop);
		
	}
	
	
	logger.log(this, "starting ListenSelector..", Logger.NORMAL);
	interfaceLoopThread.start();
	}catch(IOException e) {System.err.println("couldn't create interfaces");e.printStackTrace();}
        
	synchronized(waitForBegin) {
	    waitForBegin.notifyAll();
	}
    }

    public boolean begun() {
	return begun.state();
    }

    /** Joins the current thread with all the interface threads,
      * meaning this method doesn't return until they all stop running
      * (or the current thread is interrupted).
      * @throws CoreException  if the interface threads don't exist yet
      *                        (Core not begun)
      */
    public void join() throws CoreException {
        /*if (interfaceThreads == null) throw new CoreException("Core not begun");
        try {
            for (int i=0; i<interfaceThreads.length; ++i)
                interfaceThreads[i].join();
        }*/
	if (interfaceLoopThread == null) throw new CoreException("Core not begun");
	try {
		interfaceLoopThread.join();
	}
        catch (InterruptedException e) {}
    }

    /**
     * Wait for the Core to start
     */
    public void waitForBegin() {
	if(timer != null) return;
	synchronized(waitForBegin) {
	    while(!begun.state()) {
		try {
		    waitForBegin.wait(1000);
		} catch (InterruptedException e) {} // FIXME?
	    }
	}
    }
    
    /**
     * Stops all running interfaces and their threads
     * (join() will return).
     * @throws CoreException  if this Core wasn't begun yet
     */
    public void stop() throws CoreException {
        if (interfaces == null) throw new CoreException("Core not begun");
	//try{
        for (int i = 0 ; i < interfaces.length ; i++)
            interfaces[i].listen(false);
	//}catch(IOException e){System.err.println("couldn't stop interfaces");e.printStackTrace();}
    }

    /**
     * Sees if the Transport is supported by any available interface.
     */
    public boolean hasInterfaceFor(Transport t) {
        if (interfaces == null) throw new CoreException("Core not begun");
        for (int i = 0 ; i < interfaces.length ; i++) {
            if (interfaces[i].transport().equals(t))
                return true;
        }
        return false;
    }

    
    
    /**
     * Returns an open connection to a node, either by making a new one
     * taking one from the cache.
     * @param peer The Address of the other Freenet node/client
     * @return A ConnectionHandler that handles the new connection
     * @exception ConnectFailedException  if the Connection could not be opened
     *                                    or a new node did not respond to 
     *                                    handshake.
     */
    public final ConnectionHandler makeConnection(Peer p)
	throws CommunicationException {
        return makeConnection(p, 0);
    }
    
    /**
     * Returns an open connection to a node, either by making a new one
     * taking one from the cache.
     * @param peer     The peer to connect to.
     * @param timeout  The time to wait before throwing a 
     *                 ConnectFailedException when establishing new 
     *                 connections.
     */
    public final ConnectionHandler makeConnection(Peer p, long timeout) 
                                        throws CommunicationException {
        //if (connections == null)
        //    throw new CoreException("Core not begun");
        return connections.getConnection(this, p, timeout);
    }


    
    /**
     * Send the message using the appropriate protocol over a free or 
     * new connection.
     */
    //public OutputStream sendMessage(Message m, Peer destination, long timeout) 
    //                                            throws CommunicationException {
    //    try {
    //        return makeConnection(destination, timeout).sendMessage(m);
    //    } catch (ConnectFailedException e) {
    //        throw new SendFailedException(e);
    //    }
    //}

    /**
     * Send the message using the appropriate protocol over a free or 
     * new connection.
     */
    //public OutputStream sendMessage(Message m, Peer destination) 
    //                            throws CommunicationException {
    //    return sendMessage(m, destination, 0);
    //}
    


    
    // Digest for signatures
    private static final Digest ctx = SHA1.getInstance();

    /**
     * Signs a FieldSet using DSS.
     * @param fs     The FieldSet to sign.
     * @param field  The field name to insert the signature as in the FieldSet.
     *               If this is null then it is not inserted.
     * @return       The signature
     */
    public CryptoElement sign(FieldSet fs, String field) {
        byte[] b;
        synchronized(ctx) {
            if (field != null)
                fs.hashUpdate(ctx,new String[] {field});
            else
                fs.hashUpdate(ctx);
            b = ctx.digest();
        }
        
        CryptoElement sig = privateKey.sign(b);
        
        if (field != null)
            fs.add(field, sig.writeAsField());
        return sig;
    }

    /**
     * Signs a digest.
     * @param b  The digest to sign
     */
    public CryptoElement sign(byte[] b) {
        return privateKey.sign(b);
    }

    /**
     * @return  the Core's Ticker
     * @throws CoreException  if the Core hasn't begun yet
     */
    public final Ticker ticker() {
        if (timer == null) throw new CoreException("Core not begun");
        return timer;
    }

    /**
     * Schedule the MO to run on the ticker immediately.
     */
    public final void schedule(MessageObject mo) {
        if (timer == null) throw new CoreException("Core not begun");
        timer.add(0, mo);
    }
    
    /**
     * Schedule the MO to run on the ticker after a delay.
     */
    public final void schedule(long millis, MessageObject mo) {
        if (timer == null) throw new CoreException("Core not begun");
        timer.add(millis, mo);
    }


    
    // statistical data
    // we might want to move this somewhere..

    /** @return The upper bound of a one sided 97.5% confidence interval
      *         for the time it should take to get a reply based on the
      *         the hopTimeExpected and hopTimeDeviation values from
      *         the config (and the assumption that the time is normal).
      *         In milliseconds.
      */
    public static final long hopTime(int htl) {
        return (long) (htl*hopTimeExpected + 1.96*Math.sqrt(htl)*hopTimeDeviation);
    }

    /** @return  The expected time to get the StoreData after the
      *          Accepted/InsertReply is received (i.e. counting from
      *          when the transfer upstream is started).
      * @param htl     the number of hops upstream of this node
      * @param length  the trailing-field length of the insert
      */
    public static final long storeDataTime(int htl, long length) {
        return 2*( hopTime(htl+1) + length );
                                    // roughly 1000 bytes/second
                                    // and totally ignoring covariances
                                    // and what have you, with a big
                                    // fat doubling......
    }
}



