/*  BSD-License:

Copyright (c) 2010 by Matthias Bunte, Germany

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
  * Neither the names of the authors the name nicai-systems nor
    the names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

/*! @file    systemtimer.h
 *  @brief   System timer - the base of the library
 *  @author  Matthias Bunte (m_bunte@arcor.de)
 *  @date    2010-02-02
 */

#ifndef _BEELIB_SYSTEMTIMER_H_
#define _BEELIB_SYSTEMTIMER_H_

#include "stdlib.h"         //
#include "iodefs.h"

#ifdef __cplusplus
extern "C" {
#endif

// Mit dem Timer-System wird es moeglich, Funktionen vom folgenden
// Typ zu einem Zeitpunkt innerhalb der naechsten Minute aufzurufen.
// Die Aufloesung betraegt 1 ms.

// Pointer auf Funktionen, die durch das Timer-System aufrufbar sind, typisieren
// Kein Rueckgabewert, Uebergabewert vom Typ uint16_t
typedef  void (*pSystemTimer_Fkt_t)(uint16_t);

// Initialisierung
void SystemTimer_Init(void);

// Initialisierungszustand auslesen
uint8_t SystemTimer_Initialized(void);

// Pruefe, ob schon initialisiert wurde und wenn nicht, initialisiere
#define SystemTimer_CheckAndInit() do { \
  if (!SystemTimer_Initialized()) SystemTimer_Init(); \
} while (0)

// Liefert die aktuelle Systemzeit zurueck
// Rueckgabewert in Millisekunden
uint16_t SystemTimer_GetTime(void);

// Liefert den Wert, den der Timer nach der Zeit Offset hat
// Einheit von Offset und Rueckgabewert ist Milisekunden
uint16_t SystemTimer_GetOffset(uint16_t Offset);

// Prueft, ob der Timerwert "TimerVal" abgelaufen ist
// Der Rueckgabewert ist 0, falls die Zeit nicht abgelaufen ist
// und nicht 0, falls die Zeit abgelaufen ist
uint8_t SystemTimer_IsExpired(uint16_t TimerVal);

// SystemTimer_SchedulerAdd() - Funktion zur Timerliste zufuegen
// TimerVal ist der Offset von jetzt in ms, wann die Funktion fruehestens ausgefuehrt werden soll
// pFunktion ist der Zeiger auf die Funktion, die ausgefuehrt werden soll
// Wert ist der Uebergabewert, der der Funktion pFunktion uebergeben wird, sobald sie ausgefuehrt wird
void SystemTimer_SchedulerAdd(uint16_t TimerVal, pSystemTimer_Fkt_t pFunktion, uint16_t Wert);

// SystemTimer_SchedulerDel() - Funktion von Timerliste entfernen
// pFunktion ist der Zeiger auf die Funktion, die geloescht werden soll
void SystemTimer_SchedulerDel(pSystemTimer_Fkt_t pFunktion);

// SystemTimer_SchedulerRun() - Liste von gespeicherten Timereintraegen
// durchsuchen und ausfuehren
//
// Um das Timer-System zu verwenden, muss diese Funktion regelmaessig in der
// Hauptschleife des Programms aufgerufen werden. Damit die Funktionen nicht
// zu starke Verspaetungen bekommen, sollte diese Funktion mindestens einmal pro
// ms aufgerufen werden.
void SystemTimer_SchedulerRun(void);

#ifdef __cplusplus
} // extern "C"
#endif

#endif // _BEELIB_SYSTEMTIMER_H_

