/*
 * running.c: Motor- und passende LED Steuerung
 *
 *
 */

/*  BSD-License:

Copyright (c) 2010 by Matthias Bunte, Germany

All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

  * Redistributions of source code must retain the above copyright notice,
    this list of conditions and the following disclaimer.
  * Redistributions in binary form must reproduce the above copyright notice,
    this list of conditions and the following disclaimer in the documentation
    and/or other materials provided with the distribution.
  * Neither the names of the authors the name nicai-systems nor
    the names of its contributors may be used to endorse or promote products
    derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

#include "stdlib.h"         //
#include "iodefs.h"         // Grund-Include-Datei fuer die bee
#include "leds.h"            // LED-Steuerung
#include "motorpid.h"       // Motorregelung

#define RUNNING_MOTORSTOPP     (0)   // Stopp-Geschwindigkeit (in % von Max)
#define RUNNING_MOTORFORWARD   (100)  // Fahr-Geschwindigkeit (in % von Max)
#define RUNNING_RUECKWEG       (15)  // 15 cm Rueckwaertsfahrt bei Wendemanoevern
#define RUNNING_RECHTSKURVE    (40)  // 
#define RUNNING_LINKSKURVE     (-RUNNING_RECHTSKURVE)  // 
#define RUNNING_KURVENWEG      (20)  // Strecke der Aussenkurven

// Hierin sind die Ablaufschritte der mehrstufigen Aktionen abgebildet 
typedef enum {
  Running_AblaufKeineAktion = 0,
  Running_AblaufWendenLinks,
  Running_AblaufWendenRechts,
  Running_AblaufGerade,
  Running_AblaufZurueck,
  Running_AblaufStopp,
} Running_Ablauf_t;

static uint8_t Running_Speed = RUNNING_MOTORFORWARD;
static Running_Ablauf_t Running_Ablauf = 0;
static pMotorPID_EventFkt_t pRunning_EventFkt;
static MotorPID_Event_t MotorPID_EventVal;

// 
void Running_MotorPID_EventFkt(uint16_t MotorPID_Event);

// Abfrage des letzten MotorPID_Events
MotorPID_Event_t Running_GetMotorPID_Event(void)
{
  return MotorPID_EventVal;
}

// Library initialisieren
void Running_Init(pMotorPID_EventFkt_t pEventFkt)
{
  // LED-Steuerung initialisieren
//  led_CheckAndInit();
  LEDS_INIT();
  // Ablaufsteuerung zuruecksetzen
  Running_Ablauf = Running_AblaufKeineAktion;
  // Motorregelung initialisieren
  MotorPID_Init(Running_MotorPID_EventFkt);
  // Eventfunktion zur Weiterleitung uebernehmen
  pRunning_EventFkt = pEventFkt;
  // Interne Eventvariable initialisieren
  MotorPID_EventVal = MotorPID_EventInvalid;
}

// Motoren geradeaus fahren
void Running_Gerade(void)
{
  // Gelbe LEDs einschalten
  LEDS_OFF( LEDS_L_RD );
  LEDS_OFF( LEDS_R_RD );
  LEDS_ON( LEDS_L_YE );
  LEDS_ON( LEDS_R_YE );
  // Ablaufsteuerung zuruecksetzen
  Running_Ablauf = Running_AblaufKeineAktion;
  // Geradeausfahren
  MotorPID_StartGerade(Running_Speed);
//  MotorPID_StartStrecke(Running_Speed, 30);
}

// Motoren zurueck fahren
void Running_Zurueck(void)
{
  // Rote LEDs einschalten
  LEDS_ON( LEDS_L_RD );
  LEDS_ON( LEDS_R_RD );
  LEDS_OFF( LEDS_L_YE );
  LEDS_OFF( LEDS_R_YE );
  // Ablaufsteuerung zuruecksetzen
  Running_Ablauf = Running_AblaufKeineAktion;
  // Geradeausfahren
  MotorPID_StartGerade(Running_Speed);
//  MotorPID_StartStrecke(-Running_Speed, 30);
}

// Motoren anhalten
void Running_Stopp(void)
{
  // Ablaufsteuerung zuruecksetzen
  Running_Ablauf = Running_AblaufKeineAktion;
  // Motoren ausschalten
  MotorPID_Stopp();
  // Alle LEDs ausschalten
  LEDS_OFF( LEDS_ALL );
}

// Wendevorgang links
void Running_WendenNachLinks(void)
{
  // Motoren ausschalten
  MotorPID_Stopp();
  // Linke LEDs leuchten lassen
  LEDS_OFF(LEDS_R_RD);
  LEDS_OFF(LEDS_R_YE);
  LEDS_ON(LEDS_L_RD);
  LEDS_ON(LEDS_L_YE);
  // 
  Running_Ablauf = Running_AblaufWendenLinks;
  // Biene mit halber Geschwindigkeit zurueck fahren lassen
  MotorPID_StartStrecke(-(Running_Speed / 2), RUNNING_RUECKWEG);
}

// Wendevorgang links - zweiter Teil
static void Running_WendenLinks(void)
{
  // Nach dem Wendemanoever wieder geradeaus fahren
  Running_Ablauf = Running_AblaufGerade;
  // Biene rueckwaerts eine Linkskurve fahren lassen
  MotorPID_StartKurvenStrecke(-(Running_Speed / 2), RUNNING_LINKSKURVE, RUNNING_KURVENWEG);
}

// Wendevorgang rechts
void Running_WendenNachRechts(void)
{
  // Motoren ausschalten
  MotorPID_Stopp();
  // Rechte LEDs leuchten lassen
  LEDS_OFF(LEDS_L_RD);
  LEDS_OFF(LEDS_L_YE);
  LEDS_ON(LEDS_R_RD);
  LEDS_ON(LEDS_R_YE);
  // 
  Running_Ablauf = Running_AblaufWendenRechts;
  // Biene mit halber Geschwindigkeit zurueck fahren lassen
  MotorPID_StartStrecke(-(Running_Speed / 2), RUNNING_RUECKWEG);
}

// Wendevorgang rechts - zweiter Teil
static void Running_WendenRechts(void)
{
  // Nach dem Wendemanoever wieder geradeaus fahren
  Running_Ablauf = Running_AblaufGerade;
  // Biene rueckwaerts eine Rechtskurve fahren lassen
  MotorPID_StartKurvenStrecke(-(Running_Speed / 2), RUNNING_RECHTSKURVE, RUNNING_KURVENWEG);
}

// 
static void Running_EventFkt(void)
{
  switch (Running_Ablauf)
  {
  case Running_AblaufGerade:
    // Geradeausfahrt
    Running_Gerade();
    break;
  case Running_AblaufZurueck:
    // Zurueckfahrt
    Running_Zurueck();
    break;
  case Running_AblaufStopp:
    // Anhalten
    Running_Stopp();
    break;
  case Running_AblaufWendenLinks:
    // Zweiter Teil Wendemanoever Links
    Running_WendenLinks();
    break;
  case Running_AblaufWendenRechts:
    // Zweiter Teil Wendemanoever Rechts
    Running_WendenRechts();
    break;
  default:
    break;
  }
}

// 
void Running_MotorPID_EventFkt(uint16_t MotorPID_Event)
{
  switch (MotorPID_Event)
  {
  case MotorPID_EventFinished:
    // Nachdem ein Event fertig ist, naechste Funktion aufrufen
    Running_EventFkt();
    break;
  default:
    // Falls bei der Initialisierung eine Funktion zur Eventauswertung angegeben
    // wurde, diese hier aufrufen, sofern es sich nicht um eine interne Aktion handelte
    if (pRunning_EventFkt)
    {
      pRunning_EventFkt(MotorPID_Event);
    }
    else
    {
      MotorPID_EventVal = MotorPID_Event;
    }
    break;
  }
}

