#!/bin/sh
# CONVERTLDASCACHE.sh Converts a LDAS frame cache to READFRAMEDATA format
#
# CONVERTLDASCACHE.sh reads the specified input cache file, which must be
# a standard ASCII text version of and LDAS diskCacheAPI disk cache.  The
# cache information is then rewritten to the specified output cache file
# in the format required by READFRAMEDATA.
#
# usage: convertldascache.sh ldasDiskCache frameCacheFile
#
#   ldasDiskCache    input ASCII LDAS diskCacheAPI frame cache file
#   frameCacheFile   output READFRAMEDATA formatted frame cache file
#
# If they are not specified, the default frameCacheFile is framecache.txt
# and the default ldasDiskCache is /data/node1/ldas/frame_cache_dump.
#
# The resulting READFRAMEDATA cache file consists of whitespace delimited
# ASCII text and contains one line for each contiguous data segment with a
# common site, type, stride, and directory.  Each line consists of the
# following six columns.
#
#   * site designator (e.g. 'H' or 'L')
#   * frame file type (e.g. 'RDS_R_L3')
#   * GPS start time of segment
#   * GPS stop time of segment
#   * frame file stride in seconds
#   * full path name of directory
#
# The data segments are inclusive of the specified start time, but
# exclusive of the specified stop time, such that the segment duration
# is simply the difference between the stop and start times.
#
# See also READFRAMEDATA, LOADFRAMECACHE, and CREATEFRAMECACHE.pl.

# Shourov K. Chatterji
# shourov@ligo.caltech.edu

# $Id: convertldascache.sh,v 1.1 2008/09/05 14:49:48 ecm Exp $

# location of ascii ldas disk cache
if [ $# -lt 1 ]; then
  ldasDiskCache=/usr1/ldas/frame_cache_dump
else
  ldasDiskCache=$1
fi

# location of output frame cache file
if [ $# -lt 2 ]; then
  frameCacheFile=framecache.txt
else
  frameCacheFile=$2
fi

# check for already existing frame cache file
if [ -f ${frameCacheFile} ]; then
  echo "$0: ERROR: frame cache file already exists." 1>&2
  exit 1
fi

# translate framecache
sed -e 's|,| |g' -e 's|[{}]||g' ${ldasDiskCache} | \
  awk ' { for (i = 8; i < NF; i = i + 2) \
          print $2, $3, $i, $(i+1), $5, $1 } ' \
  >${frameCacheFile}.tmp

# sort frame cache by preference
grep ' /data/node' ${frameCacheFile}.tmp | sort >${frameCacheFile}
grep -v ' /data/node' ${frameCacheFile}.tmp | sort >>${frameCacheFile}

# remove temporary frame cahe
rm -f ${frameCacheFile}.tmp
