function spectrum = welch(data,window,overlap,numberOfFreqBins)
#WELCH  Power spectrum estimation using Welch method.
#       Ref: Digital Signal Processing, Proakis & Manolakis
#       p. 911

error(nargchk(4, 5, nargin));

## check and normalize window
blockSize = length(window);
blockHalfSize = (blockSize-1)/2.0;
window=window/norm(window);

if (blockSize > numberOfFreqBins)
  error("welch: numberOfFreqBins should be larger than the size of the window");
endif;

## validate input args
if ~isvector(data) || ~isvector(window) 
  error("welch: data and window should be vectors");
endif;

data=data(:);
window=window(:);

## segment partition
segmentSize = length(data);
blockOverlap=floor(overlap*blockSize);
numberOfBlocks=ceil((segmentSize-blockSize)/(blockSize-blockOverlap));
blockStarts = (0:numberOfBlocks-1)*(blockSize-blockOverlap);
times = -blockHalfSize:blockHalfSize;

## compute periodogram
periodogram = zeros (numberOfFreqBins,numberOfBlocks);  
for block=1:numberOfBlocks,

  blockCenter = blockStarts(block) + blockHalfSize+1; 
  periodogram(1:blockSize,block) = data(blockCenter + times).*window;

endfor;

periodogram = abs(fft(periodogram)).^2; 

## marginalize
if numberOfBlocks > 1
  spectrum = mean(periodogram');
else
  spectrum = periodogram;
endif;