function  [y, h] = resamplehighpass( x, p, q, cutoff, h )

if nargchk(3,5,nargin)
  usage("resamplehighpass.m: [y, h] = resamplehighpass( x, p, q[, cutoff, h] )");
end;

if any([p q]<=0)|any([p q]~=floor([p q])),
  error("resamplehighpass.m: p and q must be positive integers");
endif

## simplify decimation and interpolation factors

great_common_divisor=gcd(p,q);
if (great_common_divisor>1)
  p=p/great_common_divisor;
  q=q/great_common_divisor;
endif

## characteristics of the highpass filter

if (nargin < 4)
  highpass_cutoff_f = 0.0;
else
  if ~(isscalar(cutoff) && (cutoff < .5) )
    error("resamplehighpass.m: cutoff should be a scalar less than 1/2");
  endif
  highpass_cutoff_f = cutoff/p;
endif

## filter design if required

if (nargin < 5)
  ## properties of the antialiasing filter
  
  log10_rejection = -3.0;
  stopband_cutoff_f = 1.0/(2.0 * max(p,q));
  roll_off_width = stopband_cutoff_f / 10.0;
  
  ## determine filter length
  ## use empirical formula from [2] Chap 7, Eq. (7.63) p 476
  
  rejection_dB = -20.0*log10_rejection;
  L = ceil((rejection_dB-8.0) / (28.714 * roll_off_width));
  
  ## ideal bandpass filter (sinc)
  
  t=(-L:L)';
  ideal_filter=2*p*(stopband_cutoff_f * sinc(2*stopband_cutoff_f*t) ...
		    - highpass_cutoff_f * sinc(2*highpass_cutoff_f*t));  
  
  ## determine parameter of Kaiser window
  ## use empirical formula from [2] Chap 7, Eq. (7.62) p 474
  
  if ((rejection_dB>=21)&(rejection_dB<=50))
    beta = 0.5842 * (rejection_dB-21.0)^0.4 + 0.07886 * (rejection_dB-21.0);
  elseif (rejection_dB>50)
    beta = 0.1102 * (rejection_dB-8.7);
  else
    beta = 0.0;
  endif
  
  ## apodize ideal filter response
  
  h=kaiser(2*L+1,beta).*ideal_filter;
  
endif

## check if input is a row vector
isrowvector=false;
if ((rows(x)==1)&(columns(x)>1))
   x=x(:);
   isrowvector=true;
endif

## check if filter is a vector
if ~isvector(h)
  error("resamplehighpass.m: the filter h should be a vector");
endif

Lx = length(x);
Lh = length(h);
L = ( Lh - 1 )/2.0;
Ly = ceil(Lx*p/q);

## pre and postpad filter response

nz_pre = floor(q-mod(L,q));
hpad = prepad(h,Lh+nz_pre);

offset = floor((L+nz_pre)/q);
nz_post = 0;
while ceil( ( (Lx-1)*p + nz_pre + Lh + nz_post )/q ) - offset < Ly
    nz_post++;
endwhile
hpad = postpad(hpad,Lh + nz_pre + nz_post);

## filtering

xfilt = upfirdn(x,hpad,p,q);
y = xfilt(offset+1:offset+Ly,:);

if isrowvector,
   y=y.';
endif

endfunction

%!test
%! N=512;
%! p=3; q=5;
%! r=p/q;
%! NN=ceil(r*N);
%! t=0:N-1;
%! tt=0:NN-1;
%! err=zeros(N/2,1);
%! for n = 0:N/2-1,
%!   phi0=2*pi*rand;
%!   f0=n/N;
%!   x=sin(2*pi*f0*t' + phi0);
%!   [y,h]=resamplehighpass(x,p,q);
%!   xx=sin(2*pi*f0/r*tt' + phi0);
%!   t0=ceil((length(h)-1)/2/q);
%!   idx=t0+1:NN-t0;
%!   err(n+1)=max(abs(y(idx)-xx(idx)));
%! endfor;
%! rolloff=.1;
%! rejection=10^-3;
%! idx_inband=1:ceil((1-rolloff/2)*r*N/2)-1;
%! assert(max(err(idx_inband))<rejection);

%!test
%! N=512;
%! p=3; q=5;
%! r=p/q;
%! NN=ceil(r*N);
%! t=0:N-1;
%! tt=0:NN-1;
%! reject=zeros(N/2,1);
%! for n = 0:N/2-1,
%!   phi0=2*pi*rand;
%!   f0=n/N;
%!   x=sin(2*pi*f0*t' + phi0);
%!   [y,h]=resamplehighpass(x,p,q);
%!   t0=ceil((length(h)-1)/2/q);
%!   idx=t0+1:NN-t0;
%!   reject(n+1)=max(abs(y(idx)));  
%! endfor;
%! rolloff=.1;
%! rejection=10^-3;
%! idx_stopband=ceil((1+rolloff/2)*r*N/2)+1:N/2;
%! assert(max(reject(idx_stopband))<=rejection);

%!test
%! N=1024;
%! p=2; q=7;
%! r=p/q;
%! NN=ceil(r*N);
%! t=0:N-1;
%! tt=0:NN-1;
%! err=zeros(N/2,1);
%! for n = 0:N/2-1,
%!   phi0=2*pi*rand;
%!   f0=n/N;
%!   x=sin(2*pi*f0*t' + phi0);
%!   [y,h]=resamplehighpass(x,p,q);
%!   xx=sin(2*pi*f0/r*tt' + phi0);
%!   t0=ceil((length(h)-1)/2/q);
%!   idx=t0+1:NN-t0;
%!   err(n+1)=max(abs(y(idx)-xx(idx)));
%! endfor;
%! rolloff=.1;
%! rejection=10^-3;
%! idx_inband=1:ceil((1-rolloff/2)*r*N/2)-1;
%! assert(max(err(idx_inband))<rejection);

%!test
%! N=512;
%! p=3; q=5; fcut=0.1;
%! r=p/q;
%! NN=ceil(r*N);
%! t=0:N-1;
%! tt=0:NN-1;
%! reject=zeros(N/2,1);
%! for n = 0:N/2-1,
%!   phi0=2*pi*rand;
%!   f0=n/N;
%!   x=sin(2*pi*f0*t' + phi0);
%!   [y,h]=resamplehighpass(x,p,q,fcut);
%!   t0=ceil((length(h)-1)/2/q);
%!   idx=t0+1:NN-t0;
%!   reject(n+1)=max(abs(y(idx)));  
%! endfor;
%! rolloff=.1;
%! rejection=10^-3;
%! idx_stopband=1:ceil((fcut-r/2*rolloff/2)*N);
%! assert(max(reject(idx_stopband))<=rejection);
