function y = olafilter(transf,ntransf,window,stride,x);

  error(nargchk(5,5,nargin));

  %% if row vector, transpose
  isrowvector=false;
  if ((rows(x)==1)&(columns(x)>1))
    x=x(:);
    isrowvector=true;
  endif

  %% get input dimensions
  [r_x, c_x] = size(x);
  r_w = length(window);
  N = length(transf);

  %% validate input parameters
  if ~( isscalar(ntransf) && (ntransf==fix(ntransf)) )
    error("olafilter: ntransf should be an integer");
  endif

  overlap = fix((N - r_w - ntransf)/2.0);
  if ( overlap < 0 )
    error("olafilter: length(transf) should be larger length(window)+ntransf");
  endif

  if ~( isscalar(stride) && (stride==fix(stride)) )
    error("olafilter: stride should be an integer");
  endif

  %% create windowing and transfer functions matrices
  w=repmat(window,1,c_x);
  tf=repmat(transf,1,c_x);

  %% main loop
  y = zeros (r_x, c_x);

  lo = 1;
  while (lo + r_w/2.0 < r_x)

    hi = min (lo + r_w -1, r_x);

    %% select and window data block
    tmp = zeros (N, c_x);
    tmp(1:(hi-lo+1),:) = w(1:hi-lo+1,:) .* x(lo:hi,:);

    %% filter block
    tmp = ifft (tf .* fft (tmp));

    %% overlap and add
    lo_add=max(1,(lo-overlap));
    hi_add=min(min(r_x,hi+overlap),N+lo-1);
    y(lo_add:hi_add,:) += tmp([ (N+(lo_add-lo+1)):N 1:(hi_add-lo+1)],:);

    %% next block
    lo += stride;

  endwhile
    
  if (isreal (transf) && isreal (x))
    y = real (y);
  endif

  if isrowvector,
    y=y.';
  endif

endfunction

%% test constant overlap and add with Hanning window

%!test
%! Lw = 101;
%! m = (1:Lw)'/(Lw+1);
%! window = .5 * (1 - cos(2*pi*m));
%! stride = (Lw+1)/2.0;
%! N = 1024;
%! x = ones(N,1);
%! Nf = 2^(nextpow2(Lw)+1);
%! tf = ones(Nf,1);
%! y = olafilter(tf,Lw,window,stride,x);
%! hLw = (Lw + 1)/2;
%! err = abs(y(hLw+1:end-hLw) - 1.0);
%! assert(max(err),0.0,1e-15);

%% test constant overlap and add with Hamming window

%!test
%! Lw = 101;
%! m = (0:Lw-1)'/(Lw-1);
%! window = .54 - .46*cos(2*pi*m);
%! window(1) /= 2.0;
%! window(Lw) /= 2.0;
%! stride = (Lw-1)/2.0;
%! window = stride * window/sum(window);
%! N = 1024;
%! x = ones(N,1);
%! Nf = 2^(nextpow2(Lw)+1);
%! tf = ones(Nf,1);
%! y = olafilter(tf,Lw,window,stride,x);
%! hLw = (Lw + 1)/2;
%! err = abs(y(hLw+1:end-hLw) - 1.0);
%! assert(max(err),0.0,1e-15);

%% test constant overlap and add with Kaiser window
%% note: the Kaiser window only have approximate COLA property

%!test
%! Lw = 51;
%! beta=15; factor=5.5;
%! window = kaiser(Lw,beta);
%! stride = floor(Lw/factor);
%! window = stride*window/sum(window);
%! N = 2048;
%! x = ones(N,1);
%! Nf = 2^(nextpow2(Lw)+1);
%! tf = ones(Nf,1);
%! y = olafilter(tf,Lw,window,stride,x);
%! edge = fix(stride/max(window))+(Lw+1)/2;
%! err = abs(y(edge+1:end-edge) - 1.0);
%! assert(max(err),0.0,1e-6);
