function triggers=bccsearch(data,chirplet,whitening,samplingFreq,traceFlag)

  %% verify correct number of input arguments
  error(nargchk(4, 5, nargin));

  %% defaults
  if nargin<5 || isempty(traceFlag)
    traceFlag=false;
  endif

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                         parse command line arguments                       %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  nChannels=length(data);

  % returns if nothing to do
  if nChannels==0
    triggers=[];
    return;
  endif
  
  % number of samples in a block
  nBlockSamples=length(data{1});

  % size of the edges affected by whitening, to be ignored
  nEdgeEffect = floor(whitening.duration * samplingFreq);

  % number of samples in a tile
  nTileSamples = floor(chirplet.tileDuration * samplingFreq);

  % number of overlapping samples of two successive tiles
  nTileOverlapSamples = floor(chirplet.tileOverlap * samplingFreq);

  % number of tiles in a block
  nTiles = 1 + floor((nBlockSamples - nEdgeEffect  - nTileSamples)/ ... 
		     (nTileSamples - nTileOverlapSamples));

  if ( nTiles <= 0 )
    triggers=cell(nChannels,1);    
    warning("bccsearch: no data to analyze");
    return;
  endif

  % frequency bin size in a tile
  frequencyBin = samplingFreq/(2*nTileSamples);

  % chain 1st order regularity
  nRegularity1 = round(chirplet.chirpingRateLimit1*chirplet.duration/frequencyBin);

  % chain 2nd order regularity
  nRegularity2 = round(chirplet.chirpingRateLimit2*chirplet.duration^2/frequencyBin);

  % number of samples in a chirplet
  nChirpletSamples = round(chirplet.duration * samplingFreq);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %  create Fourier matrix used for normalization in the double whitening case
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  if whitening.double
    frequencyBins = 0:nTileSamples-1;
    Fourier = 1/nTileSamples * exp(2*pi*i* ...
		   repmat(frequencyBins  ,nTileSamples,1) .* ...
		   repmat(frequencyBins.',1,nTileSamples)/nTileSamples);    
  endif

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                               main loop                                    %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  triggers=cell(nChannels,1);
                  
  for channelNumber = 1 : nChannels,
    
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                      normalization if double whitening                     %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    if whitening.double
      
      %% correlation matrix from spectrum
      %% use trick to get Corr = Fourier'*diag(1./spectrum)*Fourier very fast
      CorrelationMatrix = fft(repmat(whitening.filter{channelNumber},1,nTileSamples).*Fourier); 

      %% Weyl symbol
      normalization = weyl(CorrelationMatrix);

      clear CorrelationMatrix;

      %% strip down negative frequencies
      normalization = real(normalization(1:nTileSamples,:)); 
    
    endif

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              loop over tiles                               %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % initialize trigger structure
    triggers{nChannels}.snr = zeros(nTiles,1);
    triggers{nChannels}.index = zeros(nTiles,1);
    triggers{nChannels}.meanFrequency = zeros(nTiles,1);
    triggers{nChannels}.bandwidth = zeros(nTiles,1);
  
    if traceFlag
      triggers{nChannels}.frequency = cell(nTiles,1);
    endif

    for tile = 1 : nTiles,
      
      ## select tile
      tileStart = ceil(nEdgeEffect/2.0) + (tile - 1) * (nTileSamples - nTileOverlapSamples);
      tileIndices = tileStart:(tileStart+nTileSamples-1);
    
      ## time-frequency transform
      wignerVille = dwv(data{channelNumber}(tileIndices).');
      
      ## strip down negative frequencies
      wignerVille = wignerVille(1:nTileSamples,:); 
      
      ## path search with dynamic programming
      if whitening.double
	[statistic ignored pathIndices] = dpw(wignerVille,normalization,[],nChirpletSamples,nRegularity1,1,nRegularity2);
      else
	[statistic ignored pathIndices] = dp(wignerVille,[],nChirpletSamples,nRegularity1,1,nRegularity2);
      endif
      
      ## convert path indices to frequencies in Hertz
      pathFreq = samplingFreq*pathIndices/(2.0*nTileSamples);
      
      ## fill in trigger structure
      triggers{channelNumber}.snr(tile) = statistic;
      triggers{channelNumber}.timeInBlock(tile) = (tileStart-1)/samplingFreq;
      triggers{channelNumber}.meanFrequency(tile) = mean(pathFreq);
      triggers{channelNumber}.bandwidth(tile) = max(pathFreq)-min(pathFreq);
      
      ## fill in chirp description if required
      if traceFlag
	triggers{nChannels}.frequency{tile} = pathFreq;
      endif
      
    endfor
    
  endfor

endfunction