#!/bin/sh
#
# bccpipeline.sh
#
# Shell script to run the BCC Pipeline after setting the appropriate
# envioronment variables.
#
# usage: bccpipeline.sh startTime stopTime [parameterFile frameCacheFile outputDirectory]
#
#   startTime           gps start time of analysis
#   stopTime            gps stop time of analysis
#   parameterFile       bccpipeline parameter file
#   frameCacheFile      readframedata formatted frame cache file
#   outputDirectory     directory to write results to
#
# If the specified stop time is less than the specified start time, it is
# instead assumed to be the duration of the analysis.  Non-integer start
# and stop times are truncated to the nearest integer.
#
# If no parameter and frame cache file are specified, Bccpipeline selects
# between one of two default parameter sets depending on the requested
# start time.  If the requested start time is less than 5 days old,
# Bccpipeline selects the @online parameter set if it exists.  Otherwise, the
# @default parameter set is selected.  Other predefined parameter sets of
# likely interest are @H1, @H2, @L1, @H1H2, and @H1H2L1.  Alternatively the
# user can specify the path to a custom parameter file.
#
# Similarly, if no frame cache file is specified, Bccpipeline selects between
# @online and @default frame cache files based on the requested start time.
# Other predefined parameter sets of likely interest are @S1, @S2, @S3,
# @A3, @S4, @A4, and @S5.  Alternatively, if the specified frame cache file
# argument points to a directory instead of a file, a temporary frame cache
# file will automatically be generated for the specified directory.
#
# An outputDirectory path may also be specified.  Typically, this is the
# absolute path to a directory on the local drive of a cluster node.  To
# enable recovery of results, the BCCPipeline logs the location of all
# output directories.  If not specified, the outputDirectory defaults to
# the current directory.
#
# The resulting triggers are collected into single files and stored in
# the following directory structure.
#
#   <outputDirectory>
#     results/
#       livetime-<startTime>-<stopTime>.txt
#       <channelName>-<startTime>-<stopTime>.txt
#
# These files will contain redundant overlapping triggers as well as
# overlapping livetime segments.
#
# This script sources the qsetup.sh shell script in order to setup the
# necessary environment for BCC tools.

# full path to current directory
currentDirectory=`pwd`

# full path to  installation
cd `dirname $0`/..
rootDirectory=`pwd`
cd ${currentDirectory}

# source q environment setup script
# . ${rootDirectory}/bin/bccsetup.sh

# parse command line arguments
startTime=$1
stopTime=$2
if [ $# -eq 2 ]; then
  parameterFile=
  frameCacheFile=
  outputDirectory=${currentDirectory}
elif [ $# -eq 5 ]; then
  parameterFile=$3
  frameCacheFile=$4
  outputDirectory=$5
else
  echo "usage: `basename $0` startTime stopTime [parameterFile frameCacheFile outputDirectory]" 1>&2
  exit 1
fi

# prevent core dumps
ulimit -c 0

# truncate to integer start and stop times
startTime=`echo ${startTime} | sed -e 's|\..*$||'`
stopTime=`echo ${stopTime} | sed -e 's|\..*$||'`

# handle duration argument in place of stop time
if [ ${stopTime} -lt ${startTime} ]; then
  stopTime=`expr ${startTime} + ${stopTime}`
fi

## path to log location file
#logLocationFile=${currentDirectory}/log/locate/locate_${startTime}-${stopTime}.txt

## if log location file already exists
#if [ -f ${logLocationFile} ]; then
#
#  # determine output directory from log location file
#  outputDirectory=`cat ${logLocationFile} 2>/dev/null | \
#                   sed -e "s|^node\([0-9]*\)[^:]*:/usr1|/data/node\1|" \
#                       -e "s|^[^:]*:||"`
#
## end test for log location file
#fi

# create output directory if necessary
mkdir -p ${outputDirectory}

# validate output directory
if [ ! -d ${outputDirectory} ]; then
  echo "ERROR: Could not create output directory ${outputDirectory}" 1>&2
  exit 1
fi

## if output directory is on a local file system
#if [ -n "`/bin/df -l ${outputDirectory} 2>/dev/null | grep -iv filesystem`" ]; then
#
#  # create a log location file if none already exists
#  if [ ! -f ${logLocationFile} ]; then
#    mkdir -p `dirname ${logLocationFile}`
#    echo `hostname`.`hostname -d`:${outputDirectory} >${logLocationFile}
#  fi
#
## end test for output directory on local file system
#fi

# path to results directory
resultsDirectory=${outputDirectory}/results

# create results directory if necessary
mkdir -p ${resultsDirectory}

## path to log status file
#logStatusFile=${outputDirectory}/log/status/status_${startTime}-${stopTime}.txt

## create log status directory if necessary
#mkdir -p `dirname ${logStatusFile}`

# predefined parameter file directory
parameterDirectory=${rootDirectory}/parameters

# predefined frame cache file directory
frameCacheDirectory=${rootDirectory}/framecaches

# round start time down to nearest integer GPS second
integerStartTime=`echo ${startTime} | sed -e s'|\..*$||'`

# start time of online data availability
if [ "${LIGOTOOLS:-unset}" != "unset" ]; then
  onlineStartTime=`tconvert now - 5days`
else
  onlineStartTime=${integerStartTime}
fi

# default parameter file
if [ -z "${parameterFile}" ]; then
  if [ ${integerStartTime} -gt ${onlineStartTime} ]; then
    parameterFile="@online"
  else
    parameterFile="@default"
  fi
fi

# default frame cache file
if [ -z "${frameCacheFile}" ]; then
  if [ ${integerStartTime} -gt ${onlineStartTime} ]; then
    frameCacheFile="@online"
  else
    frameCacheFile="@default"
  fi
fi

# parameter file for online analysis
if [ "${parameterFile}" = "@online" ]; then
  if [ -f "${parameterDirectory}/online.txt" -o \
       -h "${parameterDirectory}/online.txt" ]; then
    parameterFile=${parameterDirectory}/online.txt
  else
    parameterFile="@default"
  fi
fi

# frame cache file for online analysis
if [ ${frameCacheFile} = "@online" ]; then
  if [ -d "/frames/full" ]; then
    frameCacheFile="/frames/full"
  elif [ -f "/virgoData/ffl/raw.ffl" ]; then
    frameCacheFile="/virgoData/ffl/raw.ffl"
  else
    frameCacheFile="@default"
  fi
fi

# frame cache file for default analysis
if [ ${frameCacheFile} = "@default" ]; then
  if [ ${integerStartTime} -lt 728955008 ]; then
    frameCacheFile="@S1"
  elif [ ${integerStartTime} -lt 751505200 ]; then
    frameCacheFile="@S2"
  elif [ ${integerStartTime} -lt 757711008 ]; then
    frameCacheFile="@S3"
  elif [ ${integerStartTime} -lt 792866944 ]; then
    frameCacheFile="@A3"
  elif [ ${integerStartTime} -lt 796144256 ]; then
    frameCacheFile="@S4"
  elif [ ${integerStartTime} -lt 815153408 ]; then
    frameCacheFile="@A4"
  else
    frameCacheFile="@S5"
  fi
fi

# handle predefined parameter files
parameterFile=`echo ${parameterFile} | \
                   sed -e "s|^@\(.*\)$|${parameterDirectory}/\1.txt|"`

# handle predefined frame cache files
frameCacheFile=`echo ${frameCacheFile} | \
                sed -e "s|^@\(.*\)$|${frameCacheDirectory}/\1.txt|"`

# generate frame cache file if requested
if [ -d "${frameCacheFile}" ]; then
  frameFileDirectory=${frameCacheFile}
  dateString=`date +%Y%m%d%H%M%S`
  frameCacheFile=/tmp/framecache_${dateString}_${integerStartTime}_$$.txt
  rm -f ${frameCacheFile}
  createframecache.pl ${frameCacheFile} ${frameFileDirectory} >/dev/null
  temporaryFrameCacheFile=true
elif [ -n "`echo ${frameCacheFile} | grep '\.ffl$'`" ]; then
  fflFile=${frameCacheFile}
  dateString=`date +%Y%m%d%H%M%S`
  frameCacheFile=/tmp/framecache_${dateString}_${integerStartTime}_$$.txt
  rm -f ${frameCacheFile}
  convertfflcache.pl ${fflFile} ${frameCacheFile}
  temporaryFrameCacheFile=true
fi

# call the bccpipeline executable
./bccpipeline ${startTime} ${stopTime} ${parameterFile} ${frameCacheFile} \
         ${resultsDirectory} 2>&1
#          ${resultsDirectory} >>${logStatusFile} 2>&1

# record bccpipeline exit status
exitStatus=$?

# remove any temporary frame cache file
if [ "${temporaryFrameCacheFile:-false}" = "true" ]; then
  rm -f ${frameCacheFile}
fi

# return bccpipeline exit status to calling function
exit ${exitStatus}
