function bccpipeline(startTime, stopTime, parameterFile, frameCacheFile, ...
                   resultsDirectory)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                             start analysis timer                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% start time of analysis
analysisTimer = clock;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                 write header                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report analysis name and time
fprintf(1, "BCCPipeline analysis\n");
fprintf(1, "Run by %s on %s at %s\n", ...
        getenv("USER"), datestr(clock, 29), datestr(clock, 13));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                         parse command line arguments                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
fprintf(1, "parsing command line arguments\n");

% verify correct number of input arguments
error(nargchk(2, 5, nargin));

% apply default arguments
if (nargin < 3) || isempty(parameterFile),
  parameterFile = "parameters.txt";
end
if (nargin < 4) || isempty(frameCacheFile),
  frameCacheFile = "framecache.txt";
end
if (nargin < 5) || isempty(resultsDirectory),
  resultsDirectory = ".";
end

% ensure numeric start and stop times
if isstr(startTime),
  startTime = str2num(startTime);
end
if isstr(stopTime),
  stopTime = str2num(stopTime);
end

% truncate to integer start and stop times
startTime = floor(startTime);
stopTime = floor(stopTime);

% handle duration argument in place of stop time
if stopTime < startTime,
  stopTime = startTime + stopTime;
end

% string start and stop times
startTimeString = int2str(startTime);
stopTimeString = int2str(stopTime);
segmentString = [startTimeString "-" stopTimeString];

% report command line arguments
fprintf(1, "  startTime:               %#09d\n", startTime);
fprintf(1, "  stopTime:                %#09d\n", stopTime);
fprintf(1, "  parameterFile:           %s\n", parameterFile);
fprintf(1, "  frameCacheFile:          %s\n", frameCacheFile);
fprintf(1, "  resultsDirectory:        %s\n", resultsDirectory);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                               read parameters                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% reoprt status
fprintf(1, "reading parameter file\n");

% read parameters
debugLevel = 1;
parameters = bccparameters(parameterFile, debugLevel);

% extract parameters from parameters structure
channelNames = parameters.channelNames;
frameTypes = parameters.frameTypes;
timeShifts = parameters.timeShifts;
injectionNames = parameters.injectionNames;
injectionTypes = parameters.injectionTypes;
injectionFactors = parameters.injectionFactors;
injectionTimeShifts = parameters.injectionTimeShifts;
#injectionMethod = parameters.injectionMethod;
sampleFrequency = parameters.sampleFrequency;
highPassCutoff = parameters.highPassCutoff;
lowPassCutoff = parameters.lowPassCutoff;
blockDuration = parameters.blockDuration;
extraBlockOverlap = parameters.extraBlockOverlap;
randomSeed = parameters.randomSeed;
debugLevel = parameters.debugLevel;
numberOfChannels = parameters.numberOfChannels;
injectionChannels = parameters.injectionChannels;
maximumTriggers = parameters.maximumTriggers;
snrThreshold = parameters.snrThreshold;

%% whitening parameters
whitening.duration = parameters.whiteningDuration;
whitening.overlap = parameters.whiteningOverlap;
whitening.double = parameters.doubleWhiteningFlag;
whitening.windowType = parameters.whiteningWindowType;

%% chirplet chain parameters
chirplet.tileDuration = parameters.tileDuration;
chirplet.tileOverlap = parameters.tileOverlap;
chirplet.duration = parameters.chirpletDuration;
chirplet.chirpingRateLimit1 = parameters.chirpingRateLimit1;
chirplet.chirpingRateLimit2 = parameters.chirpingRateLimit2;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                       initialize random number generators                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% set random number generator seeds
rand("state", randomSeed);
randn("state", randomSeed);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                           create results directory                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
fprintf(1, "creating results directory\n", resultsDirectory);

% create output directory
unix(["mkdir -p " resultsDirectory]);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            read frame file cache                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
fprintf(1, "reading frame cache\n");

% read frame file cache
frameCache = loadframecache(frameCacheFile);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                              partition segment                               %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
fprintf(1, "partioning segment\n");

% segment duration
segmentDuration = stopTime - startTime;

% force time shift alignment with sample interval
timeShifts = round(timeShifts * sampleFrequency) / sampleFrequency;

% maximum and minimum time shifts
maximumTimeShift = max(timeShifts);
minimumTimeShift = min(timeShifts);

% total segment livetime loss due to time shifts
timeShiftLoss = maximumTimeShift - minimumTimeShift;

% if segment duration is less than block duration
if (segmentDuration - timeShiftLoss) < blockDuration,

  % write error to log file
  error("bccpipeline.m: segment to short to analyze");

% otherwise, continue
end

% number of blocks in segments
numberOfBlocks = ceil((segmentDuration - timeShiftLoss - ...
                       extraBlockOverlap) / ...
                      (blockDuration - extraBlockOverlap));

% if only one block,
if numberOfBlocks == 1,

  % ignore block overlap
  blockOverlap = 0;

% otherwise,
else

  % overlap in seconds between blocks
  blockOverlap = (segmentDuration - timeShiftLoss - ...
                  blockDuration * numberOfBlocks) ./ ...
                 (1 - numberOfBlocks);
% continue
end

% report segment information
fprintf(1, "  segment duration:    %9.2f seconds\n", segmentDuration);
fprintf(1, "  time shift loss:     %9.2f seconds\n", timeShiftLoss);
fprintf(1, "  block duration:      %9.2f seconds\n", blockDuration);
fprintf(1, "  block overlap:       %9.2f seconds\n", blockOverlap);
fprintf(1, "  number of blocks:    %9u blocks\n", numberOfBlocks);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                               analyzed blocks                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
fprintf(1, "identifying analyzed blocks\n");

% name of livetime file
livetimeFile = [resultsDirectory "/livetime_" segmentString ".txt"];

% read list of previously analyzed blocks
if exist(livetimeFile) == 2,
  analyzedBlocks = textread(livetimeFile, "%u");
else
  analyzedBlocks = [];
end


% list of blocks to analyze
blocksToAnalyze = setdiff(1 : numberOfBlocks, analyzedBlocks);

% report number of previously analyzed blocks
fprintf(1, "  analyzed blocks:     %9u blocks\n", length(analyzedBlocks));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                              begin block loop                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% initialize error flag
errorFlag = 0;

% begin loop over blocks
for blockNumber = blocksToAnalyze,

  % start time of block analysis
  blockTimer = clock;

  % gps start time of block
  blockStartTime = startTime + maximumTimeShift - whitening.duration/2.0 + ...
                   (blockNumber - 1) * (blockDuration - blockOverlap);

  % force start time alignment with data samples
  blockStartTime = floor(blockStartTime * sampleFrequency) / ...
                   sampleFrequency;

  % gps stop time of block
  blockStopTime = blockStartTime + blockDuration;

  % report status
  fprintf(1, "analyzing block %u of %u at %#019.9f\n", ...
          blockNumber, numberOfBlocks, blockStartTime);

  % reset channel names and number of channels
  channelNames = parameters.channelNames;
  numberOfChannels = parameters.numberOfChannels;

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                            read detector data                              %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % report status
  fprintf(1, "  reading detector data\n");

  % read detector data
  [data, sampleFrequencies] = ...
      qreaddata(frameCache, channelNames, frameTypes, blockStartTime, ...
                blockStopTime, timeShifts, debugLevel);

  % test for missing detector data
  if any(sampleFrequencies == 0),
    fprintf(1, "bccpipeline.m: error reading detector data\n");
    errorFlag = 1;
    continue;
  end

  % standardize channel names
  for channelNumber=1:length(channelNames)
    channelNames{channelNumber} = strrep(channelNames{channelNumber}, ";", ":");
  end

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                           read injection data                              %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % if any injections are requested
  if any(injectionChannels),

    % report status
    fprintf(1, "  reading/producing injection data\n");

# if strcmp(injectionMethod,"MDC")

    % read injection data
      [injectionData, injectionSampleFrequencies] = ...
          qreaddata(frameCache, injectionNames, injectionTypes, blockStartTime, ...
                    blockStopTime, injectionTimeShifts, debugLevel);

    % test for missing injection data
      if any(injectionSampleFrequencies(injectionChannels) == 0),
	fprintf(1, "bccpipeline.m: error reading injection data\n");
	errorFlag = 1;
	continue;
      endif

    % test for inconsistent injection data
	if any(injectionSampleFrequencies(injectionChannels) ~= ...
               sampleFrequencies(injectionChannels)),
	  error("bccpipeline.m: inconsistent detector and injection data");
	endif
	
# elseif strcmp(injectionMethod,"ONLINE")
	
# injectionData = bccinject(injectionNames,injectionTypes,blockStartTime, ...
#  blockStopTime,sampleFrequencies(injectionChannels))

# else
	
# error("bccpipeline.m: unknown injection method %s",injectionMethod);

# endif

    % add injection data to detector data
      for channelNumber = injectionChannels,
	data{channelNumber} = data{channelNumber} + ...
	    injectionFactors(channelNumber) * injectionData{channelNumber};
	endfor

    % free injection memory
    clear injectionData;
    clear injectionSampleFrequencies;

  % end test for requested injections
  end

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                     resample and highpass filter data                      %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % report status
  fprintf(1, "  resampling and highpass filtering data\n");

  % resample data
  data = bccresamplehighpass(data, sampleFrequencies, sampleFrequency, highPassCutoff);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              whitening data                                %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % report status
  fprintf(1, "  whitening data\n");


  % whitening data
  [data, whitening] = bccwhitening(data,whitening,chirplet.tileDuration,sampleFrequency);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              best chirplet chain                           %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % report status
  fprintf(1, "  trigger generation\n");

  % generate triggers
  triggers=bccsearch(data,chirplet,whitening,sampleFrequency);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              downselect triggers                           %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % report status
  fprintf(1, "  downselect triggers\n");

  % downselect triggers
  triggers=bccselect(triggers,channelNames,blockStartTime,snrThreshold,maximumTriggers);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              write triggers                                %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  %% write status to log file
  fprintf(1, "  writing triggers\n");

  %% write triggers
  for channelNumber = 1 : numberOfChannels,
    
    triggerFile = sprintf("%s/%s_%s.txt",resultsDirectory,channelNames{channelNumber},segmentString);
    triggerFileID=fopen(triggerFile, "at");

    %% test for error
    if (triggerFileID == -1),
      error("bccpipeline.m: cannot open file for writing");
    endif;

    fieldNames = fieldnames(triggers{channelNumber});
    numberOfFields = length(fieldNames);
    numberOfTriggers=length(triggers{channelNumber}.time);
    
    fprintf(triggerFileID,"%% ");
    for fieldNumber = 1 : numberOfFields,
      fprintf(triggerFileID,"%s ",fieldNames(fieldNumber));
    endfor;
    fprintf(triggerFileID,"\n");
    
    for triggerNumber=1:numberOfTriggers
      
      for fieldNumber = 1 : numberOfFields,
 	fprintf(triggerFileID,"%#010.4f ",triggers{channelNumber}.(fieldNames(fieldNumber))(triggerNumber));
      endfor;
      fprintf(triggerFileID,"\n");
      
    endfor;
    
    %% close file
    fclose(triggerFileID);
    
  endfor;

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              write livetime                                %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % write status to log file
  fprintf(1, "  writing livetime\n");

  % open livetime file for appending
  livetimeFileFID = fopen(livetimeFile, "at");

  % test for error
  if livetimeFileFID == -1,
    error("bccpipeline.m: cannot open livetime file for writing");
  end

  % write block livetime to livetime file
  fprintf(livetimeFileFID, "%#04u %#019.9f %#019.9f %#019.9f %#019.9f %#019.9f\n", blockNumber, blockStartTime, blockStopTime, whitening.duration, blockStartTime+whitening.duration/2.0, blockStopTime-whitening.duration/2.0);

  % close livetime file
  fclose(livetimeFileFID);

  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  %                              end block loop                                %
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % report status
  fprintf(1, "  block complete\n");

  % report total elapsed time
  fprintf(1, "    elapsed time:          %5.0f seconds\n", ...
          etime(clock, blockTimer));

% end loop over blocks
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                 write footer                                 %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% report status
fprintf(1, "segment complete\n");

% report total elapsed time
fprintf(1, "  elapsed time:            %5.0f seconds\n", ...
	etime(clock, analysisTimer));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                    return                                    %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% exit with error status
if errorFlag,
  error("bccpipeline.m: analysis had errors.");
else
  return;
end
