function parameters = bccparameters(parameterFile, debugFlag)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     process command line arguments                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% verify correct number of input arguments
error(nargchk(0, 2, nargin));

% apply default arguments
if (nargin < 1) || isempty(parameterFile),
  parameterFile = "./parameters.txt";
end
if (nargin < 2) || isempty(debugFlag),
  debugFlag = 0;
end

% validate command line arguments
if ~exist(parameterFile, "file"),
  error("could not find parameter file:\n%s\n", parameterFile);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            initialize parameters                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% initialize parameters
channelNames = [];
frameTypes = [];
timeShifts = [];
injectionNames = [];
injectionTypes = [];
injectionFactors = [];
injectionTimeShifts = [];
sampleFrequency = [];
highPassCutoff = [];
lowPassCutoff = [];
whiteningDuration = [];
whiteningOverlap = [];
whiteningWindowType = [];
doubleWhiteningFlag = [];
blockDuration = [];
extraBlockOverlap = [];
tileDuration = [];
tileOverlap = [];
chirpletDuration = [];
chirpingRateLimit1 = [];
chirpingRateLimit2 = [];
randomSeed = [];
debugLevel = [];
maximumTriggers = [];

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                               read parameters                                %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% open parameter file for reading
parameterFileID = fopen(parameterFile, "r");

% begin loop over parameter file
while ~feof(parameterFileID),

  % read one line from parameter file
  parameterLine = fgetl(parameterFileID);

  % remove any comments
  commentIndices = min([findstr(parameterLine, "#") ...
                        findstr(parameterLine, "%") ...
                        findstr(parameterLine, "//")]);
  if ~isempty(commentIndices),
    parameterLine = parameterLine(1 : (commentIndices(1) - 1));
  end

  % remove leading and trailing blanks
  parameterLine = fliplr(deblank(fliplr(deblank(parameterLine))));

  % if empty line, skip to the next line
  if isempty(parameterLine),
    continue;
  end

  % locate field separator
  colonIndex = strfind(parameterLine, ":");

  % if field separator not located, report syntax error
  if isempty(colonIndex),
    error("syntax error processing parameter file:\n%s\n", ...
          parameterLine);
  end

  % parse parameter line
  colonIndex = colonIndex(1);
  parameterName = parameterLine(1 : colonIndex - 1);
  parameterValue = parameterLine((colonIndex + 1) : end);
  parameterName = fliplr(deblank(fliplr(deblank(parameterName))));
  parameterValue = fliplr(deblank(fliplr(deblank(parameterValue))));

  % report parameter settings
  if debugFlag > 0,
    fprintf(1, "  %-25s%s\n", [parameterName ":"], parameterValue);
  end

  % assign parameters based on name
  switch parameterName,

    % assign parameters
    case "channelNames",
      channelNames = eval(parameterValue);
    case "frameTypes",
      frameTypes = eval(parameterValue);
    case "timeShifts",
      timeShifts = eval(parameterValue);
    case "injectionNames",
      injectionNames = eval(parameterValue);
    case "injectionTypes",
      injectionTypes = eval(parameterValue);
    case "injectionFactors",
      injectionFactors = eval(parameterValue);
    case "injectionTimeShifts",
      injectionTimeShifts = eval(parameterValue);
    case "sampleFrequency",
      sampleFrequency = eval(parameterValue);
    case "highPassCutoff",
      highPassCutoff = eval(parameterValue);
    case "lowPassCutoff",
      lowPassCutoff = eval(parameterValue);
    case "whiteningDuration",
      whiteningDuration = eval(parameterValue);
    case "whiteningOverlap",
      whiteningOverlap = eval(parameterValue);
    case "whiteningWindowType",
      whiteningWindowType = eval(parameterValue);
    case "doubleWhiteningFlag",
      doubleWhiteningFlag = eval(parameterValue);
    case "blockDuration",
      blockDuration = eval(parameterValue);
    case "extraBlockOverlap",
      extraBlockOverlap = eval(parameterValue);
    case "tileOverlap",
      tileOverlap = eval(parameterValue);
    case "tileDuration",
      tileDuration = eval(parameterValue);
    case "chirpletDuration",
      chirpletDuration = eval(parameterValue);
    case "chirpingRateLimit1",
      chirpingRateLimit1 = eval(parameterValue);
    case "chirpingRateLimit2",
      chirpingRateLimit2 = eval(parameterValue);
    case "randomSeed",
      randomSeed = eval(parameterValue);
    case "debugLevel",
      debugLevel = eval(parameterValue);
    case "maximumTriggers",
      maximumTriggers = eval(parameterValue);
    case "snrThreshold",
      snrThreshold = eval(parameterValue);

    % handle unknown parameters
    otherwise,
     error("unknown parameter %s\n", parameterName);

  % end assign parameters based on name
  end

% end loop over parameter file entries
end

% close parameter file
fclose(parameterFileID);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          handle missing parameters                           %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% test for unspecified parameters
if isempty(channelNames),
  error("channelNames not specified");
end
if isempty(frameTypes),
  error("frameTypes not specified");
end
if isempty(sampleFrequency),
  error("sampleFrequency not specified");
end
if isempty(blockDuration),
  error("blockDuration not specified");
end
if isempty(whiteningDuration),
  error("whiteningDuration not specified");
end
if isempty(tileOverlap),
  error("tileOverlap not specified");
end
if isempty(tileDuration),
  error("tileDuration not specified");
end
if isempty(chirpletDuration),
  error("chirpletDuration not specified");
end
if isempty(chirpingRateLimit1),
  error("chirpingRateLimit1 not specified");
end
if isempty(chirpingRateLimit2),
  error("chirpingRateLimit2 not specified");
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            construct cell arrays                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% force cell arrays and vectors
if ~iscell(channelNames),
  channelNames = mat2cell(channelNames, size(channelNames, 1), ...
                          size(channelNames, 2));
end
if ~iscell(frameTypes),
  frameTypes = mat2cell(frameTypes, size(frameTypes, 1), ...
                        size(frameTypes, 2));
end
if ~iscell(injectionNames) && ~isempty(injectionNames),
  injectionNames = mat2cell(injectionNames, size(injectionNames, 1), ...
                            size(injectionNames, 2));
end
if ~iscell(injectionTypes) && ~isempty(injectionTypes),
  injectionTypes = mat2cell(injectionTypes, size(injectionTypes, 1), ...
                            size(injectionTypes, 2));
end
if iscell(timeShifts) && ~isempty(timeShifts),
  timeShifts = [timeShifts{:}];
end
if iscell(injectionFactors) && ~isempty(injectionFactors),
  injectionFactors = [injectionFactors{:}];
end
if iscell(injectionTimeShifts) && ~isempty(injectionTimeShifts),
  injectionTimeShifts = [injectionTimeShifts{:}];
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                              default parameters                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% number of channels
numberOfChannels = length(channelNames);

% apply default values
if isempty(timeShifts),
  timeShifts = zeros(1, numberOfChannels);
end
if isempty(injectionNames),
  injectionNames = cell(1, numberOfChannels);
  [injectionNames{:}] = deal("NONE");
end
if isempty(injectionTypes),
  injectionTypes = cell(1, numberOfChannels);
  [injectionTypes{:}] = deal("NONE");
end
if isempty(injectionFactors),
  injectionFactors = zeros(1, numberOfChannels);
end
if isempty(injectionTimeShifts),
  injectionTimeShifts = zeros(1, numberOfChannels);
end
if isempty(highPassCutoff),
  highPassCutoff = 0.0;
end
if isempty(whiteningOverlap),
  whiteningOverlap = 0.0;
end
if isempty(whiteningWindowType),
  whiteningWindowType = "hanning";
end
if isempty(doubleWhiteningFlag),
  doubleWhiteningFlag = true;
end
if isempty(randomSeed),
  randomSeed = sum(1e6 * clock);
end
if isempty(debugLevel),
  debugLevel = 1;
end

% channels with requested injections
injectionChannels = find(~strcmp(upper(injectionNames), "NONE") & ...
                         ~strcmp(upper(injectionTypes), "NONE") & ...
                         (injectionFactors ~= 0));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                              reshape parameters                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% force row arrays and vectors
channelNames = channelNames(:);
frameTypes = frameTypes(:);
timeShifts = timeShifts(:);
injectionNames = injectionNames(:);
injectionTypes = injectionTypes(:);
injectionFactors = injectionFactors(:);
injectionTimeShifts = injectionTimeShifts(:);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                             validate parameters                              %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% validate number of frame types
if length(frameTypes) ~= numberOfChannels,
  error("number of frameTypes and channelNames are inconsistent");
end

% validate number of time shifts
if (length(timeShifts) ~= numberOfChannels) && ~isempty(timeShifts),
  error("number of timeShifts and channelNames are inconsistent");
end

% validate number of injection names
if (length(injectionNames) ~= numberOfChannels) && ~isempty(injectionNames),
  error("number of injectionNames and channelNames are inconsistent");
end

% validate number of injection types
if (length(injectionTypes) ~= numberOfChannels) && ~isempty(injectionTypes),
  error("number of injectionTypes and channelNames are inconsistent");
end

% validate number of injection factors
if (length(injectionFactors) ~= numberOfChannels) && ~isempty(injectionFactors),
  error("number of injectionFactors and channelNames are inconsistent");
end

% validate number of injection time shifts
if (length(injectionTimeShifts) ~= numberOfChannels) && ...
      ~isempty(injectionTimeShifts),
  error("number of injectionTimeShifts and channelNames are inconsistent");
end

% validate analysis mode
sites = unique(regexprep(channelNames, ".:.*$", ""));
numberOfSites = length(sites);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                        construct parameters structure                        %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% insert parameters into parameters structure
parameters.channelNames = channelNames;
parameters.frameTypes = frameTypes;
parameters.timeShifts = timeShifts;
parameters.injectionNames = injectionNames;
parameters.injectionTypes = injectionTypes;
parameters.injectionFactors = injectionFactors;
parameters.injectionTimeShifts = injectionTimeShifts;
parameters.sampleFrequency = sampleFrequency;
parameters.highPassCutoff = highPassCutoff;
parameters.lowPassCutoff = lowPassCutoff;
parameters.whiteningDuration = whiteningDuration;
parameters.blockDuration = blockDuration;
parameters.extraBlockOverlap = extraBlockOverlap;
parameters.whiteningDuration = whiteningDuration;
parameters.whiteningOverlap = whiteningOverlap;
parameters.whiteningWindowType = whiteningWindowType;
parameters.doubleWhiteningFlag = doubleWhiteningFlag;
parameters.tileDuration = tileDuration;
parameters.tileOverlap = tileOverlap;
parameters.chirpletDuration = chirpletDuration;
parameters.chirpingRateLimit1 = chirpingRateLimit1;
parameters.chirpingRateLimit2 = chirpingRateLimit2;
parameters.randomSeed = randomSeed;
parameters.debugLevel = debugLevel;
parameters.numberOfChannels = numberOfChannels;
parameters.injectionChannels = injectionChannels;
parameters.maximumTriggers = maximumTriggers;
parameters.snrThreshold = snrThreshold;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          return to calling function                          %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% return to calling function
return;
