/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "resample.h"

#ifdef  __cplusplus
extern "C" {
#endif

static double bessel_I0(double x)
{
    int k;
    double w, t, y;
    const double a[65] = {
        8.5246820682016865877e-11, 2.5966600546497407288e-9, 
        7.9689994568640180274e-8, 1.9906710409667748239e-6, 
        4.0312469446528002532e-5, 6.4499871606224265421e-4, 
        0.0079012345761930579108, 0.071111111109207045212, 
        0.444444444444724909, 1.7777777777777532045, 
        4.0000000000000011182, 3.99999999999999998, 
        1.0000000000000000001, 
        1.1520919130377195927e-10, 2.2287613013610985225e-9, 
        8.1903951930694585113e-8, 1.9821560631611544984e-6, 
        4.0335461940910133184e-5, 6.4495330974432203401e-4, 
        0.0079013012611467520626, 0.071111038160875566622, 
        0.44444450319062699316, 1.7777777439146450067, 
        4.0000000132337935071, 3.9999999968569015366, 
        1.0000000003426703174, 
        1.5476870780515238488e-10, 1.2685004214732975355e-9, 
        9.2776861851114223267e-8, 1.9063070109379044378e-6, 
        4.0698004389917945832e-5, 6.4370447244298070713e-4, 
        0.0079044749458444976958, 0.071105052411749363882, 
        0.44445280640924755082, 1.7777694934432109713, 
        4.0000055808824003386, 3.9999977081165740932, 
        1.0000004333949319118, 
        2.0675200625006793075e-10, -6.1689554705125681442e-10, 
        1.2436765915401571654e-7, 1.5830429403520613423e-6, 
        4.2947227560776583326e-5, 6.3249861665073441312e-4, 
        0.0079454472840953930811, 0.070994327785661860575, 
        0.44467219586283000332, 1.7774588182255374745, 
        4.0003038986252717972, 3.9998233869142057195, 
        1.0000472932961288324, 
        2.7475684794982708655e-10, -3.8991472076521332023e-9, 
        1.9730170483976049388e-7, 5.9651531561967674521e-7, 
        5.1992971474748995357e-5, 5.7327338675433770752e-4, 
        0.0082293143836530412024, 0.069990934858728039037, 
        0.44726764292723985087, 1.7726685170014087784, 
        4.0062907863712704432, 3.9952750700487845355, 
        1.0016354346654179322
    };
    const double b[70] = {
        6.7852367144945531383e-8, 4.6266061382821826854e-7, 
        6.9703135812354071774e-6, 7.6637663462953234134e-5, 
        7.9113515222612691636e-4, 0.0073401204731103808981, 
        0.060677114958668837046, 0.43994941411651569622, 
        2.7420017097661750609, 14.289661921740860534, 
        59.820609640320710779, 188.78998681199150629, 
        399.8731367825601118, 427.56411572180478514, 
        1.8042097874891098754e-7, 1.2277164312044637357e-6, 
        1.8484393221474274861e-5, 2.0293995900091309208e-4, 
        0.0020918539850246207459, 0.019375315654033949297, 
        0.15985869016767185908, 1.1565260527420641724, 
        7.1896341224206072113, 37.354773811947484532, 
        155.80993164266268457, 489.5211371158540918, 
        1030.9147225169564806, 1093.5883545113746958, 
        4.8017305613187493564e-7, 3.261317843912380074e-6, 
        4.9073137508166159639e-5, 5.3806506676487583755e-4, 
        0.0055387918291051866561, 0.051223717488786549025, 
        0.42190298621367914765, 3.0463625987357355872, 
        18.895299447327733204, 97.915189029455461554, 
        407.13940115493494659, 1274.3088990480582632, 
        2670.9883037012547506, 2815.7166284662544712, 
        1.2789926338424623394e-6, 8.6718263067604918916e-6, 
        1.3041508821299929489e-4, 0.001428224737372747892, 
        0.014684070635768789378, 0.13561403190404185755, 
        1.1152592585977393953, 8.0387088559465389038, 
        49.761318895895479206, 257.2684232313529138, 
        1066.8543146269566231, 3328.3874581009636362, 
        6948.8586598121634874, 7288.4893398212481055, 
        3.409350368197032893e-6, 2.3079025203103376076e-5, 
        3.4691373283901830239e-4, 0.003794994977222908545, 
        0.038974209677945602145, 0.3594948380414878371, 
        2.9522878893539528226, 21.246564609514287056, 
        131.28727387146173141, 677.38107093296675421, 
        2802.3724744545046518, 8718.5731420798254081, 
        18141.348781638832286, 18948.925349296308859
    };
    const double c[45] = {
        2.5568678676452702768e-15, 3.0393953792305924324e-14, 
        6.3343751991094840009e-13, 1.5041298011833009649e-11, 
        4.4569436918556541414e-10, 1.746393051427167951e-8, 
        1.0059224011079852317e-6, 1.0729838945088577089e-4, 
        0.05150322693642527738, 
        5.2527963991711562216e-15, 7.202118481421005641e-15, 
        7.2561421229904797156e-13, 1.482312146673104251e-11, 
        4.4602670450376245434e-10, 1.7463600061788679671e-8, 
        1.005922609132234756e-6, 1.0729838937545111487e-4, 
        0.051503226936437300716, 
        1.3365917359358069908e-14, -1.2932643065888544835e-13, 
        1.7450199447905602915e-12, 1.0419051209056979788e-11, 
        4.58047881980598326e-10, 1.7442405450073548966e-8, 
        1.0059461453281292278e-6, 1.0729837434500161228e-4, 
        0.051503226940658446941, 
        5.3771611477352308649e-14, -1.1396193006413731702e-12, 
        1.2858641335221653409e-11, -5.9802086004570057703e-11, 
        7.3666894305929510222e-10, 1.6731837150730356448e-8, 
        1.0070831435812128922e-6, 1.0729733111203704813e-4, 
        0.051503227360726294675, 
        3.7819492084858931093e-14, -4.8600496888588034879e-13, 
        1.6898350504817224909e-12, 4.5884624327524255865e-11, 
        1.2521615963377513729e-10, 1.8959658437754727957e-8, 
        1.0020716710561353622e-6, 1.073037119856927559e-4, 
        0.05150322383300230775
    };

    w = fabs(x);

    if (w < 8.5) {
      t = w * w * 0.0625;
      k = 13 * ((int) t);
      y = (((((((((((a[k] * t + a[k + 1]) * t +
		    a[k + 2]) * t + a[k + 3]) * t + a[k + 4]) * t +
		 a[k + 5]) * t + a[k + 6]) * t + a[k + 7]) * t +
	      a[k + 8]) * t + a[k + 9]) * t + a[k + 10]) * t +
	   a[k + 11]) * t + a[k + 12];
    } else if (w < 12.5) {
      k = (int) w;
      t = w - k;
      k = 14 * (k - 8);
      y = ((((((((((((b[k] * t + b[k + 1]) * t +
		     b[k + 2]) * t + b[k + 3]) * t + b[k + 4]) * t +
                  b[k + 5]) * t + b[k + 6]) * t + b[k + 7]) * t +
	       b[k + 8]) * t + b[k + 9]) * t + b[k + 10]) * t +
	    b[k + 11]) * t + b[k + 12]) * t + b[k + 13];
    } else {
      t = 60 / w;
      k = 9 * ((int) t);
      y = ((((((((c[k] * t + c[k + 1]) * t +
		 c[k + 2]) * t + c[k + 3]) * t + c[k + 4]) * t +
	      c[k + 5]) * t + c[k + 6]) * t + c[k + 7]) * t +
	   c[k + 8]) * sqrt(t) * exp(w);
    }

    return y;
}

static double get_beta(double A)
{
  /* expression taken from Oppenheim and Schafer, Discrete-Time Signal Proc. */
  /* Prentice Hall. Chap 7, Eq. (7.62) p 474 */

  if (A>50)
    return 0.1102*(A-8.7);
  else if ((A>=21)&(A<=50))
    return 0.5842*pow(A-21.0,0.4)+0.07886*(A-21.0);
  else
    return 0.0;
}

static void multiply_by_Kaiser(vector *h, double beta)
{
  int n;
  double I0_beta,x,w,alpha;

  I0_beta = 1.0/bessel_I0(beta);
  alpha = 1.0/(double)(h->length-1);
  for (n=1; n<h->length-1; n++)
    {
      x = alpha * (double) n;
      x = beta*sqrt(1.0-x*x);
      w = bessel_I0(x)*I0_beta;
      h->data[n] *= w;
    }
  h->data[h->length-1] *= I0_beta;
}

static int window_size(double A, double df, int I, double fu)
{
  int M, n;
  double u, c, sum, err, s, e;

  /* expression taken from Oppenheim and Schafer, Discrete-Time Signal Proc. */
  /* Prentice Hall. Chap 7, Eq. (7.63) p 476 */
  M=(int) ceil(1.374875*(A-8.0)/df);

  u=2.0*M_PI*fu;
  c=2.0*I/M_PI;
  sum=2.0*I*fu;
  for (n=1; n<M ; n++)
    sum+=c*sin(u*n)/(double) n;
  err=fabs(I-sum);

  /* adjust (increase) length until sum of the ideal filter is minimized */
  /* with final negative minimum */
  /* note: this is an estimate. better if we use the sum of the ideal filter times Kaiser win */
  s=sum + c*sin(u*M)/(double) M;
  e=I-s;
  while ((err>0)|((err<=0)&(e<0)))
    {
      sum=s;
      err=e;
      M++;
      s+=c*sin(u*M)/(double) M;
      e=I-s;
    }

  return M;
}

static void ideal_filter(vector *h, int I, double fl, double fu)
{
  int n;
  double u,l,c;
  
  u=2.0*M_PI*fu;
  l=2.0*M_PI*fl;
  c=I/M_PI;

  /* compute sinus card */
  h->data[0]=2.0*I*(fu-fl);
  for (n=1; n<h->length ; n++)
    h->data[n]=c*(sin(u*n)-sin(l*n))/(double) n;  
  
}

void smallest_block_size(bcc_status* status, int* out, const  resampling_params *p)
{
  int D, I, M, N;
  double f;
    
  CHECKSTATUSPTR(status);

  ASSERT(p,status,RSP_ENULL,RSP_MSGENULL);

  D=p->decimate;
  I=p->interpolate;
  f=p->upper_cutoff_f/(double) D;
  M=window_size(-20.0*p->log10_reject,p->roll_off_width/(double) D,I,f);
  N=ceil(2.0*M/(double) I);
  
  *out=ceil(N/(double) D)*D;

  RETURN(status);
}

void resample_init_buffer(bcc_status* status, resampling_buffers* b, const int raw_length, const resampling_params *p)
{
  char infostr[BCCInfoLength];
  vector *h=NULL;
  double beta,fu,fl;
  int D,I,M,N,P,m,k,t,n;

  CHECKSTATUSPTR(status);

  ASSERT(b,status,RSP_ENULL,RSP_MSGENULL); 
  ASSERT(p,status,RSP_ENULL,RSP_MSGENULL); 

  D=p->decimate;
  I=p->interpolate;

  b->proc=NULL;
  TRY(create_vector(status,&b->proc,(I*raw_length)/D),status);

  fu=p->upper_cutoff_f/(double) D; // upper f cut-off, converted to interpolated stream rate

  ASSERT(fu<= 1.0/(2.0 *(double) max_i(D,I)),status,RSP_EREGF,RSP_MSGEREGF);

  fl=p->lower_cutoff_f/(double) D; // lower f cut-off, converted to interpolated stream rate
  
  /* compute the ideal bandpass filter */
  M=window_size(-20.0*p->log10_reject,p->roll_off_width/(double) D,I,fu); // window length
  TRY(create_vector(status,&h,(int) M),status);
  ideal_filter(h,I,fl,fu);
  sprintf(infostr,"Resampling: size of the impulse response of interp. filter is %d",M);
  bcc_log(status,infostr);
      
  /* windowing with Kaiser window */
  beta=get_beta(-20.*p->log10_reject); // parameter of Kaiser window
  multiply_by_Kaiser(h,beta);
  
  /* init buffers for overlap and add*/
  P=div(M,D).quot;
  ASSERT((P<I*raw_length/(2.0*D)),status,RSP_EWINZ,RSP_MSGEWINZ);
  b->left_over=NULL;
  TRY(create_vector(status,&b->left_over,P),status);
  b->after=NULL;
  TRY(create_vector(status,&b->after,P),status);
  b->before=NULL;
  TRY(create_vector(status,&b->before,P),status);
/*   sprintf(infostr,"Resampling: size of overlap and add buffers is %d",P); */
/*   bcc_log(status,infostr); */
  
  /* compute polyphase matrix */
  N=div(M,I).quot;
  ASSERT((N<raw_length),status,RSP_EWINZ,RSP_MSGEWINZ);
  b->interp=NULL;
  TRY(create_matrix(status,&b->interp,2*N+1,I),status);
  for (m=0; m<I; m++) 
    {
      t=(m*D)%I;
      for (n=-N; n<=N; n++)
	{
	  k=abs(n*I+t);
	  if (k<h->length)
	    b->interp->data2[m][n+N]=h->data[k];
	  else
	    b->interp->data2[m][n+N]=0.0;
	}
    }

  
/*   for (m=0; m<I; m++) */
/*     { */
/*       for (n=-N; n<=N; n++) */
/* 	printf("%g ",b->interp->data2[m][n+N]); */
/*       printf("\n"); */
/*     } */

  TRY(destroy_vector(status,&h),status);
  
  RETURN(status);  
}

void resample_clear_buffer(bcc_status* status, resampling_buffers* b)
{
  CHECKSTATUSPTR(status);
  
  ASSERT(b,status,RSP_ENULL,RSP_MSGENULL);

  TRY(reset_vector(status,b->left_over),status);
  TRY(reset_vector(status,b->after),status);
  TRY(reset_vector(status,b->before),status);
  
  RETURN(status);  
}

void resample_purge_buffer(bcc_status* status, resampling_buffers* b)
{

  CHECKSTATUSPTR(status);
  
  ASSERT(b,status,RSP_ENULL,RSP_MSGENULL);

  TRY(destroy_vector(status,&b->left_over),status);
  TRY(destroy_vector(status,&b->after),status);
  TRY(destroy_vector(status,&b->before),status);
  TRY(destroy_matrix(status,&b->interp),status);
  
  RETURN(status);
}

void resample_one_block(bcc_status* status, vector *out, vector *before, vector *after, const vector *in, const int D, const int I, const matrix *interp)
{
  int N,n,m,mm,t;
  double accum;

  CHECKSTATUSPTR(status);

  ASSERT(out,status,RSP_ENULL,RSP_MSGENULL); 
  ASSERT(before,status,RSP_ENULL,RSP_MSGENULL); 
  ASSERT(after,status,RSP_ENULL,RSP_MSGENULL); 
  ASSERT(in,status,RSP_ENULL,RSP_MSGENULL); 
  ASSERT(interp,status,RSP_ENULL,RSP_MSGENULL); 
  
  /* check resampling parameters */
  ASSERT(out->length%I==0,status,RSP_EVECZ,RSP_MSGEVECZ); 
  ASSERT(in->length%D==0,status,RSP_EVECZ,RSP_MSGEVECZ);
  ASSERT(out->length==(in->length*I)/(double) D,status,RSP_EVECZ,RSP_MSGEVECZ); 

  N=(interp->rows-1)/2;

  /* filter block with polyphase: result inside current block */
  m=0;
  while (m<out->length)
    {
      t= div(D*m,I).quot;
      n=(N>t?t:N);
      mm=m%I;
      accum=0.0;
      while ((n>=-N)&(t-n<in->length))
	{
	  accum += interp->data2[mm][n+N]*in->data[t-n];
	  n--;
	}
      out->data[m]=accum;
      m++;
    }
  
  /* filter block with polyphase: result before current block */
  m=out->length-1;
  while (m>=out->length-before->length)
    {
      t= div(D*m,I).quot-in->length;
      n=t;
      mm=m%I;
      accum=0.0;
      while (n>=-N)
	{
	  accum += interp->data2[mm][n+N]*in->data[t-n];
	  n--;
	}
      before->data[m-(out->length-before->length)]=accum;
      m--;
    }

  /* filter block with polyphase: result after current block */
  m=out->length;
  while (m<out->length+after->length)
    {
      t= div(D*m,I).quot;
      n=N;
      mm=m%I;
      accum=0.0;
      while (t-n<in->length)
	{
	  accum += interp->data2[mm][n+N]*in->data[t-n];
	  n--;
	}
      after->data[m-out->length]=accum;
      m++;
    }

  RETURN(status);
}

void resample(bcc_status* status, time_series *out, resampling_buffers *b, const time_series *in, const resampling_params *p)
{
  int k,N;

  CHECKSTATUSPTR(status);

  ASSERT(in,status,RSP_ENULL,RSP_MSGENULL);
  ASSERT(out,status,RSP_ENULL,RSP_MSGENULL);
  ASSERT(b,status,RSP_ENULL,RSP_MSGENULL);
  ASSERT(b->left_over,status,RSP_ENULL,RSP_MSGENULL);
  ASSERT(p,status,RSP_ENULL,RSP_MSGENULL);

  N=b->left_over->length;

  /* construct output vector */
  TRY(reset_time_series(status,out),status);

  memcpy(out->vec->data,b->left_over->data,N*sizeof(double));

  memcpy(out->vec->data+N,b->after->data,N*sizeof(double));

  /* resample current input */
  TRY(reset_vector(status,b->proc),status);
  TRY(resample_one_block(status,b->proc,b->before,b->after,in->vec,p->decimate,p->interpolate,b->interp),status);

  for (k=0; k<N; k++)
    out->vec->data[k+N]+=b->proc->data[k];

  for (k=0; k<N; k++)
    out->vec->data[k]+=b->before->data[k];
  
  memcpy(out->vec->data+2*N,b->proc->data+N,(b->proc->length-2*N)*sizeof(double));

  /* save left_over */
  memcpy(b->left_over->data,b->proc->data+b->proc->length-N,N*sizeof(double));

  /* update timing characteristics */
  out->deltaT=(in->deltaT * p->decimate)/((double) p->interpolate);
  out->epoch=in->epoch - (double) N * out->deltaT;

  RETURN(status);
}

#ifdef  __cplusplus
}
#endif
